﻿/********************************************************************************//*!
 * @file  sample_cache_manage.c 
 * @brief  To use HWLib for the Cache Management example program. 
 * 
 * @details  Run the API category below.
 *           - Cache Management API
 *               - System Level Cache Management API
 *               - L1 Cache Management API
 *               - L2 Cache Management API
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/12/18  ---  #6456  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 * @date 2014/08/18  ---  -----  v14.0化 (誤植訂正、操作説明の表示を追加). 
 * @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
 * @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
 * @date 2018/07/02  ---  #25940  All in One ベアメタル・アプリケーション v17.1 への対応を実施.
 * @date 2019/01/18  ---  -----   All in One ベアメタル・アプリケーション v18.1 への対応を実施.
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes 
 ***********************************************************************************/
#include <stdio.h>
#include <string.h>
#if 0
#include <arm_neon.h>
#endif
#include "hwlib.h"
#include "socal/socal.h"
#include "socal/hps.h"
#include "alt_generalpurpose_io.h"
#include "alt_globaltmr.h"
#include "alt_clock_manager.h"
#include "alt_interrupt.h"
#include "alt_cache.h"

#include "sample_app_setting.h"
#include "util_time_measurement.h"
#include "util_interrupt_log.h"
#include "alt_printf.h"

/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/
void sample_cache_manage_init(void);
void sample_cache_manage_setting_gic(void);
void sample_cache_manage_callback(uint32_t icciar, void *context);
void sample_cache_manage_test_print_usage(void);
void sample_cache_manage_test_main(void);
int  sample_cache_manage_test_cmd(char* options);
// prototype (Common Functions) //

// Test Function //
int mul_f32_test_funciton(int mulNval);
void mul_f32a(float * __restrict a, float * __restrict b, float * __restrict p, int size);
int verify(float *a, float *b, int size, int place);
#if 0
void vector_mul_f32a(float * __restrict a, float * __restrict b, float * __restrict p);
void vector_mul_f32i(float *a, float *b, float *p);
#endif

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/
#define N (8192)

/* L2 Cache Interrupt Conditions */
#define ALT_CACHE_L2_ICOND_ALL		((ALT_CACHE_L2_INTERRUPT_DECERR)|(ALT_CACHE_L2_INTERRUPT_SLVERR)|\
									 (ALT_CACHE_L2_INTERRUPT_ERRRD)|(ALT_CACHE_L2_INTERRUPT_ERRRT)|\
									 (ALT_CACHE_L2_INTERRUPT_ERRWD)|(ALT_CACHE_L2_INTERRUPT_ERRWT)|\
									 (ALT_CACHE_L2_INTERRUPT_PARRD)|(ALT_CACHE_L2_INTERRUPT_PARRT)|\
									 (ALT_CACHE_L2_INTERRUPT_ECNTR))

/***********************************************************************************
 *  variables 
 ***********************************************************************************/
int data_i[N*8] __attribute__ ((aligned (ALT_CACHE_LINE_SIZE)));
float data_f[N*8] __attribute__ ((aligned (ALT_CACHE_LINE_SIZE)));
float out[N*8] __attribute__ ((aligned (ALT_CACHE_LINE_SIZE)));
float ref[N*8] __attribute__ ((aligned (ALT_CACHE_LINE_SIZE)));
uint32_t cache_all_en = ALT_E_TRUE;

/***********************************************************************************
 *  functions 
 ***********************************************************************************/

/********************************************************************************//*!
 * @fn  void sample_cache_manage_init(void) 
 * @brief  Initializing the sample program. 
 * 
 * @details  Call the initialization functions for the Cache management. 
 *           Other, do the following.
 *           -# Configure the Helio Board GPIO settings. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/12/18  ---  #6456  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_cache_manage_init(void)
{
	ALT_STATUS_CODE result_code;

#if 0	// main() 内で cpu0_init() により alt_cache_system_enable() 実施済み
	cpu0_init();
#endif
	/* Usage:  ALT_STATUS_CODE  alt_cache_system_enable (void);	*/
	printf("-- HWLib test : alt_cache_system_enable() --\n");
	result_code = alt_cache_system_enable();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_cache_system_enable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}

	cache_all_en = ALT_E_TRUE;

	// L2 Cache Combined Interrupt GIC Config
	sample_cache_manage_setting_gic();

	// L2 Cache Controller Interrupt Enable
	/* Usage:  ALT_STATUS_CODE  alt_cache_l2_int_enable (uint32_t interrupt);	*/
	/* Usage:  ALT_STATUS_CODE  alt_cache_l2_int_disable (uint32_t interrupt);	*/
	/* Usage:  ALT_STATUS_CODE  alt_cache_l2_int_status_clear (uint32_t interrupt);	*/
	// Disable All L2 Cache Interrupt
	printf("-- HWLib test : alt_cache_l2_int_disable() --\n");
	result_code = alt_cache_l2_int_disable((uint32_t)ALT_CACHE_L2_ICOND_ALL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_cache_l2_int_disable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	// Forced Clear All Interrupt Status
	printf("-- HWLib test : alt_cache_l2_int_status_clear() --\n");
	result_code = alt_cache_l2_int_status_clear((uint32_t)ALT_CACHE_L2_ICOND_ALL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_cache_l2_int_status_clear() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	// Enable All L2 Cache Interrupt
	printf("-- HWLib test : alt_cache_l2_int_enable() --\n");
	result_code = alt_cache_l2_int_enable((uint32_t)ALT_CACHE_L2_ICOND_ALL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_cache_l2_int_enable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  void sample_cache_manage_setting_gic(void) 
 * @brief  Setting of GIC (L2 Cache Combined interrupt).
 * 
 * @details    In order to receive a L2 Cache Combined interrupt setting of GIC.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/12/18  ---  #6456  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_cache_manage_setting_gic(void)
{
	ALT_STATUS_CODE result_code;
	uint32_t ui32temp = 0;
	ALT_INT_TRIGGER_t trigger_type = ALT_INT_TRIGGER_NA;

	// Print initial value (GIC: ALT_INT_INTERRUPT_L2_COMBINED_IRQ)
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_secure(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_enabled(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_pending(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_active(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_priority_get(ALT_INT_INTERRUPT_t int_id, uint32_t* priority);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_target_get(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t* target);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_get(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t* trigger_type);	*/
	/****************************************************************************************/
	printf("initial interrupt setting (GIC: ALT_INT_INTERRUPT_L2_COMBINED_IRQ)\n");
	printf("- alt_int_dist_is_secure() --> result=%d\n", (int)alt_int_dist_is_secure(ALT_INT_INTERRUPT_L2_COMBINED_IRQ));
	printf("- alt_int_dist_is_enabled() --> result=%d\n", (int)alt_int_dist_is_enabled(ALT_INT_INTERRUPT_L2_COMBINED_IRQ));
	printf("- alt_int_dist_is_pending() --> result=%d\n", (int)alt_int_dist_is_pending(ALT_INT_INTERRUPT_L2_COMBINED_IRQ));
	printf("- alt_int_dist_is_active() --> result=%d\n", (int)alt_int_dist_is_active(ALT_INT_INTERRUPT_L2_COMBINED_IRQ));
	printf("- alt_int_dist_priority_get() --> result=%d out=0x%08X\n", (int)alt_int_dist_priority_get(ALT_INT_INTERRUPT_L2_COMBINED_IRQ,&ui32temp), (int)ui32temp);
	printf("- alt_int_dist_target_get() --> result=%d out=0x%08X\n", (int)alt_int_dist_target_get(ALT_INT_INTERRUPT_L2_COMBINED_IRQ,&ui32temp), (int)ui32temp);
	printf("- alt_int_dist_trigger_get() --> result=%d out=%d\n", (int)alt_int_dist_trigger_get(ALT_INT_INTERRUPT_L2_COMBINED_IRQ,&trigger_type), (int)trigger_type);
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_isr_register(ALT_INT_INTERRUPT_t int_id, alt_int_callback_t callback, void* context);	*/
	/*  typedef void(* alt_int_callback_t)(uint32_t icciar, void *context) */
	printf("-- HWLib test : alt_int_isr_register() --\n");
	result_code = alt_int_isr_register(ALT_INT_INTERRUPT_L2_COMBINED_IRQ, sample_cache_manage_callback, NULL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_isr_register() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_target_set(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t target);	*/
	printf("-- HWLib test : alt_int_dist_target_set() --\n");
	result_code = alt_int_dist_target_set(ALT_INT_INTERRUPT_L2_COMBINED_IRQ, 1);		// arg2=targetCPU (Only Core#0)
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_target_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_set(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t trigger_type);	*/
	printf("-- HWLib test : alt_int_dist_trigger_set() --\n");
	result_code = alt_int_dist_trigger_set(ALT_INT_INTERRUPT_L2_COMBINED_IRQ, ALT_INT_TRIGGER_EDGE);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_trigger_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_priority_set(ALT_INT_INTERRUPT_t int_id, uint32_t priority);	*/
	printf("-- HWLib test : alt_int_dist_priority_set() --\n");
	result_code = alt_int_dist_priority_set(ALT_INT_INTERRUPT_L2_COMBINED_IRQ, 99);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_priority_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_enable(ALT_INT_INTERRUPT_t int_id);	*/
	printf("-- HWLib test : alt_int_dist_enable() --\n");
	result_code = alt_int_dist_enable(ALT_INT_INTERRUPT_L2_COMBINED_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_secure_enable(ALT_INT_INTERRUPT_t int_id);	*/
	printf("-- HWLib test : alt_int_dist_secure_enable() --\n");
	result_code = alt_int_dist_secure_enable(ALT_INT_INTERRUPT_L2_COMBINED_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  void sample_cache_manage_callback(uint32_t icciar, void *context) 
 * @brief  L2 Cache Combined Interrupt(ALT_INT_INTERRUPT_L2_COMBINED_IRQ) callback function.
 * 
 * @details  working opportunity an interrupt from the L2 Cache, indicate 
 *           an interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued. 
 * @param[in] context  The user provided context.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/12/18  ---  #6456  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_cache_manage_callback(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;

	/* Usage: ALT_STATUS_CODE alt_int_dist_pending_clear(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_L2_COMBINED_IRQ);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	//printf("[INTERRUPT]L2 Cache Combined Interrupt is occurred!! status=0x%08X\n", (int)alt_cache_l2_int_status_get());
	util_intlog_record(ALT_INT_INTERRUPT_L2_COMBINED_IRQ, (int)icciar, (int)alt_cache_l2_int_status_get());

	// Forced Clear All Interrupt Status
	(void)alt_cache_l2_int_status_clear((uint32_t)ALT_CACHE_L2_ICOND_ALL);

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_cache_manage_is_enable_all(void) 
 * @brief  Displays the get all caching State (enabled / disabled). 
 * 
 * @details  Run the following API specifically. 
 *           - <<L1 Cache Management API>>
 *               - alt_cache_l1_instruction_is_enabled()  
 *               - alt_cache_l1_data_is_enabled()  
 *               - alt_cache_l1_parity_is_enabled()  
 *               - alt_cache_l1_branch_is_enabled()  
 *               - alt_cache_l1_prefetch_is_enabled()  
 *           - <<L2 Cache Management API>>
 *               - alt_cache_l2_prefetch_is_enabled()  
 *               - alt_cache_l2_parity_is_enabled()  
 *               - alt_cache_l2_is_enabled()  
 *               - alt_cache_l2_int_status_get()  
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/12/18  ---  #6456  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_cache_manage_is_enable_all(void)
{
	/***********************************************************************/
	// <<L1 Cache Management API>>
	/* Usage:  bool  alt_cache_l1_instruction_is_enabled (void);	*/
	/* Usage:  bool  alt_cache_l1_data_is_enabled (void);	*/
	/* Usage:  bool  alt_cache_l1_parity_is_enabled (void);	*/
	/* Usage:  bool  alt_cache_l1_branch_is_enabled (void);	*/
	/* Usage:  bool  alt_cache_l1_prefetch_is_enabled (void);	*/
	// <<L2 Cache Management API>>
	/* Usage:  bool  alt_cache_l2_prefetch_is_enabled (void);	*/
	/* Usage:  bool  alt_cache_l2_parity_is_enabled (void);	*/
	/* Usage:  bool  alt_cache_l2_is_enabled (void);	*/
	/* Usage:  uint32_t  alt_cache_l2_int_status_get (void);	*/
	/***********************************************************************/
	printf("==== Cache Management Status L1 & L2 =====\n");
	printf("- alt_cache_l1_instruction_is_enabled() . %d\n", (int)alt_cache_l1_instruction_is_enabled());
	printf("- alt_cache_l1_data_is_enabled() ........ %d\n", (int)alt_cache_l1_data_is_enabled());
	printf("- alt_cache_l1_parity_is_enabled() ...... %d\n", (int)alt_cache_l1_parity_is_enabled());
	printf("- alt_cache_l1_branch_is_enabled() ...... %d\n", (int)alt_cache_l1_branch_is_enabled());
	printf("- alt_cache_l1_prefetch_is_enabled() .... %d\n", (int)alt_cache_l1_prefetch_is_enabled());
	printf("- alt_cache_l2_is_enabled() ............. %d\n", (int)alt_cache_l2_is_enabled());
	printf("- alt_cache_l2_parity_is_enabled() ...... %d\n", (int)alt_cache_l2_parity_is_enabled());
	printf("- alt_cache_l2_prefetch_is_enabled() .... %d\n", (int)alt_cache_l2_prefetch_is_enabled());
	printf("- alt_cache_l2_int_status_get() . 0x%08X\n", (int)alt_cache_l2_int_status_get());
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  void sample_cache_manage_test_print_usage(void) 
 * @brief  Display usage of the sample program. 
 * 
 * @details  Following information displays on the console.
 *            - Overview of HPS for push button functions (or operations)
 *            - Overview of HPS for DIP switch functions (or operations)
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/18  ---  #xxxxx  新規作成. 
 **//*******************************************************************************/
void sample_cache_manage_test_print_usage(void)
{
	printf("\n");
	printf("+--- Usage ( Function of PUSH SW and SLIDE SW )  ---------------+\n");
	printf("+ PUSH SW #0 .... Depends on the State of the SLIDE SW#0.\n");
	printf("  - SLIDE SW#0=ON ----> Exit Test loop!!! \n");
	printf("  - SLIDE SW#0=OFF ---> Switch State of All Cache Enable/Disable. \n");
	printf("+ PUSH SW #1 .... Run the cache system maintenance APIs.\n");
	printf("  - SLIDE SW#1=ON ----> API: alt_cache_system_invalidate(). \n");
	printf("  - SLIDE SW#2=ON ----> API: alt_cache_system_clean(). \n");
	printf("  - SLIDE SW#3=ON ----> API: alt_cache_system_purge(). \n");
	printf("+ PUSH SW #2 .... Run the L1/L2 cache maintenance APIs.\n");
	printf("  + SLIDE SW#0=ON (Depends on SLIDE SW#1-3) \n");
	printf("    - SLIDE SW#1=ON --> API: alt_cache_l2_data_invalidate_all(). \n");
	printf("    - SLIDE SW#2=ON --> API: alt_cache_l2_data_clean_all(). \n");
	printf("    - SLIDE SW#3=ON --> API: alt_cache_l2_data_purge_all(). \n");
	printf("  + SLIDE SW#0=OFF (Depends on SLIDE SW#1-3) \n");
	printf("    - SLIDE SW#1=ON --> API: alt_cache_l1_data_invalidate_all(). \n");
	printf("    - SLIDE SW#2=ON --> API: alt_cache_l1_data_clean_all(). \n");
	printf("    - SLIDE SW#3=ON --> API: alt_cache_l1_data_purge_all(). \n");
	printf("+ PUSH SW #3 .... Run the benchmark function and display the elapsed time. \n");
	printf("+---------------------------------------------------------------+\n\n");
	
#if 0
	　無限ループ内で以下の処理を行う。

	　 - HPS用PUSHSW0の押下契機
	　　　　HPS用DIPSW1がON --> 無限ループを終了します。
	　　　　　　　　　　　　　　(テストプログラム終了)
	　　　　HPS用DIPSW1がOFF -> L1,L2 Cache機能を全てEnable/DisableにするAPIを実行。
	　　　　　　　　　　　　　　(SW押下毎にEnableとDisableを交互に実行します)

	　 - HPS用PUSHSW1の押下契機
	　　　　HPS用DIPSW2がON --> alt_cache_system_invalidate を実行
	　　　　HPS用DIPSW3がON --> alt_cache_system_clean を実行
	　　　　HPS用DIPSW4がON --> alt_cache_system_purge を実行
	　　　　(Cache情報クリア系APIの試行)

	　 - HPS用PUSHSW2の押下契機
	　　　　HPS用DIPSW1がOFF
	　　　　　　HPS用DIPSW2がON --> alt_cache_l1_data_invalidate_all を実行
	　　　　　　HPS用DIPSW3がON --> alt_cache_l1_data_clean_all を実行
	　　　　　　HPS用DIPSW4がON --> alt_cache_l1_data_purge_all を実行
	　　　　　　(L1 Cache情報の一括クリア用APIの試行)
	　　　　HPS用DIPSW1がON
	　　　　　　HPS用DIPSW2がON --> alt_cache_l2_data_invalidate_all を実行
	　　　　　　HPS用DIPSW3がON --> alt_cache_l2_data_clean_all を実行
	　　　　　　HPS用DIPSW4がON --> alt_cache_l2_data_purge_all を実行
	　　　　　　(L2 Cache情報の一括クリア用APIの試行)

	　 - HPS用PUSHSW3の押下契機
	　　　　キャッシュ効果の検証用に関数 mul_f32_test_funciton を起動して
	　　　　処理時間の計測および表示を行います。
#endif
	
	return;
}

/********************************************************************************//*!
 * @fn  void sample_cache_manage_test_main(void) 
 * @brief  Main routine of the sample program. 
 * 
 * @details  Infinite loop structure works. Make the detection of various types 
 *           of switches in the loop, it performs in various testing processes.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/12/18  ---  #6456  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_cache_manage_test_main(void)
{
	ALT_STATUS_CODE	result_code;
	uint32_t switch_raw = 0;
	uint32_t switch_raw_bk = 0;
	uint32_t switch_detect = 0;
	uint32_t switch_detect_on = 0;
	uint32_t switch_number = 0;
	bool disp_usage = true;
	

	util_intlog_print();

	printf("==== Start While(1) loop process!!! ============\n");
	switch_raw_bk = sample_detect_switch();
	while(1)
	{
		if(disp_usage)
		{
			sample_cache_manage_test_print_usage();
			disp_usage = false;
		}

		// ---- Check the Slide-Switch and Push-Switch. ---- 
		switch_raw = sample_detect_switch();
		switch_detect = switch_raw ^ switch_raw_bk;
		switch_detect_on  |= switch_detect & switch_raw;
		
		// Slide-Switch 
		if(switch_detect & SAMPLE_SWITCH_BIT_SLIDEALL)
		{
			switch_number = switch_raw & SAMPLE_SWITCH_BIT_NUM;
			switch_number *= 1;		// To avoid warnings.

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE0){
				//printf("SAMPLE_SWITCH_BIT_SLIDE0\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE1){
				//printf("SAMPLE_SWITCH_BIT_SLIDE1\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE2){
				//printf("SAMPLE_SWITCH_BIT_SLIDE2\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE3){
				//printf("SAMPLE_SWITCH_BIT_SLIDE3\n");
			}
			disp_usage = false;
		}
		
		
		// Push-Switch 0
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH0)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH0;
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH1;
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH2;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH0\n");
			if((switch_raw & SAMPLE_SWITCH_BIT_SLIDE0) == 0){
				if(cache_all_en == ALT_E_TRUE)
				{
					/* Usage:  ALT_STATUS_CODE  alt_cache_system_disable (void);	*/
					printf("-- HWLib test : alt_cache_system_disable() --\n");
					result_code = alt_cache_system_disable();
					if(result_code != ALT_E_SUCCESS){
						printf("ERROR!!: alt_cache_system_disable() result=%d\n", (int)result_code);
					} else {
						printf("SUCCESS!!\n");
					}
					cache_all_en = ALT_E_FALSE;
				} else /* if(cache_all_en == ALT_E_FALSE) */
				{
					/* Usage:  ALT_STATUS_CODE  alt_cache_system_enable (void);	*/
					printf("-- HWLib test : alt_cache_system_enable() --\n");
					result_code = alt_cache_system_enable();
					if(result_code != ALT_E_SUCCESS){
						printf("ERROR!!: alt_cache_system_enable() result=%d\n", (int)result_code);
					} else {
						printf("SUCCESS!!\n");
					}
					cache_all_en = ALT_E_TRUE;
				}
				sample_cache_manage_is_enable_all();
				disp_usage = true;
			} else {
				break;	// Exit PUSHSW0(SW8) becomes ON.
			}
		}
		// Push-Switch 1
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH1)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH1;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH1\n");
			if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1){
				/* Usage:  ALT_STATUS_CODE  alt_cache_system_invalidate (void *vaddress, size_t length);	*/
				printf("-- HWLib test : alt_cache_system_invalidate() --\n");
				result_code = alt_cache_system_invalidate((void*)&data_i, (sizeof(int) * N * 8));
				if(result_code == ALT_E_SUCCESS){
					result_code = alt_cache_system_invalidate((void*)&data_f, (sizeof(float) * N * 8));
				}
				if(result_code == ALT_E_SUCCESS){
					result_code = alt_cache_system_invalidate((void*)&out, (sizeof(float) * N * 8));
				}
				if(result_code == ALT_E_SUCCESS){
					result_code = alt_cache_system_invalidate((void*)&ref, (sizeof(float) * N * 8));
				}
				if(result_code != ALT_E_SUCCESS){
					printf("ERROR!!: alt_cache_system_invalidate() result=%d\n", (int)result_code);
				} else {
					printf("SUCCESS!!\n");
				}
			}
			if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE2){
				/* Usage:  ALT_STATUS_CODE  alt_cache_system_clean (void *vaddress, size_t length);	*/
				printf("-- HWLib test : alt_cache_system_clean() --\n");
				result_code = alt_cache_system_clean((void*)&data_i, (sizeof(int) * N * 8));
				if(result_code == ALT_E_SUCCESS){
					result_code = alt_cache_system_clean((void*)&data_f, (sizeof(float) * N * 8));
				}
				if(result_code == ALT_E_SUCCESS){
					result_code = alt_cache_system_clean((void*)&out, (sizeof(float) * N * 8));
				}
				if(result_code == ALT_E_SUCCESS){
					result_code = alt_cache_system_clean((void*)&ref, (sizeof(float) * N * 8));
				}
				if(result_code != ALT_E_SUCCESS){
					printf("ERROR!!: alt_cache_system_clean() result=%d\n", (int)result_code);
				} else {
					printf("SUCCESS!!\n");
				}
			}
			if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE3){
				/* Usage:  ALT_STATUS_CODE  alt_cache_system_purge (void *vaddress, size_t length);	*/
				printf("-- HWLib test : alt_cache_system_purge() --\n");
				result_code = alt_cache_system_purge((void*)&data_i, (sizeof(int) * N * 8));
				if(result_code == ALT_E_SUCCESS){
					result_code = alt_cache_system_purge((void*)&data_f, (sizeof(float) * N * 8));
				}
				if(result_code == ALT_E_SUCCESS){
					result_code = alt_cache_system_purge((void*)&out, (sizeof(float) * N * 8));
				}
				if(result_code == ALT_E_SUCCESS){
					result_code = alt_cache_system_purge((void*)&ref, (sizeof(float) * N * 8));
				}
				if(result_code != ALT_E_SUCCESS){
					printf("ERROR!!: alt_cache_system_purge() result=%d\n", (int)result_code);
				} else {
					printf("SUCCESS!!\n");
				}
			}
			disp_usage = true;
		}
		// Push-Switch 2
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH2)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH2;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH2\n");
			if((switch_raw & SAMPLE_SWITCH_BIT_SLIDE0) == 0){
				if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1){
					/* Usage:  ALT_STATUS_CODE  alt_cache_l1_data_invalidate_all (void);	*/
					printf("-- HWLib test : alt_cache_l1_data_invalidate_all() --\n");
					result_code = alt_cache_l1_data_invalidate_all();
					if(result_code != ALT_E_SUCCESS){
						printf("ERROR!!: alt_cache_l1_data_invalidate_all() result=%d\n", (int)result_code);
					} else {
						printf("SUCCESS!!\n");
					}
				}
				if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE2){
					/* Usage:  ALT_STATUS_CODE  alt_cache_l1_data_clean_all (void);	*/
					printf("-- HWLib test : alt_cache_l1_data_clean_all() --\n");
					result_code = alt_cache_l1_data_clean_all();
					if(result_code != ALT_E_SUCCESS){
						printf("ERROR!!: alt_cache_l1_data_clean_all() result=%d\n", (int)result_code);
					} else {
						printf("SUCCESS!!\n");
					}
				}
				if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE3){
					/* Usage:  ALT_STATUS_CODE  alt_cache_l1_data_purge_all (void);	*/
					printf("-- HWLib test : alt_cache_l1_data_purge_all() --\n");
					result_code = alt_cache_l1_data_purge_all();
					if(result_code != ALT_E_SUCCESS){
						printf("ERROR!!: alt_cache_l1_data_purge_all() result=%d\n", (int)result_code);
					} else {
						printf("SUCCESS!!\n");
					}
				}
			} else {
				if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1){
					/* Usage:  ALT_STATUS_CODE  alt_cache_l2_invalidate_all (void);	*/
					printf("-- HWLib test : alt_cache_l2_invalidate() --\n");
					result_code = alt_cache_l2_invalidate((void*)&data_i, (sizeof(int) * N * 8));
					if(result_code == ALT_E_SUCCESS){
						result_code = alt_cache_l2_invalidate((void*)&data_f, (sizeof(float) * N * 8));
					}
					if(result_code == ALT_E_SUCCESS){
						result_code = alt_cache_l2_invalidate((void*)&out, (sizeof(float) * N * 8));
					}
					if(result_code == ALT_E_SUCCESS){
						result_code = alt_cache_l2_invalidate((void*)&ref, (sizeof(float) * N * 8));
					}
					if(result_code != ALT_E_SUCCESS){
						printf("ERROR!!: alt_cache_l2_invalidate() result=%d\n", (int)result_code);
					} else {
						printf("SUCCESS!!\n");
					}
				}
				if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE2){
					/* Usage:  ALT_STATUS_CODE  alt_cache_l2_clean_all (void);	*/
					printf("-- HWLib test : alt_cache_l2_clean() --\n");
					result_code = alt_cache_l2_clean((void*)&data_i, (sizeof(int) * N * 8));
					if(result_code == ALT_E_SUCCESS){
						result_code = alt_cache_l2_clean((void*)&data_f, (sizeof(float) * N * 8));
					}
					if(result_code == ALT_E_SUCCESS){
						result_code = alt_cache_l2_clean((void*)&out, (sizeof(float) * N * 8));
					}
					if(result_code == ALT_E_SUCCESS){
						result_code = alt_cache_l2_clean((void*)&ref, (sizeof(float) * N * 8));
					}
					if(result_code != ALT_E_SUCCESS){
						printf("ERROR!!: alt_cache_l2_clean() result=%d\n", (int)result_code);
					} else {
						printf("SUCCESS!!\n");
					}
				}
				if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE3){
					/* Usage:  ALT_STATUS_CODE  alt_cache_l2_purge_all (void);	*/
					printf("-- HWLib test : alt_cache_l2_purge() --\n");
					result_code = alt_cache_l2_purge((void*)&data_i, (sizeof(int) * N * 8));
					if(result_code == ALT_E_SUCCESS){
						result_code = alt_cache_l2_purge((void*)&data_f, (sizeof(float) * N * 8));
					}
					if(result_code == ALT_E_SUCCESS){
						result_code = alt_cache_l2_purge((void*)&out, (sizeof(float) * N * 8));
					}
					if(result_code == ALT_E_SUCCESS){
						result_code = alt_cache_l2_purge((void*)&ref, (sizeof(float) * N * 8));
					}
					if(result_code != ALT_E_SUCCESS){
						printf("ERROR!!: alt_cache_l2_purge() result=%d\n", (int)result_code);
					} else {
						printf("SUCCESS!!\n");
					}
				}
			}
			disp_usage = true;
		}
		// Push-Switch 3
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH3)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH3;

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH3\n");
			mul_f32_test_funciton((int)(switch_number & 7));
			disp_usage = true;
		}
		
		util_intlog_print();

		switch_raw_bk = switch_raw;
	}
	printf("==== End While(1) loop process. ================\n");

	util_intlog_print();

	return;
}

// All APIs
/***********************************************************************/
// <<System Level Cache Management API>>
/* Usage:  ALT_STATUS_CODE  alt_cache_system_enable (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_system_disable (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_system_invalidate (void *vaddress, size_t length);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_system_clean (void *vaddress, size_t length);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_system_purge (void *vaddress, size_t length);	*/
// <<L1 Cache Management API>>
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_enable_all (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_disable_all (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_instruction_enable (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_instruction_disable (void);	*/
/* Usage:  bool  alt_cache_l1_instruction_is_enabled (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_instruction_invalidate (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_data_enable (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_data_disable (void);	*/
/* Usage:  bool  alt_cache_l1_data_is_enabled (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_data_invalidate (void *vaddress, size_t length);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_data_invalidate_all (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_data_clean (void *vaddress, size_t length);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_data_clean_all (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_data_purge (void *vaddress, size_t length);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_data_purge_all (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_parity_enable (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_parity_disable (void);	*/
/* Usage:  bool  alt_cache_l1_parity_is_enabled (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_branch_enable (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_branch_disable (void);	*/
/* Usage:  bool  alt_cache_l1_branch_is_enabled (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_branch_invalidate (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_prefetch_enable (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l1_prefetch_disable (void);	*/
/* Usage:  bool  alt_cache_l1_prefetch_is_enabled (void);	*/
// <<L2 Cache Management API>>
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_init (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_uninit (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_prefetch_enable (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_prefetch_disable (void);	*/
/* Usage:  bool  alt_cache_l2_prefetch_is_enabled (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_parity_enable (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_parity_disable (void);	*/
/* Usage:  bool  alt_cache_l2_parity_is_enabled (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_enable (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_disable (void);	*/
/* Usage:  bool  alt_cache_l2_is_enabled (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_sync (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_invalidate (void *paddress, size_t length);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_invalidate_all (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_clean (void *paddress, size_t length);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_clean_all (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_purge (void *paddress, size_t length);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_purge_all (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_int_enable (uint32_t interrupt);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_int_disable (uint32_t interrupt);	*/
/* Usage:  uint32_t  alt_cache_l2_int_status_get (void);	*/
/* Usage:  ALT_STATUS_CODE  alt_cache_l2_int_status_clear (uint32_t interrupt);	*/
/***********************************************************************/

////////////////////////////// Common Functions //////////////////////////////////////// 

///////////////////// Test Functions /////////////////////////////////
/********************************************************************************//*!
 * @fn  int mul_f32_test_funciton(int mulNval) 
 * @brief  Implementation of various operations for validate the effect of cash. 
 * 
 * @details  The specified memory region to measure below the processing time to. 
 *           - memset (Using different data held 2 times)
 *           - mul_f32a 
 *           - verify (Collation of the result)
 * @param[in] mulNval  Memory region specified value(multiples of N(=8192)) 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/12/18  ---  #6456  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
int mul_f32_test_funciton(int mulNval)
{
	int i;
	int size;
	int vresult;
	int recbase;

	size = (mulNval + 1) * N;
	recbase = mulNval * 4;

	mul_f32a((float *)data_i, data_f, ref, size);

	// CPU Only の実行時間の計測
	printf("######## ARM Only [size=%d] ######## : \n", size);
	printf("TIME#n..memset / #n+1..memset / #n+2..mul_f32a / #n+3..verify\n");
    for(i=0;i<10;i++){
    	util_time_record_start_point(recbase + 0);
    	memset(data_i, 0xDEAD, size);
    	util_time_record_end_point(recbase + 0);
    	util_time_record_start_point(recbase + 1);
    	memset(out   , 0x1234, size);
    	util_time_record_end_point(recbase + 1);
    	util_time_record_start_point(recbase + 2);
    	mul_f32a((float *)data_i, data_f, out, size);
    	util_time_record_end_point(recbase + 2);
    	util_time_record_start_point(recbase + 3);
    	vresult = verify(out, ref, size, i);
    	util_time_record_end_point(recbase + 3);
    	if(vresult == size){
    		printf("Verify Success for %d!\n", i);
    	} else {
    		printf("Verify Error at 0x%x for %d\n", vresult, i);
    	}
    	util_time_print_result_by_seconds(recbase + 0);
    	util_time_print_result_by_seconds(recbase + 1);
    	util_time_print_result_by_seconds(recbase + 2);
    	util_time_print_result_by_seconds(recbase + 3);
    }
    printf("\n");
#if 0
	// 通常関数の実行時間の計測
    for(i=0;i<10;i++){
    	memset(data_i, 0xDEAD, N);
    	memset(out   , 0x1234, N);
    	stime = GetTickCount();
    	vector_mul_f32a((float *)data_i, data_f, out);
    	etime = GetTickCount();
    	diff = etime-stime;
    	verify(out, ref, N, i);
    	printf("ARM/NEON Non-Vectorized : %d ms\n", (int)diff);
    }
    printf("\n");

	// ベクタライズされた関数の実行時間の計測
    for(i=0;i<10;i++){
    	memset(data_i, 0xDEAD, N);
    	memset(out   , 0x1234, N);
    	stime = GetTickCount();
    	vector_mul_f32i((float *)data_i, data_f, out);
    	etime = GetTickCount();
    	diff = etime-stime;
    	verify(out, ref, N, i);
    	printf("ARM/NEON Vectorized : %d ms\n", (int)diff);
    }
    printf("\n");
#endif

    return 0;
}

/********************************************************************************//*!
 * @fn  void mul_f32a(float * __restrict a, float * __restrict b, float * __restrict p, int size) 
 * @brief  Floating-point multiplication.  
 * 
 * @details  Performs the number specified in the 4th argument(size). 
 * @param[in] a  Operation parameter storage A 
 * @param[in] b  Operation parameter storage B 
 * @param[out] p  Result storage 
 * @param[in] size  Number of operations 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/12/18  ---  #6456  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void mul_f32a(float * __restrict a, float * __restrict b, float * __restrict p, int size)
{
	int i;
	for( i = 0; i < size; i++ )
	{
		p[i] = (float)a[i] * b[i];
	}
}

/********************************************************************************//*!
 * @fn  int verify(float *a, float *b, int size,  int place) 
 * @brief  Floating point verification. 
 * 
 * @details  Performs the number specified in the 4th argument(size). 
 *           If it detects the mismatch in the middle number of the return value. 
 * @param[in] a  Operation parameter storage A 
 * @param[in] b  Operation parameter storage B 
 * @param[in] size  Number of operations 
 * @param[out] place  Don't used 
 * @retval int  result (OK:=size / NG:<size) 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/12/18  ---  #6456  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
int verify(float *a, float *b, int size,  int place)
{
	int i;
	for( i = 0; i < size; i++ )
	{
		if((*a++) != (*b++)){
			return(i);
		}
	}
	return(size);
}

#if 0
/********************************************************************************//*!
 * @fn  void vector_mul_f32a(float * __restrict a, float * __restrict b, float * __restrict p) 
 * @brief  Floating-point multiplication.  
 * 
 * @details  In the attempt to count fixed N(=8192) times operation. 
 * @param[in] a  Operation parameter storage A 
 * @param[in] b  Operation parameter storage B 
 * @param[out] p  Result storage 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/12/18  ---  #6456  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void vector_mul_f32a(float * __restrict a, float * __restrict b, float * __restrict p)
{
	int i;
	for (i=0; i < N ; i++) {
		p[i] = a[i] * b[i];
	}
}

/********************************************************************************//*!
 * @fn  void vector_mul_f32i(float *a, float *b, float *p) 
 * @brief  Using the NEON processors built-in function, floating point arithmetic, 
 * 
 * @details  In the attempt to count fixed N(=8192) times operation. 
 * @param[in] a  Operation parameter storage A 
 * @param[in] b  Operation parameter storage B 
 * @param[out] p  Result storage 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/12/18  ---  #6456  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void vector_mul_f32i(float *a, float *b, float *p)
{
	int i;
	float32x4_t a4, b4, p4;
	float32_t *pa4 = a;
	float32_t *pb4 = b;
	float32_t *pp4 = p;

	for (i=0; i< N/4; i++) {
		a4 = vld1q_f32(pa4);
		b4 = vld1q_f32(pb4);
		p4 = vmulq_f32(a4, b4);
		vst1q_f32(pp4, p4);
		pa4+=4;
		pb4+=4;
		pp4+=4;
	}
}
#endif


/************************************/
/*	Test Command (Cache Management)	*/
/************************************/
int sample_cache_manage_test_cmd(char* options)
{
    ALT_STATUS_CODE status = ALT_E_SUCCESS;
    
    printf("\r\nUser Application Start!\r\n");

    //
    // Start the interrupt system
    //
    if (status == ALT_E_SUCCESS)
    {
    	util_intlog_init();
        status = socfpga_int_start();
    }

    // Printing Current PLL Setting. And Set Global Timer for Time Measurement.
    util_time_init();

    //// Sample Function Call!!! |---->
    printf(">>>> Execute sample_cache_manage_init(); !!! <<<<\r\n");
    sample_cache_manage_init();

    printf(">>>> sample_cache_manage_test(); !!! <<<<\r\n");
    sample_cache_manage_test_main();
    //// Sample Function Call!!! <----|

    // Printing All Results of Time Measurement. And Uninit Global Timer.
    util_time_uninit();

    //
    // Stop the interrupt system
    //
    socfpga_int_stop();
	util_intlog_print();

    printf("Finished running the sample !!!\r\n");
    if (status == ALT_E_SUCCESS)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}
/***********************************************************************************
 * end of file 
 ***********************************************************************************/
