﻿/********************************************************************************//*!
 * @file  sample_dma_mem.c
 * @brief  Example program using HPS internal DMA (DMA-330).
 * 
 * @details  Try the DMA transfer under the conditions below.
 *           - MMU and L1, L2 cache enabled
 *           - ACP Port enabled (SMP mode, SCN=enabled)
 *           - Various source and destination cache settings (NonCache, Write-Throught,
 *             write-back, write-back with write allocate)
 *           - ACP port usage option (unused, via the ACP)
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2015/03/02  ---  #18042  新規作成.
 * @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
 * @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
 * @date 2018/07/02  ---  #25940  All in One ベアメタル・アプリケーション v17.1 への対応を実施.
 * @date 2019/01/18  ---  -----   All in One ベアメタル・アプリケーション v18.1 への対応を実施.
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes 
 ***********************************************************************************/
#include <stdio.h>
#include <string.h>
#include "hwlib.h"
#include "socal/socal.h"
#include "socal/hps.h"
#include "socal/alt_sysmgr.h"
#include "alt_mpu_registers.h"
#include "alt_generalpurpose_io.h"
#include "alt_globaltmr.h"
#include "alt_clock_manager.h"
#include "alt_interrupt.h"
#include "alt_address_space.h"
#include "alt_cache.h"
#include "alt_mmu.h"
#include "alt_dma.h"
#include "alt_dma_program.h"

#include "sample_app_setting.h"
#include "util_time_measurement.h"
#include "util_interrupt_log.h"
#include "alt_printf.h"

/***********************************************************************************
 *  externs 
 ***********************************************************************************/
/* defined on alt_dma_custom.c */
extern ALT_STATUS_CODE alt_dma_channel_program_memory_to_memory(ALT_DMA_CHANNEL_t channel,
                                            ALT_DMA_PROGRAM_t * program,
                                            void * dst,       int dst_cache_ctrl,
                                            const void * src, int src_cache_ctrl,
                                            size_t size);

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/
typedef enum {
	AXCACHEe_NON = 0,				// WA,RA=0/C=0/B=0 (000b) .. Noncacheableand nonbufferable
	AXCACHEe_BUFF,					// WA,RA=0/C-0/B=1 (001b) .. Bufferable only
	AXCACHEe_CACHE,					// WA,RA=0/C=1/B=0 (010b) .. Cacheable, but do not allocate
	AXCACHEe_CACHEBUFF,				// WA,RA=0/C=1/B=1 (011b) .. Cacheable and bufferable, but do not allocate
	AXCACHEe_RESERVE1,				// WA,RA=1/C=0/B=0 (100b) .. Reserved
	AXCACHEe_RESERVE2,				// WA,RA=1/C=0/B=1 (101b) .. Reserved
	AXCACHEe_CACHE_ALLOC,			// WA,RA=1/C=1/B=0 (110b) .. Cacheable write-through, allocate on both reads and writes
	AXCACHEe_CACHEBUFF_ALLOC		// WA,RA=1/C=1/B=1 (111b) .. Cacheable write-back, allocate on both reads and writes
} AXCACHEe;

#define DMA_TEST_SIZE	(16 * _MB)		/* ＠＠通常は2MB以内 (DALT_DMA_PROGRAM_PROVISION_BUFFER_SIZEを拡張すれば大きなサイズも可:128MBまで確認済み) */

#ifndef ARRAY_COUNT
#define ARRAY_COUNT(array) (sizeof(array) / sizeof(array[0]))
#endif

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/
// prototype (Common Functions) //
static void sample_dma_mem_test_init(void);
static void sample_dma_mem_test_main(ALT_DMA_CHANNEL_t channel, bool acp_en, bool cacheope_en);
static int sample_dma_mem_test_execute(	ALT_DMA_CHANNEL_t channel,	// DMA CH番号(0~7) 
										void* dst_address,			// CPUから見た送信先アドレス 
										void* src_address,			// CPUから見た送信元アドレス 
										AXCACHEe dst_axcache,		// DMAC->DST の AXI信号(AWCACHE)  
										AXCACHEe src_axcache,		// SRC->DMAC の AXI信号(ARCACHE)  
										size_t trans_size,			// DMA転送サイズ 
										uint32_t test_data,			// テストデータ初期値(転送アドレスのオフセット値を加算した値を転送)
										bool acp_en,				// ACP有効？ 
										bool cacheope_en		);	// CACHE操作有効？ 
void sample_dma_mem_callback_DMA_IRQ0(uint32_t icciar, void *context);
void sample_dma_mem_callback_DMA_IRQ1(uint32_t icciar, void *context);
void sample_dma_mem_callback_DMA_IRQ2(uint32_t icciar, void *context);
void sample_dma_mem_callback_DMA_IRQ3(uint32_t icciar, void *context);
void sample_dma_mem_callback_DMA_IRQ4(uint32_t icciar, void *context);
void sample_dma_mem_callback_DMA_IRQ5(uint32_t icciar, void *context);
void sample_dma_mem_callback_DMA_IRQ6(uint32_t icciar, void *context);
void sample_dma_mem_callback_DMA_IRQ7(uint32_t icciar, void *context);
void sample_dma_mem_callback_DMA_IRQ_ABORT(uint32_t icciar, void *context);
void sample_dma_mem_callback_DMA_ECC_CORRECTED_IRQ(uint32_t icciar, void *context);
void sample_dma_mem_callback_DMA_ECC_UNCORRECTED_IRQ(uint32_t icciar, void *context);
static void sample_dma_mem_setting_gic(void);
static void sample_dma_mem_print_manager_status(void);
static void sample_dma_mem_print_ch_status(ALT_DMA_CHANNEL_t channel, bool detail);
static void sample_dma_mem_print_all_status(void);
// prototype (Common Functions) //
static void sample_memset_address32(uint32_t* start, size_t size) __WARN_UNUSED__ ;
static void sample_memset_incrementdata(uint32_t* start, uint32_t testdata, size_t size) __WARN_UNUSED__ ;
static void sample_memset_incrementdata_4byte(uint32_t* start, uint32_t testdata, size_t size) __WARN_UNUSED__ ;


/***********************************************************************************
 *  variables 
 ***********************************************************************************/
#if 0
uint32_t dma_mem_src_buf_nc[DMA_TEST_SIZE/sizeof(uint32_t)] __attribute__ ((aligned(4096))) __attribute__ ((section (".common1_region"))) ;
uint32_t dma_mem_dest_buf_nc[DMA_TEST_SIZE/sizeof(uint32_t)] __attribute__ ((aligned(4096))) __attribute__ ((section (".common1_region"))) ;
uint32_t dma_mem_src_buf_wt[DMA_TEST_SIZE/sizeof(uint32_t)] __attribute__ ((aligned(4096))) __attribute__ ((section (".common2_region"))) ;
uint32_t dma_mem_src_buf_wb[DMA_TEST_SIZE/sizeof(uint32_t)] __attribute__ ((aligned(4096))) __attribute__ ((section (".common3_region"))) ;
uint32_t dma_mem_src_buf_wba[DMA_TEST_SIZE/sizeof(uint32_t)] __attribute__ ((aligned(4096))) __attribute__ ((section (".common4_region"))) ;
uint32_t dma_mem_dest_buf[DMA_TEST_SIZE/sizeof(uint32_t)] __attribute__ ((aligned(4096))) __attribute__ ((section (".common4_region"))) ;
#else
uint32_t* dma_mem_src_buf_nc = (uint32_t*)(SDRAM1_VIRTUAL_ADDR + DMA_TEST_SIZE * 0);
uint32_t* dma_mem_dest_buf_nc = (uint32_t*)(SDRAM1_PHYSICAL_ADDR + DMA_TEST_SIZE * 1);
uint32_t* dma_mem_src_buf_wt = (uint32_t*)SDRAM2_VIRTUAL_ADDR;
uint32_t* dma_mem_src_buf_wb = (uint32_t*)SDRAM3_VIRTUAL_ADDR;
uint32_t* dma_mem_src_buf_wba = (uint32_t*)(SDRAM4_VIRTUAL_ADDR + DMA_TEST_SIZE * 0);
uint32_t* dma_mem_dest_buf = (uint32_t*)(SDRAM4_VIRTUAL_ADDR + DMA_TEST_SIZE * 1);
#endif

size_t	dma_mem_xfer_size = DMA_TEST_SIZE;			// DMA転送サイズ

ALT_DMA_CFG_t dma_mem_config;			/*!< DMA Configration Table. */
ALT_DMA_PROGRAM_t dma_mem_program[8];	/*!< Micro code program buffer for the DMAC (The index# of the array means the DMA channel#) */

/*! Status String Definitions of DMA Manager */
//typedef enum ALT_DMA_MANAGER_STATE_e
//{
//    ALT_DMA_MANAGER_STATE_STOPPED     = 0, /*!< Stopped */
//    ALT_DMA_MANAGER_STATE_EXECUTING   = 1, /*!< Executing */
//    ALT_DMA_MANAGER_STATE_CACHE_MISS  = 2, /*!< Cache Miss */
//    ALT_DMA_MANAGER_STATE_UPDATING_PC = 3, /*!< Updating PC */
//    ALT_DMA_MANAGER_STATE_WFE         = 4, /*!< Waiting for Event */
//    ALT_DMA_MANAGER_STATE_FAULTING    = 15 /*!< Faulting */
//}
//ALT_DMA_MANAGER_STATE_t;
const char* StatusStringDMAMEM_Manager[16] = {
	"STOPPED(0)",
	"EXECUTING(1)",
	"CACHE_MISS(2)",
	"UPDATING_PC(3)",
	"WFE(4)",
	"---(5)",
	"---(6)",
	"---(7)",
	"---(8)",
	"---(9)",
	"---(10)",
	"---(11)",
	"---(12)",
	"---(13)",
	"---(14)",
	"FAULTING(15)"
};
/*! Status String Definitions of DMA Channel */
//typedef enum ALT_DMA_CHANNEL_STATE_e
//{
//    ALT_DMA_CHANNEL_STATE_STOPPED             = 0,  /*!< Stopped */
//    ALT_DMA_CHANNEL_STATE_EXECUTING           = 1,  /*!< Executing */
//    ALT_DMA_CHANNEL_STATE_CACHE_MISS          = 2,  /*!< Cache Miss */
//    ALT_DMA_CHANNEL_STATE_UPDATING_PC         = 3,  /*!< Updating PC */
//    ALT_DMA_CHANNEL_STATE_WFE                 = 4,  /*!< Waiting for Event */
//    ALT_DMA_CHANNEL_STATE_AT_BARRIER          = 5,  /*!< At Barrier */
//    ALT_DMA_CHANNEL_STATE_WFP                 = 7,  /*!< Waiting for Peripheral */
//    ALT_DMA_CHANNEL_STATE_KILLING             = 8,  /*!< Killing */
//    ALT_DMA_CHANNEL_STATE_COMPLETING          = 9,  /*!< Completing */
//    ALT_DMA_CHANNEL_STATE_FAULTING_COMPLETING = 14, /*!< Faulting Completing */
//    ALT_DMA_CHANNEL_STATE_FAULTING            = 15  /*!< Faulting */
//}
//ALT_DMA_CHANNEL_STATE_t;
const char* StatusStringDMAMEM_Channel[16] = {
	"STOPPED(0)",
	"EXECUTING(1)",
	"CACHE_MISS(2)",
	"UPDATING_PC(3)",
	"WFE(4)",
	"AT_BARRIER(5)",
	"---(6)",
	"WFP(7)",
	"KILLING(8)",
	"COMPLETING(9)",
	"---(10)",
	"---(11)",
	"---(12)",
	"---(13)",
	"FAULTING_COMPLETING(14)",
	"FAULTING(15)"
};

/*! Interrupt Callback Function Table */
const CALLBACK_FUNKt	SAMPLE_CallBacks[11] =
{
		sample_dma_mem_callback_DMA_IRQ0,
		sample_dma_mem_callback_DMA_IRQ1,
		sample_dma_mem_callback_DMA_IRQ2,
		sample_dma_mem_callback_DMA_IRQ3,
		sample_dma_mem_callback_DMA_IRQ4,
		sample_dma_mem_callback_DMA_IRQ5,
		sample_dma_mem_callback_DMA_IRQ6,
		sample_dma_mem_callback_DMA_IRQ7,
		sample_dma_mem_callback_DMA_IRQ_ABORT,
		sample_dma_mem_callback_DMA_ECC_CORRECTED_IRQ,
		sample_dma_mem_callback_DMA_ECC_UNCORRECTED_IRQ
};

volatile int	SAMPLE_CallBackCount[11];	/*!< Interrupt detection counters */

/***********************************************************************************
 *  functions 
 ***********************************************************************************/

/********************************************************************************//*!
 * @fn  void sample_dma_mem_test_print_usage(void) 
 * @brief  Display usage of the sample program. 
 * 
 * @details  Following information displays on the console.
 *            - Overview of HPS for push button functions (or operations)
 *            - Overview of HPS for DIP switch functions (or operations)
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2015/03/02  ---  #18042  新規作成.
 **//*******************************************************************************/
void sample_dma_mem_test_print_usage(void)
{
	printf("\n");
	printf("+--- Usage ( Function of PUSH SW and SLIDE SW )  ---------------+\n");
	printf("+ PUSH SW #0 .... Exit Test loop!!!\n");
	printf("+ PUSH SW #1 .... Display DMA Status Registers.\n");
	printf("+ PUSH SW #2 .... Execute DMA TEST (TEST Option: Non-ACP).\n");
	printf("+ PUSH SW #3 .... Execute DMA TEST (TEST Option: via ACP Port).\n");
	printf("+ SLIDE SW#0 .... TEST Option: Cache Maintenance ON/OFF.\n");
	printf("+ SLIDE SW#1:3 .. TEST Option: Seltct DMA Channel 0~7.\n");
	printf("+---------------------------------------------------------------+\n");
	printf("+ Note: If your target device is Arria 10 SoC, then cacheable AXI\n");
	printf("+  transactions are routed to the ACP automatically.\n");
	printf("+---------------------------------------------------------------+\n\n");
	
#if 0
	ループ中にHPS用PUSHSWの操作を検出した場合、それぞれ以下の処理を実行します。
		- PUSH SW #0 .... ループを抜けてプログラム終了。
		- PUSH SW #1 .... DMAレジスタ表示（Management Thread, Ch Thead のステータスを表示）
		- PUSH SW #2 .... DMAテストを実行。(ACP未使用)。
		- PUSH SW #3 .... DMAテストを実行。ACPポート経由で転送。
		- SLIDE SW#0 .... オプション選択：キャッシュメンテナスON/OFF 
		- SLIDE SW#1:3 .. オプション選択：DMA Channel 0～7 
#endif
	
	return;
}


////////////////////////////// Common Functions //////////////////////////////////////// 
/********************************************************************************//*!
 * @fn  void sample_dma_mem_test_init(void)
 * @brief  Initializing the DMA functions of hwlib.
 *
 * @details  Call the initialization function for the DMA Controller API.
 * @param[in,out] void  none
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2015/03/02  ---  #18042  新規作成.
 **//*******************************************************************************/
static void sample_dma_mem_test_init(void)
{
	ALT_STATUS_CODE result_code;
	int i;

	/* GIC Setting for DMA Interrupts */
	sample_dma_mem_setting_gic();

	/* Execute initialize dmac function of hwlib  */
	/* Usage:	ALT_STATUS_CODE  alt_dma_init (const ALT_DMA_CFG_t *dma_cfg);	*/
	// Set the parameters for alt_dma_init().
	for(i=0; i<4; i++){
		dma_mem_config.periph_mux[i] = ALT_DMA_PERIPH_MUX_DEFAULT;	// Handle FPGA / CAN muxing (default is FPGA)
	}
	dma_mem_config.manager_sec = ALT_DMA_SECURITY_DEFAULT;			// Handle Manager Secure / NonSecure (default is Secure)
	for(i=0; i<ARRAY_COUNT(dma_mem_config.irq_sec); i++){
		dma_mem_config.irq_sec[i] = ALT_DMA_SECURITY_DEFAULT;		// Handle IRQ Secure / NonSecure (default is Secure)
	}
	for(i=0; i<32; i++){
		dma_mem_config.periph_sec[i] = ALT_DMA_SECURITY_DEFAULT;	// peripheral Secure / NonSecure (default is Secure)
	}
	result_code = alt_dma_init(&dma_mem_config);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_dma_init() .... result=%d\n", (int)result_code);
	}

	/* Allocate all DMA Channels */
	/* Usage:	ALT_STATUS_CODE  alt_dma_channel_alloc (ALT_DMA_CHANNEL_t channel);	*/
	for(i=ALT_DMA_CHANNEL_0; i<=ALT_DMA_CHANNEL_7; i++){
		result_code = alt_dma_channel_alloc((ALT_DMA_CHANNEL_t)i);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_dma_channel_alloc(%d) .... result=%d\n", i, (int)result_code);
		}
	}

	/* Test for interrupt callback registration. */
	printf("Test for interrupt callback registration.\n");
	/* Usage:	ALT_STATUS_CODE  alt_dma_send_event (ALT_DMA_EVENT_t evt_num);	*/
	for(i=ALT_DMA_EVENT_0; i<=ALT_DMA_EVENT_7; i++){
		util_time_record_start_point(3);
		result_code = alt_dma_send_event((ALT_DMA_EVENT_t)i);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_dma_send_event(%d) .... result=%d\n", i, (int)result_code);
		}
	}
	printf("\n");

	/* Print all DMA statuses */
	sample_dma_mem_print_all_status();
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  void sample_dma_mem_test_main(void)
 * @brief  Excecute DMA TESTs.
 *
 * @details  Run the following 8 pattern DMA transfer.
 *         - TEST.01 : NonCache Buffer --> NonCache Buffer
 *         - TEST.02 : Cacheable[Write-Through(WT)] Buffer --> NonCache Buffer
 *         - TEST.03 : Cacheable[Write-Back(WB)] Buffer --> NonCache Buffer
 *         - TEST.04 : Cacheable[Write-Back with Allocate(WBA)] Buffer --> NonCache Buffer
 *         - TEST.05 : NonCache Buffer --> Cacheable[WBA] Buffer
 *         - TEST.06 : Cacheable[WT] Buffer --> Cacheable[WBA] Buffer
 *         - TEST.07 : Cacheable[WB] Buffer --> Cacheable[WBA] Buffer
 *         - TEST.08 : Cacheable[WBA] Buffer --> Cacheable[WBA] Buffer
 *
 * @param[in,out] void  none
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2015/03/02  ---  #18042  新規作成.
 **//*******************************************************************************/
static void sample_dma_mem_test_main(ALT_DMA_CHANNEL_t channel, bool acp_en, bool cacheope_en)
{
	uint32_t test_data;
	void*	dst_address;		// Destination address of CPU View
	void*	src_address;		// Source address of CPU View
	AXCACHEe dst_axcache = AXCACHEe_CACHE;	// AXI Signal of DMAC->DST (AWCACHE)
	AXCACHEe src_axcache = AXCACHEe_CACHE;	// AXI Signal of SRC->DMAC (ARCACHE)
	int result_memcmp;

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	//// TEST.01 : NonCache(dma_mem_src_buf_nc) --> NonCache(dma_mem_dest_buf_nc)
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	printf("#### Execute DMA TEST.01 : NonCache Buffer --> NonCache Buffer ###\n");
	// Setting Test parameters. （source/destination address, and test_data(initial)）
	dst_address = (void*)dma_mem_dest_buf_nc;		// Destination address of CPU View
	src_address = (void*)dma_mem_src_buf_nc;				// Source address of CPU View
	test_data = 0x11000000;
	// Execute DMA Transfer Test.
	result_memcmp = sample_dma_mem_test_execute(channel, dst_address, src_address, dst_axcache, src_axcache, dma_mem_xfer_size, test_data, acp_en, cacheope_en);
	printf("TEST.01 Verification Result ... %s\n\n", (result_memcmp == 0 ? "OK" : "NG"));


	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	//// TEST.02 : Cacheable[WT](dma_mem_src_buf_wt) --> NonCache(dma_mem_dest_buf_nc)
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	printf("#### Execute DMA TEST.02 : Cacheable[WT] Buffer --> NonCache Buffer ###\n");
	// Setting Test parameters. （source/destination address, and test_data(initial)）
	dst_address = (void*)dma_mem_dest_buf_nc;		// Destination address of CPU View
	src_address = (void*)dma_mem_src_buf_wt;				// Source address of CPU View
	test_data = 0x22000000;
	// Execute DMA Transfer Test.
	result_memcmp = sample_dma_mem_test_execute(channel, dst_address, src_address, dst_axcache, src_axcache, dma_mem_xfer_size, test_data, acp_en, cacheope_en);
	printf("TEST.02 Verification Result ... %s\n\n", (result_memcmp == 0 ? "OK" : "NG"));


	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	//// TEST.03 : Cacheable[WB](dma_mem_src_buf_wb) --> NonCache(dma_mem_dest_buf_nc)
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	printf("#### Execute DMA TEST.03 : Cacheable[WB] Buffer --> NonCache Buffer ###\n");
	// Setting Test parameters. （source/destination address, and test_data(initial)）
	dst_address = (void*)dma_mem_dest_buf_nc;		// Destination address of CPU View
	src_address = (void*)dma_mem_src_buf_wb;				// Source address of CPU View
	test_data = 0x33000000;
	// Execute DMA Transfer Test.
	result_memcmp = sample_dma_mem_test_execute(channel, dst_address, src_address, dst_axcache, src_axcache, dma_mem_xfer_size, test_data, acp_en, cacheope_en);
	printf("TEST.03 Verification Result ... %s\n\n", (result_memcmp == 0 ? "OK" : "NG"));

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	//// TEST.04 : Cacheable[WBA](dma_mem_src_buf_wba) --> NonCache(dma_mem_dest_buf_nc)
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	printf("#### Execute DMA TEST.04 : Cacheable[WBA] Buffer --> NonCache Buffer ###\n");
	// Setting Test parameters. （source/destination address, and test_data(initial)）
	dst_address = (void*)dma_mem_dest_buf_nc;		// Destination address of CPU View
	src_address = (void*)dma_mem_src_buf_wba;			// Source address of CPU View
	test_data = 0x44000000;
	// Execute DMA Transfer Test.
	result_memcmp = sample_dma_mem_test_execute(channel, dst_address, src_address, dst_axcache, src_axcache, dma_mem_xfer_size, test_data, acp_en, cacheope_en);
	printf("TEST.04 Verification Result ... %s\n\n", (result_memcmp == 0 ? "OK" : "NG"));

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	//// TEST.05 : NonCache(dma_mem_src_buf_nc) --> Cacheable[WBA](dma_mem_dest_buf)
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	printf("#### Execute DMA TEST.05 : NonCache Buffer --> Cacheable[WBA] Buffer ###\n");
	// Setting Test parameters. （source/destination address, and test_data(initial)）
	dst_address = (void*)dma_mem_dest_buf;			// Destination address of CPU View
	src_address = (void*)dma_mem_src_buf_nc;				// Source address of CPU View
	test_data = 0x55000000;
	// Execute DMA Transfer Test.
	result_memcmp = sample_dma_mem_test_execute(channel, dst_address, src_address, dst_axcache, src_axcache, dma_mem_xfer_size, test_data, acp_en, cacheope_en);
	printf("TEST.05 Verification Result ... %s\n\n", (result_memcmp == 0 ? "OK" : "NG"));


	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	//// TEST.06 : Cacheable[WT](dma_mem_src_buf_wt) --> Cacheable[WBA](dma_mem_dest_buf)
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	printf("#### Execute DMA TEST.06 : Cacheable[WT] Buffer --> Cacheable[WBA] Buffer ###\n");
	// Setting Test parameters. （source/destination address, and test_data(initial)）
	dst_address = (void*)dma_mem_dest_buf;			// Destination address of CPU View
	src_address = (void*)dma_mem_src_buf_wt;				// Source address of CPU View
	test_data = 0x66000000;
	// Execute DMA Transfer Test.
	result_memcmp = sample_dma_mem_test_execute(channel, dst_address, src_address, dst_axcache, src_axcache, dma_mem_xfer_size, test_data, acp_en, cacheope_en);
	printf("TEST.06 Verification Result ... %s\n\n", (result_memcmp == 0 ? "OK" : "NG"));


	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	//// TEST.07 : Cacheable[WB](dma_mem_src_buf_wb) --> Cacheable[WBA](dma_mem_dest_buf)
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	printf("#### Execute DMA TEST.07 : Cacheable[WB] Buffer --> Cacheable[WBA] Buffer ###\n");
	// Setting Test parameters. （source/destination address, and test_data(initial)）
	dst_address = (void*)dma_mem_dest_buf;			// Destination address of CPU View
	src_address = (void*)dma_mem_src_buf_wb;				// Source address of CPU View
	test_data = 0x77000000;
	// Execute DMA Transfer Test.
	result_memcmp = sample_dma_mem_test_execute(channel, dst_address, src_address, dst_axcache, src_axcache, dma_mem_xfer_size, test_data, acp_en, cacheope_en);
	printf("TEST.07 Verification Result ... %s\n\n", (result_memcmp == 0 ? "OK" : "NG"));

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	//// TEST.08 : Cacheable[WBA](dma_mem_src_buf_wba) --> Cacheable[WBA](dma_mem_dest_buf)
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	printf("#### Execute DMA TEST.08 : Cacheable[WBA] Buffer --> Cacheable[WBA] Buffer ###\n");
	// Setting Test parameters. （source/destination address, and test_data(initial)）
	dst_address = (void*)dma_mem_dest_buf;			// Destination address of CPU View
	src_address = (void*)dma_mem_src_buf_wba;			// Source address of CPU View
	test_data = 0x88000000;
	// Execute DMA Transfer Test.
	result_memcmp = sample_dma_mem_test_execute(channel, dst_address, src_address, dst_axcache, src_axcache, dma_mem_xfer_size, test_data, acp_en, cacheope_en);
	printf("TEST.08 Verification Result ... %s\n\n", (result_memcmp == 0 ? "OK" : "NG"));

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	printf("Finish all DMA TESTs!!! (01 ~ 08).\n");

	return;
}

/********************************************************************************//*!
 * @fn  static int sample_dma_mem_test_execute(ALT_DMA_CHANNEL_t channel, void* dst_address, void* src_address,
 * 											AXCACHEe dst_axcache, AXCACHEe src_axcache, size_t trans_size,
 * 											uint32_t test_data,	bool acp_en, bool cacheope_en)
 * @brief  Execute DMA Test.
 *
 * @details  Start the DMA transfer in the following conditions.
 *           - restart is false
 *             After generating a DMA CH parameter and see DMAC Micro Code to run the DMA transfer.
 *           - restart is true
 *             Restart the DMA transfers using DMA CH parameter has already been generated.
 * @param[in] channel  target DMA CH.
 * @param[in] evnet  Event number to notify when the transfer completes.
 * @param[in] test_p  Pointer to a DMA test parameters (info address, size, etc.).
 * @param[in] restart  Specify true to start the transfer by a restart.
 * @retval void  none
 *
 * @attention  When you first start calling this function in the restart = true. for each channels.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2015/03/02  ---  #18042  新規作成.
 **//*******************************************************************************/
static int sample_dma_mem_test_execute(	ALT_DMA_CHANNEL_t channel,	// DMA CH Number(0~7)
										void* dst_address,			// Destination address of CPU View
										void* src_address,			// Source address of CPU View
										AXCACHEe dst_axcache,		// AXI Signal of DMAC->DST (AWCACHE)
										AXCACHEe src_axcache,		// AXI Signal of SRC->DMAC (ARCACHE)
										size_t trans_size,			// DMA Transfer size
										uint32_t test_data,			// Inital value of TEST Data
										bool acp_en,				// ACP Enable?
										bool cacheope_en		)	// Cache Operation Enable?
{
	ALT_STATUS_CODE result_code;

	int result_memcmp = -1;					// Verification Result. (0:OK／!0:NG)
	volatile int polling_value;
	void* dma_dst_address = dst_address;	// Destination address of L3(DMAC) View
	void* dma_src_address = src_address;	// Source address of L3(DMAC) View
	// If ACP-Enable setting is true, Adding ACP window to transfer address.
	if(acp_en)
	{
#ifdef soc_cv_av
#ifdef GNU_COMPILER
    	dma_dst_address += ACP_WINDOW_BASE;
    	dma_src_address += ACP_WINDOW_BASE;
#endif
#ifdef ARM_COMPILER
    	dma_dst_address = (char*)dma_dst_address + ACP_WINDOW_BASE;
    	dma_src_address = (char*)dma_src_address + ACP_WINDOW_BASE;
#endif
#endif /* soc_cv_av */
	}

	printf("==== DMA Test Parameters ====\n");
	printf(" - DMA CH Select ..... ALT_DMA_CHANNEL_%d\n", (int)channel);
	printf(" - Source Address ........... 0x%08X\n", (int)dma_src_address);
	printf(" - Destination Address ...... 0x%08X\n", (int)dma_dst_address);
	printf(" - Transport size ........... 0x%08X\n", (int)trans_size);
#ifdef soc_cv_av
	printf(" - Transport via ACP Port ... %s\n", (acp_en ? "True" : "False"));
#else /* soc_cv_av */
	printf(" - Transport via ACP Port ... %s\n", "Auto");
#endif /* soc_cv_av */
	printf(" - Execute cache operation .. %s\n", (cacheope_en ? "True" : "False"));


	// Write test data to the source and destination 0 clear (Also effect to on-cash transfer area).
	util_time_record_start_point(0);
	memset(dst_address, 0, trans_size);
	sample_memset_incrementdata_4byte(src_address, test_data, trans_size);
	util_time_record_end_point(0);

	// Execute CACHE maintenace operation.
	// - ACP unused and valid cache (WriteBack) of when the DMA transfers data areas must be implemented cache maintenance process.
	// - DMA Descriptor (MicroCode space) cache maintenance is care within the HWLib API so cache maintenance unnecessary.
	util_time_record_start_point(1);
	if(cacheope_en)
	{
		alt_cache_system_purge(dst_address, trans_size);		/* Destination area off the cache */
		alt_cache_system_clean(src_address, trans_size);		/* Test data synchronized to the physical memory from the cache. */
	}
	util_time_record_end_point(1);

	// Programming DMA CHX
	util_time_record_start_point(2);
	result_code = alt_dma_channel_program_memory_to_memory(channel,
	                                        &dma_mem_program[channel],
	                                        dma_dst_address, dst_axcache,
	                                        dma_src_address, src_axcache,
	                                        trans_size);
	util_time_record_end_point(2);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_dma_channel_program_memory_to_memory(%d) .. result=%d (%s[%d]) \n", channel, (int)result_code, __FILE__, __LINE__);

	} else {

		// Execute DMA CHX
		util_time_record_start_point(3);
		polling_value = SAMPLE_CallBackCount[channel];
		result_code = alt_dma_channel_exec(channel, &dma_mem_program[channel]);

		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_dma_channel_exec(%d) .. result=%d (%s[%d]) \n", channel, (int)result_code, __FILE__, __LINE__);
		} else {

			// Polling the completion of DMA CHX.
			while(polling_value == SAMPLE_CallBackCount[channel])
			{
				__asm__ ("nop");
			};
			util_intlog_print();
		}
		// util_time_record_end_point(3); ... Implemented on the DMA completion interrupt callbacks.
		fflush(stdout);
	}

#if 0	/* The following cache operations are probably unnecessary. (Required if it on the destination buffer space to cache) */
	util_time_record_start_point(5);
	if(cacheope_en)
	{
		alt_cache_system_invalidate(dst_address, trans_size);
	}
	util_time_record_end_point(5);
#endif

	// Verification of DMA Transfer data.
	util_time_record_start_point(4);
	result_memcmp = memcmp(src_address, dst_address, trans_size);
	util_time_record_end_point(4);

	// Displays the processing times.
	util_time_print_result_partial(0, 4, enumUTIL_TIME_TARGET_by_SECOND);
	printf("\n");

	return result_memcmp;
}

/********************************************************************************//*!
 * @fn  static void sample_dma_mem_print_manager_status(void)
 * @brief  Print the DMA Manager status and information.
 *
 * @details  Run the following APIs.
 *           - alt_dma_manager_state_get
 *           - alt_dma_manager_fault_status_get
 * @param[in,out] void  none
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/26  ---  #13246  新規作成. 
 **//*******************************************************************************/
static void sample_dma_mem_print_manager_status(void)
{
	ALT_STATUS_CODE result_code;
	ALT_DMA_MANAGER_STATE_t dma_manager_state;
	ALT_DMA_MANAGER_FAULT_t dma_manager_fault_state;
	const char*	status_string_p;

	printf("==== Print DMA Manager Status ====\n");
	// -- DMA Manager Status --
	/* Usage:	ALT_STATUS_CODE  alt_dma_manager_state_get (ALT_DMA_MANAGER_STATE_t *state);	*/
	result_code = alt_dma_manager_state_get(&dma_manager_state);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
#ifdef GNU_COMPILER	// 99
	if((dma_manager_state < ALT_DMA_MANAGER_STATE_STOPPED)||(ALT_DMA_MANAGER_STATE_FAULTING < dma_manager_state)){
#endif
#ifdef ARM_COMPILER	// 99
	if(ALT_DMA_MANAGER_STATE_FAULTING < dma_manager_state){
#endif
		status_string_p = NULL;
	} else {
		status_string_p = StatusStringDMAMEM_Manager[dma_manager_state];
	}
	printf("alt_dma_manager_state_get()       : %s .... result=%d\n", status_string_p, (int)result_code);
	if(dma_manager_state == ALT_DMA_MANAGER_STATE_FAULTING){
		// -- DMA Manager Fault Status --
		/* Usage:	ALT_STATUS_CODE  alt_dma_manager_fault_status_get (ALT_DMA_MANAGER_FAULT_t *fault);	*/
		result_code = alt_dma_manager_fault_status_get(&dma_manager_fault_state);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: ");
		}
		printf("alt_dma_manager_fault_status_get(): 0x%08X .... result=%d\n", (int)dma_manager_fault_state, (int)result_code);
	}

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_dma_mem_print_ch_status(ALT_DMA_CHANNEL_t channel, bool detail)
 * @brief  Print the DMA Channel status and information.
 *
 * @details  Run the following APIs for specified DMA CH.
 *           - alt_dma_channel_state_get
 *           - alt_dma_channel_fault_status_get
 *           - alt_dma_event_int_status_get_raw
 *           - alt_dma_int_status_get
 *           - alt_dma_channel_reg_get
 *           - alt_dma_program_progress_reg
 * @param[in] channel  target DMA CH.
 * @param[in] detail  To display more information if true.
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/26  ---  #13246  新規作成. 
 **//*******************************************************************************/
static void sample_dma_mem_print_ch_status(ALT_DMA_CHANNEL_t channel, bool detail)
{
	ALT_STATUS_CODE result_code;
	ALT_DMA_CHANNEL_STATE_t dma_channel_state;
	ALT_DMA_CHANNEL_FAULT_t dma_channel_fault_state;
	ALT_DMA_EVENT_t evt_num;
	uint32_t reg_DMA_CCR;
	uint32_t reg_DMA_SAR;
	uint32_t reg_DMA_DAR;
	uint32_t progress;
	uint32_t progress_percent;
	const char*	status_string_p;

	printf("==== Print DMA Status [CH:%d] ====\n", (int)channel);
	// -- DMA CH Resources Status --
	/* Usage:	ALT_STATUS_CODE  alt_dma_channel_state_get (ALT_DMA_CHANNEL_t channel, ALT_DMA_CHANNEL_STATE_t *state);	*/
	result_code = alt_dma_channel_state_get(channel, &dma_channel_state);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
#ifdef GNU_COMPILER	// 99
	if((dma_channel_state < ALT_DMA_CHANNEL_STATE_STOPPED)||(ALT_DMA_CHANNEL_STATE_FAULTING < dma_channel_state)){
#endif
#ifdef ARM_COMPILER	// 99
	if(ALT_DMA_CHANNEL_STATE_FAULTING < dma_channel_state){
#endif
		status_string_p = NULL;
	} else {
		status_string_p = StatusStringDMAMEM_Channel[dma_channel_state];
	}
	printf("alt_dma_channel_state_get()       : %s .... result=%d\n", status_string_p, (int)result_code);
	if((dma_channel_state == ALT_DMA_CHANNEL_STATE_FAULTING_COMPLETING)||(dma_channel_state == ALT_DMA_CHANNEL_STATE_FAULTING)){
		// -- DMA CH Falut Status --
		/* Usage:	ALT_STATUS_CODE  alt_dma_channel_fault_status_get (ALT_DMA_CHANNEL_t channel, ALT_DMA_CHANNEL_FAULT_t *fault);	*/
		result_code = alt_dma_channel_fault_status_get(channel, &dma_channel_fault_state);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: ");
		}
		printf("alt_dma_channel_fault_status_get(): 0x%08X .... result=%d\n", (int)dma_channel_fault_state, (int)result_code);
	}

	if(detail){
		evt_num = (ALT_DMA_EVENT_t)(ALT_DMA_EVENT_0 + channel);
		// -- DMA CH event or int status --
		/* Usage:	ALT_STATUS_CODE  alt_dma_event_int_status_get_raw (ALT_DMA_EVENT_t evt_num);	*/
		printf("alt_dma_event_int_status_get_raw(): .... result=%d\n", (int)alt_dma_event_int_status_get_raw(evt_num));
		// -- DMA CH int status --
		/* Usage:	ALT_STATUS_CODE  alt_dma_int_status_get (ALT_DMA_EVENT_t irq_num);	*/
		printf("alt_dma_int_status_get()          : .... result=%d\n", (int)alt_dma_int_status_get(evt_num));
	}

	/* Usage:	ALT_STATUS_CODE  alt_dma_channel_reg_get (ALT_DMA_CHANNEL_t channel, ALT_DMA_PROGRAM_REG_t reg, uint32_t *val);	*/
	result_code = alt_dma_channel_reg_get(channel, ALT_DMA_PROGRAM_REG_CCR, &reg_DMA_CCR);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_dma_channel_reg_get(CCR)      : 0x%08X .... result=%d\n", (int)reg_DMA_CCR, (int)result_code);
	result_code = alt_dma_channel_reg_get(channel, ALT_DMA_PROGRAM_REG_SAR, &reg_DMA_SAR);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_dma_channel_reg_get(SAR)      : 0x%08X .... result=%d\n", (int)reg_DMA_SAR, (int)result_code);
	result_code = alt_dma_channel_reg_get(channel, ALT_DMA_PROGRAM_REG_DAR, &reg_DMA_DAR);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_dma_channel_reg_get(DAR)      : 0x%08X .... result=%d\n", (int)reg_DMA_DAR, (int)result_code);

	if(detail){
		/* Usage:	ALT_STATUS_CODE  alt_dma_program_progress_reg (ALT_DMA_PROGRAM_t *pgm, ALT_DMA_PROGRAM_REG_t reg, uint32_t current, uint32_t *progress);	*/
		result_code = alt_dma_program_progress_reg(&dma_mem_program[channel], ALT_DMA_PROGRAM_REG_SAR, reg_DMA_SAR, &progress);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: ");
		} else {
#ifdef GNU_COMPILER	// 99
			if((progress < 0)||(progress > dma_mem_xfer_size)){
#endif
#ifdef ARM_COMPILER	// 99
			if(progress > dma_mem_xfer_size){
#endif
				progress_percent = 0;
			} else {
				progress_percent = (uint32_t)((progress / dma_mem_xfer_size) * 100);
			}
		}
		printf("alt_dma_program_progress_reg(SAR) : 0x%08X (%3d%%) .... result=%d\n", (int)progress, (int)progress_percent, (int)result_code);
		
		result_code = alt_dma_program_progress_reg(&dma_mem_program[channel], ALT_DMA_PROGRAM_REG_DAR, reg_DMA_DAR, &progress);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: ");
		} else {
#ifdef GNU_COMPILER	// 99
			if((progress < 0)||(progress > dma_mem_xfer_size)){
#endif
#ifdef ARM_COMPILER	// 99
			if(progress > dma_mem_xfer_size){
#endif
				progress_percent = 0;
			} else {
				progress_percent = (uint32_t)((progress / dma_mem_xfer_size) * 100);
			}
		}
		printf("alt_dma_program_progress_reg(DAR) : 0x%08X (%3d%%) .... result=%d\n", (int)progress, (int)progress_percent, (int)result_code);
	}

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_dma_mem_print_all_status(void)
 * @brief  Print all DMA status and informations.
 *
 * @details  Run the following Functions.
 *           - sample_dma_mem_print_manager_status
 *           - sample_dma_mem_print_ch_status (for all DMA CHs)
 * @param[in,out] void  none
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/26  ---  #13246  新規作成. 
 **//*******************************************************************************/
static void sample_dma_mem_print_all_status(void)
{
	int	i;

	printf("======== Print All DMA Status ========\n");
	// -- DMA Manager Status --
	sample_dma_mem_print_manager_status();
	// -- DMA CH Resources Status --
	for(i=0; i<8; i++){
		// -- DMA CH Status --
		sample_dma_mem_print_ch_status((ALT_DMA_CHANNEL_t)i, false);
	}

	return;
}

/********************************************************************************//*!
 * @fn  void sample_dma_mem_callback_xxxx(uint32_t icciar, void *context)
 * @brief  DMA interrupt callback functions.
 *
 * @details  working opportunity an interrupt from DMAC, indicate
 *           an interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued.
 * @param[in] context  The user provided context.
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/26  ---  #13246  新規作成. 
 **//*******************************************************************************/
void sample_dma_mem_callback_DMA_IRQ0(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_0);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_0) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ0);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ0) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_CallBackCount[0]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ0 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_CallBackCount[0]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ0, (int)icciar, (int)SAMPLE_CallBackCount[0]);
	return;
}
void sample_dma_mem_callback_DMA_IRQ1(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_1);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_1) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ1);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ1) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_CallBackCount[1]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ1 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_CallBackCount[1]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ1, (int)icciar, (int)SAMPLE_CallBackCount[1]);
	return;
}
void sample_dma_mem_callback_DMA_IRQ2(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_2);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_2) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ2);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ2) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_CallBackCount[2]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ2 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_CallBackCount[2]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ2, (int)icciar, (int)SAMPLE_CallBackCount[2]);
	return;
}
void sample_dma_mem_callback_DMA_IRQ3(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_3);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_3) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ3);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ3) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_CallBackCount[3]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ3 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_CallBackCount[3]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ3, (int)icciar, (int)SAMPLE_CallBackCount[3]);
	return;
}
void sample_dma_mem_callback_DMA_IRQ4(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_4);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_4) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ4);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ4) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_CallBackCount[4]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ4 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_CallBackCount[4]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ4, (int)icciar, (int)SAMPLE_CallBackCount[4]);
	return;
}
void sample_dma_mem_callback_DMA_IRQ5(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_5);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_5) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ5);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ5) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_CallBackCount[5]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ5 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_CallBackCount[5]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ5, (int)icciar, (int)SAMPLE_CallBackCount[5]);
	return;
}
void sample_dma_mem_callback_DMA_IRQ6(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_6);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_6) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ6);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ6) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_CallBackCount[6]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ6 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_CallBackCount[6]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ6, (int)icciar, (int)SAMPLE_CallBackCount[6]);
	return;
}
void sample_dma_mem_callback_DMA_IRQ7(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_7);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_7) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ7);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ7) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_CallBackCount[7]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ7 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_CallBackCount[7]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ7, (int)icciar, (int)SAMPLE_CallBackCount[7]);
	return;
}
void sample_dma_mem_callback_DMA_IRQ_ABORT(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_ABORT);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_ABORT) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ_ABORT);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ_ABORT) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_CallBackCount[8]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ_ABORT is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_CallBackCount[8]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ_ABORT, (int)icciar, (int)SAMPLE_CallBackCount[8]);

	//sample_dma_mem_print_all_status();

	return;
}
void sample_dma_mem_callback_DMA_ECC_CORRECTED_IRQ(uint32_t icciar, void *context)
{
#ifdef soc_cv_av
	ALT_STATUS_CODE result_code;
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_ECC_CORRECTED_IRQ);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_ECC_CORRECTED_IRQ) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_CallBackCount[9]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_ECC_CORRECTED_IRQ is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_CallBackCount[9]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_ECC_CORRECTED_IRQ, (int)icciar, (int)SAMPLE_CallBackCount[9]);
#endif /* soc_cv_av */
	return;
}
void sample_dma_mem_callback_DMA_ECC_UNCORRECTED_IRQ(uint32_t icciar, void *context)
{
#ifdef soc_cv_av
	ALT_STATUS_CODE result_code;
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_ECC_UNCORRECTED_IRQ);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_ECC_UNCORRECTED_IRQ) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_CallBackCount[10]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_ECC_UNCORRECTED_IRQ is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_CallBackCount[10]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_ECC_UNCORRECTED_IRQ, (int)icciar, (int)SAMPLE_CallBackCount[10]);
#endif /* soc_cv_av */
	return;
}

/********************************************************************************//*!
 * @fn  void sample_dma_mem_setting_gic(void)
 * @brief  Setting of GIC (DMA interrupt).
 *
 * @details    In order to receive a DMA interrupt setting of GIC.
 * @param[in,out] void  none
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/26  ---  #13246  新規作成. 
 **//*******************************************************************************/
static void sample_dma_mem_setting_gic(void)
{
	ALT_STATUS_CODE result_code;

    //	ALT_INT_INTERRUPT_DMA_IRQ0                = 136, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ1                = 137, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ2                = 138, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ3                = 139, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ4                = 140, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ5                = 141, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ6                = 142, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ7                = 143, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ_ABORT           = 144, /*!< # */
    //	soc_cv_av:	ALT_INT_INTERRUPT_DMA_ECC_CORRECTED_IRQ   = 145, /*!< # */
    //	soc_cv_av:	ALT_INT_INTERRUPT_DMA_ECC_UNCORRECTED_IRQ = 146, /*!< # */
    //	soc_a10:	ALT_INT_INTERRUPT_SERR_GLOBAL = 34, /*!< # */
    //	soc_a10:	ALT_INT_INTERRUPT_DERR_GLOBAL = 32, /*!< # */
    //	/*!<
    //	 * Interrupts sourced from the DMA Controller.
    //	 *  * All interrupts in this group are level triggered.
    //	 */

	int i;

	for(i=ALT_INT_INTERRUPT_DMA_IRQ0; i<=ALT_INT_INTERRUPT_DMA_IRQ_ABORT; i++)
	{
		/* Usage: ALT_STATUS_CODE alt_int_isr_register(ALT_INT_INTERRUPT_t int_id, alt_int_callback_t callback, void* context);	*/
		result_code = alt_int_isr_register((ALT_INT_INTERRUPT_t)i, SAMPLE_CallBacks[(i - ALT_INT_INTERRUPT_DMA_IRQ0)], NULL);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_int_isr_register(%d) result=%d\n", i, (int)result_code);
		}

		/* Usage: ALT_STATUS_CODE alt_int_dist_target_set(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t target);	*/
		/* ##'target=1' the specified notification interrupts to only core0.  */
		result_code = alt_int_dist_target_set((ALT_INT_INTERRUPT_t)i, 1);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_int_dist_target_set(%d, 1) result=%d\n", i, (int)result_code);
		}

		/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_set(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t trigger_type);	*/
		result_code = alt_int_dist_trigger_set((ALT_INT_INTERRUPT_t)i, ALT_INT_TRIGGER_LEVEL);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_int_dist_trigger_set(%d, ALT_INT_TRIGGER_LEVEL) result=%d\n", i, (int)result_code);
		}

		/* Usage: ALT_STATUS_CODE alt_int_dist_priority_set(ALT_INT_INTERRUPT_t int_id, uint32_t priority);	*/
		result_code = alt_int_dist_priority_set((ALT_INT_INTERRUPT_t)i, 0);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_int_dist_priority_set(%d, 0) result=%d\n", i, (int)result_code);
		}

		// To reset the counter interrupt detection.
		SAMPLE_CallBackCount[(i - ALT_INT_INTERRUPT_DMA_IRQ0)] = 0;

		if(i <= ALT_INT_INTERRUPT_DMA_IRQ_ABORT)
		{
			// To specify the notification methods (send events or interrupt notification) for each DMA resources of the DMA controller.
			/* Usage:	ALT_STATUS_CODE  alt_dma_event_int_select (ALT_DMA_EVENT_t evt_num, ALT_DMA_EVENT_SELECT_t opt);	*/
			result_code = alt_dma_event_int_select((ALT_DMA_EVENT_t)(ALT_DMA_EVENT_0 + i - ALT_INT_INTERRUPT_DMA_IRQ0), ALT_DMA_EVENT_SELECT_SIG_IRQ);
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_dma_event_int_select(%d, ALT_DMA_EVENT_SELECT_SIG_IRQ) result=%d\n", (int)(ALT_DMA_EVENT_0 + i - ALT_INT_INTERRUPT_DMA_IRQ0), (int)result_code);
			}
		}

		/* No need to call the API below. Because all interrupts are secure by default. */
		/* Usage: ALT_STATUS_CODE  alt_int_dist_secure_enable (ALT_INT_INTERRUPT_t int_id);	*/

		/* Usage: ALT_STATUS_CODE  alt_int_dist_enable (ALT_INT_INTERRUPT_t int_id);	*/
		result_code = alt_int_dist_enable((ALT_INT_INTERRUPT_t)i);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_int_dist_enable(%d) result=%d\n", i, (int)result_code);
		}

	}

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_dma_mem_setting_testdata(uint32_t* start, uint32_t* endnext)
 * @brief  Writes the data for transfer test.
 *
 * @details  Writes the address value area of memory that is specified for the transfer test.
 * @param[in] start  start address
 * @param[in] endnext  Next to the end address
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/26  ---  #13246  新規作成.
 **//*******************************************************************************/
static void sample_memset_address32(uint32_t* start, size_t size)
{
#ifdef GNU_COMPILER
	uint32_t* endnext = (void*)start + size;
#endif
#ifdef ARM_COMPILER
	int i_start = (int)(uint32_t*)start;
	int i_size  = (size_t)size;
	int i_end = i_start + i_size;
	uint32_t* endnext = (uint32_t*)(int)i_end;
#endif
	
	while(start < endnext){
		*start = (uint32_t)start;
		start++;
		if(start < (uint32_t*)0x00000004)	break;
	}

	return;
}

static void sample_memset_incrementdata(uint32_t* start, uint32_t testdata, size_t size)
{
#ifdef GNU_COMPILER
	uint32_t* endnext = (void*)start + size;
#endif
#ifdef ARM_COMPILER
	int i_start = (int)(uint32_t*)start;
	int i_size  = (size_t)size;
	int i_end = i_start + i_size;
	uint32_t* endnext = (uint32_t*)(int)i_end;
#endif
	
	while(start < endnext){
		*start = testdata++;
		start++;
		if(start < (uint32_t*)0x00000004)	break;
	}

	return;
}

static void sample_memset_incrementdata_4byte(uint32_t* start, uint32_t testdata, size_t size)
{
#ifdef GNU_COMPILER
	uint32_t* endnext = (void*)start + size;
#endif
#ifdef ARM_COMPILER
	int i_start = (int)(uint32_t*)start;
	int i_size  = (size_t)size;
	int i_end = i_start + i_size;
	uint32_t* endnext = (uint32_t*)(int)i_end;
#endif

	while(start < endnext){
		*start++ = testdata;
		testdata += sizeof(uint32_t);
		if(start < (uint32_t*)0x00000004)	break;
	}

	return;
}


/****************************************/
/*	Test Command (DMA memory to memory)	*/
/****************************************/
int sample_dma_mem_test_cmd(char* options)
{
	//ALT_STATUS_CODE result_code;
	uint32_t switch_raw = 0;
	uint32_t switch_raw_bk = 0;
	uint32_t switch_detect = 0;
	uint32_t switch_detect_on = 0;
	uint32_t switch_number = 0;
	bool disp_usage = true;
	
	ALT_DMA_CHANNEL_t channel;	// DMA CH番号(0~7)
	bool acp_en;				// ACP有効？
	bool cacheope_en;	// CACHE操作有効？ 
	
	printf("\r\nUser Application Start!\r\n");
	
	// CPU and board settings.
	util_intlog_init();
#if 0	// main() 内で cpu0_init() 実施済み
	cpu0_init();
#endif

	// Initializing the dmac functions of hwlib. 
	sample_dma_mem_test_init();
	
	/* ################################################ */
	/* ## Implement the test setting process here!!! ## */
	/* ################################################ */
	// Print address of DMA target area.
	printf("== List of TEST Buffers ==\n");
	printf("- dma_mem_src_buf_nc (non-cache) ............. addr:0x%08X\n", (int)dma_mem_src_buf_nc);
	printf("- dma_mem_dest_buf_nc (non-cache) ............ addr:0x%08X\n", (int)dma_mem_dest_buf_nc);
	printf("- dma_mem_src_buf_wt (write-through) ......... addr:0x%08X\n", (int)dma_mem_src_buf_wt);
	printf("- dma_mem_src_buf_wb (write-back) ............ addr:0x%08X\n", (int)dma_mem_src_buf_wb);
	printf("- dma_mem_src_buf_wba (write-back/allocate) .. addr:0x%08X\n", (int)dma_mem_src_buf_wba);
	printf("- dma_mem_dest_buf (write-back/allocate) ..... addr:0x%08X\n", (int)dma_mem_dest_buf);
	printf("(DMA_TEST_SIZE = 0x%08X)\n", (int)DMA_TEST_SIZE);
	printf("\n");

	util_intlog_print();
	
	printf("==== Start While(1) loop process!!! ============\n");
	switch_raw_bk = sample_detect_switch();
	switch_raw_bk ^= SAMPLE_SWITCH_BIT_SLIDE0;
	while(1)
	{
		if(disp_usage)
		{
			sample_dma_mem_test_print_usage();
			disp_usage = false;
		}

		// ---- Check the Slide-Switch and Push-Switch. ---- 
		switch_raw = sample_detect_switch();
		switch_detect = switch_raw ^ switch_raw_bk;
		switch_detect_on  |= switch_detect & switch_raw;
		
		// Slide-Switch 
		if(switch_detect & SAMPLE_SWITCH_BIT_SLIDEALL)
		{
			switch_number = switch_raw & SAMPLE_SWITCH_BIT_NUM;
			switch_number *= 1;		// To avoid warnings.

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE0){
				//printf("SAMPLE_SWITCH_BIT_SLIDE0\n");
				// Select TEST Option: Cache Maintenance (ON:Enable/OFF:Disable)
				if(switch_number & SAMPLE_SWITCH_BIT_SLIDE0){
					cacheope_en = true;
				} else {
					cacheope_en = false;
				}
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE1){
				//printf("SAMPLE_SWITCH_BIT_SLIDE1\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE2){
				//printf("SAMPLE_SWITCH_BIT_SLIDE2\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE3){
				//printf("SAMPLE_SWITCH_BIT_SLIDE3\n");
			}
			
			// Select test settings from DIPSW value. 
			channel = (ALT_DMA_CHANNEL_t)(ALT_DMA_CHANNEL_0 + (int)(switch_number >> 1));

			disp_usage = false;
		}
		
		
		// Push-Switch 0
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH0)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH0;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH0\n");
			
			//sample_dma_mem_print_all_status();
			break;	// Exit Push-Switch 0 and 1 becomes ON.
		}
		// Push-Switch 1
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH1)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH1;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH1\n");
			
			sample_dma_mem_print_manager_status();
			sample_dma_mem_print_ch_status(channel, true);
			disp_usage = true;
		}
		// Push-Switch 2
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH2)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH2;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH2\n");
			
			acp_en = false;
			sample_dma_mem_test_main(channel, acp_en, cacheope_en);
			disp_usage = true;
		}
		// Push-Switch 3
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH3)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH3;

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH3\n");
			
			acp_en = true;
			sample_dma_mem_test_main(channel, acp_en, cacheope_en);
			disp_usage = true;
		}
		
		util_intlog_print();

		switch_raw_bk = switch_raw;
	}
	printf("==== End While(1) loop process. ================\n");

	util_intlog_print();

	printf("Finished running the sample !!!\r\n");

	return 0;
}
/***********************************************************************************
 * end of file 
 ***********************************************************************************/
