﻿/********************************************************************************//*!
 * @file  sample_watchdog.c 
 * @brief  To use HWLib for the watchdog timer (and reset manager) example program. 
 * 
 * @details  Run the API category below.
 *           - The Watchdog Timer Manager API  
 *               - Watchdog Timer Enable, Disable, Restart, Status  
 *               - Watchdog Timer Counter Configuration  
 *               - Watchdog Timer Interrupt Management  
 *               - Watchdog Timer Miscellaneous Configuration 
 *           - The Reset Manager  
 *               - Reset Status  
 *               - Reset Control  
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 * @date 2014/08/18  ---  -----  v14.0化 (誤植訂正、操作説明の表示を追加). 
 * @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
 * @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
 * @date 2018/07/02  ---  #25940  All in One ベアメタル・アプリケーション v17.1 への対応を実施.
 * @date 2019/01/18  ---  -----   All in One ベアメタル・アプリケーション v18.1 への対応を実施.
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes 
 ***********************************************************************************/
#include <stdio.h>
#include <string.h>
#include "hwlib.h"
#include "socal/socal.h"
#include "socal/hps.h"
#include "alt_generalpurpose_io.h"
#include "alt_globaltmr.h"
#include "alt_clock_manager.h"
#include "alt_interrupt.h"
#include "alt_watchdog.h"
#include "alt_reset_manager.h"

#include "sample_app_setting.h"
#include "util_time_measurement.h"
#include "util_interrupt_log.h"
#include "alt_printf.h"

/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/
static void sample_wdog_print_specific_wdt_counter(ALT_WDOG_TIMER_t tmr_id);
static void sample_wdog_print_specific_wdt_settings(ALT_WDOG_TIMER_t tmr_id);
static void sample_wdog_print_all_wdt_settings(void);
static void sample_wdog_print_all_wdt_settings2(void);
static void sample_wdog_start_all(void);
static void sample_wdog_stop_all(void);
static void sample_wdog_ienable_all(void);
static void sample_wdog_idisable_all(void);
static void sample_wdog_print_setting_gic(ALT_INT_INTERRUPT_t int_id);
static void sample_wdog_print_all_int_settings(void);
void sample_wdog_test_init(void);
void sample_wdog_test_uninit(void);
void sample_wdog_setting_gic(void);
void sample_wdog_callback_CPU(uint32_t icciar, void *context);
void sample_wdog_callback_WDOG0(uint32_t icciar, void *context);
void sample_wdog_callback_WDOG1(uint32_t icciar, void *context);
void sample_wdog_test_print_usage(void);
void sample_wdog_test_main(void);
int  sample_wdog_test_cmd(char* options);
// prototype (Common Functions) //
static void sample_resetmng_print_trigger(void);

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/

/***********************************************************************************
 *  variables 
 ***********************************************************************************/

/***********************************************************************************
 *  functions 
 ***********************************************************************************/

/********************************************************************************//*!
 * @fn  static void sample_wdog_print_specific_wdt_counter(ALT_WDOG_TIMER_t tmr_id) 
 * @brief  Display the value of specific watchdog timer count registers. 
 * 
 * @details  Displays the register value that you can get with the following APIs.
 *           - alt_wdog_counter_get_current(). 
 *           - alt_wdog_counter_get_init(). 
 *           - alt_wdog_counter_get_max(). 
 *           - alt_wdog_counter_get_current_millisecs(). 
 *           - alt_wdog_counter_get_init_millisecs(). 
 *           - alt_wdog_counter_get_max_millisecs(). 
 * @param[in] tmr_id  watchdog timer identification types. 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_wdog_print_specific_wdt_counter(ALT_WDOG_TIMER_t tmr_id)
{
	/* Usage:  uint32_t  alt_wdog_counter_get_current (ALT_WDOG_TIMER_t tmr_id); 	*/
	printf("CounterValue[cur/ini/max]= 0x%08X / 0x%08X / 0x%08X\n",
			(int)alt_wdog_counter_get_current(tmr_id),
			(int)alt_wdog_counter_get_init(tmr_id),
			(int)alt_wdog_counter_get_max(tmr_id));
	printf("MilliSeconds[cur/ini/max]= %d / %d /%d\n",
			(int)alt_wdog_counter_get_curtime_millisecs(tmr_id),
			(int)alt_wdog_counter_get_inittime_millisecs(tmr_id),
			(int)alt_wdog_counter_get_max_millisecs(tmr_id));

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_wdog_print_specific_wdt_settings(ALT_WDOG_TIMER_t tmr_id) 
 * @brief  Display the value of specific watchdog timer settings. 
 * 
 * @details  Displays timer settings that you can get with the various APIs.
 * @param[in] tmr_id  watchdog timer identification types. 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_wdog_print_specific_wdt_settings(ALT_WDOG_TIMER_t tmr_id)
{
	uint64_t	ui64val;
	uint32_t	ui32val;

	if(tmr_id == ALT_WDOG_CPU){
		/* Usage:  uint32_t  alt_wdog_core_prescaler_get (void); 	*/
		printf("alt_wdog_core_prescaler_get() ............. result=%d\n", (int)alt_wdog_core_prescaler_get());
	} else {
		/* Usage:  uint32_t  alt_wdog_compcode_get (ALT_WDOG_TIMER_t tmr_id); 	*/
		ui32val = alt_wdog_compcode_get(tmr_id);
		printf("alt_wdog_compcode_get() ................... result=0x%08X(%d)\n", (int)ui32val, (int)ui32val);
		/* Usage:  uint32_t  alt_wdog_ver_get (ALT_WDOG_TIMER_t tmr_id); 	*/
		ui32val = alt_wdog_ver_get(tmr_id);
		printf("alt_wdog_ver_get() ........................ result=0x%08X(%d)\n", (int)ui32val, (int)ui32val);
	}
	/* Usage:  bool  alt_wdog_tmr_is_enabled (ALT_WDOG_TIMER_t tmr_id); 	*/
	printf("alt_wdog_tmr_is_enabled() ................. result=%d\n", (int)alt_wdog_tmr_is_enabled(tmr_id));
	/* Usage:  uint32_t  alt_wdog_counter_get_current (ALT_WDOG_TIMER_t tmr_id); 	*/
	ui32val = alt_wdog_counter_get_current(tmr_id);
	printf("alt_wdog_counter_get_current() ............ result=0x%08X(%d)\n", (int)ui32val, (int)ui32val);
	/* Usage:  uint32_t  alt_wdog_counter_get_curtime_millisecs (ALT_WDOG_TIMER_t tmr_id); 	*/
	ui32val = alt_wdog_counter_get_curtime_millisecs(tmr_id);
	printf("alt_wdog_counter_get_curtime_millisecs() .. result=%d[msec]\n", (int)ui32val);
	/* Usage:  uint32_t  alt_wdog_counter_get_init (ALT_WDOG_TIMER_t tmr_id); 	*/
	ui32val = alt_wdog_counter_get_init(tmr_id);
	printf("alt_wdog_counter_get_init() ............... result=0x%08X(%d)\n", (int)ui32val, (int)ui32val);
	/* Usage:  uint64_t  alt_wdog_counter_get_inittime_nanosecs (ALT_WDOG_TIMER_t tmr_id); 	*/
	ui64val = alt_wdog_counter_get_inittime_nanosecs(tmr_id);
	printf("alt_wdog_counter_get_inittime_nanosecs() .. result=%lld[nsec]\n", (long long int)ui64val);
	/* Usage:  uint32_t  alt_wdog_counter_get_inittime_millisecs (ALT_WDOG_TIMER_t tmr_id); 	*/
	ui32val = alt_wdog_counter_get_inittime_millisecs(tmr_id);
	printf("alt_wdog_counter_get_inittime_millisecs() . result=%d[msec]\n", (int)ui32val);
	/* Usage:  uint32_t  alt_wdog_counter_get_max (ALT_WDOG_TIMER_t tmr_id); 	*/
	ui32val = alt_wdog_counter_get_max(tmr_id);
	printf("alt_wdog_counter_get_max() ................ result=0x%08X(%d)\n", (int)ui32val, (int)ui32val);
	/* Usage:  uint64_t  alt_wdog_counter_get_max_nanosecs (ALT_WDOG_TIMER_t tmr_id); 	*/
	ui64val = alt_wdog_counter_get_max_nanosecs(tmr_id);
	printf("alt_wdog_counter_get_max_nanosecs() ....... result=%lld[nsec]\n", (long long int)ui64val);
	/* Usage:  uint32_t  alt_wdog_counter_get_max_millisecs (ALT_WDOG_TIMER_t tmr_id); 	*/
	ui32val = alt_wdog_counter_get_max_millisecs(tmr_id);
	printf("alt_wdog_counter_get_max_millisecs() ...... result=%d[msec]\n", (int)ui32val);
	/* Usage:  bool  alt_wdog_int_is_pending (ALT_WDOG_TIMER_t tmr_id); 	*/
	printf("alt_wdog_int_is_pending() ................. result=%d\n", (int)alt_wdog_int_is_pending(tmr_id));
	/* Usage:  bool  alt_wdog_int_is_enabled (ALT_WDOG_TIMER_t tmr_id); 	*/
	printf("alt_wdog_int_is_enabled() ................. result=%d\n", (int)alt_wdog_int_is_enabled(tmr_id));
	/* Usage:  int32_t  alt_wdog_response_mode_get (ALT_WDOG_TIMER_t tmr_id); 	*/
	ui32val = alt_wdog_response_mode_get(tmr_id);
	printf("alt_wdog_response_mode_get() .............. result=%d\n", (int)alt_wdog_int_is_enabled(tmr_id));

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_wdog_print_all_wdt_settings(void) 
 * @brief  Display the value of all watchdog timer settings. 
 * 
 * @details  Call sample_wdog_print_specific_wdt_settings for the watchdog timers of all.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_wdog_print_all_wdt_settings(void)
{
	printf("<< ALT_WDOG_CPU >>\n");
	sample_wdog_print_specific_wdt_settings(ALT_WDOG_CPU);
	printf("\n");

	printf("<< ALT_WDOG0 >>\n");
	sample_wdog_print_specific_wdt_settings(ALT_WDOG0);
	printf("\n");

	printf("<< ALT_WDOG1 >>\n");
	sample_wdog_print_specific_wdt_settings(ALT_WDOG1);
	printf("\n");

	printf("<< ALT_WDOG0_INIT (=ALT_WDOG0) >>\n");
	sample_wdog_print_specific_wdt_settings(ALT_WDOG0_INIT);
	printf("\n");

	printf("<< ALT_WDOG1_INIT (=ALT_WDOG1) >>\n");
	sample_wdog_print_specific_wdt_settings(ALT_WDOG1_INIT);
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_wdog_print_all_wdt_settings2(void) 
 * @brief  Display the value of all watchdog timer settings (Except for the initial timer). 
 * 
 * @details  Call sample_wdog_print_specific_wdt_settings for the watchdog 
 *           timers except for the initial timer.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_wdog_print_all_wdt_settings2(void)
{
	printf("<< ALT_WDOG_CPU >>\n");
	sample_wdog_print_specific_wdt_settings(ALT_WDOG_CPU);
	printf("\n");

	printf("<< ALT_WDOG0 >>\n");
	sample_wdog_print_specific_wdt_settings(ALT_WDOG0);
	printf("\n");

	printf("<< ALT_WDOG1 >>\n");
	sample_wdog_print_specific_wdt_settings(ALT_WDOG1);
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_wdog_start_all(void) 
 * @brief  Start the timers for the watchdog timers.
 * 
 * @details  Trial related to the watchdog timer start APIs.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_wdog_start_all(void)
{
	ALT_STATUS_CODE	result_code;

	/* Usage:  ALT_STATUS_CODE  alt_wdog_start (ALT_WDOG_TIMER_t tmr_id); 	*/
	printf("-- HWLib test : alt_wdog_start() --\n");
	printf(" - ALT_WDOG_CPU timer start!!\n");
	result_code = alt_wdog_start(ALT_WDOG_CPU);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_start(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_WDOG0 timer start!!\n");
	result_code = alt_wdog_start(ALT_WDOG0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_start(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_WDOG1 timer start!!\n");
	result_code = alt_wdog_start(ALT_WDOG1);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_start(); result=%d\n", (int)result_code);
	}
	printf("\n");


	return;
}

/********************************************************************************//*!
 * @fn  static void sample_wdog_stop_all(void) 
 * @brief  Stop the timers for the watchdog timers.
 * 
 * @details  Trial related to the watchdog timer stop APIs.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_wdog_stop_all(void)
{
	ALT_STATUS_CODE	result_code;

	/* Usage:  ALT_STATUS_CODE  alt_wdog_stop (ALT_WDOG_TIMER_t tmr_id); 	*/
	printf("-- HWLib test : alt_wdog_stop() --\n");
	printf(" - ALT_WDOG_CPU timer stop!!\n");
	result_code = alt_wdog_stop(ALT_WDOG_CPU);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_stop(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_WDOG0 timer stop!!\n");
	result_code = alt_wdog_stop(ALT_WDOG0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_stop(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_WDOG1 timer stop!!\n");
	result_code = alt_wdog_stop(ALT_WDOG1);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_stop(); result=%d\n", (int)result_code);
	}
	printf("\n");


	return;
}

/********************************************************************************//*!
 * @fn  static void sample_wdog_ienable_all(void) 
 * @brief  Enabling interrupts of all watchdog timers. 
 * 
 * @details  Trial related to the interrupt enable APIs.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_wdog_ienable_all(void)
{
	ALT_STATUS_CODE	result_code;

	/* Usage:  bool  alt_wdog_int_if_pending_clear (ALT_WDOG_TIMER_t tmr_id); 	*/
	printf("-- HWLib test : alt_wdog_int_if_pending_clear() --\n");
	printf("alt_wdog_int_if_pending_clear(ALT_WDOG_CPU); result=%d\n", (int)alt_wdog_int_if_pending_clear(ALT_WDOG_CPU));
	printf("alt_wdog_int_if_pending_clear(ALT_WDOG0); result=%d\n", (int)alt_wdog_int_if_pending_clear(ALT_WDOG0));
	printf("alt_wdog_int_if_pending_clear(ALT_WDOG1); result=%d\n", (int)alt_wdog_int_if_pending_clear(ALT_WDOG1));
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_enable(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_enable(ALT_INT_INTERRUPT_PPI_TIMER_WATCHDOG);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable(PPI_TIMER_WATCHDOG) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_enable(ALT_INT_INTERRUPT_WDOG0_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable(WDOG0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_enable(ALT_INT_INTERRUPT_WDOG1_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable(WDOG1_IRQ) result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_int_dist_secure_enable(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_secure_enable(ALT_INT_INTERRUPT_PPI_TIMER_WATCHDOG);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable(PPI_TIMER_WATCHDOG) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_secure_enable(ALT_INT_INTERRUPT_WDOG0_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable(WDOG0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_secure_enable(ALT_INT_INTERRUPT_WDOG1_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable(WDOG1_IRQ) result=%d\n", (int)result_code);
	}

	/* Usage:  ALT_STATUS_CODE  alt_wdog_int_enable (ALT_WDOG_TIMER_t tmr_id); 	*/
	printf("-- HWLib test : alt_wdog_int_enable() --\n");
	printf(" - ALT_WDOG_CPU timer interrupt enable!!\n");
	result_code = alt_wdog_int_enable(ALT_WDOG_CPU);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_int_enable(); result=%d\n", (int)result_code);
	}

	printf("\n");


	return;
}

/********************************************************************************//*!
 * @fn  static void sample_wdog_idisable_all(void) 
 * @brief  Disabling interrupts of all watchdog timers. 
 * 
 * @details  Trial related to the interrupt disable APIs. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_wdog_idisable_all(void)
{
	ALT_STATUS_CODE	result_code;

	/* Usage:  ALT_STATUS_CODE  alt_wdog_int_disable (ALT_WDOG_TIMER_t tmr_id); 	*/
	printf("-- HWLib test : alt_wdog_int_disable() --\n");
	printf(" - ALT_WDOG_CPU timer interrupt disable!!\n");
	result_code = alt_wdog_int_disable(ALT_WDOG_CPU);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_int_disable(); result=%d\n", (int)result_code);
	}
	printf("\n");


	return;
}

/********************************************************************************//*!
 * @fn  void sample_wdog_test_init(void) 
 * @brief  Initializing the sample program. 
 * 
 * @details  Call the initialization function for the watchdog timer API. 
 *           Other, do the following.
 *           -# Print and clear the reset factor. 
 *           -# Configure the Helio Board GPIO settings. 
 *           -# Setting of watchdog timers (Use HWLib). 
 *           -# Print watchdog timer settings. 
 *           -# Setting of GIC (Watchdog timer interrupt). 
 *           -# Print GIC settings of all watchdog timer interrupts. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_wdog_test_init(void)
{
	ALT_STATUS_CODE result_code;

	sample_resetmng_print_trigger();

#if 0	// main() 内で cpu0_init() 実施済み
	cpu0_init();
#endif

	/* Usage:  ALT_STATUS_CODE  alt_wdog_init (void); 	*/
	printf("-- HWLib test : alt_wdog_init() --\n");
	result_code = alt_wdog_init();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_init() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	printf("======== Print Initial WDT Settings ========\n");
	sample_wdog_print_all_wdt_settings();

	/* Usage:  ALT_STATUS_CODE  alt_wdog_core_prescaler_set (uint32_t val); 	*/
	printf("-- HWLib test : alt_wdog_core_prescaler_set(0) --\n");
	result_code = alt_wdog_core_prescaler_set(0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_core_prescaler_set(0) result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage:  ALT_STATUS_CODE  alt_wdog_counter_set (ALT_WDOG_TIMER_t tmr_id, uint32_t val); 	*/
	printf("-- HWLib test : alt_wdog_counter_set() --\n");
	printf(" - ALT_WDOG_CPU counter set to 1000000000\n");
	result_code = alt_wdog_counter_set(ALT_WDOG_CPU, 1000000000);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_counter_set(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_WDOG0 counter set to 15(0x80000000)\n");
	result_code = alt_wdog_counter_set(ALT_WDOG0, 15);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_counter_set(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_WDOG0 counter set to 14(0x40000000)\n");
	result_code = alt_wdog_counter_set(ALT_WDOG1, 14);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_counter_set(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_WDOG0 counter set to 13(0x20000000)\n");
	result_code = alt_wdog_counter_set(ALT_WDOG0_INIT, 13);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_counter_set(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_WDOG0 counter set to 12(0x10000000)\n");
	result_code = alt_wdog_counter_set(ALT_WDOG1_INIT, 12);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_counter_set(); result=%d\n", (int)result_code);
	}
	printf("\n");

	/* Usage:  ALT_STATUS_CODE  alt_wdog_response_mode_set (ALT_WDOG_TIMER_t tmr_id, ALT_WDOG_RESET_TYPE_t type); 	*/
	printf("-- HWLib test : alt_wdog_response_mode_set() --\n");
	printf(" - ALT_WDOG_CPU response mode set to ALT_WDOG_TIMER_MODE_ONESHOT\n");
	result_code = alt_wdog_response_mode_set(ALT_WDOG_CPU, ALT_WDOG_TIMER_MODE_FREERUN);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_response_mode_set(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_WDOG0 response mode set to ALT_WDOG_INT_THEN_RESET\n");
	result_code = alt_wdog_response_mode_set(ALT_WDOG0, ALT_WDOG_INT_THEN_RESET);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_response_mode_set(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_WDOG1 response mode set to ALT_WDOG_INT_THEN_RESET\n");
	result_code = alt_wdog_response_mode_set(ALT_WDOG1, ALT_WDOG_INT_THEN_RESET);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_response_mode_set(); result=%d\n", (int)result_code);
	}
	printf("\n");

	printf("======== Print Test WDT Settings ========\n");
	sample_wdog_print_all_wdt_settings();

	printf("======== Print Initial Interrupt Settings ========\n");
	sample_wdog_print_all_int_settings();

	// Interrupt GIC Config
	sample_wdog_setting_gic();

	printf("======== Print Test Interrupt Settings ========\n");
	sample_wdog_print_all_int_settings();


	return;
}

/********************************************************************************//*!
 * @fn  void sample_wdog_test_uninit(void) 
 * @brief  Uninitializing the sample program. 
 * 
 * @details  Call the uninitialization function for the watchdog timer API.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_wdog_test_uninit(void)
{
	ALT_STATUS_CODE result_code;

	/* Usage:  ALT_STATUS_CODE  alt_wdog_uninit (void); 	*/
	printf("-- HWLib test : alt_wdog_uninit() --\n");
	result_code = alt_wdog_uninit();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_wdog_uninit() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");


	return;
}


/********************************************************************************//*!
 * @fn  static void sample_wdog_print_setting_gic(ALT_INT_INTERRUPT_t int_id) 
 * @brief  Print GIC settings of specific watchdog timer interrupts.
 * 
 * @details  Displays the specific interrupt settings of detail.
 * @param[in] int_id  interrupt identification types.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_wdog_print_setting_gic(ALT_INT_INTERRUPT_t int_id)
{
	uint32_t ui32temp = 0;
	ALT_INT_TRIGGER_t trigger_type = ALT_INT_TRIGGER_NA;

	//// Print initial value
	///****************************************************************************************/
	///* Usage: ALT_STATUS_CODE alt_int_dist_is_secure(ALT_INT_INTERRUPT_t int_id);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_is_enabled(ALT_INT_INTERRUPT_t int_id);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_is_pending(ALT_INT_INTERRUPT_t int_id);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_is_active(ALT_INT_INTERRUPT_t int_id);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_priority_get(ALT_INT_INTERRUPT_t int_id, uint32_t* priority);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_target_get(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t* target);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_trigger_get(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t* trigger_type);	*/
	///****************************************************************************************/
	printf("alt_int_dist_is_secure() ...... result=%d\n", (int)alt_int_dist_is_secure(int_id));
	printf("alt_int_dist_is_enabled() ..... result=%d\n", (int)alt_int_dist_is_enabled(int_id));
	printf("alt_int_dist_is_pending() ..... result=%d\n", (int)alt_int_dist_is_pending(int_id));
	printf("alt_int_dist_is_active() ...... result=%d\n", (int)alt_int_dist_is_active(int_id));
	printf("alt_int_dist_priority_get() ... result=%d out=0x%08X\n", (int)alt_int_dist_priority_get(int_id, &ui32temp), (int)ui32temp);
	printf("alt_int_dist_target_get() ..... result=%d out=0x%08X\n", (int)alt_int_dist_target_get(int_id, &ui32temp), (int)ui32temp);
	printf("alt_int_dist_trigger_get() .... result=%d out=%d\n", (int)alt_int_dist_trigger_get(int_id, &trigger_type), (int)trigger_type);

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_wdog_print_all_int_settings(void) 
 * @brief  Print GIC settings of all watchdog timer interrupts. 
 * 
 * @details  Displays the three types of interrupt settings.
 *           - CPU Private Watchdog Timer (ALT_WDOG_CPU) 
 *           - L4 Watchdog 0 (ALT_WDOG0) 
 *           - L4 Watchdog 1 (ALT_WDOG1) 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_wdog_print_all_int_settings(void)
{
	printf("<< ALT_INT_INTERRUPT_PPI_TIMER_WATCHDOG >>\n");
	sample_wdog_print_setting_gic(ALT_INT_INTERRUPT_PPI_TIMER_WATCHDOG);
	printf("\n");

	printf("<< ALT_INT_INTERRUPT_WDOG0_IRQ >>\n");
	sample_wdog_print_setting_gic(ALT_INT_INTERRUPT_WDOG0_IRQ);
	printf("\n");

	printf("<< ALT_INT_INTERRUPT_WDOG0_IRQ >>\n");
	sample_wdog_print_setting_gic(ALT_INT_INTERRUPT_WDOG1_IRQ);
	printf("\n");

	return;
}


/********************************************************************************//*!
 * @fn  void sample_wdog_setting_gic(void) 
 * @brief  Setting of GIC (Watchdog timer interrupt).
 * 
 * @details    In order to receive a watchdog timer interrupt setting of GIC.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_wdog_setting_gic(void)
{
	ALT_STATUS_CODE result_code;


	/* Usage: ALT_STATUS_CODE alt_int_isr_register(ALT_INT_INTERRUPT_t int_id, alt_int_callback_t callback, void* context);	*/
	/*  typedef void(* alt_int_callback_t)(uint32_t icciar, void *context) */
	result_code = alt_int_isr_register(ALT_INT_INTERRUPT_PPI_TIMER_WATCHDOG, sample_wdog_callback_CPU, NULL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_isr_register(PPI_TIMER_WATCHDOG) result=%d\n", (int)result_code);
	}
	result_code = alt_int_isr_register(ALT_INT_INTERRUPT_WDOG0_IRQ, sample_wdog_callback_WDOG0, NULL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_isr_register(WDOG0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_isr_register(ALT_INT_INTERRUPT_WDOG1_IRQ, sample_wdog_callback_WDOG1, NULL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_isr_register(WDOG1_IRQ) result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_int_dist_target_set(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t target);	*/
	result_code = alt_int_dist_target_set(ALT_INT_INTERRUPT_WDOG0_IRQ, 1);		// arg2=targetCPU (Only Core#0)
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_target_set(WDOG0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_target_set(ALT_INT_INTERRUPT_WDOG1_IRQ, 1);		// arg2=targetCPU (Only Core#0)
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_target_set(WDOG1_IRQ) result=%d\n", (int)result_code);
	}

	///* Usage: ALT_STATUS_CODE alt_int_dist_trigger_set(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t trigger_type);	*/
	//printf("-- HWLib test : alt_int_dist_trigger_set() --\n");
	//result_code = alt_int_dist_trigger_set(ALT_INT_INTERRUPT_CLKMGR_IRQ, ALT_INT_TRIGGER_EDGE);
	//if(result_code != ALT_E_SUCCESS){
	//	printf("ERROR!!: alt_int_dist_trigger_set() result=%d\n", (int)result_code);
	//} else {
	//	printf("SUCCESS!!\n");
	//}
	//printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_priority_set(ALT_INT_INTERRUPT_t int_id, uint32_t priority);	*/
	result_code = alt_int_dist_priority_set(ALT_INT_INTERRUPT_PPI_TIMER_WATCHDOG, 99);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_priority_set(PPI_TIMER_WATCHDOG) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_priority_set(ALT_INT_INTERRUPT_WDOG0_IRQ, 99);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_priority_set(WDOG0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_priority_set(ALT_INT_INTERRUPT_WDOG1_IRQ, 99);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_priority_set(WDOG1_IRQ) result=%d\n", (int)result_code);
	}

	return;
}

/********************************************************************************//*!
 * @fn  void sample_wdog_callback_CPU(uint32_t icciar, void *context) 
 * @brief  CPU Private Watchdog Timer (ALT_WDOG_CPU) interrupt callback function.
 * 
 * @details  working opportunity an interrupt from the ALT_WDOG_CPU, indicate an 
 *           interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued. 
 * @param[in] context  The user provided context.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_wdog_callback_CPU(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;

	// WDT Pending Clear
	/* Usage:  ALT_STATUS_CODE  alt_wdog_int_clear (ALT_WDOG_TIMER_t tmr_id); 	*/
	result_code = alt_wdog_int_clear(ALT_WDOG_CPU);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_wdog_int_clear(ALT_WDOG_CPU) result=%d\n", (int)result_code);
	}

	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_PPI_TIMER_WATCHDOG);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	//printf("[INTERRUPT] ALT_INT_INTERRUPT_PPI_TIMER_WATCHDOG is occurred!!\n");
	util_intlog_record(ALT_INT_INTERRUPT_PPI_TIMER_WATCHDOG, (int)icciar, *(int*)context);

	return;
}

/********************************************************************************//*!
 * @fn  void sample_wdog_callback_WDOG0(uint32_t icciar, void *context) 
 * @brief  L4 Watchdog 0 (ALT_WDOG0) interrupt callback function.
 * 
 * @details  working opportunity an interrupt from the ALT_WDOG0, indicate an 
 *           interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued. 
 * @param[in] context  The user provided context.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_wdog_callback_WDOG0(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;

	// WDT Pending Clear
	/* Usage:  ALT_STATUS_CODE  alt_wdog_int_clear (ALT_WDOG_TIMER_t tmr_id); 	*/
	result_code = alt_wdog_int_clear(ALT_WDOG0);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_wdog_int_clear(ALT_WDOG0) result=%d\n", (int)result_code);
	}

	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_WDOG0_IRQ);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	//printf("[INTERRUPT] ALT_INT_INTERRUPT_WDOG0_IRQ is occurred!!\n");
	util_intlog_record(ALT_INT_INTERRUPT_WDOG0_IRQ, (int)icciar, *(int*)context);
	util_intlog_print();

	/******************************************************************************/
	/* Do not use the automatic warm reset by WDT, cold reset with the HWLib API. */
	/******************************************************************************/
	/* Usage:  ALT_STATUS_CODE  alt_reset_cold_reset (void);	*/
	fprintf(stderr, "[INTERRUPT]-- HWLib test : alt_reset_cold_reset() --\n");
	result_code = alt_reset_cold_reset();
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_reset_cold_reset() result=%d\n", (int)result_code);
	}

	return;
}

/********************************************************************************//*!
 * @fn  void sample_wdog_callback_WDOG1(uint32_t icciar, void *context) 
 * @brief  L4 Watchdog 1 (ALT_WDOG1) interrupt callback function.
 * 
 * @details  working opportunity an interrupt from the ALT_WDOG1, indicate an 
 *           interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued. 
 * @param[in] context  The user provided context.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_wdog_callback_WDOG1(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;

	// WDT Pending Clear
	/* Usage:  ALT_STATUS_CODE  alt_wdog_int_clear (ALT_WDOG_TIMER_t tmr_id); 	*/
	result_code = alt_wdog_int_clear(ALT_WDOG1);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_wdog_int_clear(ALT_WDOG1) result=%d\n", (int)result_code);
	}

	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_WDOG1_IRQ);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	//printf("[INTERRUPT] ALT_INT_INTERRUPT_WDOG1_IRQ is occurred!!\n");
	util_intlog_record(ALT_INT_INTERRUPT_WDOG1_IRQ, (int)icciar, *(int*)context);
	util_intlog_print();

	/******************************************************************************/
	/* Do not use the automatic warm reset by WDT, cold reset with the HWLib API. */
	/******************************************************************************/
	/* Usage:  	ALT_STATUS_CODE  alt_reset_warm_reset (uint32_t warm_reset_delay,
	 * 								uint32_t nRST_pin_clk_assertion, bool sdram_refresh,
	 * 								bool fpga_mgr_handshake, bool scan_mgr_handshake,
	 * 								bool fpga_handshake, bool etr_stall);	*/
	fprintf(stderr, "[INTERRUPT]-- HWLib test : alt_reset_warm_reset() --\n");
#ifdef soc_cv_av
	result_code = alt_reset_warm_reset(255, 0x7FFFF, false, false, false, false, false);
#else /* soc_cv_av */
	result_code = alt_reset_warm_reset(255, 0x7FFFF, false, false, false, false);
#endif /* soc_cv_av */
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_reset_warm_reset() result=%d\n", (int)result_code);
	}

	return;
}

/********************************************************************************//*!
 * @fn  void sample_wdog_test_print_usage(void) 
 * @brief  Display usage of the sample program. 
 * 
 * @details  Following information displays on the console.
 *            - Overview of HPS for push button functions (or operations)
 *            - Overview of HPS for DIP switch functions (or operations)
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/18  ---  #xxxxx  新規作成. 
 **//*******************************************************************************/
void sample_wdog_test_print_usage(void)
{
	printf("\n");
	printf("+--- Usage ( Function of PUSH SW and SLIDE SW )  ---------------+\n");
	printf("+ PUSH SW #0 .... Exit Test loop!!!\n");
	printf("+ PUSH SW #1 .... Reset the CPU private watchdog timer counter.\n");
	printf("+ PUSH SW #2 .... Reset the L4 watchdog timer0 counter.\n");
	printf("+ PUSH SW #3 .... Reset the L4 watchdog timer1 counter.\n");
	printf("+ SLIDE SW#0:3 .. Change the timer freqency dynamically.\n"); 
	printf("                         4-bit value set to prescaler intact.\n");
	printf("+---------------------------------------------------------------+\n\n");
	
#if 0
	　無限ループ内で以下の処理を行う。
	　 - HPS用DIPSW1-4の変更契機で、CPU Private Watchdog Timer用の
	　　 プリスケーラ設定値を変更します。
	　　 DIPSW1をMSB、DIPSW4をLSBとして4bitの値をそのまま設定値に
	　　 適用します。

	　 - HPS用PUSHSW1の押下契機で CPU Private Watchdog Timer を
	　　 リセットします。

	　 - HPS用PUSHSW2の押下契機で L4 Watchdog 0 をリセットします。

	　 - HPS用PUSHSW3の押下契機で L4 Watchdog 1 をリセットします。

	　 - HPS用PUSHSW0の押下契機で無限ループを終了します。
	　　 (テストプログラム終了)
#endif
	
	return;
}

/********************************************************************************//*!
 * @fn  void sample_wdog_test_main(void) 
 * @brief  Main routine of the sample program. 
 * 
 * @details  Infinite loop structure works. Make the detection of various types 
 *           of switches in the loop, it performs in various testing processes.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_wdog_test_main(void)
{
	ALT_STATUS_CODE	result_code;
	uint32_t switch_raw = 0;
	uint32_t switch_raw_bk = 0;
	uint32_t switch_detect = 0;
	uint32_t switch_detect_on = 0;
	uint32_t switch_number = 0;
	bool disp_usage = true;

	// WDT Timer Start & Interrupt Enable
	sample_wdog_ienable_all();
	sample_wdog_start_all();

	util_intlog_print();

	printf("==== Start While(1) loop process!!! ============\n");
	switch_raw_bk = sample_detect_switch();
	while(1)
	{
		if(disp_usage)
		{
			sample_wdog_test_print_usage();
			disp_usage = false;
		}

		// ---- Check the Slide-Switch and Push-Switch. ---- 
		switch_raw = sample_detect_switch();
		switch_detect = switch_raw ^ switch_raw_bk;
		switch_detect_on  |= switch_detect & switch_raw;
		
		// Slide-Switch 
		if(switch_detect & SAMPLE_SWITCH_BIT_SLIDEALL)
		{
			switch_number = switch_raw & SAMPLE_SWITCH_BIT_NUM;
			switch_number *= 1;		// To avoid warnings.

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE0){
				//printf("SAMPLE_SWITCH_BIT_SLIDE0\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE1){
				//printf("SAMPLE_SWITCH_BIT_SLIDE1\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE2){
				//printf("SAMPLE_SWITCH_BIT_SLIDE2\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE3){
				//printf("SAMPLE_SWITCH_BIT_SLIDE3\n");
			}
			
			// Change ALT_WDOG_CPU's Prescaler Value.
			result_code = alt_wdog_stop(ALT_WDOG_CPU);
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_wdog_stop() result=%d\n", (int)result_code);
			}
			/* Usage:  ALT_STATUS_CODE  alt_wdog_core_prescaler_set (uint32_t val); 	*/
			printf("-- HWLib test : alt_wdog_core_prescaler_set(%d) --\n", (int)switch_number);
			result_code = alt_wdog_core_prescaler_set(switch_number);
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_wdog_core_prescaler_set(%d) result=%d\n", (int)switch_number, (int)result_code);
			} else {
				printf("SUCCESS!!\n");
			}
			result_code = alt_wdog_start(ALT_WDOG_CPU);
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_wdog_stop() result=%d\n", (int)result_code);
			}
			
			disp_usage = false;
		}
		
		
		// Push-Switch 0
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH0)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH0;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH0\n");
			// WDT Timer Stop & Interrupt Disable
			sample_wdog_stop_all();
			sample_wdog_idisable_all();

			printf("======== Print Last WDT Settings ========\n");
			sample_wdog_print_all_wdt_settings2();

			break;	// Exit Push-Switch 0 and 1 becomes ON.
		}
		// Push-Switch 1
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH1)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH1;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH1\n");
			printf("-- HWLib test : alt_wdog_reset(ALT_WDOG_CPU) --\n");
			sample_wdog_print_specific_wdt_counter(ALT_WDOG_CPU);
			/* Usage:  ALT_STATUS_CODE  alt_wdog_reset (ALT_WDOG_TIMER_t tmr_id); 	*/
			result_code = alt_wdog_reset(ALT_WDOG_CPU);
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_wdog_reset() result=%d\n", (int)result_code);
			}
			sample_wdog_print_specific_wdt_counter(ALT_WDOG_CPU);
			printf("\n");
			disp_usage = true;
		}
		// Push-Switch 2
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH2)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH2;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH2\n");
			printf("-- HWLib test : alt_wdog_reset(ALT_WDOG0) --\n");
			sample_wdog_print_specific_wdt_counter(ALT_WDOG0);
			/* Usage:  ALT_STATUS_CODE  alt_wdog_reset (ALT_WDOG_TIMER_t tmr_id); 	*/
			result_code = alt_wdog_reset(ALT_WDOG0);
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_wdog_reset() result=%d\n", (int)result_code);
			}
			sample_wdog_print_specific_wdt_counter(ALT_WDOG0);
			printf("\n");
			disp_usage = true;
		}
		// Push-Switch 3
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH3)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH3;

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH3\n");
			printf("-- HWLib test : alt_wdog_reset(ALT_WDOG1) --\n");
			sample_wdog_print_specific_wdt_counter(ALT_WDOG1);
			/* Usage:  ALT_STATUS_CODE  alt_wdog_reset (ALT_WDOG_TIMER_t tmr_id); 	*/
			result_code = alt_wdog_reset(ALT_WDOG1);
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_wdog_reset() result=%d\n", (int)result_code);
			}
			sample_wdog_print_specific_wdt_counter(ALT_WDOG1);
			printf("\n");
			disp_usage = true;
		}
		
		util_intlog_print();

		switch_raw_bk = switch_raw;
	}
	printf("==== End While(1) loop process. ================\n");

	util_intlog_print();

	return;
}


// All APIs
/***********************************************************************/
/* Usage:  ALT_STATUS_CODE  alt_wdog_init (void); 	*/
/* Usage:  ALT_STATUS_CODE  alt_wdog_uninit (void); 	*/
/* Usage:  ALT_STATUS_CODE  alt_wdog_stop (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  ALT_STATUS_CODE  alt_wdog_start (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  bool  alt_wdog_tmr_is_enabled (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  ALT_STATUS_CODE  alt_wdog_reset (ALT_WDOG_TIMER_t tmr_id); 	*/

/* Usage:  ALT_STATUS_CODE  alt_wdog_counter_set (ALT_WDOG_TIMER_t tmr_id, uint32_t val); 	*/
/* Usage:  uint32_t  alt_wdog_counter_get_current (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  uint32_t  alt_wdog_counter_get_curtime_millisecs (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  uint32_t  alt_wdog_counter_get_init (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  uint64_t  alt_wdog_counter_get_inittime_nanosecs (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  uint32_t  alt_wdog_counter_get_inittime_millisecs (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  ALT_STATUS_CODE  alt_wdog_core_prescaler_set (uint32_t val); 	*/
/* Usage:  uint32_t  alt_wdog_core_prescaler_get (void); 	*/
/* Usage:  uint32_t  alt_wdog_counter_get_max (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  uint64_t  alt_wdog_counter_get_max_nanosecs (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  uint32_t  alt_wdog_counter_get_max_millisecs (ALT_WDOG_TIMER_t tmr_id); 	*/

/* Usage:  ALT_STATUS_CODE  alt_wdog_int_disable (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  ALT_STATUS_CODE  alt_wdog_int_enable (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  bool  alt_wdog_int_is_pending (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  bool  alt_wdog_int_is_enabled (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  ALT_STATUS_CODE  alt_wdog_int_clear (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  bool  alt_wdog_int_if_pending_clear (ALT_WDOG_TIMER_t tmr_id); 	*/

/* Usage:  ALT_STATUS_CODE  alt_wdog_response_mode_set (ALT_WDOG_TIMER_t tmr_id, ALT_WDOG_RESET_TYPE_t type); 	*/
/* Usage:  int32_t  alt_wdog_response_mode_get (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  uint32_t  alt_wdog_compcode_get (ALT_WDOG_TIMER_t tmr_id); 	*/
/* Usage:  uint32_t  alt_wdog_ver_get (ALT_WDOG_TIMER_t tmr_id); 	*/
/***********************************************************************/

////////////////////////////// Common Functions //////////////////////////////////////// 
/********************************************************************************//*!
 * @fn  static void sample_resetmng_print_trigger(void) 
 * @brief  Display reset factors. And the clear.
 * 
 * @details  View event information indicates the reset factors and clears.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/21  ---  #6296  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_resetmng_print_trigger(void)
{
	ALT_STATUS_CODE result_code;

	printf("==== Print Reset Trigger, and Clear ====\n");
#if 0
	typedef enum ALT_RESET_EVENT_e
	{
	    /*! Power-On Voltage Detector Cold Reset */
	    ALT_RESET_EVENT_PORVOLTRST          = 0x00000001,

	    /*! nPOR Pin Cold Reset                  */
	    ALT_RESET_EVENT_NPORPINRST          = 0x00000002,

	    /*! FPGA Core Cold Reset                 */
	    ALT_RESET_EVENT_FPGACOLDRST         = 0x00000004,

	    /*! CONFIG_IO Cold Reset                 */
	    ALT_RESET_EVENT_CONFIGIOCOLDRST     = 0x00000008,

	    /*! Software Cold Reset                  */
	    ALT_RESET_EVENT_SWCOLDRST           = 0x00000010,

	    /*! nRST Pin Warm Reset                  */
	    ALT_RESET_EVENT_NRSTPINRST          = 0x00000100,

	    /*! FPGA Core Warm Reset                 */
	    ALT_RESET_EVENT_FPGAWARMRST         = 0x00000200,

	    /*! Software Warm Reset                  */
	    ALT_RESET_EVENT_SWWARMRST           = 0x00000400,

	    /*! MPU Watchdog 0 Warm Reset            */
	    ALT_RESET_EVENT_MPUWD0RST           = 0x00001000,

	    /*! MPU Watchdog 1 Warm Reset            */
	    ALT_RESET_EVENT_MPUWD1RST           = 0x00002000,

	    /*! L4 Watchdog 0 Warm Reset             */
	    ALT_RESET_EVENT_L4WD0RST            = 0x00004000,

	    /*! L4 Watchdog 1 Warm Reset             */
	    ALT_RESET_EVENT_L4WD1RST            = 0x00008000,

	    /*! FPGA Core Debug Reset                */
	    ALT_RESET_EVENT_FPGADBGRST          = 0x00040000,

	    /*! DAP Debug Reset                      */
	    ALT_RESET_EVENT_CDBGREQRST          = 0x00080000,

	    /*! SDRAM Self-Refresh Timeout           */
	    ALT_RESET_EVENT_SDRSELFREFTIMEOUT   = 0x01000000,

	    /*! FPGA manager handshake Timeout       */
	    ALT_RESET_EVENT_FPGAMGRHSTIMEOUT    = 0x02000000,

	    /*! SCAN manager handshake Timeout       */
	    ALT_RESET_EVENT_SCANHSTIMEOUT       = 0x04000000,

	    /*! FPGA handshake Timeout               */
	    ALT_RESET_EVENT_FPGAHSTIMEOUT       = 0x08000000,

	    /*! ETR Stall Timeout                    */
	    ALT_RESET_EVENT_ETRSTALLTIMEOUT     = 0x10000000
	} ALT_RESET_EVENT_t;
#endif

	/* Usage: uint32_t alt_reset_event_get  ( void   ) ;  */
	printf("-- HWLib test : alt_reset_event_get() .. result=0x%08X --\n", (int)alt_reset_event_get());


	/* Usage: ALT_STATUS_CODE alt_reset_event_clear( uint32_t  event_mask );  */
	printf("-- HWLib test : alt_reset_event_clear() --\n");
	result_code = alt_reset_event_clear((uint32_t)0xFFFFFFFF);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_reset_event_clear() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	return;
}


/********************************************************/
/*	Test Command (Watchdog timer (and reset manager))	*/
/********************************************************/
int sample_wdog_test_cmd(char* options)
{
	volatile uint32_t	waitstep;

    ALT_STATUS_CODE status = ALT_E_SUCCESS;
    
    printf("\r\nUser Application Start!\r\n");

    //
    // Start the interrupt system
    //
    if (status == ALT_E_SUCCESS)
    {
    	util_intlog_init();
        status = socfpga_int_start();
    }

    // Wait for 1024steps
    for(waitstep=1024; waitstep>0; waitstep--)
    {}


    // Printing Current PLL Setting. And Set Global Timer for Time Measurement.
    util_time_init();

    //// Sample Function Call!!! |---->
    printf(">>>> Execute sample_wdog_test_init(); !!! <<<<\r\n");
    sample_wdog_test_init();

    printf(">>>> Execute sample_wdog_test_main(); !!! <<<<\r\n");
    sample_wdog_test_main();

    printf(">>>> Execute sample_wdog_test_uninit(); !!! <<<<\r\n");
    sample_wdog_test_uninit();
    //// Sample Function Call!!! <----|

    // Printing All Results of Time Measurement. And Uninit Global Timer.
    util_time_uninit();

    //
    // Stop the interrupt system
    //
    socfpga_int_stop();
	util_intlog_print();

    printf("Finished running the sample !!!\r\n");
    if (status == ALT_E_SUCCESS)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}
/***********************************************************************************
 * end of file 
 ***********************************************************************************/
