﻿/********************************************************************************//*!
 * @file  sample_globaltmr.c 
 * @brief  To use HWLib for the Global Timer example program. 
 * 
 * @details  Run the API category below.
 *           - The Global Timer Manager API  
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/07  ---  #6203  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 * @date 2014/08/18  ---  -----  v14.0化 (誤植訂正、操作説明の表示を追加). 
 * @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
 * @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
 * @date 2018/07/02  ---  #25940  All in One ベアメタル・アプリケーション v17.1 への対応を実施.
 * @date 2019/01/18  ---  -----   All in One ベアメタル・アプリケーション v18.1 への対応を実施.
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes 
 ***********************************************************************************/
#include <stdio.h>
#include <string.h>
#include "hwlib.h"
#include "socal/socal.h"
#include "socal/hps.h"
#include "alt_generalpurpose_io.h"
#include "alt_globaltmr.h"
#include "alt_interrupt.h"

#include "sample_app_setting.h"
#include "util_time_measurement.h"
#include "util_interrupt_log.h"
#include "alt_printf.h"

/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/
void sample_globaltmr_setting_gic(void);
void sample_globaltmr_callback(uint32_t icciar, void *context);
void sample_globaltmr_print_get_result(void);
void sample_globaltmr_alldisable(void);
void sample_globaltmr_allenable(void);
void sample_globaltmr_paraminit(void);
void sample_globaltmr_test_main(void);
void sample_globaltmr_test_print_usage(void);
int  sample_globaltmr_test_cmd(char* options);

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/

/***********************************************************************************
 *  variables 
 ***********************************************************************************/

/***********************************************************************************
 *  functions 
 ***********************************************************************************/

/********************************************************************************//*!
 * @fn  void sample_globaltmr_setting_gic(void) 
 * @brief  Setting of GIC (Global Timer interrupt).
 * 
 * @details    In order to receive a Global Timer interrupt setting of GIC.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/07  ---  #6203  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_globaltmr_setting_gic(void)
{
	ALT_STATUS_CODE result_code;
	uint32_t ui32temp = 0;
	ALT_INT_TRIGGER_t trigger_type = ALT_INT_TRIGGER_NA;

	// Print initial value (GIC: GPIO2)
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_secure(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_enabled(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_pending(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_active(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_priority_get(ALT_INT_INTERRUPT_t int_id, uint32_t* priority);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_target_get(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t* target);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_get(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t* trigger_type);	*/
	/****************************************************************************************/
	printf("-- HWLib test : alt_int_dist_is_secure() --\n");
	printf("arg1=ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL --> result=%d\n", (int)alt_int_dist_is_secure(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL));
	printf("\n");
	printf("-- HWLib test : alt_int_dist_is_enabled() --\n");
	printf("arg1=ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL --> result=%d\n", (int)alt_int_dist_is_enabled(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL));
	printf("\n");
	printf("-- HWLib test : alt_int_dist_is_pending() --\n");
	printf("arg1=ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL --> result=%d\n", (int)alt_int_dist_is_pending(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL));
	printf("\n");
	printf("-- HWLib test : alt_int_dist_is_active() --\n");
	printf("arg1=ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL --> result=%d\n", (int)alt_int_dist_is_active(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL));
	printf("\n");
	printf("-- HWLib test : alt_int_dist_priority_get() --\n");
	printf("arg1=ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL --> result=%d out=0x%08X\n", (int)alt_int_dist_priority_get(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL,&ui32temp), (int)ui32temp);
	printf("\n");
	printf("-- HWLib test : alt_int_dist_target_get() --\n");
	printf("arg1=ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL --> result=%d out=0x%08X\n", (int)alt_int_dist_target_get(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL,&ui32temp), (int)ui32temp);
	printf("\n");
	printf("-- HWLib test : alt_int_dist_trigger_get() --\n");
	printf("arg1=ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL --> result=%d out=%d\n", (int)alt_int_dist_trigger_get(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL,&trigger_type), (int)trigger_type);
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_isr_register(ALT_INT_INTERRUPT_t int_id, alt_int_callback_t callback, void* context);	*/
	/*  typedef void(* alt_int_callback_t)(uint32_t icciar, void *context) */
	printf("-- HWLib test : alt_int_isr_register() --\n");
	result_code = alt_int_isr_register(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL, sample_globaltmr_callback, NULL);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_isr_register() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");

	///* Usage: ALT_STATUS_CODE alt_int_dist_target_set(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t target);	*/
	//printf("-- HWLib test : alt_int_dist_target_set() --\n");
	//result_code = alt_int_dist_target_set(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL, 1);		// arg2=targetCPU (Only Core#0)
	//if(result_code != ALT_E_SUCCESS)
	//{
	//	printf("ERROR!!: alt_int_dist_target_set() result=%d\n", (int)result_code);
	//}
	//else
	//{
	//	printf("SUCCESS!!\n");
	//}
	//printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_set(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t trigger_type);	*/
	printf("-- HWLib test : alt_int_dist_trigger_set() --\n");
	result_code = alt_int_dist_trigger_set(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL, ALT_INT_TRIGGER_EDGE);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_trigger_set() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_priority_set(ALT_INT_INTERRUPT_t int_id, uint32_t priority);	*/
	printf("-- HWLib test : alt_int_dist_priority_set() --\n");
	result_code = alt_int_dist_priority_set(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL, 99);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_priority_set() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_enable(ALT_INT_INTERRUPT_t int_id);	*/
	printf("-- HWLib test : alt_int_dist_enable() --\n");
	result_code = alt_int_dist_enable(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_enable() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_secure_enable(ALT_INT_INTERRUPT_t int_id);	*/
	printf("-- HWLib test : alt_int_dist_secure_enable() --\n");
	result_code = alt_int_dist_secure_enable(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_enable() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");


	return;
}

/********************************************************************************//*!
 * @fn  void sample_globaltmr_callback(uint32_t icciar, void *context) 
 * @brief  Global Timer interrupt callback function.
 * 
 * @details  working opportunity an interrupt from the Global Timer, indicate an 
 *           interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued. 
 * @param[in] context  The user provided context.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/07  ---  #6203  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_globaltmr_callback(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;

	/* Usage: ALT_STATUS_CODE alt_globaltmr_int_clear_pending(void);	*/
	result_code = alt_globaltmr_int_clear_pending();
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_globaltmr_int_clear_pending() result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_int_dist_pending_clear(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	//printf("[INTERRUPT]Global Timer Interrupt is occurred!!\n");
	util_intlog_record(ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL, (int)icciar, *(int*)context);

	return;
}


/********************************************************************************//*!
 * @fn  void sample_globaltmr_print_get_result(void) 
 * @brief  Global timer information get and display. 
 * 
 * @details  Use the global timer API and briefly displays the available information. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/07  ---  #6203  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_globaltmr_print_get_result(void)
{
	uint32_t ui32data[2];
	ALT_STATUS_CODE	result_code;

	// All APIs for information getting are performed. And a result is printed.
	// Counters (Register Values)
	/***********************************************************************/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_get(uint32_t* highword, uint32_t* lowword);	*/
	/* Usage: uint64_t alt_globaltmr_get64(void);	*/
	/* Usage: uint32_t alt_globaltmr_counter_get_low32(void);	*/
	/* Usage: uint32_t alt_globaltmr_counter_get_hi32(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_comp_get(uint32_t* highword, uint32_t* lowword);	*/
	/* Usage: uint64_t alt_globaltmr_comp_get64(void);	*/
	/* Usage: uint32_t alt_globaltmr_prescaler_get(void);	*/
	/* Usage: uint32_t alt_globaltmr_autoinc_get(void);	*/
	/* Usage: uint32_t alt_globaltmr_maxcounter_get(void);	*/
	/***********************************************************************/
	printf("-- HWLib test : alt_globaltmr_get() --\n");
	result_code = alt_globaltmr_get(&ui32data[0], &ui32data[1]);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_get() result=%d\n", (int)result_code);
	} else {
		printf("result { highword=0x%08X, lowword=0x%08X }\n", (int)ui32data[0], (int)ui32data[1]);
	}
	printf("-- HWLib test : alt_globaltmr_get64() --\n");
	printf("result=0x%016llX\n", (long long int)alt_globaltmr_get64());
	printf("-- HWLib test : alt_globaltmr_counter_get_low32(), alt_globaltmr_counter_get_hi32()--\n");
	ui32data[0] = alt_globaltmr_counter_get_hi32();
	ui32data[1] = alt_globaltmr_counter_get_low32();
	printf("result { highword=0x%08X, lowword=0x%08X }\n", (int)ui32data[0], (int)ui32data[1]);
	printf("-- HWLib test : alt_globaltmr_comp_get() --\n");
	result_code = alt_globaltmr_comp_get(&ui32data[0], &ui32data[1]);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_comp_get() result=%d\n", (int)result_code);
	} else {
		printf("result { highword=0x%08X, lowword=0x%08X }\n", (int)ui32data[0], (int)ui32data[1]);
	}
	printf("-- HWLib test : alt_globaltmr_comp_get64() --\n");
	printf("result=0x%016llX\n", (long long int)alt_globaltmr_comp_get64());
	printf("-- HWLib test : alt_globaltmr_prescaler_get() --\n");
	printf("result=0x%08X\n", (int)alt_globaltmr_prescaler_get());
	printf("-- HWLib test : alt_globaltmr_autoinc_get() --\n");
	printf("result=0x%08X\n", (int)alt_globaltmr_autoinc_get());
	printf("-- HWLib test : alt_globaltmr_maxcounter_get() --\n");
	printf("result=0x%08X\n", (int)alt_globaltmr_maxcounter_get());

	// Statuses
	/***********************************************************************/
	/* Usage: bool alt_globaltmr_is_comp_mode(void);	*/
	/* Usage: bool alt_globaltmr_is_autoinc_mode(void);	*/
	/* Usage: bool alt_globaltmr_int_is_enabled(void);	*/
	/* Usage: bool alt_globaltmr_int_is_pending(void);	*/
	/***********************************************************************/
	printf("-- HWLib test : alt_globaltmr_is_comp_mode() --\n");
	printf("result=%d\n", (int)alt_globaltmr_is_comp_mode());
	printf("-- HWLib test : alt_globaltmr_is_autoinc_mode() --\n");
	printf("result=%d\n", (int)alt_globaltmr_is_autoinc_mode());
	printf("-- HWLib test : alt_globaltmr_int_is_enabled() --\n");
	printf("result=%d\n", (int)alt_globaltmr_int_is_enabled());
	printf("-- HWLib test : alt_globaltmr_int_is_pending() --\n");
	printf("result=%d\n", (int)alt_globaltmr_int_is_pending());
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  void sample_globaltmr_alldisable(void) 
 * @brief  To disable all features of the global timer.  
 * 
 * @details  Stop a global timer. Also, disable comparator mode, auto-increment 
 *           mode, interrupt notification. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/07  ---  #6203  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_globaltmr_alldisable(void)
{
	ALT_STATUS_CODE	result_code;

	// All Functions of Global Timer are disabled.
	/***********************************************************************/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_stop(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_comp_mode_stop(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_autoinc_mode_stop(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_int_disable(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_int_clear_pending(void);	*/
	/***********************************************************************/
	printf("-- HWLib test : alt_globaltmr_stop() --\n");
	result_code = alt_globaltmr_stop();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_stop() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("-- HWLib test : alt_globaltmr_comp_mode_stop() --\n");
	result_code = alt_globaltmr_comp_mode_stop();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_comp_mode_stop() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("-- HWLib test : alt_globaltmr_autoinc_mode_stop() --\n");
	result_code = alt_globaltmr_autoinc_mode_stop();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_autoinc_mode_stop() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("-- HWLib test : alt_globaltmr_int_disable() --\n");
	result_code = alt_globaltmr_int_disable();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_int_disable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("-- HWLib test : alt_globaltmr_int_clear_pending() --\n");
	result_code = alt_globaltmr_int_clear_pending();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_int_clear_pending() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  void sample_globaltmr_allenable(void) 
 * @brief  To enable all features of the global timer.  
 * 
 * @details  Start a global timer. Also, enable comparator mode, auto-increment 
 *           mode, interrupt notification. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/07  ---  #6203  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_globaltmr_allenable(void)
{
	ALT_STATUS_CODE	result_code;

	// All Functions of Global Timer are enabled.
	/***********************************************************************/
	/* Usage: bool alt_globaltmr_int_if_pending_clear(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_start(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_comp_mode_start(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_autoinc_mode_start(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_int_enable(void);	*/
	/***********************************************************************/
	printf("-- HWLib test : alt_globaltmr_int_if_pending_clear() --\n");
	printf("result=%d\n", (int)alt_globaltmr_int_if_pending_clear());
	printf("-- HWLib test : alt_globaltmr_start() --\n");
	result_code = alt_globaltmr_start();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_start() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("-- HWLib test : alt_globaltmr_comp_mode_start() --\n");
	result_code = alt_globaltmr_comp_mode_start();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_comp_mode_start() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("-- HWLib test : alt_globaltmr_autoinc_mode_start() --\n");
	result_code = alt_globaltmr_autoinc_mode_start();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_autoinc_mode_start() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("-- HWLib test : alt_globaltmr_int_enable() --\n");
	result_code = alt_globaltmr_int_enable();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_int_enable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  void sample_globaltmr_paraminit(void) 
 * @brief  Initialize parameters for global timer.
 * 
 * @details    Set the initial value of the prescaler, AutoIncrement, comparator 
 *             as a global timer parameter initialization.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/07  ---  #6203  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_globaltmr_paraminit(void)
{
	ALT_STATUS_CODE	result_code;

	// Parameter setting for the global timer.
	/***********************************************************************/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_prescaler_set(uint32_t val);	*/
	/*			val ... Valid range is 0-255. Actual clock divisor ratio is this number plus one. */
	/***********************************************************************/
	printf("-- HWLib test : alt_globaltmr_prescaler_set() --\n");
	//result_code = alt_globaltmr_prescaler_set(255);	// Set the maximum value(255).
	result_code = alt_globaltmr_prescaler_set(0);	// Set the minimum value(0).
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_prescaler_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");
	/***********************************************************************/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_autoinc_set(uint32_t inc);	*/
	/***********************************************************************/
	printf("-- HWLib test : alt_globaltmr_autoinc_set() --\n");
	result_code = alt_globaltmr_autoinc_set(0x40000000);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_autoinc_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");
	/***********************************************************************/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_comp_set64(uint64_t compval);	*/
	/***********************************************************************/
	printf("-- HWLib test : alt_globaltmr_comp_set64() --\n");
	result_code = alt_globaltmr_comp_set64((uint64_t)(alt_globaltmr_comp_get64() + 0x0000000040000000ll));
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_comp_set64() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  void sample_globaltmr_test_main(void) 
 * @brief  Main routine of the sample program. 
 * 
 * @details  Infinite loop structure works. Make the detection of various types 
 *           of switches in the loop, it performs in various testing processes.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/07  ---  #6203  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_globaltmr_test_main(void)
{
	uint32_t switch_raw = 0;
	uint32_t switch_raw_bk = 0;
	uint32_t switch_detect = 0;
	uint32_t switch_detect_on = 0;
	uint32_t switch_number = 0;
	bool disp_usage = true;
	
	uint32_t ui32high, ui32low;
	uint32_t ui32lowbk = 0;
	uint32_t ui32hcmp, ui32lcmp;
	uint32_t ui32hcmpbk, ui32lcmpbk;
	
	ALT_STATUS_CODE	result_code;

	result_code = alt_globaltmr_comp_get(&ui32hcmpbk, &ui32lcmpbk);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_comp_get() result=%d\n", (int)result_code);
	}

	util_intlog_print();

	printf("==== Start While(1) loop process!!! ============\n");
	switch_raw_bk = sample_detect_switch();
	while(1)
	{
		if(disp_usage)
		{
			sample_globaltmr_test_print_usage();
			disp_usage = false;
		}

		// ---- Check the Slide-Switch and Push-Switch. ---- 
		switch_raw = sample_detect_switch();
		switch_detect = switch_raw ^ switch_raw_bk;
		switch_detect_on  |= switch_detect & switch_raw;
		
		// ---- Check the Global Timer Registers. ----
		result_code = alt_globaltmr_get(&ui32high, &ui32low);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_globaltmr_get() result=%d\n", (int)result_code);
		}
		
		// Slide-Switch 
		if(switch_detect & SAMPLE_SWITCH_BIT_SLIDEALL)
		{
			switch_number = switch_raw & SAMPLE_SWITCH_BIT_NUM;
			switch_number *= 1;		// To avoid warnings.

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE0){
				//printf("SAMPLE_SWITCH_BIT_SLIDE0\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE1){
				//printf("SAMPLE_SWITCH_BIT_SLIDE1\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE2){
				//printf("SAMPLE_SWITCH_BIT_SLIDE2\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE3){
				//printf("SAMPLE_SWITCH_BIT_SLIDE3\n");
			}
			
			/* Usage: ALT_STATUS_CODE alt_globaltmr_prescaler_set(uint32_t val);	*/
			/*			val ... Valid range is 0-255. Actual clock divisor ratio is this number plus one. */
			result_code = alt_globaltmr_prescaler_set(switch_number);
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_globaltmr_prescaler_set() result=%d\n", (int)result_code);
			} else {
				printf("Change Prescaler Setting: alt_globaltmr_prescaler_set(%d)\n", (int)switch_number);
			}
			
			disp_usage = false;
		}
		
		
		// Push-Switch 0
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH0)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH0;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH0\n");
			break;	// Exit Push-Switch 0 and 1 becomes ON.
		}
		// Push-Switch 1
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH1)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH1;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH1\n");
			
			result_code = alt_globaltmr_comp_get(&ui32hcmpbk, &ui32lcmpbk);
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_globaltmr_comp_get() result=%d\n", (int)result_code);
			} else {
				result_code = alt_globaltmr_comp_set((ui32high + 1), ui32lowbk);
				if(result_code != ALT_E_SUCCESS){
					printf("ERROR!!: alt_globaltmr_comp_set() result=%d\n", (int)result_code);
				} else {
					result_code = alt_globaltmr_comp_get(&ui32hcmp, &ui32lcmp);
					if(result_code != ALT_E_SUCCESS){
						printf("ERROR!!: alt_globaltmr_comp_get() result=%d\n", (int)result_code);
					} else {
						printf("* Change Comparator Val:  0x%08X:%08X --> 0x%08X:%08X\n", 
							(int)ui32hcmpbk, (int)ui32lcmpbk, (int)ui32hcmp, (int)ui32lcmp);
					}
				}
			}
			
			disp_usage = true;
		}
		// Push-Switch 2
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH2)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH2;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH2\n");
			
			result_code = alt_globaltmr_comp_get(&ui32hcmp, &ui32lcmp);
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_globaltmr_comp_get() result=%d\n", (int)result_code);
			} else {
				printf("* Display Comparator Val: 0x%08X:%08X\n", (int)ui32hcmp, (int)ui32lcmp);
			}
			
			disp_usage = true;
		}
		// Push-Switch 3
		//if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH3)
		//&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		if(switch_raw & SAMPLE_SWITCH_BIT_PUSH3)
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH3;

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH3\n");
			
			printf("GlobalTimer Counter = 0x%08X:%08X\n", (int)ui32high, (int)ui32low);
			
		} else {
			// Global Timer Counter Value
			if((ui32low ^ ui32lowbk) & 0xFC000000)
			{
				printf("GlobalTimer Counter = 0x%08X:%08X\n", (int)ui32high, (int)ui32low);
			}
		}
		
		
		
		util_intlog_print();

		ui32lowbk = ui32low;
		switch_raw_bk = switch_raw;
	}
	printf("==== End While(1) loop process. ================\n");
	
	util_intlog_print();

	return;
}

/********************************************************************************//*!
 * @fn  void sample_globaltmr_test_print_usage(void) 
 * @brief  Display usage of the sample program. 
 * 
 * @details  Following information displays on the console.
 *            - Overview of HPS for push button functions (or operations)
 *            - Overview of HPS for DIP switch functions (or operations)
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/18  ---  #xxxxx  新規作成. 
 **//*******************************************************************************/
void sample_globaltmr_test_print_usage(void)
{
	printf("\n");
	printf("+--- Usage ( Function of PUSH SW and SLIDE SW )  ---------------+\n");
	printf("+ PUSH SW #0 .... Exit Test loop!!!\n");
	printf("+ PUSH SW #1 .... Update the Comparator value to delay the timer interrupt.\n");
	printf("+ PUSH SW #2 .... Display the current Comparator value.\n");
	printf("+ PUSH SW #3 .... Display the Global Timer counter.\n");
	printf("                  (Display continuously during push-down)\n");
	printf("+ SLIDE SW#0:3 .. Change the timer freqency dynamically.\n"); 
	printf("                  4-bit value set to prescaler intact.\n");
	printf("+---------------------------------------------------------------+\n\n");
	
#if 0
	　無限ループ内で以下の処理を行う。
	　 - Timerカウンタ値の下から26bit目以上（下位32bitのbit#31:26）
	　　 が変化する毎にカウンタ値をコンソール出力。
	　　 HPS用PUSHSW3が押されている間は無条件にコンソール出力。
	　　 (Timerカウンタの進み具合の確認用)
	
	　 - HPS用PUSHSW1の押下契機でComparatorのレジスタ値(64bit)を
	　　 変更。上位32bitは"カウンタの上位32bit値 + 1"、下位32bit
	　　 は Timerカウンタの下位32bit値 に設定し、変更後の値をコンソール出力。
	　　 (Comparatorの動作確認用にComparatorのレジスタ値を少し進める)
	
	　 - HPS用PUSHSW2の押下契機でComparatorのレジスタ値(64bit)を
	　　 コンソール出力。(AutoIncrement機能の確認用)
	
	　 - HPS用DIPSW1-4の変更契機で、Prescalerの設定値を変更。
	　　 DIPSW1をMSB、DIPSW4をLSBとして4bitの値をそのまま
	　　 Prescalerに設定し、設定内容をコンソール出力。
	　　 (Prescalerの設定値に連動して、Timerカウンタの動作速度が
	　　 変化する事が確認できます)

	　 - HPS用PUSHSW0の押下契機で無限ループを終了します。
	　　 (テストプログラム終了)
#endif
	
	return;
}


/********************************/
/*	Test Command (Global Timer)	*/
/********************************/
int sample_globaltmr_test_cmd(char* options)
{
    ALT_STATUS_CODE status = ALT_E_SUCCESS;
    
    printf("\r\nUser Application Start!\r\n");

    //
    // Start the interrupt system
    //
    if (status == ALT_E_SUCCESS)
    {
		util_intlog_init();
        status = socfpga_int_start();
    }

    //// Sample Function Call!!! |---->
	// GPIO Setting for using the DIPSW and PUSHSW
#if 0	// main() 内で cpu0_init() 実施済み
	cpu0_init();
#endif

	// All Functions of Global Timer are disabled.
	sample_globaltmr_alldisable();

	// Configuration for Global Timer Interrupt (GIC Setting)
	sample_globaltmr_setting_gic();

	// Parameter setting for the global timer.
	sample_globaltmr_paraminit();

	// All Functions of Global Timer are enabled.
	sample_globaltmr_allenable();

	// Print initial Values
	sample_globaltmr_print_get_result();

	// Do Test!!
	sample_globaltmr_test_main();
    //// Sample Function Call!!! <----|

    //
    // Stop the interrupt system
    //
    socfpga_int_stop();
	util_intlog_print();

    printf("Finished running the sample !!!\r\n");
    if (status == ALT_E_SUCCESS)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}

// All APIs
/***********************************************************************/
/* Usage: ALT_STATUS_CODE alt_globaltmr_stop(void);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_start(void);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_get(uint32_t* highword, uint32_t* lowword);	*/
/* Usage: uint64_t alt_globaltmr_get64(void);	*/
/* Usage: uint32_t alt_globaltmr_counter_get_low32(void);	*/
/* Usage: uint32_t alt_globaltmr_counter_get_hi32(void);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_comp_set(uint32_t highword, uint32_t loword);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_comp_set64(uint64_t compval);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_comp_get(uint32_t* highword, uint32_t* lowword);	*/
/* Usage: uint64_t alt_globaltmr_comp_get64(void);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_comp_mode_start(void);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_comp_mode_stop(void);	*/
/* Usage: bool alt_globaltmr_is_comp_mode(void);	*/
/* Usage: uint32_t alt_globaltmr_prescaler_get(void);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_prescaler_set(uint32_t val);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_autoinc_set(uint32_t inc);	*/
/* Usage: uint32_t alt_globaltmr_autoinc_get(void);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_autoinc_mode_start(void);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_autoinc_mode_stop(void);	*/
/* Usage: bool alt_globaltmr_is_autoinc_mode(void);	*/
/* Usage: uint32_t alt_globaltmr_maxcounter_get(void);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_int_disable(void);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_int_enable(void);	*/
/* Usage: bool alt_globaltmr_int_is_enabled(void);	*/
/* Usage: ALT_STATUS_CODE alt_globaltmr_int_clear_pending(void);	*/
/* Usage: bool alt_globaltmr_int_is_pending(void);	*/
/* Usage: bool alt_globaltmr_int_if_pending_clear(void);	*/
/***********************************************************************/

/***********************************************************************************
 * end of file 
 ***********************************************************************************/
