﻿//*******************************************************************************
// @file  sample_gpio.c 
// @brief  To use HWLib for the general purpose I/O example program. 
// 
// @details  Run the API category below.
//           - The General Purpose Input/Output Manager API  
//               - General-Purpose IO Configuration Functions  
//               - General-Purpose IO Interrupt Functions  
//               - General-Purpose IO via Bit Index  
//               - General-Purpose IO Utility Functions 
// 
// @note  nothing. 
// 
// @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
// @date 2013/11/06  ---  #6192  新規作成. 
// @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
// @date 2014/08/18  ---  -----  v14.0化 (誤植訂正、操作説明の表示を追加). 
// @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
// @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
// @date 2018/07/02  ---  #25940  All in One ベアメタル・アプリケーション v17.1 への対応を実施.
// @date 2019/01/18  ---  -----   All in One ベアメタル・アプリケーション v18.1 への対応を実施.
// 
// attention
// Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
//   This software is licensed "AS IS". 
//   Please perform use of this software by a user's own responsibility and expense.
//   It cannot guarantee in the maker side about the damage which occurred by the ab-
//   ility not to use or use this software, and all damage that occurred secondarily.
//*******************************************************************************

/***********************************************************************************
 *  includes 
 ***********************************************************************************/
#include <stdio.h>
#include <string.h>
#include "hwlib.h"
#include "socal/socal.h"
#include "socal/hps.h"
#include "alt_generalpurpose_io.h"
#include "alt_interrupt.h"

#include "sample_app_setting.h"
#include "util_time_measurement.h"
#include "util_interrupt_log.h"
#include "alt_printf.h"

#if 9999

#if GEN_SYS_HEADER
#include "hps_system.h"
#else
#define LED_PIO_BASE    (0xff210040)
#define DIPSW_PIO_BASE  (0xff210080)
#define BUTTON_PIO_BASE (0xff2100c0)
#endif

#endif


/***********************************************************************************
 *  externs 
 ***********************************************************************************/


/***********************************************************************************
 *  proto types 
 ***********************************************************************************/
void sample_gpio_iointerrupt(void);
void sample_gpio_callback(uint32_t icciar, void *context);
void sample_gpio_iopolling(void);
void sample_gpio_iconfig(void);
void sample_gpio_config(void);
void sample_gpio_utility(void);
void sample_gpio_test_print_usage(void);
int  sample_gpio_test_cmd(char* options);

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/
#define ALT_GPIO_PORTBIT_ALL	((ALT_GPIO_BIT0) | \
								 (ALT_GPIO_BIT1) | \
								 (ALT_GPIO_BIT2) | \
								 (ALT_GPIO_BIT3) | \
								 (ALT_GPIO_BIT4) | \
								 (ALT_GPIO_BIT5) | \
								 (ALT_GPIO_BIT6) | \
								 (ALT_GPIO_BIT7) | \
								 (ALT_GPIO_BIT8) | \
								 (ALT_GPIO_BIT9) | \
								 (ALT_GPIO_BIT10) | \
								 (ALT_GPIO_BIT11) | \
								 (ALT_GPIO_BIT12) | \
								 (ALT_GPIO_BIT13) | \
								 (ALT_GPIO_BIT14) | \
								 (ALT_GPIO_BIT15) | \
								 (ALT_GPIO_BIT16) | \
								 (ALT_GPIO_BIT17) | \
								 (ALT_GPIO_BIT18) | \
								 (ALT_GPIO_BIT19) | \
								 (ALT_GPIO_BIT20) | \
								 (ALT_GPIO_BIT21) | \
								 (ALT_GPIO_BIT22) | \
								 (ALT_GPIO_BIT23) | \
								 (ALT_GPIO_BIT24) | \
								 (ALT_GPIO_BIT25) | \
								 (ALT_GPIO_BIT26) | \
								 (ALT_GPIO_BIT27) | \
								 (ALT_GPIO_BIT28))

#if TARGET_BOARD_NO==1	/* 1:helio */
//#define HELIO_GPIOPORTC_PUSHSW3		(ALT_GPIO_BIT20)
#define GPIOIRQ_PORT	ALT_GPIO_PORTC
#define GPIOIRQ_BIT		ALT_GPIO_BIT20
#define GPIOIRQ_1BIT	ALT_HLGPI_7
#define GPIOIRQ_IRQNUM	ALT_INT_INTERRUPT_GPIO2

#define ALT_GPIO_PORTBIT_DIPS1		(ALT_GPIO_BIT26)
#define ALT_GPIO_PORTBIT_DIPS2		(ALT_GPIO_BIT25)
#define ALT_GPIO_PORTBIT_DIPS3		(ALT_GPIO_BIT24)
#define ALT_GPIO_PORTBIT_DIPS4		(ALT_GPIO_BIT23)
#define ALT_GPIO_PORTBIT_DIPSWALL	((ALT_GPIO_BIT26)|(ALT_GPIO_BIT25)|(ALT_GPIO_BIT24)|(ALT_GPIO_BIT23))
#define ALT_GPIO_PORTBIT_PUSH3		(ALT_GPIO_BIT22)
#define ALT_GPIO_PORTBIT_PUSH1		(ALT_GPIO_BIT21)
#define ALT_GPIO_PORTBIT_PUSH4		(ALT_GPIO_BIT20)
#define ALT_GPIO_PORTBIT_PUSH2		(ALT_GPIO_BIT19)
#define ALT_GPIO_PORTBIT_PUSHSWALL	((ALT_GPIO_BIT22)|(ALT_GPIO_BIT21)|(ALT_GPIO_BIT20)|(ALT_GPIO_BIT19))
#define ALT_GPIO_PORTBIT_SWALL		((ALT_GPIO_PORTBIT_DIPSWALL)|(ALT_GPIO_PORTBIT_PUSHSWALL))

/** memo: port asign ***************/
/* Signal Name <--> HPS Port Name  */
/* USER_DIPSW_HPS0 <--> HPS_GPI13  */
/* USER_DIPSW_HPS1 <--> HPS_GPI12  */
/* USER_DIPSW_HPS2 <--> HPS_GPI11  */
/* USER_DIPSW_HPS3 <--> HPS_GPI10  */
/* USER_PB_HPS2 <--> HPS_GPI9      */
/* USER_PB_HPS0 <--> HPS_GPI8      */
/* USER_PB_HPS3 <--> HPS_GPI7      */
/* USER_PB_HPS1 <--> HPS_GPI6      */
/* USER_HPS_GPIO61 <--> HPS_GPIO61 */
/***********************************/

#elif TARGET_BOARD_NO==2	/* 2:atlas */
//#define ATLAS_GPIOPORTB_PUSH2		(ALT_GPIO_BIT25)
#define GPIOIRQ_PORT	ALT_GPIO_PORTB
#define GPIOIRQ_BIT		ALT_GPIO_BIT25
#define GPIOIRQ_1BIT	ALT_GPIO_1BIT_54
#define GPIOIRQ_IRQNUM	ALT_INT_INTERRUPT_GPIO1
#elif TARGET_BOARD_NO==3	/* 3:sodia */
//#define SODIA_GPIOPORTC_PUSHSW3		(ALT_GPIO_BIT24)
#define GPIOIRQ_PORT	ALT_GPIO_PORTC
#define GPIOIRQ_BIT		ALT_GPIO_BIT24
#define GPIOIRQ_1BIT	ALT_HLGPI_11
#define GPIOIRQ_IRQNUM	ALT_INT_INTERRUPT_GPIO2
#elif TARGET_BOARD_NO==4	/* 4:c5socdk */
//#define C5SOCDK_GPIOPORTC_PUSHSW3		(ALT_GPIO_BIT24)
#define GPIOIRQ_PORT	ALT_GPIO_PORTC
#define GPIOIRQ_BIT		ALT_GPIO_BIT24
#define GPIOIRQ_1BIT	ALT_HLGPI_11
#define GPIOIRQ_IRQNUM	ALT_INT_INTERRUPT_GPIO2
#elif TARGET_BOARD_NO==6	/* 6:DE10-Nano */
//#define ATLAS_GPIOPORTB_PUSH2		(ALT_GPIO_BIT25)
#define GPIOIRQ_PORT	ALT_GPIO_PORTB
#define GPIOIRQ_BIT		ALT_GPIO_BIT25
#define GPIOIRQ_1BIT	ALT_GPIO_1BIT_54
#define GPIOIRQ_IRQNUM	ALT_INT_INTERRUPT_GPIO1
#endif	/* TARGET_BOARD_NO */

/***********************************************************************************
 *  variables 
 ***********************************************************************************/

/***********************************************************************************
 *  functions 
 ***********************************************************************************/

/********************************************************************************//*!
 * @fn  void sample_gpio_iointerrupt(void) 
 * @brief  Interrupt controller Setting for GPIO interrupts.
 * 
 * @details  In order to receive the GPIO interrupts, interrupt callback routine 
 *           registered, priority setting, configuration of the Distributor, and 
 *           interrupt enable setting.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/06  ---  #6192  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gpio_iointerrupt(void)
{
	ALT_STATUS_CODE result_code;
	uint32_t ui32temp = 0;
	ALT_INT_TRIGGER_t trigger_type = ALT_INT_TRIGGER_NA;

	// Print initial value (GIC: GPIO2)
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_secure(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_enabled(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_pending(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_active(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_priority_get(ALT_INT_INTERRUPT_t int_id, uint32_t* priority);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_target_get(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t* target);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_get(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t* trigger_type);	*/
	/****************************************************************************************/
	printf("-- HWLib test : alt_int_dist_is_secure() --\n");
	printf("arg1=GPIOIRQ_IRQNUM(%d) --> result=%d\n", (int)GPIOIRQ_IRQNUM, (int)alt_int_dist_is_secure(GPIOIRQ_IRQNUM));
	printf("\n");
	printf("-- HWLib test : alt_int_dist_is_enabled() --\n");
	printf("arg1=GPIOIRQ_IRQNUM(%d) --> result=%d\n", (int)GPIOIRQ_IRQNUM, (int)alt_int_dist_is_enabled(GPIOIRQ_IRQNUM));
	printf("\n");
	printf("-- HWLib test : alt_int_dist_is_pending() --\n");
	printf("arg1=GPIOIRQ_IRQNUM(%d) --> result=%d\n", (int)GPIOIRQ_IRQNUM, (int)alt_int_dist_is_pending(GPIOIRQ_IRQNUM));
	printf("\n");
	printf("-- HWLib test : alt_int_dist_is_active() --\n");
	printf("arg1=GPIOIRQ_IRQNUM(%d) --> result=%d\n", (int)GPIOIRQ_IRQNUM, (int)alt_int_dist_is_active(GPIOIRQ_IRQNUM));
	printf("\n");
	printf("-- HWLib test : alt_int_dist_priority_get() --\n");
	printf("arg1=GPIOIRQ_IRQNUM(%d) --> result=%d out=0x%08X\n", (int)GPIOIRQ_IRQNUM, (int)alt_int_dist_priority_get(GPIOIRQ_IRQNUM,&ui32temp), (int)ui32temp);
	printf("\n");
	printf("-- HWLib test : alt_int_dist_target_get() --\n");
	printf("arg1=GPIOIRQ_IRQNUM(%d) --> result=%d out=0x%08X\n", (int)GPIOIRQ_IRQNUM, (int)alt_int_dist_target_get(GPIOIRQ_IRQNUM,&ui32temp), (int)ui32temp);
	printf("\n");
	printf("-- HWLib test : alt_int_dist_trigger_get() --\n");
	printf("arg1=GPIOIRQ_IRQNUM(%d) --> result=%d out=%d\n", (int)GPIOIRQ_IRQNUM, (int)alt_int_dist_trigger_get(GPIOIRQ_IRQNUM,&trigger_type), (int)trigger_type);
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_isr_register(ALT_INT_INTERRUPT_t int_id, alt_int_callback_t callback, void* context);	*/
	/*  typedef void(* alt_int_callback_t)(uint32_t icciar, void *context) */
	printf("-- HWLib test : alt_int_isr_register() --\n");
	result_code = alt_int_isr_register(GPIOIRQ_IRQNUM, sample_gpio_callback, NULL);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_isr_register() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_target_set(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t target);	*/
	printf("-- HWLib test : alt_int_dist_target_set() --\n");
	result_code = alt_int_dist_target_set(GPIOIRQ_IRQNUM, 1);		// arg2=targetCPU (Only Core#0)
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_target_set() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_set(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t trigger_type);	*/
	printf("-- HWLib test : alt_int_dist_trigger_set() --\n");
	result_code = alt_int_dist_trigger_set(GPIOIRQ_IRQNUM, ALT_INT_TRIGGER_LEVEL);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_trigger_set() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_priority_set(ALT_INT_INTERRUPT_t int_id, uint32_t priority);	*/
	printf("-- HWLib test : alt_int_dist_priority_set() --\n");
	result_code = alt_int_dist_priority_set(GPIOIRQ_IRQNUM, 99);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_priority_set() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_enable(ALT_INT_INTERRUPT_t int_id);	*/
	printf("-- HWLib test : alt_int_dist_enable() --\n");
	result_code = alt_int_dist_enable(GPIOIRQ_IRQNUM);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_enable() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_secure_enable(ALT_INT_INTERRUPT_t int_id);	*/
	printf("-- HWLib test : alt_int_dist_secure_enable() --\n");
	result_code = alt_int_dist_secure_enable(GPIOIRQ_IRQNUM);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_enable() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");


	return;
}

/********************************************************************************//*!
 * @fn  void sample_gpio_callback(uint32_t icciar, void *context) 
 * @brief  GPIO interrupt callback function.
 * 
 * @details  working opportunity an interrupt from the GPIO, indicate an 
 *           interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued. 
 * @param[in] context  The user provided context.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/06  ---  #6192  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gpio_callback(uint32_t icciar, void *context)
{
	uint32_t ui32result;
	ALT_STATUS_CODE result_code;

	/* Usage: uint32_t alt_gpio_port_int_status_get(ALT_GPIO_PORT_t gpio_pid);				*/
	ui32result = alt_gpio_port_int_status_get(GPIOIRQ_PORT);
	//printf("[INTERRUPT]alt_gpio_port_int_status_get(GPIOIRQ_PORT) -.-> result=0x%08X\n", (int)ui32result);
	util_intlog_record(GPIOIRQ_IRQNUM, (int)icciar, (int)ui32result);

	/* Usage: ALT_STATUS_CODE alt_gpio_port_int_status_clear(ALT_GPIO_PORT_t gpio_pid, uint32_t clrmask);	*/
	result_code = alt_gpio_port_int_status_clear(GPIOIRQ_PORT, ui32result);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_gpio_port_int_status_clear() result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_int_dist_pending_clear(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_pending_clear(GPIOIRQ_IRQNUM);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}
}

/********************************************************************************//*!
 * @fn  void sample_gpio_iopolling(void) 
 * @brief  Main routine of the sample program. 
 * 
 * @details  Infinite loop structure works. Make the detection of various types 
 *           of switches in the loop, it performs in various testing processes.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/06  ---  #6192  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gpio_iopolling(void)
{
	ALT_STATUS_CODE result_code;
	uint32_t switch_raw = 0;
	uint32_t switch_raw_bk = 0;
	uint32_t switch_detect = 0;
	uint32_t switch_detect_on = 0;
	uint32_t switch_number = 0;
	bool disp_usage = true;
	uint32_t ui32result;
	uint32_t ui32backup;
	
	printf("==== Start While(1) loop process!!! ============\n");
	switch_raw_bk = sample_detect_switch();
	ui32backup = alt_gpio_port_data_read(GPIOIRQ_PORT, (uint32_t)GPIOIRQ_BIT);
	printf("Initial value GPIOIRQ_PORT(%d) = 0x%08X\n", (int)GPIOIRQ_PORT, (int)ui32backup);
	while(1)
	{
		if(disp_usage)
		{
			sample_gpio_test_print_usage();
			disp_usage = false;
		}

		// ---- Check the Slide-Switch and Push-Switch. ---- 
		switch_raw = sample_detect_switch();
		switch_detect = switch_raw ^ switch_raw_bk;
		switch_detect_on  |= switch_detect & switch_raw;
		
		ui32result = alt_gpio_port_data_read(GPIOIRQ_PORT, GPIOIRQ_BIT);
		if(ui32result != ui32backup)
		{
			printf("Change detection GPIOIRQ_PORT(%d) = 0x%08X --> 0x%08X\n", (int)GPIOIRQ_PORT, (int)ui32backup, (int)ui32result);
		}
		
		// Slide-Switch 
		if(switch_detect & SAMPLE_SWITCH_BIT_SLIDEALL)
		{
			ALT_GPIO_PIN_TYPE_t type;
			ALT_GPIO_PIN_POL_t pol;
			
			switch_number = switch_raw & SAMPLE_SWITCH_BIT_NUM;
			switch_number *= 1;		// To avoid warnings.

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE0){
				//printf("SAMPLE_SWITCH_BIT_SLIDE0\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE1){
				//printf("SAMPLE_SWITCH_BIT_SLIDE1\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE2){
				//printf("SAMPLE_SWITCH_BIT_SLIDE2\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE3){
				//printf("SAMPLE_SWITCH_BIT_SLIDE3\n");
			}
			
			switch(switch_number & (SAMPLE_SWITCH_BIT_SLIDE0 | SAMPLE_SWITCH_BIT_SLIDE1))
			{
			case 0:		/* SLIDE SW#0:1 = 0 ... Rising-Edge */
				printf("Resetting PUSH SW #3: Rising-Edge\n");
				type = ALT_GPIO_PIN_EDGE_TRIG_INT;
				pol = ALT_GPIO_PIN_ACTIVE_HIGH;
				break;
			case 1:		/* SLIDE SW#0:1 = 1 ... Falling-Edge */
				printf("Resetting PUSH SW #3: Falling-Edge\n");
				type = ALT_GPIO_PIN_EDGE_TRIG_INT;
				pol = ALT_GPIO_PIN_ACTIVE_LOW;
				break;
			case 2:		/* SLIDE SW#0:1 = 2 ... High-Level */
				printf("Resetting PUSH SW #3: High-Level\n");
				type = ALT_GPIO_PIN_LEVEL_TRIG_INT;
				pol = ALT_GPIO_PIN_ACTIVE_HIGH;
				break;
			/* case 3: */
			default:	/* SLIDE SW#0:1 = 3 ... Low-Level */
				printf("Resetting PUSH SW #3: Low-Level\n");
				type = ALT_GPIO_PIN_LEVEL_TRIG_INT;
				pol = ALT_GPIO_PIN_ACTIVE_LOW;
				break;
			}
			
			// setting GPIOIRQ_PORT/BIT  via normal configuration func.
			/****************************************************************************************/
			/* Usage: ALT_STATUS_CODE alt_gpio_port_config(	ALT_GPIO_PORT_t gpio_pid,
			 * 												uint32_t mask,
			 * 												ALT_GPIO_PIN_DIR_t dir,
			 * 												ALT_GPIO_PIN_TYPE_t type,
			 * 												ALT_GPIO_PIN_POL_t pol,
			 * 												ALT_GPIO_PIN_DEBOUNCE_t debounc,
			 * 												uint32_t data); 						*/
			/****************************************************************************************/
			printf("-- HWLib test : alt_gpio_port_config() --\n");
			result_code = alt_gpio_port_config(GPIOIRQ_PORT, GPIOIRQ_BIT,
					ALT_GPIO_PIN_INPUT, type, pol, ALT_GPIO_PIN_NODEBOUNCE, 0);
			if(result_code != ALT_E_SUCCESS)
			{
				printf("ERROR!!: alt_gpio_port_config() result=%d\n", (int)result_code);
			}
			else
			{
				printf("SUCCESS!!\n");
			}
			printf("\n");
			
			disp_usage = false;
		}
		
		
		// Push-Switch 0
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH0)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH0;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH0\n");
			break;	// Exit Push-Switch 0 and 1 becomes ON.
		}
		// Push-Switch 1
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH1)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH1;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH1\n");
			disp_usage = true;
		}
		// Push-Switch 2
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH2)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH2;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH2\n");
			disp_usage = true;
		}
		// Push-Switch 3
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH3)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH3;

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH3\n");
			disp_usage = true;
		}
		
		util_intlog_print();

		ui32backup = ui32result;
		switch_raw_bk = switch_raw;
	}
	printf("==== End While(1) loop process. ================\n");

	util_intlog_print();

	return;
}

/********************************************************************************//*!
 * @fn  void sample_gpio_iconfig(void) 
 * @brief  Internal setting of GPIO block for interrupts. 
 * 
 * @details  In order to receive the GPIO interrupts, polarity (Edge or Level, 
 *           Rise(High) or Fall(Low)..), mask, and interrupt enable setting.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/06  ---  #6192  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gpio_iconfig(void)
{
	uint32_t ui32result;
	ALT_STATUS_CODE result_code;

	// Print initial value
	/****************************************************************************************/
	/* Usage: uint32_t alt_gpio_port_int_type_get(ALT_GPIO_PORT_t gpio_pid, uint32_t mask);	*/
	/* Usage: uint32_t alt_gpio_port_int_pol_get(ALT_GPIO_PORT_t gpio_pid, uint32_t mask);	*/
	/* Usage: uint32_t alt_gpio_port_int_enable_get(ALT_GPIO_PORT_t gpio_pid);				*/
	/* Usage: uint32_t alt_gpio_port_int_mask_get(ALT_GPIO_PORT_t gpio_pid);				*/
	/* Usage: uint32_t alt_gpio_port_int_status_get(ALT_GPIO_PORT_t gpio_pid);				*/
	/****************************************************************************************/
	printf("-- HWLib test : alt_gpio_port_int_type_get() --\n");
	ui32result = alt_gpio_port_int_type_get(ALT_GPIO_PORTA, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTA, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_int_type_get(ALT_GPIO_PORTB, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTB, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_int_type_get(ALT_GPIO_PORTC, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTC, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_int_pol_get() --\n");
	ui32result = alt_gpio_port_int_pol_get(ALT_GPIO_PORTA, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTA, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_int_pol_get(ALT_GPIO_PORTB, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTB, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_int_pol_get(ALT_GPIO_PORTC, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTC, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_int_enable_get() --\n");
	ui32result = alt_gpio_port_int_enable_get(ALT_GPIO_PORTA);
	printf("arg1=ALT_GPIO_PORTA --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_int_enable_get(ALT_GPIO_PORTB);
	printf("arg1=ALT_GPIO_PORTB --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_int_enable_get(ALT_GPIO_PORTC);
	printf("arg1=ALT_GPIO_PORTC --> result=0x%08X\n", (int)ui32result);
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_int_mask_get() --\n");
	ui32result = alt_gpio_port_int_mask_get(ALT_GPIO_PORTA);
	printf("arg1=ALT_GPIO_PORTA --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_int_mask_get(ALT_GPIO_PORTB);
	printf("arg1=ALT_GPIO_PORTB --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_int_mask_get(ALT_GPIO_PORTC);
	printf("arg1=ALT_GPIO_PORTC --> result=0x%08X\n", (int)ui32result);
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_int_status_get() --\n");
	ui32result = alt_gpio_port_int_status_get(ALT_GPIO_PORTA);
	printf("arg1=ALT_GPIO_PORTA --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_int_status_get(ALT_GPIO_PORTB);
	printf("arg1=ALT_GPIO_PORTB --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_int_status_get(ALT_GPIO_PORTC);
	printf("arg1=ALT_GPIO_PORTC --> result=0x%08X\n", (int)ui32result);
	printf("\n");

	// setting gpio interrupts
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_port_int_type_set(ALT_GPIO_PORT_t gpio_pid, uint32_t mask, uint32_t config); */
	/* Usage: ALT_STATUS_CODE alt_gpio_port_int_pol_set(ALT_GPIO_PORT_t gpio_pid, uint32_t mask, uint32_t config); 	*/
	/* Usage: ALT_STATUS_CODE alt_gpio_port_int_enable(ALT_GPIO_PORT_t gpio_pid, uint32_t config);					*/
	/* Usage: ALT_STATUS_CODE alt_gpio_port_int_disable(ALT_GPIO_PORT_t gpio_pid, uint32_t config);					*/
	/* Usage: ALT_STATUS_CODE alt_gpio_port_int_mask_set(ALT_GPIO_PORT_t gpio_pid, uint32_t mask, uint32_t val);	*/
	/****************************************************************************************/
	printf("-- HWLib test : alt_gpio_port_int_enable() --\n");
	result_code = alt_gpio_port_int_enable(ALT_GPIO_PORTA, 0);
	result_code |= alt_gpio_port_int_enable(ALT_GPIO_PORTB, 0);
	result_code |= alt_gpio_port_int_enable(ALT_GPIO_PORTC, 0);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_int_enable() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!! ALT_GPIO_PORTA=0x%08X / PORTB=0x%08X / PORTC=0x%08X\n",
				(int)alt_gpio_port_int_enable_get(ALT_GPIO_PORTA),
				(int)alt_gpio_port_int_enable_get(ALT_GPIO_PORTB),
				(int)alt_gpio_port_int_enable_get(ALT_GPIO_PORTC));
	}
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_int_type_set() --\n");
	result_code = alt_gpio_port_int_type_set(GPIOIRQ_PORT, GPIOIRQ_BIT, GPIOIRQ_BIT);	// Edge Sense
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_int_type_set() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_int_pol_set() --\n");
	result_code = alt_gpio_port_int_pol_set(GPIOIRQ_PORT, GPIOIRQ_BIT, GPIOIRQ_BIT);	// RisingEdge 
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_int_pol_set() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_int_disable() --\n");
	result_code = alt_gpio_port_int_disable(ALT_GPIO_PORTA, 0);
	result_code |= alt_gpio_port_int_disable(ALT_GPIO_PORTB, 0);
	result_code |= alt_gpio_port_int_disable(ALT_GPIO_PORTC, 0);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_int_disable() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!! ALT_GPIO_PORTA=0x%08X / PORTB=0x%08X / PORTC=0x%08X\n",
				(int)alt_gpio_port_int_enable_get(ALT_GPIO_PORTA),
				(int)alt_gpio_port_int_enable_get(ALT_GPIO_PORTB),
				(int)alt_gpio_port_int_enable_get(ALT_GPIO_PORTC));
	}
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_int_mask_set() --\n");
	result_code = alt_gpio_port_int_mask_set(GPIOIRQ_PORT, ~GPIOIRQ_BIT, ALT_GPIO_PORTBIT_ALL);	// DIPSW..Interrupt Mask
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_int_mask_set() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_int_enable() --\n");
	result_code = alt_gpio_port_int_enable(GPIOIRQ_PORT, GPIOIRQ_BIT);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_int_enable() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!! ALT_GPIO_PORTx=0x%08X\n", (int)alt_gpio_port_int_enable_get(GPIOIRQ_PORT));
	}
	printf("\n");



	return;
}

/********************************************************************************//*!
 * @fn  void sample_gpio_config(void) 
 * @brief  GPIO configuration for the sample programs run.
 * 
 * @details  Performs the following processing. 
 *           - Setting DIPSW via normal configuration func. 
 *           - Setting OUT-PORT（GPIO61） via normal configuration func. 
 *           - Setting PUSHSW via bit index configuration func. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/06  ---  #6192  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gpio_config(void)
{
	uint32_t ui32result;
	ALT_STATUS_CODE result_code;
	ALT_GPIO_CONFIG_RECORD_t gpio_config_array[10];
	ALT_GPIO_CONFIG_RECORD_t* p_gpiocfg;

	// Print initial value
	/****************************************************************************************/
	/* Usage: uint32_t alt_gpio_port_datadir_get(ALT_GPIO_PORT_t gpio_pid, uint32_t mask);	*/
	/* Usage: uint32_t alt_gpio_port_debounce_get(ALT_GPIO_PORT_t gpio_pid, uint32_t mask);	*/
	/* Usage: ALT_STATUS_CODE alt_gpio_port_sync_get(ALT_GPIO_PORT_t gpio_pid);				*/
	/* Usage: uint32_t alt_gpio_port_data_read(ALT_GPIO_PORT_t gpio_pid, uint32_t mask);	*/
	/****************************************************************************************/
	printf("-- HWLib test : alt_gpio_port_datadir_get() --\n");
	ui32result = alt_gpio_port_datadir_get(ALT_GPIO_PORTA, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTA, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_datadir_get(ALT_GPIO_PORTB, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTB, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_datadir_get(ALT_GPIO_PORTC, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTC, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_datadir_get() --\n");
	ui32result = alt_gpio_port_debounce_get(ALT_GPIO_PORTA, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTA, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_debounce_get(ALT_GPIO_PORTB, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTB, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_debounce_get(ALT_GPIO_PORTC, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTC, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_sync_get() --\n");
	printf("arg1=ALT_GPIO_PORTA --> result=%d\n", (int)alt_gpio_port_sync_get(ALT_GPIO_PORTA));
	printf("arg1=ALT_GPIO_PORTB --> result=%d\n", (int)alt_gpio_port_sync_get(ALT_GPIO_PORTB));
	printf("arg1=ALT_GPIO_PORTC --> result=%d\n", (int)alt_gpio_port_sync_get(ALT_GPIO_PORTC));
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_data_read() --\n");
	ui32result = alt_gpio_port_data_read(ALT_GPIO_PORTA, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTA, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_data_read(ALT_GPIO_PORTB, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTB, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	ui32result = alt_gpio_port_data_read(ALT_GPIO_PORTC, ALT_GPIO_PORTBIT_ALL);
	printf("arg1=ALT_GPIO_PORTC, arg2=ALLBIT --> result=0x%08X\n", (int)ui32result);
	printf("\n");

#if TARGET_BOARD_NO==1	/* 1:helio */
	// setting DIPSW via normal configuration func.
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_port_datadir_set(ALT_GPIO_PORT_t gpio_pid, uint32_t mask, uint32_t config); */
	/* Usage: ALT_STATUS_CODE alt_gpio_port_debounce_set(ALT_GPIO_PORT_t gpio_pid, uint32_t mask, uint32_t config); */
	/* Usage: ALT_STATUS_CODE alt_gpio_port_sync_set(ALT_GPIO_PORT_t gpio_pid, uint32_t config); */
	/****************************************************************************************/
	printf("-- HWLib test : alt_gpio_port_datadir_set() --\n");
	result_code = alt_gpio_port_datadir_set(ALT_GPIO_PORTC, ALT_GPIO_PORTBIT_DIPSWALL, 0);	// all input port
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_datadir_set() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_debounce_set() --\n");
	result_code = alt_gpio_port_debounce_set(ALT_GPIO_PORTC, ALT_GPIO_PORTBIT_DIPSWALL, ALT_GPIO_PORTBIT_DIPSWALL);	// all debounce enable
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_debounce_set() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");
	printf("-- HWLib test : alt_gpio_port_sync_set() --\n");
	result_code = alt_gpio_port_sync_set(ALT_GPIO_PORTC, 0);	// Do not synchronize to internal clock signal.
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_sync_set() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");

	// setting OUT-PORT（GPIO61） via normal configuration func.
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_port_config(	ALT_GPIO_PORT_t gpio_pid,
	 * 												uint32_t mask,
	 * 												ALT_GPIO_PIN_DIR_t dir,
	 * 												ALT_GPIO_PIN_TYPE_t type,
	 * 												ALT_GPIO_PIN_POL_t pol,
	 * 												ALT_GPIO_PIN_DEBOUNCE_t debounc,
	 * 												uint32_t data); 						*/
	/****************************************************************************************/
	printf("-- HWLib test : alt_gpio_port_config() --\n");
	result_code = alt_gpio_port_config(ALT_GPIO_PORTC, ALT_GPIO_BIT3,
			ALT_GPIO_PIN_OUTPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_BIT3);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");
#endif	/* TARGET_BOARD_NO */

	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_bitconfig_get(ALT_GPIO_1BIT_t signal_num, ALT_GPIO_CONFIG_RECORD_t* config); */
	/****************************************************************************************/
	printf("-- HWLib test : alt_gpio_bitconfig_get() --\n");
	result_code = alt_gpio_bitconfig_get(ALT_GPIO_1BIT_61, &gpio_config_array[0]);
	result_code |= alt_gpio_bitconfig_get(ALT_HLGPI_6, &gpio_config_array[1]);
	result_code |= alt_gpio_bitconfig_get(ALT_HLGPI_7, &gpio_config_array[2]);
	result_code |= alt_gpio_bitconfig_get(ALT_HLGPI_8, &gpio_config_array[3]);
	result_code |= alt_gpio_bitconfig_get(ALT_HLGPI_9, &gpio_config_array[4]);
	result_code |= alt_gpio_bitconfig_get(ALT_HLGPI_10, &gpio_config_array[5]);
	result_code |= alt_gpio_bitconfig_get(ALT_HLGPI_11, &gpio_config_array[6]);
	result_code |= alt_gpio_bitconfig_get(ALT_HLGPI_12, &gpio_config_array[7]);
	result_code |= alt_gpio_bitconfig_get(ALT_HLGPI_13, &gpio_config_array[8]);
	gpio_config_array[9].signal_number = ALT_END_OF_GPIO_SIGNALS;
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_bitconfig_get() result=%d\n", (int)result_code);
	} else
	{
		for(int i=0; i<10; i++)
		{
			p_gpiocfg = &gpio_config_array[i];
			printf("gpio_config_array[%d] = {sig#=%d, dire=%d, type=%d, pola=%d, debo=%d, data=%d}\n",
					i, (int)p_gpiocfg->signal_number, (int)p_gpiocfg->direction, (int)p_gpiocfg->type,
					(int)p_gpiocfg->polarity, (int)p_gpiocfg->debounce, (int)p_gpiocfg->data);
		}
	}
	printf("\n");

#if TARGET_BOARD_NO==1	/* 1:helio */
	/* USER_PB_HPS0 <--> HPS_GPI9      */
	/* USER_PB_HPS1 <--> HPS_GPI8      */
	/* USER_PB_HPS2 <--> HPS_GPI7      */
	/* USER_PB_HPS3 <--> HPS_GPI6      */
	// setting PUSHSW via bit index configuration func.
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_bit_config(	ALT_GPIO_1BIT_t signal_num,
	 * 												ALT_GPIO_PIN_DIR_t dir,
	 * 												ALT_GPIO_PIN_TYPE_t type,
	 * 												ALT_GPIO_PIN_POL_t pol,
	 * 												ALT_GPIO_PIN_DEBOUNCE_t debounce,
	 * 												ALT_GPIO_PIN_DATA_t data);				*/
	/****************************************************************************************/
	printf("-- HWLib test : alt_gpio_bit_config() --\n");
	result_code = alt_gpio_bit_config(ALT_HLGPI_9, ALT_GPIO_PIN_INPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT,
			ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE, ALT_GPIO_PIN_DATAZERO);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_bit_config() result=%d\n", (int)result_code);
	}
	else
	{
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_group_config(ALT_GPIO_CONFIG_RECORD_t* config_array, uint32_t len); */
	/****************************************************************************************/
	printf("-- HWLib test : alt_gpio_group_config() --\n");
	for(int i=1; i<10; i++)
	{
		p_gpiocfg = &gpio_config_array[i];
		p_gpiocfg->direction = ALT_GPIO_PIN_INPUT;
		p_gpiocfg->type = ALT_GPIO_PIN_LEVEL_TRIG_INT;
		p_gpiocfg->polarity = ALT_GPIO_PIN_ACTIVE_HIGH;
		p_gpiocfg->debounce = ALT_GPIO_PIN_NODEBOUNCE;
		p_gpiocfg->data = ALT_GPIO_PIN_DATAZERO;
	}
	result_code = alt_gpio_group_config(gpio_config_array, 10);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_group_config() result=%d\n", (int)result_code);
	}
	printf("\n");
#endif	/* TARGET_BOARD_NO */

	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_group_config_get(ALT_GPIO_CONFIG_RECORD_t* config_array, uint32_t len);	*/
	/****************************************************************************************/
	printf("-- HWLib test : alt_gpio_group_config_get() --\n");
	result_code = alt_gpio_group_config_get(gpio_config_array, 10);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_group_config_get() result=%d\n", (int)result_code);
	} else
	{
		for(int i=0; i<10; i++)
		{
			p_gpiocfg = &gpio_config_array[i];
			printf("gpio_config_array[%d] = {sig#=%d, dire=%d, type=%d, pola=%d, debo=%d, data=%d}\n",
					i, (int)p_gpiocfg->signal_number, (int)p_gpiocfg->direction, (int)p_gpiocfg->type,
					(int)p_gpiocfg->polarity, (int)p_gpiocfg->debounce, (int)p_gpiocfg->data);
		}
	}
	printf("\n");

	return;
}


/********************************************************************************//*!
 * @fn  void sample_gpio_utility(void) 
 * @brief  Try calling the GPIO utility API. 
 * 
 * @details  Run the API of the subcategory "General-Purpose IO Utility Functions". 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/06  ---  #6192  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gpio_utility(void)
{
	uint32_t ui32result;

	/***********************************************************************/
	/* Usage: uint32_t alt_gpio_port_idcode_get(ALT_GPIO_PORT_t gpio_pid); */
	/***********************************************************************/
	printf("-- HWLib test: alt_gpio_port_idcode_get() --\n");
	ui32result = alt_gpio_port_idcode_get(ALT_GPIO_PORTA);
	printf("arg1=ALT_GPIO_PORTA(%d) --> result=0x%08X(%d)\n", (int)ALT_GPIO_PORTA, (int)ui32result, (int)ui32result);
	ui32result = alt_gpio_port_idcode_get(ALT_GPIO_PORTB);
	printf("arg1=ALT_GPIO_PORTB(%d) --> result=0x%08X(%d)\n", (int)ALT_GPIO_PORTB, (int)ui32result, (int)ui32result);
	ui32result = alt_gpio_port_idcode_get(ALT_GPIO_PORTC);
	printf("arg1=ALT_GPIO_PORTC(%d) --> result=0x%08X(%d)\n", (int)ALT_GPIO_PORTC, (int)ui32result, (int)ui32result);
	printf("\n");

	/********************************************************************/
	/* Usage: uint32_t alt_gpio_port_ver_get(ALT_GPIO_PORT_t gpio_pid);	*/
	/********************************************************************/
	printf("-- HWLib test : alt_gpio_port_ver_get() --\n");
	ui32result = alt_gpio_port_ver_get(ALT_GPIO_PORTA);
	printf("arg1=ALT_GPIO_PORTA(%d) --> result=0x%08X (%d)\n", (int)ALT_GPIO_PORTA, (int)ui32result, (int)ui32result);
	ui32result = alt_gpio_port_ver_get(ALT_GPIO_PORTB);
	printf("arg1=ALT_GPIO_PORTB(%d) --> result=0x%08X (%d)\n", (int)ALT_GPIO_PORTB, (int)ui32result, (int)ui32result);
	ui32result = alt_gpio_port_ver_get(ALT_GPIO_PORTC);
	printf("arg1=ALT_GPIO_PORTC(%d) --> result=0x%08X (%d)\n", (int)ALT_GPIO_PORTC, (int)ui32result, (int)ui32result);
	printf("\n");

	/************************************************************************/
	/* Usage: ALT_GPIO_PORT_t alt_gpio_bit_to_pid(ALT_GPIO_1BIT_t pin_num);	*/
	/************************************************************************/
	/********************************************************************************/
	/* Usage: ALT_GPIO_PORTBIT_t alt_gpio_bit_to_port_pin(ALT_GPIO_1BIT_t pin_num);	*/
	/********************************************************************************/
	printf("-- HWLib test : alt_gpio_bit_to_pid(), alt_gpio_bit_to_port_pin() --\n");
	printf("arg1=GPIOIRQ_BIT(0x%X) --> result{pid=%d, bit=%d}\n",
			(int)GPIOIRQ_BIT, (int)alt_gpio_bit_to_pid(GPIOIRQ_1BIT), (int)alt_gpio_bit_to_port_pin(GPIOIRQ_1BIT));
	printf("\n");

	/*******************************************************************************************/
	/* Usage: ALT_GPIO_1BIT_t alt_gpio_port_pin_to_bit(ALT_GPIO_PORT_t pid, uint32_t bitmask); */
	/*******************************************************************************************/
	printf("-- HWLib test : alt_gpio_port_pin_to_bit() --\n");
	ui32result = alt_gpio_port_pin_to_bit(GPIOIRQ_PORT,GPIOIRQ_BIT);
	printf("arg1=GPIOIRQ_PORT(%d), arg2=GPIOIRQ_BIT(0x%8X) --> result0x%08X(%d)\n", 
			(int)GPIOIRQ_PORT, (int)GPIOIRQ_BIT, (int)ui32result, (int)ui32result);
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  void sample_gpio_test_print_usage(void) 
 * @brief  Display usage of the sample program. 
 * 
 * @details  Following information displays on the console.
 *            - Overview of HPS for push button functions (or operations)
 *            - Overview of HPS for DIP switch functions (or operations)
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/18  ---  #xxxxx  新規作成. 
 **//*******************************************************************************/
void sample_gpio_test_print_usage(void)
{
	printf("\n");
	printf("+--- Usage ( Function of PUSH SW and SLIDE SW )  ---------------+\n");
	printf("+ PUSH SW #0 .... Exit Test loop!!! \n");
	printf("+ PUSH SW #3 .... Generate GPIO interrupt.\n");
	printf("+ SLIDE SW#0:1 .. Select interrupt triger and polarity.\n");
	printf("+  - SLIDE SW#0:1 = 0 --- PUSH SW #3 is detected at Rising-Edge.\n");
	printf("+  - SLIDE SW#0:1 = 1 --- PUSH SW #3 is detected at Falling-Edge.\n");
	printf("+  - SLIDE SW#0:1 = 2 --- PUSH SW #3 is detected at High-Level.\n");
	printf("+  - SLIDE SW#0:1 = 3 --- PUSH SW #3 is detected at Low-Level.\n");
	printf("+---------------------------------------------------------------+\n\n");
	
#if 0
	動作確認用に割り込みコントローラ(GIC)の設定変更を行った後、
	下記の条件で割り込みが動作します。
	　 - GPIO割り込み契機で sample_gpio_callback() が起動します。
	　 - SLIDEスイッチ(DIPSW) SW#0:1 を利用して割り込みトリガーを選択します。
	　　　SLIDE SW#0:1 = 0 ... Rising-Edge
	　　　SLIDE SW#0:1 = 1 ... Falling-Edge
	　　　SLIDE SW#0:1 = 2 ... Rising-Edge
	　　　SLIDE SW#0:1 = 3 ... Falling-Edge
	　 - PUSH SW #3 を契機にGPIO割り込みを発生させます。
#endif
	
	return;
}


/************************/
/*	Test Command (GPIO)	*/
/************************/
int sample_gpio_test_cmd(char* options)
{
    ALT_STATUS_CODE status = ALT_E_SUCCESS;
    
    printf("\r\nUser Application Start!\r\n");

    //
    // Start the interrupt system
    //
    if (status == ALT_E_SUCCESS)
    {
		util_intlog_init();
        status = socfpga_int_start();
    }

    //// Sample Function Call!!! |---->
#if 9999-9999
	cpu0_init();	// main() 内で cpu0_init() 実施済み
#endif
    sample_gpio_utility();
    sample_gpio_config();
    sample_gpio_iconfig();
    sample_gpio_iointerrupt();
    sample_gpio_iopolling();
    //// Sample Function Call!!! <----|

    //
    // Stop the interrupt system
    //
    socfpga_int_stop();
	util_intlog_print();

    printf("Finished running the sample !!!\r\n");
    if (status == ALT_E_SUCCESS)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}
/***********************************************************************************
 * end of file 
 ***********************************************************************************/
