﻿/********************************************************************************//*!
 * @file  sample_app.c
 * @brief  Example program using HPS internal DMA (DMA-330).
 * 
 * @details  Try the DMA transfer under the conditions below.
 *           - MMU and L1, L2 cache enabled
 *           - ACP Port enabled (SMP mode, SCN=enabled)
 *           - Various source and destination cache settings (NonCache, Write-Throught,
 *             write-back, write-back with write allocate)
 *           - ACP port usage option (unused, via the ACP)
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2015/03/02  ---  #18042  新規作成.
 * @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
 * @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
 * @date 2021/04/23  ---  -----  SoC EDS v20.1 / Arm DS 2020.1 への対応を実施. 
 * 
 * @attention 
 * Copyright (C) 2013-2021 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes 
 ***********************************************************************************/
#include <stdio.h>
#include <string.h>
#include "alt_mmu.h"
#include "alt_cache.h"
#include "alt_watchdog.h"
#include "hwlib.h"
#include "socal/socal.h"
#include "socal/hps.h"

#include "sample_app_setting.h"
#include "util_time_measurement.h"
#include "util_interrupt_log.h"
#include "cmd.h"
#include "alt_printf.h"

/***********************************************************************************
 *  externs 
 ***********************************************************************************/

#if USED_DMA==1
extern void sample_dmac_initialize(void);
#endif

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/

/***********************************************************************************
 *  variables 
 ***********************************************************************************/
/* enable semihosting with Arm DS by defining an __auto_semihosting symbol
   will be removed by linker if semihosting is disabled */
int __auto_semihosting;

/***********************************************************************************
 *  functions 
 ***********************************************************************************/

/********************************************************************************//*!
 * @fn  void sample_test_print_usage(void) 
 * @brief  Display usage of the sample program. 
 * 
 * @details  Following information displays on the console.
 *            - Overview of HPS for push button functions (or operations)
 *            - Overview of HPS for DIP switch functions (or operations)
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2015/03/02  ---  #18042  新規作成.
 **//*******************************************************************************/
void sample_test_print_usage(void)
{
	printf("\n");
	printf("+-<< Usage: Command and Switch Functions >>----------------------------------+\n");
	printf(" SLIDESW #0 .... Select Operation Mode ( ON:Switch / OFF:Command )\n");
	printf("< Switch Mode >\n");
	printf(" PUSH SW #0 .... Exit Test loop!!!\n");
	printf(" PUSH SW #1 .... Function-A\n");
	printf(" PUSH SW #2 .... Function-B\n");
	printf(" PUSH SW #3 .... Function-C\n");
	printf(" SLIDESW #1:3 .. Option 0~7\n");
	printf("< Command Mode >\n");
	cmd_menu(NULL);
	printf("+----------------------------------------------------------------------------+\n\n");
	
	return;
}

/********************************************************************************//*!
 * @fn  void main(void)
 * @brief  Main routine of the sample program. 
 * 
 * @details  Infinite loop structure works. Make the detection of various types 
 *           of switches in the loop, it performs in various testing processes.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2015/03/02  ---  #18042  新規作成.
 **//*******************************************************************************/
int main(void)
{
	//ALT_STATUS_CODE result_code;
	uint32_t switch_raw = 0;
	uint32_t switch_raw_bk = 0;
	uint32_t switch_detect = 0;
	uint32_t switch_detect_on = 0;
	uint32_t switch_number = 0;
	bool disp_usage = true;
	bool switch_mode = true;
	
	/*********************************************************************/
	/* Stop Watch Dog Timer[WDT] set up in boot process. */
	/* Note: If you want to enable WDT, additional implementation is required. */
	/*********************************************************************/
	alt_wdog_uninit();

	// Disable caches that have already been set up in U-Boot.
	alt_cache_system_disable();

	printf("User Application Start!\r\n");
	
	// CPU and board settings.
	util_intlog_init();
	cpu0_init();

	// Initializing the dmac functions of hwlib.
#if USED_DMA==1
	sample_dmac_initialize();
#endif

	/* ################################################ */
	/* ## Implement the test setting process here!!! ## */
	/* ################################################ */

	util_intlog_print();
	
	printf("==== Start While(1) loop process!!! ============\n");
	switch_raw_bk = sample_detect_switch();
	switch_raw_bk ^= SAMPLE_SWITCH_BIT_SLIDE0;
	while(1)
	{
		if(disp_usage)
		{
			sample_test_print_usage();
			disp_usage = false;
		}

		// ---- Check the Slide-Switch and Push-Switch. ---- 
		switch_raw = sample_detect_switch();
		switch_detect = switch_raw ^ switch_raw_bk;
		switch_detect_on  |= switch_detect & switch_raw;
		
		// Push-Switch 0
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH0)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			break;	// Exit Test loop!!! 
		}
		
		// Change operation mode ?
		if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE0){
			if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0){
				printf("Enter Switch Mode!\n");
				switch_mode = true;
			} else {
				printf("Enter Command Mode! <Press Enter key to continue>\n");
				while(getchar() != '\n');	/* Clear stdin buffer */
				switch_mode = false;

				printf("\n");
				printf("Command: ");
			}
		}

		// ==== Branch by operation mode (Command Mode or Switch Mode) ==== 
		if(switch_mode){
		// ---- Switch Mode -------------------------------------------- //
			
			// Slide-Switch 
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDEALL)
			{
				switch_number = switch_raw & SAMPLE_SWITCH_BIT_NUM;
				switch_number *= 1;		// To avoid warnings.
				
				if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE1){
					printf("SAMPLE_SWITCH_BIT_SLIDE1\n");
				}
				if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE2){
					printf("SAMPLE_SWITCH_BIT_SLIDE2\n");
				}
				if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE3){
					printf("SAMPLE_SWITCH_BIT_SLIDE3\n");
				}
			}
			
			// Push-Switch 
			if(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)){
				
				if(switch_detect_on & SAMPLE_SWITCH_BIT_PUSH1){
					switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH1;
					printf("SAMPLE_SWITCH_BIT_PUSH1\n");
					disp_usage = true;
				}
				if(switch_detect_on & SAMPLE_SWITCH_BIT_PUSH2){
					switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH2;
					printf("SAMPLE_SWITCH_BIT_PUSH2\n");
					disp_usage = true;
				}
				if(switch_detect_on & SAMPLE_SWITCH_BIT_PUSH3){
					switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH3;
					printf("SAMPLE_SWITCH_BIT_PUSH3\n");
					disp_usage = true;
				}
			}
		} else {
		// ---- Command Mode ------------------------------------------- //
			if(cmd_execute())
			{
				break;	// Exit Test loop!!! 
			}
		}
		
		util_intlog_print();
		switch_raw_bk = switch_raw;
	}
	printf("==== End While(1) loop process. ================\n");
	
	util_intlog_print();
	
	printf("Finished running the sample !!!\r\n");
	
	return 0;
}


////////////////////////////// Common Functions //////////////////////////////////////// 

/***********************************************************************************
 * end of file 
 ***********************************************************************************/
