﻿/********************************************************************************//*!
 * @file  util_time_measurement.h 
 * @brief  Time measurement programs.
 * 
 * @details  Provides a mechanism to measure the time interval specified, using 
 *           the global timer with clock Manager.
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 * @date 2014/02/26  ---  #13246  計測結果表示用関数の仕様を一部変更.
 * @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
 * @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
 * @date 2018/06/29  ---  -----  DE10-Nano ボードへの対応を追加.
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

#ifndef __UTIL_TIME_MEASUREMENT_H__
#define __UTIL_TIME_MEASUREMENT_H__

/***********************************************************************************
 *  includes 
 ***********************************************************************************/

/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/
#define	UTIL_TIME_MAX_TRACE			32		/*!< The maximum number of time-counter info. */

/*! Timer interval is calculated using the following equation:
 * Interval = ((PRESCALER_value + 1) * (Load_value + 1) / PERIPHCLK)
 * ## PERIPHCLK = ALT_CLK_MPU_PERIPH
 */
#if TARGET_BOARD_NO==1	/* 1:helio @ 800MHz */
#define	UTIL_TIME_GLOBALTIMER_PRESCALE	1	/*!< PRESCALER_value set to 1(for MAINCLK=800MHz) or 2(for 600MHz) */
											///<       <800MHz div 4 --> 200MHz>  <600MHz div 4 --> 150MHz>
											///< - 0 --> ( 5.0nsec/count@200MHz,  6.6666..nsec/count@150MHz)
											///< - 1 --> (10.0nsec/count@200MHz, 13.3333..nsec/count@150MHz)
											///< - 2 --> (15.0nsec/count@800MHz, 19.9999..nsec/count@600MHz)
											///< - 3 --> (20.0nsec/count@800MHz)

#define	UTIL_TIME_NSEC_PER_COUNT		10	/*!< 10(10.0nsec/count@200MHz) or 20(19.9999..nsec/count@600MHz) */
#elif TARGET_BOARD_NO==2	/* 2:atlas @ 925MHz */
#define	UTIL_TIME_GLOBALTIMER_PRESCALE	36
#define	UTIL_TIME_NSEC_PER_COUNT		160
#elif TARGET_BOARD_NO==3	/* 3:sodia @ 800MHz */
#define	UTIL_TIME_GLOBALTIMER_PRESCALE	1
#define	UTIL_TIME_NSEC_PER_COUNT		10
#elif TARGET_BOARD_NO==4	/* 4:c5socdk @ 925MHz */
#define	UTIL_TIME_GLOBALTIMER_PRESCALE	36
#define	UTIL_TIME_NSEC_PER_COUNT		160
#elif TARGET_BOARD_NO==5	/* 5:a10socdk @ 1.2GHz */
#define	UTIL_TIME_GLOBALTIMER_PRESCALE	2
#define	UTIL_TIME_NSEC_PER_COUNT		10
#elif TARGET_BOARD_NO==6	/* 6:de10nano @ 800MHz */
#define	UTIL_TIME_GLOBALTIMER_PRESCALE	1
#define	UTIL_TIME_NSEC_PER_COUNT		10
#else	/* 3:sodia @ 800MHz */
#define	UTIL_TIME_GLOBALTIMER_PRESCALE	1
#define	UTIL_TIME_NSEC_PER_COUNT		10
#endif	/* TARGET_BOARD_NO */


/*! Measurement information record's status */
typedef enum{
	enumUTIL_TIME_STATUS_INIT,
	enumUTIL_TIME_STATUS_START,
	enumUTIL_TIME_STATUS_END,
} UtilTimeStatus_et;

/*! Print target of measurement result */
typedef enum{
	enumUTIL_TIME_TARGET_by_COUNT,
	enumUTIL_TIME_TARGET_by_SECOND,
	enumUTIL_TIME_TARGET_by_BOTH,
} UtilTimePrintTarget_et;

/*! Type of measurement information record. */
typedef struct{
	uint64_t			start;
	uint64_t			end;
	UtilTimeStatus_et	status;
} UtilTimeBuf_st;

/***********************************************************************************
 *  variables 
 ***********************************************************************************/

/***********************************************************************************
 *  functions 
 ***********************************************************************************/
extern void util_time_init(void);
extern void util_time_uninit(void);
extern void util_time_record_start_point(uint32_t index);
extern void util_time_record_end_point(uint32_t index);
extern void util_time_print_result_by_counter(uint32_t index);
extern void util_time_print_result_by_seconds(uint32_t index);
extern void util_time_print_result_all(UtilTimePrintTarget_et printby);
extern void util_time_print_result_partial(int startid, int endid, UtilTimePrintTarget_et printby);
extern void util_time_print_result_all_and_clear(UtilTimePrintTarget_et printby);


#endif /* __UTIL_TIME_MEASUREMENT_H__ */
/***********************************************************************************
 * end of file 
 ***********************************************************************************/
