﻿/********************************************************************************//*!
 * @file  util_interrupt_log.c 
 * @brief  Interrupt logging programs.
 * 
 * @details  Provides a mechanism to logging interrupt occurrence information.
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2015/06/26  ---  #20263  新規作成. 
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes 
 ***********************************************************************************/
#include <stdio.h>
#include "hwlib.h"
#include "alt_interrupt_common.h"

/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/
#define INTERRUPT_BUF_DEPTH		64

typedef struct {
	ALT_INT_INTERRUPT_t	kind;
	int	optional[2];
	int	totalcount;
} InterruptLog_st;

/***********************************************************************************
 *  variables 
 ***********************************************************************************/
InterruptLog_st		ILogBuf[INTERRUPT_BUF_DEPTH];
InterruptLog_st*	ILogBufTop_p;
InterruptLog_st*	ILogBufBottom_p;
uint32_t	ILogCount;

/*! Interrupt's kind messages. */
const char* ILogKindMessage[212] =
{
	"SGI0 ",	// ALT_INT_INTERRUPT_SGI0  =  0,
	"SGI1 ",	// ALT_INT_INTERRUPT_SGI1  =  1,
	"SGI2 ",	// ALT_INT_INTERRUPT_SGI2  =  2,
	"SGI3 ",	// ALT_INT_INTERRUPT_SGI3  =  3,
	"SGI4 ",	// ALT_INT_INTERRUPT_SGI4  =  4,
	"SGI5 ",	// ALT_INT_INTERRUPT_SGI5  =  5,
	"SGI6 ",	// ALT_INT_INTERRUPT_SGI6  =  6,
	"SGI7 ",	// ALT_INT_INTERRUPT_SGI7  =  7,
	"SGI8 ",	// ALT_INT_INTERRUPT_SGI8  =  8,
	"SGI9 ",	// ALT_INT_INTERRUPT_SGI9  =  9,
	"SGI10",	// ALT_INT_INTERRUPT_SGI10 = 10,
	"SGI11",	// ALT_INT_INTERRUPT_SGI11 = 11,
	"SGI12",	// ALT_INT_INTERRUPT_SGI12 = 12,
	"SGI13",	// ALT_INT_INTERRUPT_SGI13 = 13,
	"SGI14",	// ALT_INT_INTERRUPT_SGI14 = 14,
	"SGI15",	// ALT_INT_INTERRUPT_SGI15 = 15,
	"",	// 16,
	"",	// 17,
	"",	// 18,
	"",	// 19,
	"",	// 20,
	"",	// 21,
	"",	// 22,
	"",	// 23,
	"",	// 24,
	"",	// 25,
	"",	// 26,
	"PPI_TIMER_GLOBAL",	// ALT_INT_INTERRUPT_PPI_TIMER_GLOBAL   = 27,
	"",	// 28,
	"PPI_TIMER_PRIVATE",	// ALT_INT_INTERRUPT_PPI_TIMER_PRIVATE  = 29,
	"PPI_TIMER_WATCHDOG",	// ALT_INT_INTERRUPT_PPI_TIMER_WATCHDOG = 30,
	"",	// 31,
#ifdef soc_cv_av
	"CPU0_PARITYFAIL",	// ALT_INT_INTERRUPT_CPU0_PARITYFAIL         = 32,
	"CPU0_PARITYFAIL_BTAC",	// ALT_INT_INTERRUPT_CPU0_PARITYFAIL_BTAC    = 33,
	"CPU0_PARITYFAIL_GHB",	// ALT_INT_INTERRUPT_CPU0_PARITYFAIL_GHB     = 34,
	"CPU0_PARITYFAIL_I_TAG",	// ALT_INT_INTERRUPT_CPU0_PARITYFAIL_I_TAG   = 35,
	"CPU0_PARITYFAIL_I_DATA",	// ALT_INT_INTERRUPT_CPU0_PARITYFAIL_I_DATA  = 36,
	"CPU0_PARITYFAIL_TLB",	// ALT_INT_INTERRUPT_CPU0_PARITYFAIL_TLB     = 37,
	"CPU0_PARITYFAIL_D_OUTER",	// ALT_INT_INTERRUPT_CPU0_PARITYFAIL_D_OUTER = 38,
	"CPU0_PARITYFAIL_D_TAG",	// ALT_INT_INTERRUPT_CPU0_PARITYFAIL_D_TAG   = 39,
	"CPU0_PARITYFAIL_D_DATA",	// ALT_INT_INTERRUPT_CPU0_PARITYFAIL_D_DATA  = 40,
	"CPU0_DEFLAGS0",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS0           = 41,
	"CPU0_DEFLAGS1",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS1           = 42,
	"CPU0_DEFLAGS2",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS2           = 43,
	"CPU0_DEFLAGS3",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS3           = 44,
	"CPU0_DEFLAGS4",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS4           = 45,
	"CPU0_DEFLAGS5",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS5           = 46,
	"CPU0_DEFLAGS6",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS6           = 47,
	"CPU1_PARITYFAIL",	// ALT_INT_INTERRUPT_CPU1_PARITYFAIL         = 48,
	"CPU1_PARITYFAIL_BTAC",	// ALT_INT_INTERRUPT_CPU1_PARITYFAIL_BTAC    = 49,
	"CPU1_PARITYFAIL_GHB",	// ALT_INT_INTERRUPT_CPU1_PARITYFAIL_GHB     = 50,
	"CPU1_PARITYFAIL_I_TAG",	// ALT_INT_INTERRUPT_CPU1_PARITYFAIL_I_TAG   = 51,
	"CPU1_PARITYFAIL_I_DATA",	// ALT_INT_INTERRUPT_CPU1_PARITYFAIL_I_DATA  = 52,
	"CPU1_PARITYFAIL_TLB",	// ALT_INT_INTERRUPT_CPU1_PARITYFAIL_TLB     = 53,
	"CPU1_PARITYFAIL_D_OUTER",	// ALT_INT_INTERRUPT_CPU1_PARITYFAIL_D_OUTER = 54,
	"CPU1_PARITYFAIL_D_TAG",	// ALT_INT_INTERRUPT_CPU1_PARITYFAIL_D_TAG   = 55,
	"CPU1_PARITYFAIL_D_DATA",	// ALT_INT_INTERRUPT_CPU1_PARITYFAIL_D_DATA  = 56,
	"CPU1_DEFLAGS0",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS0           = 57,
	"CPU1_DEFLAGS1",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS1           = 58,
	"CPU1_DEFLAGS2",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS2           = 59,
	"CPU1_DEFLAGS3",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS3           = 60,
	"CPU1_DEFLAGS4",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS4           = 61,
	"CPU1_DEFLAGS5",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS5           = 62,
	"CPU1_DEFLAGS6",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS6           = 63,
	"SCU_PARITYFAIL0",	// ALT_INT_INTERRUPT_SCU_PARITYFAIL0 =  64,
	"SCU_PARITYFAIL1",	// ALT_INT_INTERRUPT_SCU_PARITYFAIL1 =  65,
	"SCU_EV_ABORT",	// ALT_INT_INTERRUPT_SCU_EV_ABORT    =  66,
	"L2_ECC_BYTE_WR_IRQ",	// ALT_INT_INTERRUPT_L2_ECC_BYTE_WR_IRQ     = 67,
	"L2_ECC_CORRECTED_IRQ",	// ALT_INT_INTERRUPT_L2_ECC_CORRECTED_IRQ   = 68,
	"L2_ECC_UNCORRECTED_IRQ",	// ALT_INT_INTERRUPT_L2_ECC_UNCORRECTED_IRQ = 69,
	"L2_COMBINED_IRQ",	// ALT_INT_INTERRUPT_L2_COMBINED_IRQ        = 70,
	"DDR_ECC_ERROR_IRQ",	// ALT_INT_INTERRUPT_DDR_ECC_ERROR_IRQ =  71,
	"F2S_FPGA_IRQ0 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ0  =  72,
	"F2S_FPGA_IRQ1 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ1  =  73,
	"F2S_FPGA_IRQ2 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ2  =  74,
	"F2S_FPGA_IRQ3 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ3  =  75,
	"F2S_FPGA_IRQ4 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ4  =  76,
	"F2S_FPGA_IRQ5 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ5  =  77,
	"F2S_FPGA_IRQ6 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ6  =  78,
	"F2S_FPGA_IRQ7 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ7  =  79,
	"F2S_FPGA_IRQ8 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ8  =  80,
	"F2S_FPGA_IRQ9 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ9  =  81,
	"F2S_FPGA_IRQ10",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ10 =  82,
	"F2S_FPGA_IRQ11",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ11 =  83,
	"F2S_FPGA_IRQ12",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ12 =  84,
	"F2S_FPGA_IRQ13",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ13 =  85,
	"F2S_FPGA_IRQ14",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ14 =  86,
	"F2S_FPGA_IRQ15",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ15 =  87,
	"F2S_FPGA_IRQ16",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ16 =  88,
	"F2S_FPGA_IRQ17",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ17 =  89,
	"F2S_FPGA_IRQ18",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ18 =  90,
	"F2S_FPGA_IRQ19",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ19 =  91,
	"F2S_FPGA_IRQ20",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ20 =  92,
	"F2S_FPGA_IRQ21",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ21 =  93,
	"F2S_FPGA_IRQ22",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ22 =  94,
	"F2S_FPGA_IRQ23",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ23 =  95,
	"F2S_FPGA_IRQ24",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ24 =  96,
	"F2S_FPGA_IRQ25",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ25 =  97,
	"F2S_FPGA_IRQ26",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ26 =  98,
	"F2S_FPGA_IRQ27",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ27 =  99,
	"F2S_FPGA_IRQ28",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ28 = 100,
	"F2S_FPGA_IRQ29",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ29 = 101,
	"F2S_FPGA_IRQ30",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ30 = 102,
	"F2S_FPGA_IRQ31",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ31 = 103,
	"F2S_FPGA_IRQ32",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ32 = 104,
	"F2S_FPGA_IRQ33",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ33 = 105,
	"F2S_FPGA_IRQ34",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ34 = 106,
	"F2S_FPGA_IRQ35",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ35 = 107,
	"F2S_FPGA_IRQ36",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ36 = 108,
	"F2S_FPGA_IRQ37",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ37 = 109,
	"F2S_FPGA_IRQ38",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ38 = 110,
	"F2S_FPGA_IRQ39",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ39 = 111,
	"F2S_FPGA_IRQ40",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ40 = 112,
	"F2S_FPGA_IRQ41",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ41 = 113,
	"F2S_FPGA_IRQ42",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ42 = 114,
	"F2S_FPGA_IRQ43",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ43 = 115,
	"F2S_FPGA_IRQ44",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ44 = 116,
	"F2S_FPGA_IRQ45",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ45 = 117,
	"F2S_FPGA_IRQ46",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ46 = 118,
	"F2S_FPGA_IRQ47",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ47 = 119,
	"F2S_FPGA_IRQ48",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ48 = 120,
	"F2S_FPGA_IRQ49",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ49 = 121,
	"F2S_FPGA_IRQ50",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ50 = 122,
	"F2S_FPGA_IRQ51",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ51 = 123,
	"F2S_FPGA_IRQ52",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ52 = 124,
	"F2S_FPGA_IRQ53",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ53 = 125,
	"F2S_FPGA_IRQ54",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ54 = 126,
	"F2S_FPGA_IRQ55",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ55 = 127,
	"F2S_FPGA_IRQ56",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ56 = 128,
	"F2S_FPGA_IRQ57",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ57 = 129,
	"F2S_FPGA_IRQ58",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ58 = 130,
	"F2S_FPGA_IRQ59",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ59 = 131,
	"F2S_FPGA_IRQ60",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ60 = 132,
	"F2S_FPGA_IRQ61",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ61 = 133,
	"F2S_FPGA_IRQ62",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ62 = 134,
	"F2S_FPGA_IRQ63",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ63 = 135,
	"DMA_IRQ0",	// ALT_INT_INTERRUPT_DMA_IRQ0                = 136,
	"DMA_IRQ1",	// ALT_INT_INTERRUPT_DMA_IRQ1                = 137,
	"DMA_IRQ2",	// ALT_INT_INTERRUPT_DMA_IRQ2                = 138,
	"DMA_IRQ3",	// ALT_INT_INTERRUPT_DMA_IRQ3                = 139,
	"DMA_IRQ4",	// ALT_INT_INTERRUPT_DMA_IRQ4                = 140,
	"DMA_IRQ5",	// ALT_INT_INTERRUPT_DMA_IRQ5                = 141,
	"DMA_IRQ6",	// ALT_INT_INTERRUPT_DMA_IRQ6                = 142,
	"DMA_IRQ7",	// ALT_INT_INTERRUPT_DMA_IRQ7                = 143,
	"DMA_IRQ_ABORT",	// ALT_INT_INTERRUPT_DMA_IRQ_ABORT           = 144,
	"DMA_ECC_CORRECTED_IRQ",	// ALT_INT_INTERRUPT_DMA_ECC_CORRECTED_IRQ   = 145,
	"DMA_ECC_UNCORRECTED_IRQ",	// ALT_INT_INTERRUPT_DMA_ECC_UNCORRECTED_IRQ = 146,
	"EMAC0_IRQ",	// ALT_INT_INTERRUPT_EMAC0_IRQ                    = 147,
	"EMAC0_TX_ECC_CORRECTED_IRQ",	// ALT_INT_INTERRUPT_EMAC0_TX_ECC_CORRECTED_IRQ   = 148,
	"EMAC0_TX_ECC_UNCORRECTED_IRQ",	// ALT_INT_INTERRUPT_EMAC0_TX_ECC_UNCORRECTED_IRQ = 149,
	"EMAC0_RX_ECC_CORRECTED_IRQ",	// ALT_INT_INTERRUPT_EMAC0_RX_ECC_CORRECTED_IRQ   = 150,
	"EMAC0_RX_ECC_UNCORRECTED_IRQ",	// ALT_INT_INTERRUPT_EMAC0_RX_ECC_UNCORRECTED_IRQ = 151,
	"EMAC1_IRQ",	// ALT_INT_INTERRUPT_EMAC1_IRQ                    = 152,
	"EMAC1_TX_ECC_CORRECTED_IRQ",	// ALT_INT_INTERRUPT_EMAC1_TX_ECC_CORRECTED_IRQ   = 153,
	"EMAC1_TX_ECC_UNCORRECTED_IRQ",	// ALT_INT_INTERRUPT_EMAC1_TX_ECC_UNCORRECTED_IRQ = 154,
	"EMAC1_RX_ECC_CORRECTED_IRQ",	// ALT_INT_INTERRUPT_EMAC1_RX_ECC_CORRECTED_IRQ   = 155,
	"EMAC1_RX_ECC_UNCORRECTED_IRQ",	// ALT_INT_INTERRUPT_EMAC1_RX_ECC_UNCORRECTED_IRQ = 156,
	"USB0_IRQ",	// ALT_INT_INTERRUPT_USB0_IRQ             = 157,
	"USB0_ECC_CORRECTED",	// ALT_INT_INTERRUPT_USB0_ECC_CORRECTED   = 158,
	"USB0_ECC_UNCORRECTED",	// ALT_INT_INTERRUPT_USB0_ECC_UNCORRECTED = 159,
	"USB1_IRQ",	// ALT_INT_INTERRUPT_USB1_IRQ             = 160,
	"USB1_ECC_CORRECTED",	// ALT_INT_INTERRUPT_USB1_ECC_CORRECTED   = 161,
	"USB1_ECC_UNCORRECTED",	// ALT_INT_INTERRUPT_USB1_ECC_UNCORRECTED = 162,
	"CAN0_STS_IRQ",	// ALT_INT_INTERRUPT_CAN0_STS_IRQ             = 163,
	"CAN0_MO_IRQ",	// ALT_INT_INTERRUPT_CAN0_MO_IRQ              = 164,
	"CAN0_ECC_CORRECTED_IRQ",	// ALT_INT_INTERRUPT_CAN0_ECC_CORRECTED_IRQ   = 165,
	"CAN0_ECC_UNCORRECTED_IRQ",	// ALT_INT_INTERRUPT_CAN0_ECC_UNCORRECTED_IRQ = 166,
	"CAN1_STS_IRQ",	// ALT_INT_INTERRUPT_CAN1_STS_IRQ             = 167,
	"CAN1_MO_IRQ",	// ALT_INT_INTERRUPT_CAN1_MO_IRQ              = 168,
	"CAN1_ECC_CORRECTED_IRQ",	// ALT_INT_INTERRUPT_CAN1_ECC_CORRECTED_IRQ   = 169,
	"CAN1_ECC_UNCORRECTED_IRQ",	// ALT_INT_INTERRUPT_CAN1_ECC_UNCORRECTED_IRQ = 170,
	"SDMMC_IRQ",	// ALT_INT_INTERRUPT_SDMMC_IRQ                   = 171,
	"SDMMC_PORTA_ECC_CORRECTED",	// ALT_INT_INTERRUPT_SDMMC_PORTA_ECC_CORRECTED   = 172,
	"SDMMC_PORTA_ECC_UNCORRECTED",	// ALT_INT_INTERRUPT_SDMMC_PORTA_ECC_UNCORRECTED = 173,
	"SDMMC_PORTB_ECC_CORRECTED",	// ALT_INT_INTERRUPT_SDMMC_PORTB_ECC_CORRECTED   = 174,
	"SDMMC_PORTB_ECC_UNCORRECTED",	// ALT_INT_INTERRUPT_SDMMC_PORTB_ECC_UNCORRECTED = 175,
	"NAND_IRQ",	// ALT_INT_INTERRUPT_NAND_IRQ                  = 176,
	"NANDR_ECC_CORRECTED_IRQ",	// ALT_INT_INTERRUPT_NANDR_ECC_CORRECTED_IRQ   = 177,
	"NANDR_ECC_UNCORRECTED_IRQ",	// ALT_INT_INTERRUPT_NANDR_ECC_UNCORRECTED_IRQ = 178,
	"NANDW_ECC_CORRECTED_IRQ",	// ALT_INT_INTERRUPT_NANDW_ECC_CORRECTED_IRQ   = 179,
	"NANDW_ECC_UNCORRECTED_IRQ",	// ALT_INT_INTERRUPT_NANDW_ECC_UNCORRECTED_IRQ = 180,
	"NANDE_ECC_CORRECTED_IRQ",	// ALT_INT_INTERRUPT_NANDE_ECC_CORRECTED_IRQ   = 181,
	"NANDE_ECC_UNCORRECTED_IRQ",	// ALT_INT_INTERRUPT_NANDE_ECC_UNCORRECTED_IRQ = 182,
	"QSPI_IRQ",	// ALT_INT_INTERRUPT_QSPI_IRQ                 = 183,
	"QSPI_ECC_CORRECTED_IRQ",	// ALT_INT_INTERRUPT_QSPI_ECC_CORRECTED_IRQ   = 184,
	"QSPI_ECC_UNCORRECTED_IRQ",	// ALT_INT_INTERRUPT_QSPI_ECC_UNCORRECTED_IRQ = 185,
	"SPI0_IRQ",	// ALT_INT_INTERRUPT_SPI0_IRQ = 186,
	"SPI1_IRQ",	// ALT_INT_INTERRUPT_SPI1_IRQ = 187,
	"SPI2_IRQ",	// ALT_INT_INTERRUPT_SPI2_IRQ = 188,
	"SPI3_IRQ",	// ALT_INT_INTERRUPT_SPI3_IRQ = 189,
	"I2C0_IRQ",	// ALT_INT_INTERRUPT_I2C0_IRQ = 190,
	"I2C1_IRQ",	// ALT_INT_INTERRUPT_I2C1_IRQ = 191,
	"I2C2_IRQ",	// ALT_INT_INTERRUPT_I2C2_IRQ = 192,
	"I2C3_IRQ",	// ALT_INT_INTERRUPT_I2C3_IRQ = 193,
	"UART0",	// ALT_INT_INTERRUPT_UART0 = 194,
	"UART1",	// ALT_INT_INTERRUPT_UART1 = 195,
	"GPIO0",	// ALT_INT_INTERRUPT_GPIO0 = 196,
	"GPIO1",	// ALT_INT_INTERRUPT_GPIO1 = 197,
	"GPIO2",	// ALT_INT_INTERRUPT_GPIO2 = 198,
	"TIMER_L4SP_0_IRQ",	// ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ = 199,
	"TIMER_L4SP_1_IRQ",	// ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ = 200,
	"TIMER_OSC1_0_IRQ",	// ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ = 201,
	"TIMER_OSC1_1_IRQ",	// ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ = 202,
	"WDOG0_IRQ",	// ALT_INT_INTERRUPT_WDOG0_IRQ = 203,
	"WDOG1_IRQ",	// ALT_INT_INTERRUPT_WDOG1_IRQ = 204,
	"CLKMGR_IRQ",	// ALT_INT_INTERRUPT_CLKMGR_IRQ = 205,
	"MPUWAKEUP_IRQ",	// ALT_INT_INTERRUPT_MPUWAKEUP_IRQ = 206,
	"FPGA_MAN_IRQ",	// ALT_INT_INTERRUPT_FPGA_MAN_IRQ = 207,
	"NCTIIRQ0",	// ALT_INT_INTERRUPT_NCTIIRQ0 = 208,
	"NCTIIRQ1",	// ALT_INT_INTERRUPT_NCTIIRQ1 = 209,
	"RAM_ECC_CORRECTED_IRQ",	// ALT_INT_INTERRUPT_RAM_ECC_CORRECTED_IRQ   = 210,
	"RAM_ECC_UNCORRECTED_IRQ"	// ALT_INT_INTERRUPT_RAM_ECC_UNCORRECTED_IRQ = 211
#else /* soc_cv_av */
	"DERR_GLOBAL",		// ALT_INT_INTERRUPT_DERR_GLOBAL = 32,
	"CPUX_PARITYFAIL",	// ALT_INT_INTERRUPT_CPUX_PARITYFAIL = 33,
	"SERR_GLOBAL",		// ALT_INT_INTERRUPT_SERR_GLOBAL = 34,
	"CPU0_DEFLAGS0",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS0 = 35,
	"CPU0_DEFLAGS1",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS1 = 36,
	"CPU0_DEFLAGS2",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS2 = 37,
	"CPU0_DEFLAGS3",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS3 = 38,
	"CPU0_DEFLAGS4",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS4 = 39,
	"CPU0_DEFLAGS5",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS5 = 40,
	"CPU0_DEFLAGS6",	// ALT_INT_INTERRUPT_CPU0_DEFLAGS6 = 41,
	"CPU1_DEFLAGS0",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS0 = 42,
	"CPU1_DEFLAGS1",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS1 = 43,
	"CPU1_DEFLAGS2",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS2 = 44,
	"CPU1_DEFLAGS3",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS3 = 45,
	"CPU1_DEFLAGS4",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS4 = 46,
	"CPU1_DEFLAGS5",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS5 = 47,
	"CPU1_DEFLAGS6",	// ALT_INT_INTERRUPT_CPU1_DEFLAGS6 = 48,
	"SCU_EV_ABORT",		// ALT_INT_INTERRUPT_SCU_EV_ABORT = 49,
	"L2_COMBINED_IRQ",	// ALT_INT_INTERRUPT_L2_COMBINED_IRQ = 50,
	"F2S_FPGA_IRQ0 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ0  =  51,
	"F2S_FPGA_IRQ1 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ1  =  52,
	"F2S_FPGA_IRQ2 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ2  =  53,
	"F2S_FPGA_IRQ3 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ3  =  54,
	"F2S_FPGA_IRQ4 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ4  =  55,
	"F2S_FPGA_IRQ5 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ5  =  56,
	"F2S_FPGA_IRQ6 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ6  =  57,
	"F2S_FPGA_IRQ7 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ7  =  58,
	"F2S_FPGA_IRQ8 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ8  =  59,
	"F2S_FPGA_IRQ9 ",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ9  =  60,
	"F2S_FPGA_IRQ10",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ10 =  61,
	"F2S_FPGA_IRQ11",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ11 =  62,
	"F2S_FPGA_IRQ12",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ12 =  63,
	"F2S_FPGA_IRQ13",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ13 =  64,
	"F2S_FPGA_IRQ14",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ14 =  65,
	"F2S_FPGA_IRQ15",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ15 =  66,
	"F2S_FPGA_IRQ16",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ16 =  67,
	"F2S_FPGA_IRQ17",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ17 =  68,
	"F2S_FPGA_IRQ18",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ18 =  69,
	"F2S_FPGA_IRQ19",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ19 =  70,
	"F2S_FPGA_IRQ20",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ20 =  71,
	"F2S_FPGA_IRQ21",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ21 =  72,
	"F2S_FPGA_IRQ22",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ22 =  73,
	"F2S_FPGA_IRQ23",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ23 =  74,
	"F2S_FPGA_IRQ24",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ24 =  75,
	"F2S_FPGA_IRQ25",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ25 =  76,
	"F2S_FPGA_IRQ26",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ26 =  77,
	"F2S_FPGA_IRQ27",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ27 =  78,
	"F2S_FPGA_IRQ28",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ28 =  79,
	"F2S_FPGA_IRQ29",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ29 =  80,
	"F2S_FPGA_IRQ30",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ30 =  81,
	"F2S_FPGA_IRQ31",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ31 =  82,
	"F2S_FPGA_IRQ32",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ32 =  83,
	"F2S_FPGA_IRQ33",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ33 =  84,
	"F2S_FPGA_IRQ34",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ34 =  85,
	"F2S_FPGA_IRQ35",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ35 =  86,
	"F2S_FPGA_IRQ36",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ36 =  87,
	"F2S_FPGA_IRQ37",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ37 =  88,
	"F2S_FPGA_IRQ38",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ38 =  89,
	"F2S_FPGA_IRQ39",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ39 =  90,
	"F2S_FPGA_IRQ40",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ40 =  91,
	"F2S_FPGA_IRQ41",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ41 =  92,
	"F2S_FPGA_IRQ42",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ42 =  93,
	"F2S_FPGA_IRQ43",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ43 =  94,
	"F2S_FPGA_IRQ44",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ44 =  95,
	"F2S_FPGA_IRQ45",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ45 =  96,
	"F2S_FPGA_IRQ46",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ46 =  97,
	"F2S_FPGA_IRQ47",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ47 =  98,
	"F2S_FPGA_IRQ48",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ48 =  99,
	"F2S_FPGA_IRQ49",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ49 = 100,
	"F2S_FPGA_IRQ50",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ50 = 101,
	"F2S_FPGA_IRQ51",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ51 = 102,
	"F2S_FPGA_IRQ52",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ52 = 103,
	"F2S_FPGA_IRQ53",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ53 = 104,
	"F2S_FPGA_IRQ54",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ54 = 105,
	"F2S_FPGA_IRQ55",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ55 = 106,
	"F2S_FPGA_IRQ56",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ56 = 107,
	"F2S_FPGA_IRQ57",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ57 = 108,
	"F2S_FPGA_IRQ58",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ58 = 109,
	"F2S_FPGA_IRQ59",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ59 = 110,
	"F2S_FPGA_IRQ60",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ60 = 111,
	"F2S_FPGA_IRQ61",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ61 = 112,
	"F2S_FPGA_IRQ62",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ62 = 113,
	"F2S_FPGA_IRQ63",	// ALT_INT_INTERRUPT_F2S_FPGA_IRQ63 = 114,
	"DMA_IRQ0",		// ALT_INT_INTERRUPT_DMA_IRQ0      = 115,
	"DMA_IRQ1",		// ALT_INT_INTERRUPT_DMA_IRQ1      = 116,
	"DMA_IRQ2",		// ALT_INT_INTERRUPT_DMA_IRQ2      = 117,
	"DMA_IRQ3",		// ALT_INT_INTERRUPT_DMA_IRQ3      = 118,
	"DMA_IRQ4",		// ALT_INT_INTERRUPT_DMA_IRQ4      = 119,
	"DMA_IRQ5",		// ALT_INT_INTERRUPT_DMA_IRQ5      = 120,
	"DMA_IRQ6",		// ALT_INT_INTERRUPT_DMA_IRQ6      = 121,
	"DMA_IRQ7",		// ALT_INT_INTERRUPT_DMA_IRQ7      = 122,
	"DMA_IRQ_ABORT",	// ALT_INT_INTERRUPT_DMA_IRQ_ABORT = 123,
	"EMAC0_IRQ",		// ALT_INT_INTERRUPT_EMAC0_IRQ = 124,
	"EMAC1_IRQ",		// ALT_INT_INTERRUPT_EMAC1_IRQ = 125,
	"EMAC2_IRQ",		// ALT_INT_INTERRUPT_EMAC2_IRQ = 126,
	"USB0_IRQ",		// ALT_INT_INTERRUPT_USB0_IRQ = 127,
	"USB1_IRQ",		// ALT_INT_INTERRUPT_USB1_IRQ = 128,
	"HMC_ERROR",		// ALT_INT_INTERRUPT_HMC_ERROR = 129,
	"SDMMC_IRQ",		// ALT_INT_INTERRUPT_SDMMC_IRQ = 130,
	"NAND_IRQ ",		// ALT_INT_INTERRUPT_NAND_IRQ  = 131,
	"QSPI_IRQ ",		// ALT_INT_INTERRUPT_QSPI_IRQ  = 132,
	"SPI0_IRQ",		// ALT_INT_INTERRUPT_SPI0_IRQ = 133,
	"SPI1_IRQ",		// ALT_INT_INTERRUPT_SPI1_IRQ = 134,
	"SPI2_IRQ",		// ALT_INT_INTERRUPT_SPI2_IRQ = 135,
	"SPI3_IRQ",		// ALT_INT_INTERRUPT_SPI3_IRQ = 136,
	"I2C0_IRQ",		// ALT_INT_INTERRUPT_I2C0_IRQ = 137,
	"I2C1_IRQ",		// ALT_INT_INTERRUPT_I2C1_IRQ = 138,
	"I2C2_IRQ",		// ALT_INT_INTERRUPT_I2C2_IRQ = 139,
	"I2C3_IRQ",		// ALT_INT_INTERRUPT_I2C3_IRQ = 140,
	"I2C4_IRQ",		// ALT_INT_INTERRUPT_I2C4_IRQ = 141,
	"UART0",		// ALT_INT_INTERRUPT_UART0 = 142,
	"UART1",		// ALT_INT_INTERRUPT_UART1 = 143,
	"GPIO0",		// ALT_INT_INTERRUPT_GPIO0 = 144,
	"GPIO1",		// ALT_INT_INTERRUPT_GPIO1 = 145,
	"GPIO2",		// ALT_INT_INTERRUPT_GPIO2 = 146,
	"TIMER_L4SP_0_IRQ",	// ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ = 147,
	"TIMER_L4SP_1_IRQ",	// ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ = 148,
	"TIMER_OSC1_0_IRQ",	// ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ = 149,
	"TIMER_OSC1_1_IRQ",	// ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ = 150,
	"WDOG0_IRQ",		// ALT_INT_INTERRUPT_WDOG0_IRQ = 151,
	"WDOG1_IRQ",		// ALT_INT_INTERRUPT_WDOG1_IRQ = 152,
	"CLKMGR_IRQ  ",		// ALT_INT_INTERRUPT_CLKMGR_IRQ   = 153,
	"RESTMGR_IRQ ",		// ALT_INT_INTERRUPT_RESTMGR_IRQ  = 154,
	"FPGA_MAN_IRQ",		// ALT_INT_INTERRUPT_FPGA_MAN_IRQ = 155,
	"NCTIIRQ0",		// ALT_INT_INTERRUPT_NCTIIRQ0 = 156,
	"NCTIIRQ1",		// ALT_INT_INTERRUPT_NCTIIRQ1 = 157,
	"SEC_MGR_INTR",		// ALT_INT_INTERRUPT_SEC_MGR_INTR = 158,
	"DATABWERR"		// ALT_INT_INTERRUPT_DATABWERR = 159
#endif /* soc_cv_av */
};

/***********************************************************************************
 *  functions 
 ***********************************************************************************/
/* 初期化処理: 最初に必ず呼ぶ */
void util_intlog_init(void)
{
	ILogCount = 0;
	ILogBufTop_p = ILogBuf;
	ILogBufBottom_p = ILogBuf;
	
	return;
}

static InterruptLog_st* util_intlog_get_next_address(InterruptLog_st* currentp)
{
	currentp++;
	if(currentp >= &ILogBuf[INTERRUPT_BUF_DEPTH])
	{
		currentp = ILogBuf;
	}
	return currentp;
}

/* ログ記録処理: 割り込みルーチン上で呼ぶ */
void util_intlog_record(ALT_INT_INTERRUPT_t kind, int opt1, int opt2)
{
	ILogCount++;
	
	/* バッファへの追記が可能か判断： 追加後に Top == Bottom になる場合は記録しない。 */
	if(ILogBufBottom_p != util_intlog_get_next_address(ILogBufTop_p))
	{	/* 追記可能な場合 */
		
		/* ポインタ更新 */
		InterruptLog_st* recordp = ILogBufTop_p;
		ILogBufTop_p = util_intlog_get_next_address(ILogBufTop_p);
		
		/* ログを記録 */
		recordp->kind = kind;
		recordp->optional[0] = opt1;
		recordp->optional[1] = opt2;
		recordp->totalcount = ILogCount;
	}
	
	return;
}

/* ログ出力処理: 通常ルーチン上で定期的に呼ぶ */
void util_intlog_print(void)
{
	/* ログバッファが空になるまでループ (Top == Bottom で空の状態) */
	while(ILogBufBottom_p != ILogBufTop_p)
	{
		int kind = (int)ILogBufBottom_p->kind;
		printf("[IRQ#%d] %s (0x%08X,0x%08X) count=%d\r\n", kind, ILogKindMessage[kind], 
			ILogBufBottom_p->optional[0], ILogBufBottom_p->optional[1], ILogBufBottom_p->totalcount);
		
		ILogBufBottom_p = util_intlog_get_next_address(ILogBufBottom_p);
	}
	
	return;
}

/***********************************************************************************
 * end of file 
 ***********************************************************************************/
