﻿/********************************************************************************//*!
 * @file  sample_interruptctrlSGI.c 
 * @brief  To use HWLib for the Interrupt Controller (mainly SGI) example program. 
 * 
 * @details  Run the API category below.
 *           - Interrupt Controller Low-Level API [Secure] 
 *               - Interrupt Controller Global Interface [Secure] (only a part)
 *               - Software Generated Interrupts [Secure] 
 *               - Interrupt Controller CPU Interface [Secure] (only a part)
 *               - Interrupt Utility Functions [Secure] (only a part)
 *           Does not highlight the sub-categories below so have been tested in various samples.
 *               - Interrupt Controller Distributor Interface [Secure]
 *               - Interrupt Service Routine [Secure]  
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/13  ---  #13144  新規作成. 
 * @date 2014/08/18  ---  -----  v14.0化 (誤植訂正、操作説明の表示を追加). 
 * @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
 * @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
 * @date 2018/07/02  ---  #25940  All in One ベアメタル・アプリケーション v17.1 への対応を実施.
 * @date 2019/01/18  ---  -----   All in One ベアメタル・アプリケーション v18.1 への対応を実施.
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes 
 ***********************************************************************************/
#include <stdio.h>
#include <string.h>
#include "hwlib.h"
#include "socal/socal.h"
#include "socal/hps.h"
#include "alt_generalpurpose_io.h"
#include "alt_globaltmr.h"
#include "alt_clock_manager.h"
#include "alt_interrupt.h"
#include "alt_timers.h"
#include "alt_watchdog.h"

#include "sample_app_setting.h"
#include "util_time_measurement.h"
#include "util_interrupt_log.h"
#include "alt_printf.h"

/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/
void sample_intctrl_exec_utilityAPIs(void);
void sample_intctrl_exec_paramgetAPIs(void);
void sample_intctrl_exec_paramsetAPIs(void);
void sample_intctrl_exec_SGI(int sginumber, ALT_INT_SGI_TARGET_t target_filter);
static void sample_intctrl_print_setting_gic(ALT_INT_INTERRUPT_t int_id);
static void sample_intctrl_print_all_int_settings(void);
void sample_intctrl_setting_gic(void);
void sample_intctrl_callback_SGI0(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI1(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI2(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI3(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI4(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI5(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI6(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI7(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI8(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI9(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI10(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI11(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI12(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI13(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI14(uint32_t icciar, void *context);
void sample_intctrl_callback_SGI15(uint32_t icciar, void *context);
void sample_intctrl_test_init(void);
void sample_intctrl_test_uninit(void);
void sample_intctrl_test_print_usage(void);
void sample_intctrl_test_main(void);
int  sample_intctrl_test_cmd(char* options);
// prototype (Common Functions) //

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/
/*! TEST Parameters */
#define SAMPLE_PARAM_USE_SECURE_BINARY_POINT	(false)	/*!< If true then use the Secure Binary Point Register for both secure and non-secure interrupts. If false then use Secure Binary Point Register for secure interrupts and non-secure Binary Point Register for non-secure interrupts. */
#define SAMPLE_PARAM_USE_FIQ_FOR_SECURE_INT		(false)	/*!< If true then signal secure interrupts using the FIQ signal. If false then signal secure interrupts using the IRQ signal. */
#define SAMPLE_PARAM_ALLOW_SECURE_ACK_FOR_ALL	(true)	/*!< Controls whether a secure acknowledgement of an interrupt, when the highest priority pending interrupt is non-secure, causes the CPU interface to acknowledge the interrupt. If true then a secure acknowledgement of the interrupt is not completed and the Interrupt ID of the Non-secure interrupt is returned. If false then a secure acknowledgement of the interrupt is not completed and the Interrupt ID of 1022 is returned. */
#define SAMPLE_PARAM_PRIORITY_MASK		(255)		/*!< The interrupt priority mask is the group priority needed to instruct the GIC to preempt lower priority interrupt. The valid range for this value is 0 - 255. */
#define SAMPLE_PARAM_BINARY_POINT		(0)		/*!< The binary point to use. The valid range for the value is 0 - 7. */
#define SAMPLE_PARAM_BINARY_POINT_NS	(0)		/*!< The binary point to use. The valid range for the value is 0 - 7. */


/***********************************************************************************
 *  variables 
 ***********************************************************************************/
const CALLBACK_FUNKt	SAMPLE_IntCtrl_CallBacks[16] =
{
		sample_intctrl_callback_SGI0,
		sample_intctrl_callback_SGI1,
		sample_intctrl_callback_SGI2,
		sample_intctrl_callback_SGI3,
		sample_intctrl_callback_SGI4,
		sample_intctrl_callback_SGI5,
		sample_intctrl_callback_SGI6,
		sample_intctrl_callback_SGI7,
		sample_intctrl_callback_SGI8,
		sample_intctrl_callback_SGI9,
		sample_intctrl_callback_SGI10,
		sample_intctrl_callback_SGI11,
		sample_intctrl_callback_SGI12,
		sample_intctrl_callback_SGI13,
		sample_intctrl_callback_SGI14,
		sample_intctrl_callback_SGI15
};

int	SAMPLE_IntCtrl_CallBackCount[16];	/*!< Interrupt detection counters */

/***********************************************************************************
 *  functions 
 ***********************************************************************************/

/********************************************************************************//*!
 * @fn  void sample_intctrl_exec_utilityAPIs(void) 
 * @brief  Execute the Interrupt Utility Functions. 
 * 
 * @details  To run the following APIs and prints the result.
 *           - alt_int_util_cpu_count(). 
 *           - alt_int_util_int_count(). 
 *           - alt_int_util_cpu_current(). 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/13  ---  #13144  新規作成. 
 **//*******************************************************************************/
void sample_intctrl_exec_utilityAPIs(void)
{
	
	/* -- Interrupt Utility Functions [Secure] -- */
	/* Usage:	uint32_t  alt_int_util_cpu_count (void);	*/
	/* Usage:	uint32_t  alt_int_util_int_count (void);	*/
	/* Usage:	alt_int_cpu_target_t  alt_int_util_cpu_current (void);	*/
	
	/* <typedef uint32_t alt_int_cpu_target_t> */
	/******************************************************************************* 
	 * This is the CPU target type. It is used to specify a set of CPUs on the system. 
	 * If only bit 0 is set then it specifies a set of CPUs containing only CPU 0. 
	 * Multiple CPUs can be specified by setting the appropriate bit up to the number 
	 * of CPUs on the system. 
	 *******************************************************************************/
	
	printf("==== HWLib test : Execute ""Interrupt Utility Functions [Secure]"" ====\n");
	printf("alt_int_util_cpu_count() .... result=%d\n", (int)alt_int_util_cpu_count());
	printf("alt_int_util_int_count() .... result=%d\n", (int)alt_int_util_int_count());
	printf("alt_int_util_cpu_current() .... result=0x%08X\n", (int)alt_int_util_cpu_current());
	printf("\n");
	
	return;
}

/********************************************************************************//*!
 * @fn  void sample_intctrl_exec_paramgetAPIs(void) 
 * @brief  Execute the getting parameter Functions. 
 * 
 * @details  To run the following APIs and prints the result.
 *           - alt_int_cpu_config_get(). 
 *           - alt_int_cpu_priority_mask_get(). 
 *           - alt_int_cpu_binary_point_get(). 
 *           - alt_int_cpu_binary_point_get_ns(). 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/13  ---  #13144  新規作成. 
 **//*******************************************************************************/
void sample_intctrl_exec_paramgetAPIs(void)
{
	ALT_STATUS_CODE result_code;
	bool outbool1;
	bool outbool2;
	bool outbool3;
	
	/* -- Interrupt Controller CPU Interface [Secure] -- */
	/* Usage:	ALT_STATUS_CODE  alt_int_cpu_config_get (bool *use_secure_binary_point, bool *use_FIQ_for_secure_interrupts, bool *allow_secure_ack_all_interrupts);	*/
	/* Usage:	uint32_t  alt_int_cpu_priority_mask_get (void);	*/
	/* Usage:	uint32_t  alt_int_cpu_binary_point_get (void);	*/
	/* Usage:	uint32_t  alt_int_cpu_binary_point_get_ns (void);	*/
	printf("==== HWLib test : Execute the getting parameter Functions ====\n");
	result_code = alt_int_cpu_config_get(&outbool1, &outbool2, &outbool3);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_cpu_config_get() .... result=%d\n", (int)result_code);
	printf(" +-- use_secure_binary_point=%s\n", ((outbool1 == true)?"true":"false"));
	printf(" +-- use_FIQ_for_secure_interrupts=%s\n", ((outbool2 == true)?"true":"false"));
	printf(" +-- allow_secure_ack_all_interrupts=%s\n", ((outbool3 == true)?"true":"false"));
	printf("alt_int_cpu_priority_mask_get() .... result=%d\n", (int)alt_int_cpu_priority_mask_get());
	printf("alt_int_cpu_binary_point_get() .... result=%d\n", (int)alt_int_cpu_binary_point_get());
	printf("alt_int_cpu_binary_point_get_ns() .... result=%d\n", (int)alt_int_cpu_binary_point_get_ns());
	printf("\n");
	
	return;
}

/********************************************************************************//*!
 * @fn  void sample_intctrl_exec_paramsetAPIs(void) 
 * @brief  Execute the setting parameter Functions. 
 * 
 * @details  To run the following APIs and prints the result.
 *           - alt_int_cpu_config_set(). 
 *           - alt_int_cpu_priority_mask_set(). 
 *           - alt_int_cpu_binary_point_set(). 
 *           - alt_int_cpu_binary_point_set_ns(). 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/13  ---  #13144  新規作成. 
 **//*******************************************************************************/
void sample_intctrl_exec_paramsetAPIs(void)
{
	ALT_STATUS_CODE result_code;
	bool	inbool1 = SAMPLE_PARAM_USE_SECURE_BINARY_POINT;
	bool	inbool2 = SAMPLE_PARAM_USE_FIQ_FOR_SECURE_INT;
	bool	inbool3 = SAMPLE_PARAM_ALLOW_SECURE_ACK_FOR_ALL;
	
	/* -- Interrupt Controller CPU Interface [Secure] -- */
	/* Usage:	ALT_STATUS_CODE  alt_int_cpu_config_set (bool use_secure_binary_point, bool use_FIQ_for_secure_interrupts, bool allow_secure_ack_all_interrupts);	*/
	/* Usage:	ALT_STATUS_CODE  alt_int_cpu_priority_mask_set (uint32_t priority_mask);	*/
	/* Usage:	ALT_STATUS_CODE  alt_int_cpu_binary_point_set (uint32_t binary_point);	*/
	/* Usage:	ALT_STATUS_CODE  alt_int_cpu_binary_point_set_ns (uint32_t binary_point);	*/
	printf("==== HWLib test : Execute the setting parameter Functions ====\n");
	result_code = alt_int_cpu_config_set(inbool1, inbool2, inbool3);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_cpu_config_set() .... result=%d\n", (int)result_code);
	printf(" +-- use_secure_binary_point=%s\n", ((inbool1 == true)?"true":"false"));
	printf(" +-- use_FIQ_for_secure_interrupts=%s\n", ((inbool2 == true)?"true":"false"));
	printf(" +-- allow_secure_ack_all_interrupts=%s\n", ((inbool3 == true)?"true":"false"));
	result_code = alt_int_cpu_priority_mask_set((uint32_t)SAMPLE_PARAM_PRIORITY_MASK);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_cpu_priority_mask_set(%d) .... result=%d\n", SAMPLE_PARAM_PRIORITY_MASK, (int)result_code);
	result_code = alt_int_cpu_binary_point_set((uint32_t)SAMPLE_PARAM_BINARY_POINT);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_cpu_binary_point_set(%d) .... result=%d\n", SAMPLE_PARAM_BINARY_POINT, (int)result_code);
	result_code = alt_int_cpu_binary_point_set_ns((uint32_t)SAMPLE_PARAM_BINARY_POINT_NS);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_cpu_binary_point_set_ns(%d) .... result=%d\n", SAMPLE_PARAM_BINARY_POINT_NS, (int)result_code);
	printf("\n");
	
	return;
}

/********************************************************************************//*!
 * @fn  void sample_intctrl_exec_SGI(void) 
 * @brief  Execute the Software Generated Interrupts. 
 * 
 * @details  To run the following APIs and prints the result.
 *           - alt_int_util_cpu_count(). 
 *           - alt_int_util_int_count(). 
 *           - alt_int_util_cpu_current(). 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/13  ---  #13144  新規作成. 
 **//*******************************************************************************/
void sample_intctrl_exec_SGI(int sginumber, ALT_INT_SGI_TARGET_t target_filter)
{
#if 1
	ALT_STATUS_CODE result_code;
	ALT_INT_INTERRUPT_t int_id;
	
	int_id = (ALT_INT_INTERRUPT_t)(sginumber + ALT_INT_INTERRUPT_SGI0);

#ifdef GNU_COMPILER	// 99
	if((ALT_INT_INTERRUPT_SGI0 <= int_id)&&(int_id <= ALT_INT_INTERRUPT_SGI15))
#endif
#ifdef ARM_COMPILER	// 99
	if(int_id <= ALT_INT_INTERRUPT_SGI15)
#endif
	{
		/* -- Software Generated Interrupts [Secure] -- */
		/* Usage:	ALT_STATUS_CODE  alt_int_sgi_trigger (ALT_INT_INTERRUPT_t int_id, ALT_INT_SGI_TARGET_t target_filter, alt_int_cpu_target_t target_list, bool secure_only);	*/
		printf("==== HWLib test : Execute ""Software Generated Interrupts [Secure]"" ====\n");
		result_code = alt_int_sgi_trigger(int_id, target_filter, (alt_int_cpu_target_t)3, true);
		printf("alt_int_sgi_trigger(%d, %d, 3, true) .... result=%d\n", (int)int_id, (int)target_filter, (int)result_code);
		printf("\n");

		/* <typedef enum ALT_INT_INTERRUPT_e ALT_INT_INTERRUPT_t> */
		/*******************************************************************************
		 * ALT_INT_INTERRUPT_SGI0
		 * ALT_INT_INTERRUPT_SGI1
		 * ALT_INT_INTERRUPT_SGI2
		 * ALT_INT_INTERRUPT_SGI3
		 * ALT_INT_INTERRUPT_SGI4
		 * ALT_INT_INTERRUPT_SGI5
		 * ALT_INT_INTERRUPT_SGI6
		 * ALT_INT_INTERRUPT_SGI7
		 * ALT_INT_INTERRUPT_SGI8
		 * ALT_INT_INTERRUPT_SGI9
		 * ALT_INT_INTERRUPT_SGI10
		 * ALT_INT_INTERRUPT_SGI11
		 * ALT_INT_INTERRUPT_SGI12
		 * ALT_INT_INTERRUPT_SGI13
		 * ALT_INT_INTERRUPT_SGI14
		 * ALT_INT_INTERRUPT_SGI15
		 * 		Software Generated Interrupts (SGI), 0 - 15.
		 * 		・All interrupts in this group are software triggered.
		 *******************************************************************************/

		/* <typedef enum ALT_INT_SGI_TARGET_e ALT_INT_SGI_TARGET_t> */
		/*******************************************************************************
		 * ALT_INT_SGI_TARGET_LIST -- This filter list uses the target list parameter to
		 * 		specify which CPUs to send the interrupt to. If target list is 0, no
		 * 		interrupts are sent.
		 * ALT_INT_SGI_TARGET_ALL_EXCL_SENDER -- This filter list sends the interrupt
		 * 		all CPUs except the current CPU. The target list parameter is ignored.
		 * ALT_INT_SGI_TARGET_SENDER_ONLY -- This filter list sends the interrupt to the
		 * 		current CPU only. The target list parameter is ignored.
		 *******************************************************************************/

		/* <typedef uint32_t alt_int_cpu_target_t> */
		/*******************************************************************************
		 * This is the CPU target type. It is used to specify a set of CPUs on the system.
		 * If only bit 0 is set then it specifies a set of CPUs containing only CPU 0.
		 * Multiple CPUs can be specified by setting the appropriate bit up to the number
		 * of CPUs on the system.
		 *******************************************************************************/
		// Up to SoCFPGA MPCore is a maximum two-core configuration.
		// --> This parameter has 3 fixed.
	}
#endif
	return;
}

/********************************************************************************//*!
 * @fn  static void sample_intctrl_print_setting_gic(ALT_INT_INTERRUPT_t int_id)
 * @brief  Print GIC settings of specific Software-Generated interrupts.
 *
 * @details  Displays the specific interrupt settings of detail.
 * @param[in] int_id  interrupt identification types.
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/13  ---  #13144  新規作成.
 **//*******************************************************************************/
static void sample_intctrl_print_setting_gic(ALT_INT_INTERRUPT_t int_id)
{
	uint32_t ui32temp = 0;
	//ALT_INT_TRIGGER_t trigger_type = ALT_INT_TRIGGER_NA;

	//// Print initial value
	///****************************************************************************************/
	///* Usage: ALT_STATUS_CODE alt_int_dist_is_secure(ALT_INT_INTERRUPT_t int_id);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_is_enabled(ALT_INT_INTERRUPT_t int_id);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_is_pending(ALT_INT_INTERRUPT_t int_id);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_is_active(ALT_INT_INTERRUPT_t int_id);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_priority_get(ALT_INT_INTERRUPT_t int_id, uint32_t* priority);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_target_get(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t* target);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_trigger_get(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t* trigger_type);	*/
	///****************************************************************************************/
	printf("alt_int_dist_is_secure() ...... result=%d\n", (int)alt_int_dist_is_secure(int_id));

	// For Software Generated Interrupts (SGI) (interrupts 0 - 15) and Private Peripheral Interrupt (PPI) (interrupts 16 - 32),
	// interrupts must be queried on each CPU that may service the interrupt. This is done by calling this API when executing on that CPU.
	printf("alt_int_dist_is_enabled() ..... result=%d\n", (int)alt_int_dist_is_enabled(int_id));

	// For Software Generated Interrupts (SGI) (interrupts 0 - 15) and Private Peripheral Interrupt (PPI) (interrupts 16 - 32),
	// is pending is queried on a per CPU basis. This is done by calling this API when executing on that CPU.
	// # GIC architecture specification: "4.3.7 Interrupt Set-Pending Registers, GICD_ISPENDRn" and
	// # "4.3.8 Interrupt Clear-Pending Registers, GICD_ICPENDRn" in description below.
	// # Set-pending bits for SGIs are read-only and ignore writes.
	printf("alt_int_dist_is_pending() ..... result=%d\n", (int)alt_int_dist_is_pending(int_id));

	// For Software Generated Interrupts (SGI) (interrupts 0 - 15) and Private Peripheral Interrupt (PPI) (interrupts 16 - 32),
	// is active is queried on a per CPU basis. This is done by calling this API when executing on that CPU.
	printf("alt_int_dist_is_active() ...... result=%d\n", (int)alt_int_dist_is_active(int_id));

	// For Software Generated Interrupts (SGI) (interrupts 0 - 15) and Private Peripheral Interrupt (PPI) (interrupts 16 - 32),
	// priority is queried on a per CPU basis. This is done by calling this API when executing on that CPU.
	printf("alt_int_dist_priority_get() ... result=%d out=0x%08X\n", (int)alt_int_dist_priority_get(int_id, &ui32temp), (int)ui32temp);

	// For Software Generated Interrupts (SGI) (interrupts 0 - 15) and Private Peripheral Interrupt (PPI) (interrupts 16 - 32),
	// get target will return a set corresponding to the current CPU.
	printf("alt_int_dist_target_get() ..... result=%d out=0x%08X\n", (int)alt_int_dist_target_get(int_id, &ui32temp), (int)ui32temp);

	// The following APIs is not applicable SGI. (The interrupt identifier. Only PPI and SPI are valid, 16 - 1019.)
	//printf("alt_int_dist_trigger_get() .... result=%d out=%d\n", (int)alt_int_dist_trigger_get(int_id, &trigger_type), (int)trigger_type);

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_intctrl_print_all_int_settings(void)
 * @brief  Print GIC settings of all Software-Generated interrupts.
 *
 * @details  Displays the following interrupt settings.
 *           - ALT_INT_INTERRUPT_SGI0
 *           - ALT_INT_INTERRUPT_SGI1
 *           - ALT_INT_INTERRUPT_SGI2
 *           - ALT_INT_INTERRUPT_SGI3
 *           - ALT_INT_INTERRUPT_SGI4
 *           - ALT_INT_INTERRUPT_SGI5
 *           - ALT_INT_INTERRUPT_SGI6
 *           - ALT_INT_INTERRUPT_SGI7
 *           - ALT_INT_INTERRUPT_SGI8
 *           - ALT_INT_INTERRUPT_SGI9
 *           - ALT_INT_INTERRUPT_SGI10
 *           - ALT_INT_INTERRUPT_SGI11
 *           - ALT_INT_INTERRUPT_SGI12
 *           - ALT_INT_INTERRUPT_SGI13
 *           - ALT_INT_INTERRUPT_SGI14
 *           - ALT_INT_INTERRUPT_SGI15
 * @param[in,out] void  none
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/13  ---  #13144  新規作成.
 **//*******************************************************************************/
static void sample_intctrl_print_all_int_settings(void)
{
	int i;

	for(i=ALT_INT_INTERRUPT_SGI0; i<=ALT_INT_INTERRUPT_SGI15; i++)
	{
		printf("<< ALT_INT_INTERRUPT_SGI%d >>\n", i);
		sample_intctrl_print_setting_gic((ALT_INT_INTERRUPT_t)i);
		printf("\n");
	}

	return;
}

/********************************************************************************//*!
 * @fn  void sample_intctrl_setting_gic(void)
 * @brief  Setting of GIC (Software-Generated interrupt).
 *
 * @details    In order to receive a Software-Generated interrupt setting of GIC.
 * @param[in,out] void  none
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/13  ---  #13144  新規作成.
 **//*******************************************************************************/
void sample_intctrl_setting_gic(void)
{
	ALT_STATUS_CODE result_code;
	int i;

	for(i=ALT_INT_INTERRUPT_SGI0; i<=ALT_INT_INTERRUPT_SGI15; i++)
	{
		/* Usage: ALT_STATUS_CODE alt_int_isr_register(ALT_INT_INTERRUPT_t int_id, alt_int_callback_t callback, void* context);	*/
		result_code = alt_int_isr_register((ALT_INT_INTERRUPT_t)i, SAMPLE_IntCtrl_CallBacks[i], NULL);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_int_isr_register(ALT_INT_INTERRUPT_SGI%d) result=%d\n", i, (int)result_code);
		}
		
		/* No need to call the API below. Because all interrupts are secure by default. */
		/* Usage: ALT_STATUS_CODE  alt_int_dist_secure_enable (ALT_INT_INTERRUPT_t int_id);	*/
		
		/* For Software Generated Interrupts (SGI) (interrupts 0 - 15) and Private Peripheral Interrupt (PPI) (interrupts 16 - 32), */
		/* interrupts must be enabled on each CPU that will service the interrupt. */
		/* Usage: ALT_STATUS_CODE  alt_int_dist_enable (ALT_INT_INTERRUPT_t int_id);	*/
		result_code = alt_int_dist_enable((ALT_INT_INTERRUPT_t)i);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_int_dist_enable(ALT_INT_INTERRUPT_SGI%d) result=%d\n", i, (int)result_code);
		}
		
		// The following APIs is not applicable SGI. (int_id .. Only SPI are valid, 32 - 1019.)
		/* Usage: ALT_STATUS_CODE alt_int_dist_target_set(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t target);	*/

		// The following APIs is not applicable SGI. (int_id .. Only PPI and SPI are valid, 16 - 1019.)
		/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_set(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t trigger_type);	*/

		// For Software Generated Interrupts (SGI) (interrupts 0 - 15) and Private Peripheral Interrupt (PPI) (interrupts 16 - 32),
		// priority is set on a per CPU basis. This is done by calling this API when executing on that CPU.
		/* Usage: ALT_STATUS_CODE alt_int_dist_priority_set(ALT_INT_INTERRUPT_t int_id, uint32_t priority);	*/
		result_code = alt_int_dist_priority_set((ALT_INT_INTERRUPT_t)i, (i * 0x10));
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_int_dist_priority_set(ALT_INT_INTERRUPT_SGI%d) result=%d\n", i, (int)result_code);
		}

		// To reset the counter interrupt detection.
		SAMPLE_IntCtrl_CallBackCount[i] = 0;
	}

	return;
}

/********************************************************************************//*!
 * @fn  void sample_intctrl_callback_SGI0~15(uint32_t icciar, void *context)
 * @brief  ALT_INT_INTERRUPT_SGI0~15 interrupt callback functions.
 *
 * @details  working opportunity an interrupt from the ALT_INT_INTERRUPT_SGI0~15,
 *           only indicate an interrupt.
 * @param[in] icciar  The callback to use when the given interrupt is issued.
 * @param[in] context  The user provided context.
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/13  ---  #13144  新規作成.
 **//*******************************************************************************/
void sample_intctrl_callback_SGI0(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[0]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI0 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[0]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI0, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[0]);
	return;
}
void sample_intctrl_callback_SGI1(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[1]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI1 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[1]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI1, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[1]);
	return;
}
void sample_intctrl_callback_SGI2(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[2]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI2 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[2]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI2, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[2]);
	return;
}
void sample_intctrl_callback_SGI3(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[3]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI3 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[3]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI3, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[3]);
	return;
}
void sample_intctrl_callback_SGI4(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[4]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI4 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[4]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI4, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[4]);
	return;
}
void sample_intctrl_callback_SGI5(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[5]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI5 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[5]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI5, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[5]);
	return;
}
void sample_intctrl_callback_SGI6(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[6]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI6 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[6]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI6, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[6]);
	return;
}
void sample_intctrl_callback_SGI7(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[7]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI7 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[7]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI7, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[7]);
	return;
}
void sample_intctrl_callback_SGI8(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[8]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI8 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[8]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI8, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[8]);
	return;
}
void sample_intctrl_callback_SGI9(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[9]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI9 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[9]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI9, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[9]);
	return;
}
void sample_intctrl_callback_SGI10(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[10]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI10 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[10]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI10, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[10]);
	return;
}
void sample_intctrl_callback_SGI11(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[11]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI11 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[11]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI11, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[11]);
	return;
}
void sample_intctrl_callback_SGI12(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[12]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI12 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[12]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI12, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[12]);
	return;
}
void sample_intctrl_callback_SGI13(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[13]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI13 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[13]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI13, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[13]);
	return;
}
void sample_intctrl_callback_SGI14(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[14]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI14 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[14]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI14, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[14]);
	return;
}
void sample_intctrl_callback_SGI15(uint32_t icciar, void *context)
{
	SAMPLE_IntCtrl_CallBackCount[15]++;
	// SGI not required pending clearing of the GIC and other Peripheral.
	// Just writing to the ICC_AEOI register at HWLib.
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_SGI15 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_IntCtrl_CallBackCount[15]);
	util_intlog_record(ALT_INT_INTERRUPT_SGI15, (int)icciar, (int)SAMPLE_IntCtrl_CallBackCount[15]);
	return;
}

/********************************************************************************//*!
 * @fn  void sample_intctrl_test_init(void) 
 * @brief  Initializing the sample program. 
 * 
 * @details  Call the initialization function for the Interrupt Controller API. 
 *           Other, do the following.
 *           -# Configure the Helio Board GPIO settings. 
 *           -# Setting of general-purpose timers (Use HWLib). 
 *           -# Setting of GIC (General-Purpose timer interrupt). 
 *           -# Print general-purpose timer settings. 
 *           -# Start the timers for the general-purpose timers. 
 *           -# Enabling interrupts of all general-purpose timers.  
 *           -# Print GIC settings of all general-purpose timer interrupts. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/13  ---  #13144  新規作成. 
 **//*******************************************************************************/
void sample_intctrl_test_init(void)
{
	ALT_STATUS_CODE result_code;
	
#if 0	// main() 内で cpu0_init() 実施済み
	cpu0_init();
#endif
	
	// Only four API calls to try.
	// In call the initialization function immediately after the setting is disabled.
	/* -- Interrupt Controller Global Interface [Secure] -- */
	/* Usage:	ALT_STATUS_CODE  alt_int_global_enable_ns (void);	*/
	/* Usage:	ALT_STATUS_CODE  alt_int_global_enable_all (void);	*/
	/* -- Interrupt Controller CPU Interface [Secure] -- */
	/* Usage:	ALT_STATUS_CODE  alt_int_cpu_enable_ns (void);	*/
	/* Usage:	ALT_STATUS_CODE  alt_int_cpu_enable_all (void);	*/
	printf("==== HWLib TEST!: Enable all Global & individual CPU Interrupts ====\n");
	result_code = alt_int_cpu_enable_ns();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_cpu_enable_ns() .... result=%d\n", (int)result_code);
	result_code = alt_int_cpu_enable_all();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_cpu_enable_all() .... result=%d\n", (int)result_code);
	result_code = alt_int_global_enable_ns();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_global_enable_ns() .... result=%d\n", (int)result_code);
	result_code = alt_int_global_enable_all();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_global_enable_all() .... result=%d\n", (int)result_code);
	printf("\n");

	/* -- Interrupt Controller Global Interface [Secure] -- */
	/* Usage:	ALT_STATUS_CODE  alt_int_global_init (void);	*/
	/* -- Interrupt Controller CPU Interface [Secure] -- */
	/* Usage:	ALT_STATUS_CODE  alt_int_cpu_init (void);	*/
	printf("==== Execute Initialize Functions of Interrupt Controller ====\n");
	result_code = alt_int_global_init();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_global_init() .... result=%d\n", (int)result_code);
	result_code = alt_int_cpu_init();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_cpu_init() .... result=%d\n", (int)result_code);
	printf("\n");
	
	// Interrupt GIC Config
	sample_intctrl_setting_gic();

	printf("======== Print Test Interrupt Settings ========\n");
	sample_intctrl_print_all_int_settings();

	/* Execute the getting parameter Functions. */
	sample_intctrl_exec_paramgetAPIs();
	/* Execute the setting parameter Functions. */
	sample_intctrl_exec_paramsetAPIs();
	
	/* -- Interrupt Controller Global Interface [Secure] -- */
	/* Usage:	ALT_STATUS_CODE  alt_int_global_enable (void);	*/
	/* -- Interrupt Controller CPU Interface [Secure] -- */
	/* Usage:	ALT_STATUS_CODE  alt_int_cpu_enable (void);	*/
	printf("==== Enable Global & individual CPU Interrupts ====\n");
	result_code = alt_int_cpu_enable();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_cpu_enable() .... result=%d\n", (int)result_code);
	result_code = alt_int_global_enable();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_global_enable() .... result=%d\n", (int)result_code);
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  void sample_intctrl_test_uninit(void) 
 * @brief  Uninitializing the sample program. 
 * 
 * @details  Call the uninitialization function for the Interrupt Controller API.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/13  ---  #13144  新規作成. 
 **//*******************************************************************************/
void sample_intctrl_test_uninit(void)
{
	ALT_STATUS_CODE result_code;
	
	/* -- Interrupt Controller Global Interface [Secure] -- */
	/* Usage:	ALT_STATUS_CODE  alt_int_global_disable_ns (void);	*/
	/* Usage:	ALT_STATUS_CODE  alt_int_global_disable_all (void);	*/
	/* -- Interrupt Controller CPU Interface [Secure] -- */
	/* Usage:	ALT_STATUS_CODE  alt_int_cpu_disable_ns (void);	*/
	/* Usage:	ALT_STATUS_CODE  alt_int_cpu_disable_all (void);	*/
	printf("==== Disable all Global & individual CPU Interrupts ====\n");
	result_code = alt_int_global_disable_ns();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_global_disable_ns() .... result=%d\n", (int)result_code);
	result_code = alt_int_global_disable_all();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_global_disable_all() .... result=%d\n", (int)result_code);
	result_code = alt_int_cpu_disable_ns();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_cpu_disable_ns() .... result=%d\n", (int)result_code);
	result_code = alt_int_cpu_disable_all();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_cpu_disable_all() .... result=%d\n", (int)result_code);
	printf("\n");
	
	/* -- Interrupt Controller Global Interface [Secure] -- */
	/* Usage:	ALT_STATUS_CODE  alt_int_global_uninit (void);	*/
	/* -- Interrupt Controller CPU Interface [Secure] -- */
	/* Usage:	ALT_STATUS_CODE  alt_int_cpu_uninit (void);	*/
	printf("==== Execute Uninitialize Functions of Interrupt Controller ====\n");
	result_code = alt_int_global_uninit();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_global_uninit() .... result=%d\n", (int)result_code);
	result_code = alt_int_cpu_uninit();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_int_cpu_uninit() .... result=%d\n", (int)result_code);
	printf("\n");
	
	return;
}

/********************************************************************************//*!
 * @fn  void sample_intctrl_test_print_usage(void) 
 * @brief  Display usage of the sample program. 
 * 
 * @details  Following information displays on the console.
 *            - Overview of HPS for push button functions (or operations)
 *            - Overview of HPS for DIP switch functions (or operations)
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/18  ---  #xxxxx  新規作成. 
 **//*******************************************************************************/
void sample_intctrl_test_print_usage(void)
{
	printf("\n");
	printf("+--- Usage ( Function of PUSH SW and SLIDE SW )  ---------------+\n");
	printf("+ PUSH SW #0 .... Exit Test loop!!!\n");
	printf("+ PUSH SW #1 .... Issued SGI to CPU0.\n");
	printf("+ PUSH SW #2 .... Issued SGI to CPU1.(CPU1 is reset so unresponsive.)\n");
	printf("+ PUSH SW #3 .... Issued SGI to all CPUs.(Only the CPU0 callback to work.)\n");
	printf("+ SLIDE SW#0:3 .. Select the SGI number.\n");
	printf("   0000=ALT_INT_INTERRUPT_SGI0, 0001=SGI1,.., 1111=SGI15\n");
	printf("+---------------------------------------------------------------+\n\n");
	
#if 0
	HPS用DIPSW(4bit)の値で、対象とするSGI（ALT_INT_INTERRUPT_SGI0～ALT_INT_INTERRUPT_SGI15）を
	選択し、HPS用PUSHSW1～3のいずれかを押下することでSGI割り込みを発行します。
	HPS用PUSHSW1～3はそれぞれ下記の設定でSGIを発行する実装としています。
	- HPS用PUSHSW1 .. SGI発行元のコアへ割り込みを通知する。(Core#0宛) --> Core#0にてSGIのコールバックが動作。
	- HPS用PUSHSW2 .. SGI発行元のコアを除く全コアへ割り込みを通知する。(Core#1宛) --> Core#1を動かさないので何も反応しない。 
	- HPS用PUSHSW3 .. SGI発行元のコアを含む全コアへ割り込みを通知する。(Core#0,#1宛) --> Core#0のみコールバックが動作。Core#1は無反応。
	
	　無限ループ内で以下の処理を行う。
	　 - HPS用PUSHSW1の押下契機で SGI発行APIを実行します。
	　　 下記のパラメータを指定することで、Core#0宛でSGIを発行します。
	　　	int_id (1st Argument) = HPS用DIPSW(4bit)に応じて 0(ALT_INT_INTERRUPT_SGI0) ～ 15(ALT_INT_INTERRUPT_SGI15) を指定。
	　　	target_filter (2nd Argument) = ALT_INT_SGI_TARGET_SENDER_ONLY (発行元のみを指定) 

	　 - HPS用PUSHSW2の押下契機で SGI発行APIを実行します。
	　　 下記のパラメータを指定することで、Core#1宛でSGIを発行します。
	　　	int_id (1st Argument) = HPS用DIPSW(4bit)に応じて 0(ALT_INT_INTERRUPT_SGI0) ～ 15(ALT_INT_INTERRUPT_SGI15) を指定。
	　　	target_filter (2nd Argument) = ALT_INT_SGI_TARGET_ALL_EXCL_SENDER (発行元を除く全てを指定)

	　 - HPS用PUSHSW3の押下契機で SGI発行APIを実行します。
	　　 下記のパラメータを指定することで、Core#0,#1宛でSGIを発行します。
	　　	int_id (1st Argument) = HPS用DIPSW(4bit)に応じて 0(ALT_INT_INTERRUPT_SGI0) ～ 15(ALT_INT_INTERRUPT_SGI15) を指定。
	　　	target_filter (2nd Argument) = ALT_INT_SGI_TARGET_LIST (target_listで宛先を指定) 
	　　	target_list (3rd Argument) = 3 (bit#0[Core#0]=1 && bit#1[Core#1]=1)

	　 - HPS用PUSHSW0の押下契機で無限ループを終了します。
	　　 (テストプログラム終了)
#endif
	
	return;
}

/********************************************************************************//*!
 * @fn  void sample_intctrl_test_main(void) 
 * @brief  Main routine of the sample program. 
 * 
 * @details  Infinite loop structure works. Make the detection of various types 
 *           of switches in the loop, it performs in various testing processes.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/13  ---  #13144  新規作成. 
 **//*******************************************************************************/
void sample_intctrl_test_main(void)
{
	uint32_t switch_raw = 0;
	uint32_t switch_raw_bk = 0;
	uint32_t switch_detect = 0;
	uint32_t switch_detect_on = 0;
	uint32_t switch_number = 0;
	bool disp_usage = true;
	
	/* ################################################ */
	/* ## Implement the test setting process here!!! ## */
	/* ################################################ */
	// Execute the Interrupt Utility Functions.
	sample_intctrl_exec_utilityAPIs();

	util_intlog_print();

	printf("==== Start While(1) loop process!!! ============\n");
	switch_raw_bk = sample_detect_switch();
	while(1)
	{
		if(disp_usage)
		{
			sample_intctrl_test_print_usage();
			disp_usage = false;
		}

		// ---- Check the Slide-Switch and Push-Switch. ---- 
		switch_raw = sample_detect_switch();
		switch_detect = switch_raw ^ switch_raw_bk;
		switch_detect_on  |= switch_detect & switch_raw;
		
		// Slide-Switch 
		if(switch_detect & SAMPLE_SWITCH_BIT_SLIDEALL)
		{
			switch_number = switch_raw & SAMPLE_SWITCH_BIT_NUM;
			switch_number *= 1;		// To avoid warnings.

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE0){
				//printf("SAMPLE_SWITCH_BIT_SLIDE0\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE1){
				//printf("SAMPLE_SWITCH_BIT_SLIDE1\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE2){
				//printf("SAMPLE_SWITCH_BIT_SLIDE2\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE3){
				//printf("SAMPLE_SWITCH_BIT_SLIDE3\n");
			}
			disp_usage = false;
		}
		
		
		// Push-Switch 0
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH0)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH0;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH0\n");
			break;	// Exit Push-Switch 0 and 1 becomes ON.
		}
		// Push-Switch 1
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH1)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH1;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH1\n");
			printf("SGI to send to the sender.\n");
			sample_intctrl_exec_SGI(switch_number, ALT_INT_SGI_TARGET_SENDER_ONLY);
			disp_usage = true;
		}
		// Push-Switch 2
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH2)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH2;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH2\n");
			printf("SGI to send to all the core without the sender.\n");
			sample_intctrl_exec_SGI(switch_number, ALT_INT_SGI_TARGET_ALL_EXCL_SENDER);
			disp_usage = true;
		}
		// Push-Switch 3
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH3)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH3;

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH3\n");
			printf("SGI to send to all of the core.\n");
			sample_intctrl_exec_SGI(switch_number, ALT_INT_SGI_TARGET_LIST);
			disp_usage = true;
		}
		
		util_intlog_print();

		switch_raw_bk = switch_raw;
	}
	printf("==== End While(1) loop process. ================\n");

	util_intlog_print();

	return;
}


/********************************************************/
/*	Test Command (Interrupt Controller (mainly SGI))	*/
/********************************************************/
int sample_intctrl_test_cmd(char* options)
{
	volatile uint32_t	waitstep;

    ALT_STATUS_CODE status = ALT_E_SUCCESS;
    
    printf("\r\nUser Application Start!\r\n");

    //
    // Start the interrupt system
    //
    if (status == ALT_E_SUCCESS)
    {
    	util_intlog_init();
        status = socfpga_int_start();
    }

    // Wait for 1024steps
    for(waitstep=1024; waitstep>0; waitstep--)
    {}


    // Printing Current PLL Setting. And Set Global Timer for Time Measurement.
    util_time_init();

    //// Sample Function Call!!! |---->
    printf(">>>> Execute sample_intctrl_test_init(); !!! <<<<\r\n");
    sample_intctrl_test_init();

    printf(">>>> Execute sample_intctrl_test_main(); !!! <<<<\r\n");
    sample_intctrl_test_main();

    printf(">>>> Execute sample_intctrl_test_uninit(); !!! <<<<\r\n");
    sample_intctrl_test_uninit();
    //// Sample Function Call!!! <----|

    // Printing All Results of Time Measurement. And Uninit Global Timer.
    util_time_uninit();

    //
    // Stop the interrupt system
    //
    socfpga_int_stop();
	util_intlog_print();

    printf("Finished running the sample !!!\r\n");
    if (status == ALT_E_SUCCESS)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}

// All APIs to be tested in this sample.
/* -- Interrupt Controller Global Interface [Secure] -- */
/* Usage:	ALT_STATUS_CODE  alt_int_global_enable_ns (void);	*/
/* Usage:	ALT_STATUS_CODE  alt_int_global_disable_ns (void);	*/
/* Usage:	ALT_STATUS_CODE  alt_int_global_enable_all (void);	*/
/* Usage:	ALT_STATUS_CODE  alt_int_global_disable_all (void);	*/
/* -- Software Generated Interrupts [Secure] -- */
/* Usage:	ALT_STATUS_CODE  alt_int_sgi_trigger (ALT_INT_INTERRUPT_t int_id, ALT_INT_SGI_TARGET_t target_filter, alt_int_cpu_target_t target_list, bool secure_only);	*/
/* -- Interrupt Controller CPU Interface [Secure] -- */
/* Usage:	ALT_STATUS_CODE  alt_int_cpu_enable_ns (void);	*/
/* Usage:	ALT_STATUS_CODE  alt_int_cpu_disable_ns (void);	*/
/* Usage:	ALT_STATUS_CODE  alt_int_cpu_enable_all (void);	*/
/* Usage:	ALT_STATUS_CODE  alt_int_cpu_disable_all (void);	*/
/* Usage:	ALT_STATUS_CODE  alt_int_cpu_config_get (bool *use_secure_binary_point, bool *use_FIQ_for_secure_interrupts, bool *allow_secure_ack_all_interrupts);	*/
/* Usage:	ALT_STATUS_CODE  alt_int_cpu_config_set (bool use_secure_binary_point, bool use_FIQ_for_secure_interrupts, bool allow_secure_ack_all_interrupts);	*/
/* Usage:	uint32_t  alt_int_cpu_priority_mask_get (void);	*/
/* Usage:	ALT_STATUS_CODE  alt_int_cpu_priority_mask_set (uint32_t priority_mask);	*/
/* Usage:	uint32_t  alt_int_cpu_binary_point_get (void);	*/
/* Usage:	ALT_STATUS_CODE  alt_int_cpu_binary_point_set (uint32_t binary_point);	*/
/* Usage:	uint32_t  alt_int_cpu_binary_point_get_ns (void);	*/
/* Usage:	ALT_STATUS_CODE  alt_int_cpu_binary_point_set_ns (uint32_t binary_point);	*/
/* -- Interrupt Utility Functions [Secure] -- */
/* Usage:	uint32_t  alt_int_util_int_count (void);	*/
/* Usage:	alt_int_cpu_target_t  alt_int_util_cpu_current (void);	*/

////////////////////////////// Common Functions //////////////////////////////////////// 

/***********************************************************************************
 * end of file 
 ***********************************************************************************/
