﻿/********************************************************************************//*!
 * @file  sample_dmac.c
 * @brief  Example program using HPS internal DMA (DMA-330).
 * 
 * @details  Generation and execution of original DMA microcode.
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2018/03/29  ---  -----  新規作成.
 * @date 2018/07/02  ---  #25940  All in One ベアメタル・アプリケーション v17.1 への対応を実施.
 * @date 2019/01/18  ---  -----   All in One ベアメタル・アプリケーション v18.1 への対応を実施.
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes 
 ***********************************************************************************/
#include <stdio.h>
#include <string.h>
#include "hwlib.h"
#include "socal/socal.h"
#include "socal/hps.h"
#include "socal/alt_sysmgr.h"
#include "alt_types.h"
#include "alt_mpu_registers.h"
#include "alt_generalpurpose_io.h"
#include "alt_globaltmr.h"
#include "alt_clock_manager.h"
#include "alt_interrupt.h"
#include "alt_address_space.h"
#include "alt_cache.h"
#include "alt_mmu.h"
#include "alt_dma.h"
#include "alt_dma_program.h"

#include "sample_app_setting.h"
#include "util_time_measurement.h"
#include "util_interrupt_log.h"
#include "mem_util.h"
#include "alt_printf.h"


/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/

typedef enum {
	AXCACHEe_NON = 0,				// WA,RA=0/C=0/B=0 (000b) .. Noncacheableand nonbufferable
	AXCACHEe_BUFF,					// WA,RA=0/C-0/B=1 (001b) .. Bufferable only
	AXCACHEe_CACHE,					// WA,RA=0/C=1/B=0 (010b) .. Cacheable, but do not allocate
	AXCACHEe_CACHEBUFF,				// WA,RA=0/C=1/B=1 (011b) .. Cacheable and bufferable, but do not allocate
	AXCACHEe_RESERVE1,				// WA,RA=1/C=0/B=0 (100b) .. Reserved
	AXCACHEe_RESERVE2,				// WA,RA=1/C=0/B=1 (101b) .. Reserved
	AXCACHEe_CACHE_ALLOC,			// WA,RA=1/C=1/B=0 (110b) .. Cacheable write-through, allocate on both reads and writes
	AXCACHEe_CACHEBUFF_ALLOC		// WA,RA=1/C=1/B=1 (111b) .. Cacheable write-back, allocate on both reads and writes
} AXCACHEe;

typedef ALT_STATUS_CODE (*DMA_PROGRAM_FUNKt)(ALT_DMA_PROGRAM_t*, uintptr_t, uintptr_t, size_t, int, int);

#define DMA_TEST_SIZE	(2 * _MB)		/* ＠＠通常は2MB以内 (DALT_DMA_PROGRAM_PROVISION_BUFFER_SIZEを拡張すれば大きなサイズも可) */

#ifndef MIN
#define MIN(a, b) ((a) > (b) ? (b) : (a))
#endif /* MIN */

#ifndef ARRAY_COUNT
#define ARRAY_COUNT(array) (sizeof(array) / sizeof(array[0]))
#endif

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/
// prototype (Common Functions) //
void sample_dmac_initialize(void);
static int sample_dmac_test_execute(	ALT_DMA_CHANNEL_t channel,	// DMA CH番号(0~7) 
										void* dst_address,			// CPUから見た送信先アドレス 
										void* src_address,			// CPUから見た送信元アドレス 
										AXCACHEe dst_axcache,		// DMAC->DST の AXI信号(AWCACHE)  
										AXCACHEe src_axcache,		// SRC->DMAC の AXI信号(ARCACHE)  
										size_t trans_size,			// DMA転送サイズ 
										uint32_t test_data,			// テストデータ初期値(転送アドレスのオフセット値を加算した値を転送)
										bool acp_en,				// ACP有効？ 
										bool cacheope_en		);	// CACHE操作有効？ 
void sample_dmac_callback_DMA_IRQ0(uint32_t icciar, void *context);
void sample_dmac_callback_DMA_IRQ1(uint32_t icciar, void *context);
void sample_dmac_callback_DMA_IRQ2(uint32_t icciar, void *context);
void sample_dmac_callback_DMA_IRQ3(uint32_t icciar, void *context);
void sample_dmac_callback_DMA_IRQ4(uint32_t icciar, void *context);
void sample_dmac_callback_DMA_IRQ5(uint32_t icciar, void *context);
void sample_dmac_callback_DMA_IRQ6(uint32_t icciar, void *context);
void sample_dmac_callback_DMA_IRQ7(uint32_t icciar, void *context);
void sample_dmac_callback_DMA_IRQ_ABORT(uint32_t icciar, void *context);
void sample_dmac_callback_DMA_ECC_CORRECTED_IRQ(uint32_t icciar, void *context);
void sample_dmac_callback_DMA_ECC_UNCORRECTED_IRQ(uint32_t icciar, void *context);
static void sample_dmac_setting_gic(void);
static void sample_dmac_print_manager_status(void);
static void sample_dmac_print_ch_status(ALT_DMA_CHANNEL_t channel, bool detail);
static void sample_dmac_print_all_status(void);
// prototype (Common Functions) //
static ALT_STATUS_CODE sample_dma_m2m_1bytex16beat(ALT_DMA_PROGRAM_t * program,
                                                        uintptr_t segdstpa,
                                                        uintptr_t segsrcpa,
                                                        size_t segsize,
                                                        int dst_cache_ctrl,
                                                        int src_cache_ctrl);
static ALT_STATUS_CODE sample_dma_m2m_2bytex16beat(ALT_DMA_PROGRAM_t * program,
                                                        uintptr_t segdstpa,
                                                        uintptr_t segsrcpa,
                                                        size_t segsize,
                                                        int dst_cache_ctrl,
                                                        int src_cache_ctrl);
static ALT_STATUS_CODE sample_dma_m2m_4bytex16beat(ALT_DMA_PROGRAM_t * program,
                                                        uintptr_t segdstpa,
                                                        uintptr_t segsrcpa,
                                                        size_t segsize,
                                                        int dst_cache_ctrl,
                                                        int src_cache_ctrl);
static ALT_STATUS_CODE sample_dma_m2m_8bytex16beat(ALT_DMA_PROGRAM_t * program,
                                                        uintptr_t segdstpa,
                                                        uintptr_t segsrcpa,
                                                        size_t segsize,
                                                        int dst_cache_ctrl,
                                                        int src_cache_ctrl);
#if 0
static ALT_STATUS_CODE sample_dma_m2m_16bytex16beat(ALT_DMA_PROGRAM_t * program,
                                                        uintptr_t segdstpa,
                                                        uintptr_t segsrcpa,
                                                        size_t segsize,
                                                        int dst_cache_ctrl,
                                                        int src_cache_ctrl);
#endif
void sample_dma_m2m_setting(int ccr_size);
ALT_STATUS_CODE sample_dma_m2m_program(ALT_DMA_CHANNEL_t channel,
                                         ALT_DMA_PROGRAM_t * program,
                                         void * dst,		int dst_cache_ctrl,
                                         const void * src,	int src_cache_ctrl,
                                         size_t size);
void sample_dmac_test_main(ALT_DMA_CHANNEL_t channel, void* src_address, void* dst_address, size_t size);
int  sample_dmac_test_cmd(char* options);

/***********************************************************************************
 *  variables 
 ***********************************************************************************/
/* 以下、使用していないのでコメントアウト */
//uint32_t dma_source_buf_nc[DMA_TEST_SIZE/sizeof(uint32_t)] __attribute__ ((aligned(4096))) __attribute__ ((section (".common1_region"))) ;
//uint32_t dma_source_buf_wt[DMA_TEST_SIZE/sizeof(uint32_t)] __attribute__ ((aligned(4096))) __attribute__ ((section (".common2_region"))) ;
//uint32_t dma_source_buf_wb[DMA_TEST_SIZE/sizeof(uint32_t)] __attribute__ ((aligned(4096))) __attribute__ ((section (".common3_region"))) ;
//uint32_t dma_source_buf_wba[DMA_TEST_SIZE/sizeof(uint32_t)] __attribute__ ((aligned(4096))) __attribute__ ((section (".common4_region"))) ;
//uint32_t dma_destination_buf[DMA_TEST_SIZE/sizeof(uint32_t)] __attribute__ ((aligned(4096))) ;
//uint32_t dma_destination_buf_nc[DMA_TEST_SIZE/sizeof(uint32_t)] __attribute__ ((aligned(4096))) __attribute__ ((section (".common1_region"))) ;

size_t	dma_trans_size = DMA_TEST_SIZE;			// DMA転送サイズ
DMA_PROGRAM_FUNKt dma_m2m_program_func = sample_dma_m2m_8bytex16beat;

ALT_DMA_CFG_t dma_config;			/*!< DMA Configration Table. */
ALT_DMA_PROGRAM_t dma_program[8];	/*!< Micro code program buffer for the DMAC (The index# of the array means the DMA channel#) */

/*! Status String Definitions of DMA Manager */
//typedef enum ALT_DMA_MANAGER_STATE_e
//{
//    ALT_DMA_MANAGER_STATE_STOPPED     = 0, /*!< Stopped */
//    ALT_DMA_MANAGER_STATE_EXECUTING   = 1, /*!< Executing */
//    ALT_DMA_MANAGER_STATE_CACHE_MISS  = 2, /*!< Cache Miss */
//    ALT_DMA_MANAGER_STATE_UPDATING_PC = 3, /*!< Updating PC */
//    ALT_DMA_MANAGER_STATE_WFE         = 4, /*!< Waiting for Event */
//    ALT_DMA_MANAGER_STATE_FAULTING    = 15 /*!< Faulting */
//}
//ALT_DMA_MANAGER_STATE_t;
const char* StatusStringDMAManager[16] = {
	"STOPPED(0)",
	"EXECUTING(1)",
	"CACHE_MISS(2)",
	"UPDATING_PC(3)",
	"WFE(4)",
	"---(5)",
	"---(6)",
	"---(7)",
	"---(8)",
	"---(9)",
	"---(10)",
	"---(11)",
	"---(12)",
	"---(13)",
	"---(14)",
	"FAULTING(15)"
};
/*! Status String Definitions of DMA Channel */
//typedef enum ALT_DMA_CHANNEL_STATE_e
//{
//    ALT_DMA_CHANNEL_STATE_STOPPED             = 0,  /*!< Stopped */
//    ALT_DMA_CHANNEL_STATE_EXECUTING           = 1,  /*!< Executing */
//    ALT_DMA_CHANNEL_STATE_CACHE_MISS          = 2,  /*!< Cache Miss */
//    ALT_DMA_CHANNEL_STATE_UPDATING_PC         = 3,  /*!< Updating PC */
//    ALT_DMA_CHANNEL_STATE_WFE                 = 4,  /*!< Waiting for Event */
//    ALT_DMA_CHANNEL_STATE_AT_BARRIER          = 5,  /*!< At Barrier */
//    ALT_DMA_CHANNEL_STATE_WFP                 = 7,  /*!< Waiting for Peripheral */
//    ALT_DMA_CHANNEL_STATE_KILLING             = 8,  /*!< Killing */
//    ALT_DMA_CHANNEL_STATE_COMPLETING          = 9,  /*!< Completing */
//    ALT_DMA_CHANNEL_STATE_FAULTING_COMPLETING = 14, /*!< Faulting Completing */
//    ALT_DMA_CHANNEL_STATE_FAULTING            = 15  /*!< Faulting */
//}
//ALT_DMA_CHANNEL_STATE_t;
const char* StatusStringDMAChannel[16] = {
	"STOPPED(0)",
	"EXECUTING(1)",
	"CACHE_MISS(2)",
	"UPDATING_PC(3)",
	"WFE(4)",
	"AT_BARRIER(5)",
	"---(6)",
	"WFP(7)",
	"KILLING(8)",
	"COMPLETING(9)",
	"---(10)",
	"---(11)",
	"---(12)",
	"---(13)",
	"FAULTING_COMPLETING(14)",
	"FAULTING(15)"
};

/*! Interrupt Callback Function Table */
const CALLBACK_FUNKt	SAMPLE_DMAC_CallBacks[11] =
{
		sample_dmac_callback_DMA_IRQ0,
		sample_dmac_callback_DMA_IRQ1,
		sample_dmac_callback_DMA_IRQ2,
		sample_dmac_callback_DMA_IRQ3,
		sample_dmac_callback_DMA_IRQ4,
		sample_dmac_callback_DMA_IRQ5,
		sample_dmac_callback_DMA_IRQ6,
		sample_dmac_callback_DMA_IRQ7,
		sample_dmac_callback_DMA_IRQ_ABORT,
		sample_dmac_callback_DMA_ECC_CORRECTED_IRQ,
		sample_dmac_callback_DMA_ECC_UNCORRECTED_IRQ
};

volatile int	SAMPLE_DMAC_CallBackCount[11];	/*!< Interrupt detection counters */

/***********************************************************************************
 *  functions 
 ***********************************************************************************/

////////////////////////////// Common Functions //////////////////////////////////////// 
/********************************************************************************//*!
 * @fn  void sample_dmac_initialize(void)
 * @brief  Initializing the DMA functions of hwlib.
 *
 * @details  Call the initialization function for the DMA Controller API.
 * @param[in,out] void  none
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2015/03/02  ---  #18042  新規作成.
 **//*******************************************************************************/
void sample_dmac_initialize(void)
{
	ALT_STATUS_CODE result_code;
	int i;

	/* GIC Setting for DMA Interrupts */
	sample_dmac_setting_gic();

	/* Execute initialize dmac function of hwlib  */
	/* Usage:	ALT_STATUS_CODE  alt_dma_init (const ALT_DMA_CFG_t *dma_cfg);	*/
	// Set the parameters for alt_dma_init().
	for(i=0; i<4; i++){
		dma_config.periph_mux[i] = ALT_DMA_PERIPH_MUX_DEFAULT;	// Handle FPGA / CAN muxing (default is FPGA)
	}
	dma_config.manager_sec = ALT_DMA_SECURITY_DEFAULT;			// Handle Manager Secure / NonSecure (default is Secure)
	for(i=0; i<ARRAY_COUNT(dma_config.irq_sec); i++){
		dma_config.irq_sec[i] = ALT_DMA_SECURITY_DEFAULT;		// Handle IRQ Secure / NonSecure (default is Secure)
	}
	for(i=0; i<32; i++){
		dma_config.periph_sec[i] = ALT_DMA_SECURITY_DEFAULT;	// peripheral Secure / NonSecure (default is Secure)
	}
	result_code = alt_dma_init(&dma_config);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_dma_init() .... result=%d\n", (int)result_code);
	}

	/* Allocate all DMA Channels */
	/* Usage:	ALT_STATUS_CODE  alt_dma_channel_alloc (ALT_DMA_CHANNEL_t channel);	*/
	for(i=ALT_DMA_CHANNEL_0; i<=ALT_DMA_CHANNEL_7; i++){
		result_code = alt_dma_channel_alloc((ALT_DMA_CHANNEL_t)i);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_dma_channel_alloc(%d) .... result=%d\n", i, (int)result_code);
		}
	}

	/* Test for interrupt callback registration. */
	printf("Test for interrupt callback registration.\n");
	/* Usage:	ALT_STATUS_CODE  alt_dma_send_event (ALT_DMA_EVENT_t evt_num);	*/
	for(i=ALT_DMA_EVENT_0; i<=ALT_DMA_EVENT_7; i++){
		util_time_record_start_point(3);
		result_code = alt_dma_send_event((ALT_DMA_EVENT_t)i);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_dma_send_event(%d) .... result=%d\n", i, (int)result_code);
		}
	}
	printf("\n");

	/* Print all DMA statuses */
	sample_dmac_print_all_status();
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  static int sample_dmac_test_execute(ALT_DMA_CHANNEL_t channel, void* dst_address, void* src_address,
 * 											AXCACHEe dst_axcache, AXCACHEe src_axcache, size_t trans_size,
 * 											uint32_t test_data,	bool acp_en, bool cacheope_en)
 * @brief  Execute DMA Test.
 *
 * @details  Start the DMA transfer in the following conditions.
 *           - restart is false
 *             After generating a DMA CH parameter and see DMAC Micro Code to run the DMA transfer.
 *           - restart is true
 *             Restart the DMA transfers using DMA CH parameter has already been generated.
 * @param[in] channel  target DMA CH.
 * @param[in] evnet  Event number to notify when the transfer completes.
 * @param[in] test_p  Pointer to a DMA test parameters (info address, size, etc.).
 * @param[in] restart  Specify true to start the transfer by a restart.
 * @retval void  none
 *
 * @attention  When you first start calling this function in the restart = true. for each channels.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2015/03/02  ---  #18042  新規作成.
 **//*******************************************************************************/
static int sample_dmac_test_execute(	ALT_DMA_CHANNEL_t channel,	// DMA CH Number(0~7)
										void* dst_address,			// Destination address of CPU View
										void* src_address,			// Source address of CPU View
										AXCACHEe dst_axcache,		// AXI Signal of DMAC->DST (AWCACHE)
										AXCACHEe src_axcache,		// AXI Signal of SRC->DMAC (ARCACHE)
										size_t trans_size,			// DMA Transfer size
										uint32_t test_data,			// Inital value of TEST Data
										bool acp_en,				// ACP Enable?
										bool cacheope_en		)	// Cache Operation Enable?
{
	ALT_STATUS_CODE result_code;
	void*	dma_dst_address = dst_address;	// Destination address of L3(DMAC) View
	void*	dma_src_address = src_address;	// Source address of L3(DMAC) View
	int result_memcmp = -1;					// Verification Result. (0:OK／!0:NG)
	volatile int polling_value;

	// If ACP-Enable setting is true, Adding ACP window to transfer address.
	if(acp_en)
	{
#ifdef soc_cv_av
		dma_dst_address = (void*)((uint32_t)dma_dst_address + ACP_WINDOW_BASE);
		dma_src_address = (void*)((uint32_t)dma_src_address + ACP_WINDOW_BASE);
#endif /* soc_cv_av */
	}

	printf("==== DMA Test Parameters ====\n");
	printf(" - DMA CH Select ..... ALT_DMA_CHANNEL_%d\n", (int)channel);
	printf(" - Source Address ........... 0x%08X\n", (int)dma_src_address);
	printf(" - Destination Address ...... 0x%08X\n", (int)dma_dst_address);
	printf(" - Transport size ........... 0x%08X\n", (int)trans_size);
#ifdef soc_cv_av
	printf(" - Transport via ACP Port ... %s\n", (acp_en ? "True" : "False"));
#else /* soc_cv_av */
	printf(" - Transport via ACP Port ... %s\n", "Auto");
#endif /* soc_cv_av */
	printf(" - Execute cache operation .. %s\n", (cacheope_en ? "True" : "False"));


	// Write test data to the source and destination 0 clear (Also effect to on-cash transfer area).
	util_time_record_start_point(0);
	memset(dst_address, 0, trans_size);
	sample_memset_incrementdata_4byte(src_address, test_data, trans_size);
	util_time_record_end_point(0);

	// Execute CACHE maintenace operation.
	// - ACP unused and valid cache (WriteBack) of when the DMA transfers data areas must be implemented cache maintenance process.
	// - DMA Descriptor (MicroCode space) cache maintenance is care within the HWLib API so cache maintenance unnecessary.
	util_time_record_start_point(1);
	if(cacheope_en)
	{
		alt_cache_system_purge(dst_address, trans_size);		/* Destination area off the cache */
		alt_cache_system_clean(src_address, trans_size);		/* Test data synchronized to the physical memory from the cache. */
	}
	util_time_record_end_point(1);

	// Programming DMA CHX
	util_time_record_start_point(2);
	result_code = sample_dma_m2m_program(channel,
							&dma_program[channel],
							dma_dst_address, dst_axcache,
							dma_src_address, src_axcache,
							trans_size);
	util_time_record_end_point(2);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: sample_dma_m2m_program(%d) .. result=%d (%s[%d]) \n", channel, (int)result_code, __FILE__, __LINE__);

	} else {

		// Execute DMA CHX
		util_time_record_start_point(3);
		polling_value = SAMPLE_DMAC_CallBackCount[channel];
		result_code = alt_dma_channel_exec(channel, &dma_program[channel]);

		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_dma_channel_exec(%d) .. result=%d (%s[%d]) \n", channel, (int)result_code, __FILE__, __LINE__);
		} else {

			// Polling the completion of DMA CHX.
			while(polling_value == SAMPLE_DMAC_CallBackCount[channel])
			{
				__asm__ ("nop");
			};
		}
		// util_time_record_end_point(3); ... Implemented on the DMA completion interrupt callbacks.
		fflush(stdout);
	}

#if 0	/* The following cache operations are probably unnecessary. (Required if it on the destination buffer space to cache) */
	util_time_record_start_point(5);
	if(cacheope_en)
	{
		alt_cache_system_invalidate(dst_address, trans_size);
	}
	util_time_record_end_point(5);
#endif

	// Verification of DMA Transfer data.
	util_time_record_start_point(4);
	result_memcmp = memcmp(src_address, dst_address, trans_size);
	util_time_record_end_point(4);

	// Displays the processing times.
	util_time_print_result_partial(0, 4, enumUTIL_TIME_TARGET_by_SECOND);
	printf("\n");

	return result_memcmp;
}

/********************************************************************************//*!
 * @fn  static void sample_dmac_print_manager_status(void)
 * @brief  Print the DMA Manager status and information.
 *
 * @details  Run the following APIs.
 *           - alt_dma_manager_state_get
 *           - alt_dma_manager_fault_status_get
 * @param[in,out] void  none
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/26  ---  #13246  新規作成. 
 **//*******************************************************************************/
static void sample_dmac_print_manager_status(void)
{
	ALT_STATUS_CODE result_code;
	ALT_DMA_MANAGER_STATE_t dma_manager_state;
	ALT_DMA_MANAGER_FAULT_t dma_manager_fault_state;
	const char*	status_string_p;

	printf("==== Print DMA Manager Status ====\n");
	// -- DMA Manager Status --
	/* Usage:	ALT_STATUS_CODE  alt_dma_manager_state_get (ALT_DMA_MANAGER_STATE_t *state);	*/
	result_code = alt_dma_manager_state_get(&dma_manager_state);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	if(((int)dma_manager_state < (int)ALT_DMA_MANAGER_STATE_STOPPED)||((int)ALT_DMA_MANAGER_STATE_FAULTING < (int)dma_manager_state)){
		status_string_p = NULL;
	} else {
		status_string_p = StatusStringDMAManager[dma_manager_state];
	}
	printf("alt_dma_manager_state_get()       : %s .... result=%d\n", status_string_p, (int)result_code);
	if(dma_manager_state == ALT_DMA_MANAGER_STATE_FAULTING){
		// -- DMA Manager Fault Status --
		/* Usage:	ALT_STATUS_CODE  alt_dma_manager_fault_status_get (ALT_DMA_MANAGER_FAULT_t *fault);	*/
		result_code = alt_dma_manager_fault_status_get(&dma_manager_fault_state);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: ");
		}
		printf("alt_dma_manager_fault_status_get(): 0x%08X .... result=%d\n", (int)dma_manager_fault_state, (int)result_code);
	}

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_dmac_print_ch_status(ALT_DMA_CHANNEL_t channel, bool detail)
 * @brief  Print the DMA Channel status and information.
 *
 * @details  Run the following APIs for specified DMA CH.
 *           - alt_dma_channel_state_get
 *           - alt_dma_channel_fault_status_get
 *           - alt_dma_event_int_status_get_raw
 *           - alt_dma_int_status_get
 *           - alt_dma_channel_reg_get
 *           - alt_dma_program_progress_reg
 * @param[in] channel  target DMA CH.
 * @param[in] detail  To display more information if true.
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/26  ---  #13246  新規作成. 
 **//*******************************************************************************/
static void sample_dmac_print_ch_status(ALT_DMA_CHANNEL_t channel, bool detail)
{
	ALT_STATUS_CODE result_code;
	ALT_DMA_CHANNEL_STATE_t dma_channel_state;
	ALT_DMA_CHANNEL_FAULT_t dma_channel_fault_state;
	ALT_DMA_EVENT_t evt_num;
	uint32_t reg_DMA_CCR;
	uint32_t reg_DMA_SAR;
	uint32_t reg_DMA_DAR;
	uint32_t progress;
	uint32_t progress_percent;
	const char*	status_string_p;

	printf("==== Print DMA Status [CH:%d] ====\n", (int)channel);
	// -- DMA CH Resources Status --
	/* Usage:	ALT_STATUS_CODE  alt_dma_channel_state_get (ALT_DMA_CHANNEL_t channel, ALT_DMA_CHANNEL_STATE_t *state);	*/
	result_code = alt_dma_channel_state_get(channel, &dma_channel_state);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	if(((int)dma_channel_state < (int)ALT_DMA_CHANNEL_STATE_STOPPED)||((int)ALT_DMA_CHANNEL_STATE_FAULTING < (int)dma_channel_state)){
		status_string_p = NULL;
	} else {
		status_string_p = StatusStringDMAChannel[dma_channel_state];
	}
	printf("alt_dma_channel_state_get()       : %s .... result=%d\n", status_string_p, (int)result_code);
	if((dma_channel_state == ALT_DMA_CHANNEL_STATE_FAULTING_COMPLETING)||(dma_channel_state == ALT_DMA_CHANNEL_STATE_FAULTING)){
		// -- DMA CH Falut Status --
		/* Usage:	ALT_STATUS_CODE  alt_dma_channel_fault_status_get (ALT_DMA_CHANNEL_t channel, ALT_DMA_CHANNEL_FAULT_t *fault);	*/
		result_code = alt_dma_channel_fault_status_get(channel, &dma_channel_fault_state);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: ");
		}
		printf("alt_dma_channel_fault_status_get(): 0x%08X .... result=%d\n", (int)dma_channel_fault_state, (int)result_code);
	}

	if(detail){
		evt_num = (ALT_DMA_EVENT_t)(ALT_DMA_EVENT_0 + channel);
		// -- DMA CH event or int status --
		/* Usage:	ALT_STATUS_CODE  alt_dma_event_int_status_get_raw (ALT_DMA_EVENT_t evt_num);	*/
		printf("alt_dma_event_int_status_get_raw(): .... result=%d\n", (int)alt_dma_event_int_status_get_raw(evt_num));
		// -- DMA CH int status --
		/* Usage:	ALT_STATUS_CODE  alt_dma_int_status_get (ALT_DMA_EVENT_t irq_num);	*/
		printf("alt_dma_int_status_get()          : .... result=%d\n", (int)alt_dma_int_status_get(evt_num));
	}

	/* Usage:	ALT_STATUS_CODE  alt_dma_channel_reg_get (ALT_DMA_CHANNEL_t channel, ALT_DMA_PROGRAM_REG_t reg, uint32_t *val);	*/
	result_code = alt_dma_channel_reg_get(channel, ALT_DMA_PROGRAM_REG_CCR, &reg_DMA_CCR);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_dma_channel_reg_get(CCR)      : 0x%08X .... result=%d\n", (int)reg_DMA_CCR, (int)result_code);
	result_code = alt_dma_channel_reg_get(channel, ALT_DMA_PROGRAM_REG_SAR, &reg_DMA_SAR);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_dma_channel_reg_get(SAR)      : 0x%08X .... result=%d\n", (int)reg_DMA_SAR, (int)result_code);
	result_code = alt_dma_channel_reg_get(channel, ALT_DMA_PROGRAM_REG_DAR, &reg_DMA_DAR);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: ");
	}
	printf("alt_dma_channel_reg_get(DAR)      : 0x%08X .... result=%d\n", (int)reg_DMA_DAR, (int)result_code);

	if(detail){
		/* Usage:	ALT_STATUS_CODE  alt_dma_program_progress_reg (ALT_DMA_PROGRAM_t *pgm, ALT_DMA_PROGRAM_REG_t reg, uint32_t current, uint32_t *progress);	*/
		result_code = alt_dma_program_progress_reg(&dma_program[channel], ALT_DMA_PROGRAM_REG_SAR, reg_DMA_SAR, &progress);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: ");
		} else {
			if(((int)progress < 0)||(progress > dma_trans_size)){
				progress_percent = 0;
			} else {
				progress_percent = (uint32_t)((progress / dma_trans_size) * 100);
			}
		}
		printf("alt_dma_program_progress_reg(SAR) : 0x%08X (%3d%%) .... result=%d\n", (int)progress, (int)progress_percent, (int)result_code);
		
		result_code = alt_dma_program_progress_reg(&dma_program[channel], ALT_DMA_PROGRAM_REG_DAR, reg_DMA_DAR, &progress);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: ");
		} else {
			if(((int)progress < 0)||(progress > dma_trans_size)){
				progress_percent = 0;
			} else {
				progress_percent = (uint32_t)((progress / dma_trans_size) * 100);
			}
		}
		printf("alt_dma_program_progress_reg(DAR) : 0x%08X (%3d%%) .... result=%d\n", (int)progress, (int)progress_percent, (int)result_code);
	}

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_dmac_print_all_status(void)
 * @brief  Print all DMA status and informations.
 *
 * @details  Run the following Functions.
 *           - sample_dmac_print_manager_status
 *           - sample_dmac_print_ch_status (for all DMA CHs)
 * @param[in,out] void  none
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/26  ---  #13246  新規作成. 
 **//*******************************************************************************/
static void sample_dmac_print_all_status(void)
{
	int	i;

	printf("======== Print All DMA Status ========\n");
	// -- DMA Manager Status --
	sample_dmac_print_manager_status();
	// -- DMA CH Resources Status --
	for(i=0; i<8; i++){
		// -- DMA CH Status --
		sample_dmac_print_ch_status((ALT_DMA_CHANNEL_t)i, false);
	}

	return;
}

/********************************************************************************//*!
 * @fn  void sample_dmac_callback_xxxx(uint32_t icciar, void *context)
 * @brief  DMA interrupt callback functions.
 *
 * @details  working opportunity an interrupt from DMAC, indicate
 *           an interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued.
 * @param[in] context  The user provided context.
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/26  ---  #13246  新規作成. 
 **//*******************************************************************************/
void sample_dmac_callback_DMA_IRQ0(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_0);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_0) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ0);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ0) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_DMAC_CallBackCount[0]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ0 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_DMAC_CallBackCount[0]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ0, (int)icciar, (int)SAMPLE_DMAC_CallBackCount[0]);
	return;
}
void sample_dmac_callback_DMA_IRQ1(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_1);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_1) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ1);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ1) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_DMAC_CallBackCount[1]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ1 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_DMAC_CallBackCount[1]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ1, (int)icciar, (int)SAMPLE_DMAC_CallBackCount[1]);
	return;
}
void sample_dmac_callback_DMA_IRQ2(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_2);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_2) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ2);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ2) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_DMAC_CallBackCount[2]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ2 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_DMAC_CallBackCount[2]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ2, (int)icciar, (int)SAMPLE_DMAC_CallBackCount[2]);
	return;
}
void sample_dmac_callback_DMA_IRQ3(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_3);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_3) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ3);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ3) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_DMAC_CallBackCount[3]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ3 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_DMAC_CallBackCount[3]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ3, (int)icciar, (int)SAMPLE_DMAC_CallBackCount[3]);
	return;
}
void sample_dmac_callback_DMA_IRQ4(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_4);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_4) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ4);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ4) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_DMAC_CallBackCount[4]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ4 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_DMAC_CallBackCount[4]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ4, (int)icciar, (int)SAMPLE_DMAC_CallBackCount[4]);
	return;
}
void sample_dmac_callback_DMA_IRQ5(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_5);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_5) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ5);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ5) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_DMAC_CallBackCount[5]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ5 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_DMAC_CallBackCount[5]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ5, (int)icciar, (int)SAMPLE_DMAC_CallBackCount[5]);
	return;
}
void sample_dmac_callback_DMA_IRQ6(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_6);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_6) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ6);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ6) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_DMAC_CallBackCount[6]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ6 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_DMAC_CallBackCount[6]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ6, (int)icciar, (int)SAMPLE_DMAC_CallBackCount[6]);
	return;
}
void sample_dmac_callback_DMA_IRQ7(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	util_time_record_end_point(3);
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_7);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_7) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ7);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ7) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_DMAC_CallBackCount[7]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ7 is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_DMAC_CallBackCount[7]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ7, (int)icciar, (int)SAMPLE_DMAC_CallBackCount[7]);
	return;
}
void sample_dmac_callback_DMA_IRQ_ABORT(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;
	// DMAC Pending Clear
	/* Usage:	ALT_STATUS_CODE  alt_dma_int_clear (ALT_DMA_EVENT_t irq_num);	*/
	result_code = alt_dma_int_clear(ALT_DMA_EVENT_ABORT);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_dma_int_clear(ALT_DMA_EVENT_ABORT) result=%d\n", (int)result_code);
	}
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ_ABORT);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_IRQ_ABORT) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_DMAC_CallBackCount[8]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_IRQ_ABORT is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_DMAC_CallBackCount[8]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_IRQ_ABORT, (int)icciar, (int)SAMPLE_DMAC_CallBackCount[8]);

	//sample_dmac_print_all_status();

	return;
}
void sample_dmac_callback_DMA_ECC_CORRECTED_IRQ(uint32_t icciar, void *context)
{
#ifdef soc_cv_av
	ALT_STATUS_CODE result_code;
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_ECC_CORRECTED_IRQ);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_ECC_CORRECTED_IRQ) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_DMAC_CallBackCount[9]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_ECC_CORRECTED_IRQ is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_DMAC_CallBackCount[9]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_ECC_CORRECTED_IRQ, (int)icciar, (int)SAMPLE_DMAC_CallBackCount[9]);
#endif /* soc_cv_av */
	return;
}
void sample_dmac_callback_DMA_ECC_UNCORRECTED_IRQ(uint32_t icciar, void *context)
{
#ifdef soc_cv_av
	ALT_STATUS_CODE result_code;
	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_ECC_UNCORRECTED_IRQ);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "ERROR!!: alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DMA_ECC_UNCORRECTED_IRQ) result=%d\n", (int)result_code);
	}
	// Indicates the interrupt occurred.
	SAMPLE_DMAC_CallBackCount[10]++;
	//printf("[INTERRUPT] ALT_INT_INTERRUPT_DMA_ECC_UNCORRECTED_IRQ is occurred!! icciar=0x%08X counter=%d\n", (int)icciar, SAMPLE_DMAC_CallBackCount[10]);
	util_intlog_record(ALT_INT_INTERRUPT_DMA_ECC_UNCORRECTED_IRQ, (int)icciar, (int)SAMPLE_DMAC_CallBackCount[10]);
#endif /* soc_cv_av */
	return;
}

/********************************************************************************//*!
 * @fn  void sample_dmac_setting_gic(void)
 * @brief  Setting of GIC (DMA interrupt).
 *
 * @details    In order to receive a DMA interrupt setting of GIC.
 * @param[in,out] void  none
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/26  ---  #13246  新規作成. 
 **//*******************************************************************************/
static void sample_dmac_setting_gic(void)
{
	ALT_STATUS_CODE result_code;

    //	ALT_INT_INTERRUPT_DMA_IRQ0                = 136, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ1                = 137, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ2                = 138, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ3                = 139, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ4                = 140, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ5                = 141, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ6                = 142, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ7                = 143, /*!< # */
    //	ALT_INT_INTERRUPT_DMA_IRQ_ABORT           = 144, /*!< # */
    //	soc_cv_av:	ALT_INT_INTERRUPT_DMA_ECC_CORRECTED_IRQ   = 145, /*!< # */
    //	soc_cv_av:	ALT_INT_INTERRUPT_DMA_ECC_UNCORRECTED_IRQ = 146, /*!< # */
    //	soc_a10:	ALT_INT_INTERRUPT_SERR_GLOBAL = 34, /*!< # */
    //	soc_a10:	ALT_INT_INTERRUPT_DERR_GLOBAL = 32, /*!< # */
    //	/*!<
    //	 * Interrupts sourced from the DMA Controller.
    //	 *  * All interrupts in this group are level triggered.
    //	 */

	int i;

	for(i=ALT_INT_INTERRUPT_DMA_IRQ0; i<=ALT_INT_INTERRUPT_DMA_IRQ_ABORT; i++)
	{
		/* Usage: ALT_STATUS_CODE alt_int_isr_register(ALT_INT_INTERRUPT_t int_id, alt_int_callback_t callback, void* context);	*/
		result_code = alt_int_isr_register((ALT_INT_INTERRUPT_t)i, SAMPLE_DMAC_CallBacks[(i - ALT_INT_INTERRUPT_DMA_IRQ0)], NULL);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_int_isr_register(%d) result=%d\n", i, (int)result_code);
		}

		/* Usage: ALT_STATUS_CODE alt_int_dist_target_set(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t target);	*/
		/* ##'target=1' the specified notification interrupts to only core0.  */
		result_code = alt_int_dist_target_set((ALT_INT_INTERRUPT_t)i, 1);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_int_dist_target_set(%d, 1) result=%d\n", i, (int)result_code);
		}

		/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_set(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t trigger_type);	*/
		result_code = alt_int_dist_trigger_set((ALT_INT_INTERRUPT_t)i, ALT_INT_TRIGGER_LEVEL);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_int_dist_trigger_set(%d, ALT_INT_TRIGGER_LEVEL) result=%d\n", i, (int)result_code);
		}

		/* Usage: ALT_STATUS_CODE alt_int_dist_priority_set(ALT_INT_INTERRUPT_t int_id, uint32_t priority);	*/
		result_code = alt_int_dist_priority_set((ALT_INT_INTERRUPT_t)i, 0);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_int_dist_priority_set(%d, 0) result=%d\n", i, (int)result_code);
		}

		// To reset the counter interrupt detection.
		SAMPLE_DMAC_CallBackCount[(i - ALT_INT_INTERRUPT_DMA_IRQ0)] = 0;

		if(i <= ALT_INT_INTERRUPT_DMA_IRQ_ABORT)
		{
			// To specify the notification methods (send events or interrupt notification) for each DMA resources of the DMA controller.
			/* Usage:	ALT_STATUS_CODE  alt_dma_event_int_select (ALT_DMA_EVENT_t evt_num, ALT_DMA_EVENT_SELECT_t opt);	*/
			result_code = alt_dma_event_int_select((ALT_DMA_EVENT_t)(ALT_DMA_EVENT_0 + i - ALT_INT_INTERRUPT_DMA_IRQ0), ALT_DMA_EVENT_SELECT_SIG_IRQ);
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_dma_event_int_select(%d, ALT_DMA_EVENT_SELECT_SIG_IRQ) result=%d\n", (int)(ALT_DMA_EVENT_0 + i - ALT_INT_INTERRUPT_DMA_IRQ0), (int)result_code);
			}
		}

		/* No need to call the API below. Because all interrupts are secure by default. */
		/* Usage: ALT_STATUS_CODE  alt_int_dist_secure_enable (ALT_INT_INTERRUPT_t int_id);	*/

		/* Usage: ALT_STATUS_CODE  alt_int_dist_enable (ALT_INT_INTERRUPT_t int_id);	*/
		result_code = alt_int_dist_enable((ALT_INT_INTERRUPT_t)i);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_int_dist_enable(%d) result=%d\n", i, (int)result_code);
		}

	}

	return;
}

static ALT_STATUS_CODE sample_dma_m2m_1bytex16beat(ALT_DMA_PROGRAM_t * program,
                                                        uintptr_t segdstpa,
                                                        uintptr_t segsrcpa,
                                                        size_t segsize,
                                                        int dst_cache_ctrl,
                                                        int src_cache_ctrl)
{
    uint32_t burstcount;
    size_t sizeleft = segsize;
    ALT_STATUS_CODE status = ALT_E_SUCCESS;

    /* Cache control bits a bitmask to within a 3 bit value. */
    dst_cache_ctrl &= 7;
    src_cache_ctrl &= 7;

    if (status == ALT_E_SUCCESS)
    {
        status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_SAR, segsrcpa);
    }
    if (status == ALT_E_SUCCESS)
    {
        status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_DAR, segdstpa);
    }

    /* This is the number of bursts */
    burstcount = sizeleft;

    dprintf("DMA[M->M][seg]: Total Main 1-byte burst size transfer(s): %" PRIu32 ".\n", burstcount);

    /* Determine how many 16 length bursts can be done */

    if (burstcount >> 4)
    {
        uint32_t length16burstcount = burstcount >> 4;
        burstcount &= 0xf;

        dprintf("DMA[M->M][seg]:   Number of 16 burst length 1-byte transfer(s): %" PRIu32 ".\n", length16burstcount);
        dprintf("DMA[M->M][seg]:   Number of remaining 1-byte transfer(s):       %" PRIu32 ".\n", burstcount);

        /* Program in the following parameters:
         *  - SS8   : Source      burst size of 1-byte
         *  - DS8   : Destination burst size of 1-byte
         *  - SB16  : Source      burst length of 16 transfers
         *  - DB16  : Destination burst length of 16 transfers
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ALT_DMA_CCR_OPT_SB16
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ALT_DMA_CCR_OPT_DB16
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }

        while (length16burstcount > 0)
        {
            uint32_t loopcount = MIN(length16burstcount, 256);
            if (status != ALT_E_SUCCESS)
            {
                break;
            }

            length16burstcount -= loopcount;

            dprintf("DMA[M->M][seg]:   Looping %" PRIu32 "x 16 burst length 1-byte transfer(s).\n", loopcount);

            if ((status == ALT_E_SUCCESS) && (loopcount > 1))
            {
                status = alt_dma_program_DMALP(program, loopcount);
            }
            if (status == ALT_E_SUCCESS)
            {
                status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
            if (status == ALT_E_SUCCESS)
            {
                status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
            if ((status == ALT_E_SUCCESS) && (loopcount > 1))
            {
                status = alt_dma_program_DMALPEND(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
        }
    }

    /* At this point, we should have [burstcount] 1-byte transfer(s)
     * remaining. [burstcount] should be less than 16. */

    /* Do one more burst with a SB / DB of length [burstcount]. */

    if (burstcount)
    {
        /* Program in the following parameters:
         *  - SS8   : Source      burst size of 1-byte
         *  - DS8   : Destination burst size of 1-byte
         *  - SBx   : Source      burst length of [burstlength] transfer(s)
         *  - DBx   : Destination burst length of [burstlength] transfer(s)
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((burstcount - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((burstcount - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    return status;
}

static ALT_STATUS_CODE sample_dma_m2m_2bytex16beat(ALT_DMA_PROGRAM_t * program,
                                                        uintptr_t segdstpa,
                                                        uintptr_t segsrcpa,
                                                        size_t segsize,
                                                        int dst_cache_ctrl,
                                                        int src_cache_ctrl)
{
    uint32_t burstcount;
    bool correction;
    size_t sizeleft = segsize;
    ALT_STATUS_CODE status = ALT_E_SUCCESS;

    /* Cache control bits a bitmask to within a 3 bit value. */
    dst_cache_ctrl &= 7;
    src_cache_ctrl &= 7;

    if (status == ALT_E_SUCCESS)
    {
        status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_SAR, segsrcpa);
    }
    if (status == ALT_E_SUCCESS)
    {
        status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_DAR, segdstpa);
    }

    /*
     * The algorithm uses the strategy described in PL330 B.3.1.
     * It is extended for 2-byte and 1-byte unaligned cases.
     * */

    /* First see how many byte(s) we need to transfer to get src to be 4 byte aligned */
    if (segsrcpa & 0x1)
    {
        uint32_t aligncount = MIN(2 - (segsrcpa & 0x1), sizeleft);
        sizeleft -= aligncount;

        dprintf("DMA[M->M][seg]: Total pre-alignment 1-byte burst size transfer(s): %" PRIu32 ".\n", aligncount);

        /* Program in the following parameters:
         *  - SS8   : Source      burst size of 1-byte
         *  - DS8   : Destination burst size of 1-byte
         *  - SBx   : Source      burst length of [aligncount] transfer(s)
         *  - DBx   : Destination burst length of [aligncount] transfer(s)
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((aligncount - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((aligncount - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    /* This is the number of 2-byte bursts */
    burstcount = sizeleft >> 1;

    /* If bursting was done and src and dst are not mod-2 congruent, we need to
     * correct for some data left over in the MFIFO due to unaligment issues. */
    correction = (burstcount != 0) && ((segsrcpa & 0x1) != (segdstpa & 0x1));

    /* Update the size left to transfer */
    sizeleft &= 0x1;

    dprintf("DMA[M->M][seg]: Total Main 2-byte burst size transfer(s): %" PRIu32 ".\n", burstcount);

    /* Determine how many 16 length bursts can be done */

    if (burstcount >> 4)
    {
        uint32_t length16burstcount = burstcount >> 4;
        burstcount &= 0xf;

        dprintf("DMA[M->M][seg]:   Number of 16 burst length 2-byte transfer(s): %" PRIu32 ".\n", length16burstcount);
        dprintf("DMA[M->M][seg]:   Number of remaining 2-byte transfer(s):       %" PRIu32 ".\n", burstcount);

        /* Program in the following parameters:
         *  - SS16  : Source      burst size of 2-byte
         *  - DS16  : Destination burst size of 2-byte
         *  - SB16  : Source      burst length of 16 transfers
         *  - DB16  : Destination burst length of 16 transfers
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ALT_DMA_CCR_OPT_SB16
                                              | ALT_DMA_CCR_OPT_SS16
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ALT_DMA_CCR_OPT_DB16
                                              | ALT_DMA_CCR_OPT_DS16
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }

        while (length16burstcount > 0)
        {
            uint32_t loopcount = MIN(length16burstcount, 256);
            if (status != ALT_E_SUCCESS)
            {
                break;
            }

            length16burstcount -= loopcount;

            dprintf("DMA[M->M][seg]:   Looping %" PRIu32 "x 16 burst length 2-byte transfer(s).\n", loopcount);

            if ((status == ALT_E_SUCCESS) && (loopcount > 1))
            {
                status = alt_dma_program_DMALP(program, loopcount);
            }
            if (status == ALT_E_SUCCESS)
            {
                status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
            if (status == ALT_E_SUCCESS)
            {
                status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
            if ((status == ALT_E_SUCCESS) && (loopcount > 1))
            {
                status = alt_dma_program_DMALPEND(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
        }
    }

    /* At this point, we should have [burstcount] 2-byte transfer(s)
     * remaining. [burstcount] should be less than 16. */

    /* Do one more burst with a SB / DB of length [burstcount]. */

    if (burstcount)
    {
        /* Program in the following parameters:
         *  - SS16  : Source      burst size of 2-byte
         *  - DS16  : Destination burst size of 2-byte
         *  - SBx   : Source      burst length of [burstlength] transfer(s)
         *  - DBx   : Destination burst length of [burstlength] transfer(s)
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((burstcount - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS16
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((burstcount - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS16
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    /* Corrections may be needed if usrc and udst are relatively 2-byte unaligned
     * and bursts were previously used. */

    if (correction)
    {
        if (status == ALT_E_SUCCESS)
        {
            /* This is the number of 1-byte corrections DMAST needed.
             * This is determined by how many remaining data is in the MFIFO after
             * the burst(s) have completed. */
            int correctcount = (segdstpa + (2 - (segsrcpa & 0x1))) & 0x1;

            dprintf("DMA[M->M][seg]: Total correction 1-byte burst size transfer(s): %u.\n", correctcount);

            /* Program in the following parameters:
             *  - SS8   : Source      burst size of 1-byte
             *  - DS8   : Destination burst size of 1-byte
             *  - SBx   : Source      burst length of [correctcount] transfer(s)
             *  - DBx   : Destination burst length of [correctcount] transfer(s)
             *  - SC(7) : Source      cacheable write-back, allocate on reads only
             *  - DC(7) : Destination cacheable write-back, allocate on writes only
             *  - All other options default. */

            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((correctcount - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((correctcount - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    /* At this point, there should be 0 - 1 1-byte transfers remaining. */

    if (sizeleft)
    {
        dprintf("DMA[M->M][seg]: Total post 1-byte burst size transfer(s): %u.\n", sizeleft);

        /* Program in the following parameters:
         *  - SS8   : Source      burst size of 1-byte)
         *  - DS8   : Destination burst size of 1-byte)
         *  - SBx   : Source      burst length of [sizeleft] transfer(s)
         *  - DBx   : Destination burst length of [sizeleft] transfer(s)
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((sizeleft - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((sizeleft - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    return status;
}

static ALT_STATUS_CODE sample_dma_m2m_4bytex16beat(ALT_DMA_PROGRAM_t * program,
                                                        uintptr_t segdstpa,
                                                        uintptr_t segsrcpa,
                                                        size_t segsize,
                                                        int dst_cache_ctrl,
                                                        int src_cache_ctrl)
{
    uint32_t burstcount;
    bool correction;
    size_t sizeleft = segsize;
    ALT_STATUS_CODE status = ALT_E_SUCCESS;

    /* Cache control bits a bitmask to within a 3 bit value. */
    dst_cache_ctrl &= 7;
    src_cache_ctrl &= 7;

    if (status == ALT_E_SUCCESS)
    {
        status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_SAR, segsrcpa);
    }
    if (status == ALT_E_SUCCESS)
    {
        status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_DAR, segdstpa);
    }

    /*
     * The algorithm uses the strategy described in PL330 B.3.1.
     * It is extended for 2-byte and 1-byte unaligned cases.
     * */

    /* First see how many byte(s) we need to transfer to get src to be 4 byte aligned */
    if (segsrcpa & 0x3)
    {
        uint32_t aligncount = MIN(4 - (segsrcpa & 0x3), sizeleft);
        sizeleft -= aligncount;

        dprintf("DMA[M->M][seg]: Total pre-alignment 1-byte burst size transfer(s): %" PRIu32 ".\n", aligncount);

        /* Program in the following parameters:
         *  - SS8   : Source      burst size of 1-byte
         *  - DS8   : Destination burst size of 1-byte
         *  - SBx   : Source      burst length of [aligncount] transfer(s)
         *  - DBx   : Destination burst length of [aligncount] transfer(s)
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((aligncount - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((aligncount - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    /* This is the number of 4-byte bursts */
    burstcount = sizeleft >> 2;

    /* If bursting was done and src and dst are not mod-4 congruent, we need to
     * correct for some data left over in the MFIFO due to unaligment issues. */
    correction = (burstcount != 0) && ((segsrcpa & 0x3) != (segdstpa & 0x3));

    /* Update the size left to transfer */
    sizeleft &= 0x3;

    dprintf("DMA[M->M][seg]: Total Main 4-byte burst size transfer(s): %" PRIu32 ".\n", burstcount);

    /* Determine how many 16 length bursts can be done */

    if (burstcount >> 4)
    {
        uint32_t length16burstcount = burstcount >> 4;
        burstcount &= 0xf;

        dprintf("DMA[M->M][seg]:   Number of 16 burst length 4-byte transfer(s): %" PRIu32 ".\n", length16burstcount);
        dprintf("DMA[M->M][seg]:   Number of remaining 4-byte transfer(s):       %" PRIu32 ".\n", burstcount);

        /* Program in the following parameters:
         *  - SS32  : Source      burst size of 4-byte
         *  - DS32  : Destination burst size of 4-byte
         *  - SB16  : Source      burst length of 16 transfers
         *  - DB16  : Destination burst length of 16 transfers
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ALT_DMA_CCR_OPT_SB16
                                              | ALT_DMA_CCR_OPT_SS32
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ALT_DMA_CCR_OPT_DB16
                                              | ALT_DMA_CCR_OPT_DS32
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }

        while (length16burstcount > 0)
        {
            uint32_t loopcount = MIN(length16burstcount, 256);
            if (status != ALT_E_SUCCESS)
            {
                break;
            }

            length16burstcount -= loopcount;

            dprintf("DMA[M->M][seg]:   Looping %" PRIu32 "x 16 burst length 4-byte transfer(s).\n", loopcount);

            if ((status == ALT_E_SUCCESS) && (loopcount > 1))
            {
                status = alt_dma_program_DMALP(program, loopcount);
            }
            if (status == ALT_E_SUCCESS)
            {
                status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
            if (status == ALT_E_SUCCESS)
            {
                status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
            if ((status == ALT_E_SUCCESS) && (loopcount > 1))
            {
                status = alt_dma_program_DMALPEND(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
        }
    }

    /* At this point, we should have [burstcount] 4-byte transfer(s)
     * remaining. [burstcount] should be less than 16. */

    /* Do one more burst with a SB / DB of length [burstcount]. */

    if (burstcount)
    {
        /* Program in the following parameters:
         *  - SS32  : Source      burst size of 4-byte
         *  - DS32  : Destination burst size of 4-byte
         *  - SBx   : Source      burst length of [burstlength] transfer(s)
         *  - DBx   : Destination burst length of [burstlength] transfer(s)
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((burstcount - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS32
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((burstcount - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS32
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    /* Corrections may be needed if usrc and udst are relatively 4-byte unaligned
     * and bursts were previously used. */

    if (correction)
    {
        if (status == ALT_E_SUCCESS)
        {
            /* This is the number of 1-byte corrections DMAST needed.
             * This is determined by how many remaining data is in the MFIFO after
             * the burst(s) have completed. */
            int correctcount = (segdstpa + (4 - (segsrcpa & 0x3))) & 0x3;

            dprintf("DMA[M->M][seg]: Total correction 1-byte burst size transfer(s): %u.\n", correctcount);

            /* Program in the following parameters:
             *  - SS8   : Source      burst size of 1-byte
             *  - DS8   : Destination burst size of 1-byte
             *  - SBx   : Source      burst length of [correctcount] transfer(s)
             *  - DBx   : Destination burst length of [correctcount] transfer(s)
             *  - SC(7) : Source      cacheable write-back, allocate on reads only
             *  - DC(7) : Destination cacheable write-back, allocate on writes only
             *  - All other options default. */

            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((correctcount - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((correctcount - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    /* At this point, there should be 0 - 3 1-byte transfers remaining. */

    if (sizeleft)
    {
        dprintf("DMA[M->M][seg]: Total post 1-byte burst size transfer(s): %u.\n", sizeleft);

        /* Program in the following parameters:
         *  - SS8   : Source      burst size of 1-byte)
         *  - DS8   : Destination burst size of 1-byte)
         *  - SBx   : Source      burst length of [sizeleft] transfer(s)
         *  - DBx   : Destination burst length of [sizeleft] transfer(s)
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((sizeleft - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((sizeleft - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    return status;
}

static ALT_STATUS_CODE sample_dma_m2m_8bytex16beat(ALT_DMA_PROGRAM_t * program,
                                                        uintptr_t segdstpa,
                                                        uintptr_t segsrcpa,
                                                        size_t segsize,
                                                        int dst_cache_ctrl,
                                                        int src_cache_ctrl)
{
    uint32_t burstcount;
    bool correction;
    size_t sizeleft = segsize;
    ALT_STATUS_CODE status = ALT_E_SUCCESS;

    /* Cache control bits a bitmask to within a 3 bit value. */
    dst_cache_ctrl &= 7;
    src_cache_ctrl &= 7;

    if (status == ALT_E_SUCCESS)
    {
        status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_SAR, segsrcpa);
    }
    if (status == ALT_E_SUCCESS)
    {
        status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_DAR, segdstpa);
    }

    /*
     * The algorithm uses the strategy described in PL330 B.3.1.
     * It is extended for 2-byte and 1-byte unaligned cases.
     * */

    /* First see how many byte(s) we need to transfer to get src to be 8 byte aligned */
    if (segsrcpa & 0x7)
    {
        uint32_t aligncount = MIN(8 - (segsrcpa & 0x7), sizeleft);
        sizeleft -= aligncount;

        dprintf("DMA[M->M][seg]: Total pre-alignment 1-byte burst size transfer(s): %" PRIu32 ".\n", aligncount);

        /* Program in the following parameters:
         *  - SS8   : Source      burst size of 1-byte
         *  - DS8   : Destination burst size of 1-byte
         *  - SBx   : Source      burst length of [aligncount] transfer(s)
         *  - DBx   : Destination burst length of [aligncount] transfer(s)
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((aligncount - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((aligncount - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    /* This is the number of 8-byte bursts */
    burstcount = sizeleft >> 3;

    /* If bursting was done and src and dst are not mod-8 congruent, we need to
     * correct for some data left over in the MFIFO due to unaligment issues. */
    correction = (burstcount != 0) && ((segsrcpa & 0x7) != (segdstpa & 0x7));

    /* Update the size left to transfer */
    sizeleft &= 0x7;

    dprintf("DMA[M->M][seg]: Total Main 8-byte burst size transfer(s): %" PRIu32 ".\n", burstcount);

    /* Determine how many 16 length bursts can be done */

    if (burstcount >> 4)
    {
        uint32_t length16burstcount = burstcount >> 4;
        burstcount &= 0xf;

        dprintf("DMA[M->M][seg]:   Number of 16 burst length 8-byte transfer(s): %" PRIu32 ".\n", length16burstcount);
        dprintf("DMA[M->M][seg]:   Number of remaining 8-byte transfer(s):       %" PRIu32 ".\n", burstcount);

        /* Program in the following parameters:
         *  - SS64  : Source      burst size of 8-byte
         *  - DS64  : Destination burst size of 8-byte
         *  - SB16  : Source      burst length of 16 transfers
         *  - DB16  : Destination burst length of 16 transfers
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ALT_DMA_CCR_OPT_SB16
                                              | ALT_DMA_CCR_OPT_SS64
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ALT_DMA_CCR_OPT_DB16
                                              | ALT_DMA_CCR_OPT_DS64
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }

        while (length16burstcount > 0)
        {
            uint32_t loopcount = MIN(length16burstcount, 256);
            if (status != ALT_E_SUCCESS)
            {
                break;
            }

            length16burstcount -= loopcount;

            dprintf("DMA[M->M][seg]:   Looping %" PRIu32 "x 16 burst length 8-byte transfer(s).\n", loopcount);

            if ((status == ALT_E_SUCCESS) && (loopcount > 1))
            {
                status = alt_dma_program_DMALP(program, loopcount);
            }
            if (status == ALT_E_SUCCESS)
            {
                status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
            if (status == ALT_E_SUCCESS)
            {
                status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
            if ((status == ALT_E_SUCCESS) && (loopcount > 1))
            {
                status = alt_dma_program_DMALPEND(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
        }
    }

    /* At this point, we should have [burstcount] 8-byte transfer(s)
     * remaining. [burstcount] should be less than 16. */

    /* Do one more burst with a SB / DB of length [burstcount]. */

    if (burstcount)
    {
        /* Program in the following parameters:
         *  - SS64  : Source      burst size of 8-byte
         *  - DS64  : Destination burst size of 8-byte
         *  - SBx   : Source      burst length of [burstlength] transfer(s)
         *  - DBx   : Destination burst length of [burstlength] transfer(s)
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((burstcount - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS64
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((burstcount - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS64
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    /* Corrections may be needed if usrc and udst are relatively 8-byte unaligned
     * and bursts were previously used. */

    if (correction)
    {
        if (status == ALT_E_SUCCESS)
        {
            /* This is the number of 1-byte corrections DMAST needed.
             * This is determined by how many remaining data is in the MFIFO after
             * the burst(s) have completed. */
            int correctcount = (segdstpa + (8 - (segsrcpa & 0x7))) & 0x7;

            dprintf("DMA[M->M][seg]: Total correction 1-byte burst size transfer(s): %u.\n", correctcount);

            /* Program in the following parameters:
             *  - SS8   : Source      burst size of 1-byte
             *  - DS8   : Destination burst size of 1-byte
             *  - SBx   : Source      burst length of [correctcount] transfer(s)
             *  - DBx   : Destination burst length of [correctcount] transfer(s)
             *  - SC(7) : Source      cacheable write-back, allocate on reads only
             *  - DC(7) : Destination cacheable write-back, allocate on writes only
             *  - All other options default. */

            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((correctcount - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((correctcount - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    /* At this point, there should be 0 - 7 1-byte transfers remaining. */

    if (sizeleft)
    {
        dprintf("DMA[M->M][seg]: Total post 1-byte burst size transfer(s): %u.\n", sizeleft);

        /* Program in the following parameters:
         *  - SS8   : Source      burst size of 1-byte)
         *  - DS8   : Destination burst size of 1-byte)
         *  - SBx   : Source      burst length of [sizeleft] transfer(s)
         *  - DBx   : Destination burst length of [sizeleft] transfer(s)
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((sizeleft - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((sizeleft - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    return status;
}




/***********************************************************************/
/* Burst size 16 bytes is unavailable. If forced, it causes DMA abort. 
	
	# Refer to: DMA-330 Techinical Reference Manual (It applies to the following conditions)
	2.10.4 AXI data transfer size
		The DMAC can only perform data accesses up to the configured width of the AXI data bus. If
		you program the src_burst_size or dst_burst_size fields to be larger than this then the DMAC
		signals a precise abort. See Abort sources on page 2-25 and Channel Control Registers on
		page 3-25 for more information.
	
	(HPS DMAC data access interface is 64bits width.)
*//********************************************************************/
#if 0
static ALT_STATUS_CODE sample_dma_m2m_16bytex16beat(ALT_DMA_PROGRAM_t * program,
                                                        uintptr_t segdstpa,
                                                        uintptr_t segsrcpa,
                                                        size_t segsize,
                                                        int dst_cache_ctrl,
                                                        int src_cache_ctrl)
{
    uint32_t burstcount;
    bool correction;
    size_t sizeleft = segsize;
    ALT_STATUS_CODE status = ALT_E_SUCCESS;

    /* Cache control bits a bitmask to within a 3 bit value. */
    dst_cache_ctrl &= 7;
    src_cache_ctrl &= 7;

    if (status == ALT_E_SUCCESS)
    {
        status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_SAR, segsrcpa);
    }
    if (status == ALT_E_SUCCESS)
    {
        status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_DAR, segdstpa);
    }

    /*
     * The algorithm uses the strategy described in PL330 B.3.1.
     * It is extended for 2-byte and 1-byte unaligned cases.
     * */

    /* First see how many byte(s) we need to transfer to get src to be 16 byte aligned */
    if (segsrcpa & 0xF)
    {
        uint32_t aligncount = MIN(16 - (segsrcpa & 0xF), sizeleft);
        sizeleft -= aligncount;

        dprintf("DMA[M->M][seg]: Total pre-alignment 1-byte burst size transfer(s): %" PRIu32 ".\n", aligncount);

        /* Program in the following parameters:
         *  - SS8   : Source      burst size of 1-byte
         *  - DS8   : Destination burst size of 1-byte
         *  - SBx   : Source      burst length of [aligncount] transfer(s)
         *  - DBx   : Destination burst length of [aligncount] transfer(s)
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((aligncount - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((aligncount - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    /* This is the number of 16-byte bursts */
    burstcount = sizeleft >> 4;

    /* If bursting was done and src and dst are not mod-16 congruent, we need to
     * correct for some data left over in the MFIFO due to unaligment issues. */
    correction = (burstcount != 0) && ((segsrcpa & 0xF) != (segdstpa & 0xF));

    /* Update the size left to transfer */
    sizeleft &= 0xF;

    dprintf("DMA[M->M][seg]: Total Main 16-byte burst size transfer(s): %" PRIu32 ".\n", burstcount);

    /* Determine how many 16 length bursts can be done */

    if (burstcount >> 4)
    {
        uint32_t length16burstcount = burstcount >> 4;
        burstcount &= 0xf;

        dprintf("DMA[M->M][seg]:   Number of 16 burst length 16-byte transfer(s): %" PRIu32 ".\n", length16burstcount);
        dprintf("DMA[M->M][seg]:   Number of remaining 16-byte transfer(s):       %" PRIu32 ".\n", burstcount);

        /* Program in the following parameters:
         *  - SS128 : Source      burst size of 16-byte
         *  - DS128 : Destination burst size of 16-byte
         *  - SB16  : Source      burst length of 16 transfers
         *  - DB16  : Destination burst length of 16 transfers
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ALT_DMA_CCR_OPT_SB16
                                              | ALT_DMA_CCR_OPT_SS128
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ALT_DMA_CCR_OPT_DB16
                                              | ALT_DMA_CCR_OPT_DS128
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }

        while (length16burstcount > 0)
        {
            uint32_t loopcount = MIN(length16burstcount, 256);
            if (status != ALT_E_SUCCESS)
            {
                break;
            }

            length16burstcount -= loopcount;

            dprintf("DMA[M->M][seg]:   Looping %" PRIu32 "x 16 burst length 16-byte transfer(s).\n", loopcount);

            if ((status == ALT_E_SUCCESS) && (loopcount > 1))
            {
                status = alt_dma_program_DMALP(program, loopcount);
            }
            if (status == ALT_E_SUCCESS)
            {
                status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
            if (status == ALT_E_SUCCESS)
            {
                status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
            if ((status == ALT_E_SUCCESS) && (loopcount > 1))
            {
                status = alt_dma_program_DMALPEND(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
            }
        }
    }

    /* At this point, we should have [burstcount] 16-byte transfer(s)
     * remaining. [burstcount] should be less than 16. */

    /* Do one more burst with a SB / DB of length [burstcount]. */

    if (burstcount)
    {
        /* Program in the following parameters:
         *  - SS128 : Source      burst size of 16-byte
         *  - DS128 : Destination burst size of 16-byte
         *  - SBx   : Source      burst length of [burstlength] transfer(s)
         *  - DBx   : Destination burst length of [burstlength] transfer(s)
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((burstcount - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS128
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((burstcount - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS128
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    /* Corrections may be needed if usrc and udst are relatively 16-byte unaligned
     * and bursts were previously used. */

    if (correction)
    {
        if (status == ALT_E_SUCCESS)
        {
            /* This is the number of 1-byte corrections DMAST needed.
             * This is determined by how many remaining data is in the MFIFO after
             * the burst(s) have completed. */
            int correctcount = (segdstpa + (16 - (segsrcpa & 0xF))) & 0xF;

            dprintf("DMA[M->M][seg]: Total correction 1-byte burst size transfer(s): %u.\n", correctcount);

            /* Program in the following parameters:
             *  - SS8   : Source      burst size of 1-byte
             *  - DS8   : Destination burst size of 1-byte
             *  - SBx   : Source      burst length of [correctcount] transfer(s)
             *  - DBx   : Destination burst length of [correctcount] transfer(s)
             *  - SC(7) : Source      cacheable write-back, allocate on reads only
             *  - DC(7) : Destination cacheable write-back, allocate on writes only
             *  - All other options default. */

            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((correctcount - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((correctcount - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    /* At this point, there should be 0 - 15 1-byte transfers remaining. */

    if (sizeleft)
    {
        dprintf("DMA[M->M][seg]: Total post 1-byte burst size transfer(s): %u.\n", sizeleft);

        /* Program in the following parameters:
         *  - SS8   : Source      burst size of 1-byte)
         *  - DS8   : Destination burst size of 1-byte)
         *  - SBx   : Source      burst length of [sizeleft] transfer(s)
         *  - DBx   : Destination burst length of [sizeleft] transfer(s)
         *  - SC(7) : Source      cacheable write-back, allocate on reads only
         *  - DC(7) : Destination cacheable write-back, allocate on writes only
         *  - All other options default. */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAMOV(program, ALT_DMA_PROGRAM_REG_CCR,
                                            (   ((sizeleft - 1) << 4) /* SB */
                                              | ALT_DMA_CCR_OPT_SS8
                                              | ALT_DMA_CCR_OPT_SA_DEFAULT
                                              | ALT_DMA_CCR_OPT_SP_DEFAULT
                                              | ALT_DMA_CCR_OPT_SC(src_cache_ctrl)	/*ALT_DMA_CCR_OPT_SC(7)*/
                                              | ((sizeleft - 1) << 18) /* DB */
                                              | ALT_DMA_CCR_OPT_DS8
                                              | ALT_DMA_CCR_OPT_DA_DEFAULT
                                              | ALT_DMA_CCR_OPT_DP_DEFAULT
                                              | ALT_DMA_CCR_OPT_DC(dst_cache_ctrl)	/*ALT_DMA_CCR_OPT_DC(7)*/
                                              | ALT_DMA_CCR_OPT_ES_DEFAULT
                                            )
                );
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMALD(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAST(program, ALT_DMA_PROGRAM_INST_MOD_NONE);
        }
    }

    return status;
}
#endif

void sample_dma_m2m_setting(int ccr_size)
{
	switch(ccr_size)
	{
	case 1:
		dma_m2m_program_func = sample_dma_m2m_1bytex16beat;
		break;
	case 2:
		dma_m2m_program_func = sample_dma_m2m_2bytex16beat;
		break;
	case 4:
		dma_m2m_program_func = sample_dma_m2m_4bytex16beat;
		break;
	case 8:
		dma_m2m_program_func = sample_dma_m2m_8bytex16beat;
		break;
#if 0
	case 16:
		dma_m2m_program_func = sample_dma_m2m_16bytex16beat;
		break;
#endif
	default:
		dma_m2m_program_func = sample_dma_m2m_8bytex16beat;
		break;
	}
}

ALT_STATUS_CODE sample_dma_m2m_program(ALT_DMA_CHANNEL_t channel,
                                         ALT_DMA_PROGRAM_t * program,
                                         void * dst,		int dst_cache_ctrl,
                                         const void * src,	int src_cache_ctrl,
                                         size_t size)
{
    ALT_STATUS_CODE status = ALT_E_SUCCESS;
    ALT_DMA_EVENT_t evt;

    // If the size is zero, just return success.
    if (size == 0)
    {
        return ALT_E_SUCCESS;
    }

    switch(channel)
    {
    case ALT_DMA_CHANNEL_0:
        evt = ALT_DMA_EVENT_0;
        break;
    case ALT_DMA_CHANNEL_1:
        evt = ALT_DMA_EVENT_1;
        break;
    case ALT_DMA_CHANNEL_2:
        evt = ALT_DMA_EVENT_2;
        break;
    case ALT_DMA_CHANNEL_3:
        evt = ALT_DMA_EVENT_3;
        break;
    case ALT_DMA_CHANNEL_4:
        evt = ALT_DMA_EVENT_4;
        break;
    case ALT_DMA_CHANNEL_5:
        evt = ALT_DMA_EVENT_5;
        break;
    case ALT_DMA_CHANNEL_6:
        evt = ALT_DMA_EVENT_6;
        break;
    case ALT_DMA_CHANNEL_7:
        evt = ALT_DMA_EVENT_7;
        break;
    default:
        return ALT_E_BAD_ARG;
    }

    if (status == ALT_E_SUCCESS)
    {
        status = alt_dma_program_init(program);
    }

    if (size != 0)
    {
        ALT_MMU_VA_TO_PA_COALESCE_t coalesce_dst;
        ALT_MMU_VA_TO_PA_COALESCE_t coalesce_src;
        uintptr_t segpa_dst   = 0;
        uintptr_t segpa_src   = 0;
        uint32_t  segsize_dst = 0;
        uint32_t  segsize_src = 0;

        dprintf("DMA[M->M]: dst  = %p.\n",   dst);
        dprintf("DMA[M->M]: src  = %p.\n",   src);
        dprintf("DMA[M->M]: size = 0x%x.\n", size);

        /* Detect if memory regions overshoots the address space.
         * This error checking is handled by the coalescing API. */

        /* Detect if memory regions overlaps. */

        if ((uintptr_t)dst > (uintptr_t)src)
        {
            if ((uintptr_t)src + size - 1 > (uintptr_t)dst)
            {
                return ALT_E_BAD_ARG;
            }
        }
        else
        {
            if ((uintptr_t)dst + size - 1 > (uintptr_t)src)
            {
                return ALT_E_BAD_ARG;
            }
        }

        /*
         * Attempt to coalesce and make the transfer.
         */

        if (status == ALT_E_SUCCESS)
        {
            status = alt_mmu_va_to_pa_coalesce_begin(&coalesce_dst, dst, size);
        }

        if (status == ALT_E_SUCCESS)
        {
            status = alt_mmu_va_to_pa_coalesce_begin(&coalesce_src, src, size);
        }

        while (size)
        {
            uint32_t segsize;
            if (status != ALT_E_SUCCESS)
            {
                break;
            }

            /*
             * If any of dst or src segments has been completed (or not started), determine its
             * next segment.
             */

            if ((status == ALT_E_SUCCESS) && (segsize_dst == 0))
            {
                status = alt_mmu_va_to_pa_coalesce_next(&coalesce_dst, &segpa_dst, &segsize_dst);

                dprintf("DMA[M->M]: Next dst segment: PA = 0x%x, size = 0x%" PRIx32 ".\n", segpa_dst, segsize_dst);
            }

            if ((status == ALT_E_SUCCESS) && (segsize_src == 0))
            {
                status = alt_mmu_va_to_pa_coalesce_next(&coalesce_src, &segpa_src, &segsize_src);

                dprintf("DMA[M->M]: Next src segment: PA = 0x%x, size = 0x%" PRIx32 ".\n", segpa_src, segsize_src);
            }

            /*
             * Determine the largest segment to safely transfer next.
             * */

            /* This is the largest segment that can safely be transfered considering both dst and
             * src segmentation. Typically dst or src or both segment(s) will complete. */
            segsize = MIN(segsize_dst, segsize_src);

            /*
             * Transfer the largest safe segment.
             * */

            if (status == ALT_E_SUCCESS)
            {
                dprintf("DMA[M->M]: Transfering safe size = 0x%" PRIx32 ".\n", segsize);
                
                status = dma_m2m_program_func(program, segpa_dst, segpa_src, segsize, src_cache_ctrl, dst_cache_ctrl);
            }

            /*
             * Update some bookkeeping.
             * */

            segsize_dst -= segsize;
            segsize_src -= segsize;

            if (segsize_dst)
            {
                segpa_dst += segsize;

                dprintf("DMA[M->M]: Updated dst segment: PA = 0x%x, size = 0x%" PRIx32 ".\n", segpa_dst, segsize_dst);
            }

            if (segsize_src)
            {
                segpa_src += segsize;

                dprintf("DMA[M->M]: Updated src segment: PA = 0x%x, size = 0x%" PRIx32 ".\n", segpa_src, segsize_src);
            }

            /* Use MIN() to assuredly prevent infinite loop. If either the dst or src has not yet
             * completed, coalesce_end() will catch that logical error. */
            size -= MIN(segsize, size);
        }

        if (status == ALT_E_SUCCESS)
        {
            status = alt_mmu_va_to_pa_coalesce_end(&coalesce_dst);
        }

        if (status == ALT_E_SUCCESS)
        {
            status = alt_mmu_va_to_pa_coalesce_end(&coalesce_src);
        }

    } /* if (size != 0) */

    /* Send event if requested. */
    {
        if (status == ALT_E_SUCCESS)
        {
            status = alt_dma_program_DMAWMB(program);
        }

        if (status == ALT_E_SUCCESS)
        {
            dprintf("DMA[M->M]: Adding event ...\n");
            status = alt_dma_program_DMASEV(program, evt);
        }
    }

    /* Now that everything is done, end the program. */
    if (status == ALT_E_SUCCESS)
    {
        status = alt_dma_program_DMAEND(program);
    }

    /* If there was a problem assembling the program, clean up the buffer and exit. */
    if (status != ALT_E_SUCCESS)
    {
        /* Do not report the status for the clear operation. A failure should be
         * reported regardless of if the clear is successful. */
        alt_dma_program_clear(program);
        return status;
    }

    /* Execute the program on the given channel. */
    /*return alt_dma_channel_exec(channel, program);*/
    return ALT_E_SUCCESS;
}

void sample_dmac_test_main(ALT_DMA_CHANNEL_t channel, void* src_address, void* dst_address, size_t size)
{
	AXCACHEe dst_axcache = AXCACHEe_CACHE;	// AXI Signal of DMAC->DST (AWCACHE)
	AXCACHEe src_axcache = AXCACHEe_CACHE;	// AXI Signal of SRC->DMAC (ARCACHE)
	bool acp_en = false;
	bool cacheope_en = true;
	uint32_t test_data = 0x12340000;
	int result_memcmp;
	
	dma_trans_size = MIN(size, DMA_TEST_SIZE);
	
	// Execute DMA Transfer Test.
	result_memcmp = sample_dmac_test_execute(channel, dst_address, src_address, dst_axcache, src_axcache, dma_trans_size, test_data, acp_en, cacheope_en);
	printf("DMA Result ... %s\n\n", (result_memcmp == 0 ? "OK" : "NG"));

	return;
}

/************************/
/*	Test Command (DMAC)	*/
/************************/
int sample_dmac_test_cmd(char* options)
{
	alt_u32 srcaddr, dstaddr;
	alt_u32 size;
	int bytes;

	printf("dma_test \n");

	if( sscanf(options,"%lx %lx %lx %d", &srcaddr, &dstaddr, &size, &bytes) < 4) {
		printf("arg error \n");
		return 0;
	}

	sample_dma_m2m_setting(bytes);
	sample_dmac_test_main(ALT_DMA_CHANNEL_0, (void*)srcaddr, (void*)dstaddr, (size_t)size);

	return 0;
}
/***********************************************************************************
 * end of file 
 ***********************************************************************************/
