﻿/********************************************************************************//*!
 * @file  sample_ecc.c
 * @brief  To use HWLib for Error Correcting Code (ECC) Management API example program.
 * 
 * @details  Run the API category below.
 *           - Error Correcting Code (ECC) Management
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/20  ---  #13689  新規作成. 
 * @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
 * @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
 * @date 2018/07/02  ---  #25940  All in One ベアメタル・アプリケーション v17.1 への対応を実施.
 * @date 2019/01/18  ---  -----   All in One ベアメタル・アプリケーション v18.1 への対応を実施.
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

// ---- 
// Handbook 6-26
// 
// To use ECCs, the software and system must meet the following requirements:
// ## L1 and L2 cache must be configured as write-back allocate for any cacheable memory region
// ## FPGA soft IP using the ACP must only perform the following types of data writes:
// 	## 64-bit aligned in memory
// 	## 64 bit wide accesses
// 
// ---- 

/***********************************************************************************
 *  includes 
 ***********************************************************************************/
#include <stdbool.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include "hwlib.h"
#include "socal/socal.h"
#include "socal/hps.h"
#include "alt_generalpurpose_io.h"
#include "alt_globaltmr.h"
#include "alt_clock_manager.h"
#include "alt_interrupt.h"
#include "alt_address_space.h"
#include "alt_mmu.h"
#include "alt_cache.h"
#include "alt_ecc.h"
#include "socal/alt_sysmgr.h"
#include "socal/hps.h"
#include "socal/socal.h"

#include "sample_app_setting.h"
#include "util_time_measurement.h"
#include "util_interrupt_log.h"
#include "alt_printf.h"

/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/
void* sample_mmu_tlb_alloc(const size_t size, void * context);
void sample_mmu_init(void);
void sample_mmu_print_registers(void);
void sample_ecc_init(void);
void sample_ecc_setting_gic(ALT_INT_INTERRUPT_t int_id);
void sample_ecc_corrected_callback(uint32_t icciar, void *context);
void sample_ecc_uncorrected_callback(uint32_t icciar, void *context);
void sample_ecc_test_init(void);
void sample_ecc_test_uninit(void);
void sample_ecc_test_print_usage(void);
void sample_ecc_test_main(void);
int  sample_ecc_test_cmd(char* options);
// prototype (Common Functions) //

// Test Function //
void sample_memchk_by_byte(volatile uint8_t* start_address, int bytes);
void sample_memchk_by_halfword(volatile uint16_t* start_address, int bytes);
void sample_memchk_by_word(volatile uint32_t* start_address, int bytes);
void sample_memchk_by_wordword(volatile uint64_t* start_address, int bytes);
void sample_memchk_by_memset(volatile uint32_t* start_address, int bytes);
void sample_memread_for_ecccheck(void* start_address, int bytes);

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/
/* ECC test buffer size (Bytes) */
#define	ECC_TESTBUF_SIZE	( 64 * _KB )

/* MMU table buffer size */
#define	MMU_TLB_HEAP_SIZE	( 32 * _KB )

/* TLB memory should be aligned 16KB */
#define	MMU_TLB_ALIGNMENT		(16*_KB)


/******** Definition Memory Region ********/
#define MAX_MEMORY_REGION_NUM	16

#ifdef soc_cv_av
/*! ON-CHIP RAM 0 .. PA:0x00000000~0x0000FFFF <--> VA:0x00000000~0x0000FFFF */
#define	OCRAM0_PHYSICAL_ADDR		0x00000000
#define	OCRAM0_VIRTUAL_ADDR			OCRAM0_PHYSICAL_ADDR
#define	OCRAM0_SIZE					( 64 * _KB )
#else /* soc_cv_av */
/*! A10 ONCHIPRAM .. PA:0x00000000~0x0003FFFF <--> VA: (=PA) */
#define	OCRAM0_PHYSICAL_ADDR		0x00000000
#define	OCRAM0_VIRTUAL_ADDR			OCRAM0_PHYSICAL_ADDR
#define	OCRAM0_SIZE					( 256 *_KB )
#endif /* soc_cv_av */

#if 0
/*! SDRAM .. PA:0x00100000~0x007FFFFF <--> VA:0x00100000~0x007FFFFF */
#define	SDRAM1_PHYSICAL_ADDR			0x00100000
#define	SDRAM1_VIRTUAL_ADDR			SDRAM1_PHYSICAL_ADDR
#define	SDRAM1_SIZE					( 7 * _MB )

/*! SDRAM2 .. PA:0x00800000~0x00FFFFFF <--> VA:0x01000000~0x017FFFFF */
#define	SDRAM2_PHYSICAL_ADDR		0x00800000
#define	SDRAM2_VIRTUAL_ADDR			0x01000000
#define	SDRAM2_SIZE					( 8 * _MB )

/*! SDRAM3 .. PA:0x02000000~0x3FFFFFFF <--> VA:0x02000000~0x3FFFFFFF */
#define	SDRAM3_PHYSICAL_ADDR		0x02000000
#define	SDRAM3_VIRTUAL_ADDR			SDRAM3_PHYSICAL_ADDR
#define	SDRAM3_SIZE					(( 1024 * _MB ) - (32 * _MB))

/*! HPS2FPGA .. PA:0xC000000~0xFBFFFFFF <--> VA:0xC0000000~0xFBFFFFFF */
#define	HPS2FPGA_PHYSICAL_ADDR		0xC0000000
#define	HPS2FPGA_VIRTUAL_ADDR		HPS2FPGA_PHYSICAL_ADDR
#define	HPS2FPGA_SIZE				( 960 * _MB )

/*! LW_HPS2FPGA .. PA:0xFF200000~0xFF3FFFFF <--> VA:0xFF200000~0xFF3FFFFF */
#define	LW_HPS2FPGA_PHYSICAL_ADDR	0xFF200000
#define	LW_HPS2FPGA_VIRTUAL_ADDR	LW_HPS2FPGA_PHYSICAL_ADDR
#define	LW_HPS2FPGA_SIZE			( 2 * _MB )

/*! PERIPHRALS .. PA:0xFF400000~0xFFEFFFFF <--> VA:0xFF400000~0xFFEFFFFF */
#define	PERIPHRALS_PHYSICAL_ADDR	0xFF400000
#define	PERIPHRALS_VIRTUAL_ADDR		PERIPHRALS_PHYSICAL_ADDR
#define	PERIPHRALS_SIZE				( 11 * _MB )

/*! BOOTROM .. PA:0xFFFD0000~0xFFFDFFFF <--> VA:0xFFFD0000~0xFFFDFFFF */
#define	BOOTROM_PHYSICAL_ADDR		0xFFFD0000
#define	BOOTROM_VIRTUAL_ADDR		BOOTROM_PHYSICAL_ADDR
#define	BOOTROM_SIZE				( 64 *_KB )

/*! MPUSCU .. PA:0xFFFEC000~0xFFFEEFFF <--> VA:0xFFFEC000~0xFFFEEFFF */
#define	MPUSCU_PHYSICAL_ADDR		0xFFFEC000
#define	MPUSCU_VIRTUAL_ADDR			MPUSCU_PHYSICAL_ADDR
#define	MPUSCU_SIZE					( 12 * _KB)

/*! MPUL2 .. PA:0xFFFEF000~0xFFFEFFFF <--> VA:0xFFFEF000~0xFFFEFFFF */
#define	MPUL2_PHYSICAL_ADDR			0xFFFEF000
#define	MPUL2_VIRTUAL_ADDR			MPUL2_PHYSICAL_ADDR
#define	MPUL2_SIZE					( 4 * _KB)
#endif

#ifdef soc_cv_av
/*! ON-CHIP RAM F .. PA:0xFFFF0000~0xFFFFFFFF <--> VA:0xFFFF0000~0xFFFFFFFF */
#define	OCRAMF_PHYSICAL_ADDR		0xFFFF0000
#define	OCRAMF_VIRTUAL_ADDR			OCRAMF_PHYSICAL_ADDR
#define	OCRAMF_SIZE					( 64 *_KB )
#else /* soc_cv_av */
/*! A10 ONCHIPRAM .. PA:0xFFE00000~0xFFE3FFFF <--> VA: (=PA) */
#define	OCRAMF_PHYSICAL_ADDR		0xFFE00000
#define	OCRAMF_VIRTUAL_ADDR			OCRAMF_PHYSICAL_ADDR
#define	OCRAMF_SIZE					( 256 *_KB )
#endif /* soc_cv_av */


/* Debug Register Address */
#ifdef soc_cv_av
#define ALT_MPUL2_DEBUG_CTRL_OFST           0xf40
#define ALT_MPUL2_DEBUG_CTRL_ADDR           ALT_CAST(void *, (ALT_CAST(char *, ALT_MPUL2_ADDR) + ALT_MPUL2_DEBUG_CTRL_OFST))
#else /* soc_cv_av */
#define ALT_MPUL2_DEBUG_CTRL_OFST           0x40
#define ALT_MPUL2_DEBUG_CTRL_ADDR           ALT_CAST(void *, (ALT_CAST(char *, ALT_L2_REGS_L2DBG_OFST) + ALT_MPUL2_DEBUG_CTRL_OFST))
#endif /* soc_cv_av */

/***********************************************************************************
 *  variables
 ***********************************************************************************/
uint32_t testbuf_on_onchipram[ECC_TESTBUF_SIZE/sizeof(uint32_t)] __attribute__ ((section (".testbuf"))) ;

#if 0
//typedef struct ALT_MMU_MEM_REGION_s
//{
//  void *                 va;        /*!< The beginning virtual address for the memory
//                                     *   region. The address must be aligned to one of 4KiB,
//                                     *   64KiB, 1MiB, or 16MiB boundaries.
//                                     */
//  void *                 pa;        /*!< The beginning physical address mapping for the
//                                     *   virtual address of the memory region. The address
//                                     *   must be aligned to one of 4KiB, 64KiB, 1MiB, or 16MiB
//                                     *   boundaries.
//                                     */
//  uint32_t               size;      /*!< The size of the memory region in bytes. The size
//                                     *   must be a multiple of 4KiB, 64KiB, 1MiB, or 16MiB
//                                     *   sizes.
//                                     */
//  ALT_MMU_AP_t           access;    /*!< The access permissions for the memory region. */
//  ALT_MMU_ATTR_t         attributes;/*!< The memory region attributes. These attributes
//                                     *   determine the memory type (ordering), cache
//                                     *   policy, and as a possible side effect, the
//                                     *   shareablity of the memory region.
//                                     */
//  ALT_MMU_TTB_S_t        shareable; /*!< The shareability of the memory region. */
//  ALT_MMU_TTB_XN_t       execute;   /*!< Whether instructions can be executed from this
//                                     *   memory region.
//                                     */
//  ALT_MMU_TTB_NS_t       security;  /*!< Controls whether address translations made from
//                                     *   the secure state translate physical address in
//                                     *   the secure or non-secure address map.
//                                     */
//} ALT_MMU_MEM_REGION_t;
int	ValidMemoryRegion = 11;		/*!< The number of valid descriptors. */
ALT_MMU_MEM_REGION_t MemoryRegion[MAX_MEMORY_REGION_NUM] = {
	/* [0] ON-CHIP RAM	*/
	{ (void*)OCRAM0_VIRTUAL_ADDR, (void*)OCRAM0_PHYSICAL_ADDR, OCRAM0_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_WBA,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [1] SDRAM	*/
	{ (void*)SDRAM1_VIRTUAL_ADDR, (void*)SDRAM1_PHYSICAL_ADDR, SDRAM1_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_WBA,				/* attributes, */
	  ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [2] SDRAM2	*/
	{ (void*)SDRAM2_VIRTUAL_ADDR, (void*)SDRAM2_PHYSICAL_ADDR, SDRAM2_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_WBA,				/* attributes, */
	  ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [3] SDRAM3 (include .text)	*/
	{ (void*)SDRAM3_VIRTUAL_ADDR, (void*)SDRAM3_PHYSICAL_ADDR, SDRAM3_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_WBA,				/* attributes, */
	  ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [4] HPS2FPGA	*/
	{ (void*)HPS2FPGA_VIRTUAL_ADDR, (void*)HPS2FPGA_PHYSICAL_ADDR, HPS2FPGA_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_DEVICE,			/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_ENABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [5] LW_HPS2FPGA	*/
	{ (void*)LW_HPS2FPGA_VIRTUAL_ADDR, (void*)LW_HPS2FPGA_PHYSICAL_ADDR, LW_HPS2FPGA_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_DEVICE,			/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_ENABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [6] PERIPHRALS	*/
	{ (void*)PERIPHRALS_VIRTUAL_ADDR, (void*)PERIPHRALS_PHYSICAL_ADDR, PERIPHRALS_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_STRONG,			/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_ENABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [7] BOOTROM	*/
	{ (void*)BOOTROM_VIRTUAL_ADDR, (void*)BOOTROM_PHYSICAL_ADDR, BOOTROM_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_READ_ONLY,			/* access, */
	  ALT_MMU_ATTR_WBA,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [8] MPU_SCU	*/
	{ (void*)MPUSCU_VIRTUAL_ADDR, (void*)MPUSCU_PHYSICAL_ADDR, MPUSCU_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_STRONG,			/* attributes, */
	  ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, */
	  ALT_MMU_TTB_XN_ENABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [9] MPU_L2	*/
	{ (void*)MPUL2_VIRTUAL_ADDR, (void*)MPUL2_PHYSICAL_ADDR, MPUL2_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_STRONG,			/* attributes, */
	  ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, */
	  ALT_MMU_TTB_XN_ENABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [10] ON-CHIP RAM	*/
	{ (void*)OCRAMF_VIRTUAL_ADDR, (void*)OCRAMF_PHYSICAL_ADDR, OCRAMF_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_WBA,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [11] Undefined */
	{ 0, 0, 0,	/* va, pa, size, */
	  0,		/* access, */
	  0,		/* attributes, */
	  0,		/* shareable, */
	  0,		/* execute, */
	  0			/* security */
	},
	/* [12] Undefined */
	{ 0, 0, 0,	/* va, pa, size, */
	  0,		/* access, */
	  0,		/* attributes, */
	  0,		/* shareable, */
	  0,		/* execute, */
	  0			/* security */
	},
	/* [13] Undefined */
	{ 0, 0, 0,	/* va, pa, size, */
	  0,		/* access, */
	  0,		/* attributes, */
	  0,		/* shareable, */
	  0,		/* execute, */
	  0			/* security */
	},
	/* [14] Undefined */
	{ 0, 0, 0,	/* va, pa, size, */
	  0,		/* access, */
	  0,		/* attributes, */
	  0,		/* shareable, */
	  0,		/* execute, */
	  0			/* security */
	},
	/* [15] Undefined */
	{ 0, 0, 0,	/* va, pa, size, */
	  0,		/* access, */
	  0,		/* attributes, */
	  0,		/* shareable, */
	  0,		/* execute, */
	  0			/* security */
	}
};

static uint32_t mmu_tlb_heap[MMU_TLB_HEAP_SIZE] __attribute__ ((aligned MMU_TLB_ALIGNMENT)) ;
uint32_t *mmu_tlb_l1 = NULL;

const char* PageAddressString[4] = {
	"Page 0 - MPU address range 0x00000000 - 0x3FFFFFFF",
	"Page 1 - MPU address range 0x40000000 - 0x7FFFFFFF",
	"Page 2 - MPU address range 0x80000000 - 0xBFFFFFFF",
	"Page 3 - MPU address range 0xC0000000 - 0xFFFFFFFF"
};

/***********************************************************************************
 *  functions 
 ***********************************************************************************/

/********************************************************************************//*!
 * @fn  void* sample_mmu_tlb_alloc(const size_t size, void * context)
 * @brief  Memory allocation for 1st and 2nd translation tables.
 *
 * @details  User defined function used for allocating storage for first and second level translation tables.
 *           This is the function to specify the 4th argument of the API 'alt_mmu_va_space_create'.
 * @param[in] size  The size in bytes of the storage request. The argument will be either the value ALT_MMU_TTB1_SIZE or ALT_MMU_TTB2_SIZE.
 * @param[in] context  A user provided context for the allocator function.
 * @retval void*  A sixteen-word aligned pointer to the allocated memory or NULL if the storage request cannot be satisfied. 
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/03/03  ---  #13288  新規作成.
 **//*******************************************************************************/
void* sample_mmu_tlb_alloc(const size_t size, void * context)
{
	void* p = NULL;
	printf("Execute ttb_alloc Function.\n");

	/* User defined function used for allocating storage for first and second level translation tables. */

	if(size >= sizeof(mmu_tlb_heap)){
		printf("ERROR!!: mmu_tlb_alloc() 1st argument ""size"" is too large!\n");
	} else if(((uint32_t)context % MMU_TLB_ALIGNMENT) != 0){
		printf("ERROR!!: mmu_tlb_alloc() 2st argument ""context"" is not aligned!\n");
	} else {
		p = context;
	}

	return p ;
}

/********************************************************************************//*!
 * @fn  void sample_mmu_init(void)
 * @brief  Do the initial settings of the MMU.
 *
 * @details  Run the following APIs.
 *           - alt_mmu_disable
 *           - alt_mmu_va_space_create
 *           - alt_mmu_va_space_enable
 * @param[in,out] void  none
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/03/03  ---  #13288  新規作成.
 **//*******************************************************************************/
void sample_mmu_init(void)
{
	ALT_STATUS_CODE result_code;

	/* disable MMU */
	alt_mmu_disable();

	/* create MMU tables */
	/* Usage:	ALT_STATUS_CODE  alt_mmu_va_space_create (uint32_t **ttb1, const ALT_MMU_MEM_REGION_t *mem_regions, const size_t num_mem_regions, alt_mmu_ttb_alloc_t ttb_alloc, void *ttb_alloc_context);	*/
	result_code = alt_mmu_va_space_create(&mmu_tlb_l1, MemoryRegion, ValidMemoryRegion, sample_mmu_tlb_alloc, (void*)mmu_tlb_heap );
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_mmu_va_space_create() .... result=%d\n", (int)result_code);
	}

	if(result_code == ALT_E_SUCCESS){
		/* enable MMU */
		/* Usage:	ALT_STATUS_CODE  alt_mmu_va_space_enable (const uint32_t *ttb1);	*/
		result_code = alt_mmu_va_space_enable(mmu_tlb_l1);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_mmu_va_space_enable() .... result=%d\n", (int)result_code);
		}
	}

	if(result_code != ALT_E_SUCCESS){
		printf("MMU Init error\n" );
	}

	return;
}

/********************************************************************************//*!
 * @fn  void sample_mmu_print_registers(void)
 * @brief  Print the MMU related Registers.
 *
 * @details  Print the register values obtained using the API and coprocessor instructions.
 * @param[in,out] void  none
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/03/03  ---  #13288  新規作成.
 **//*******************************************************************************/
void sample_mmu_print_registers(void)
{
	/* The value returned by the following API doesn't register values. */
	/* Retrieve only the value of the base address of the translation table. */

	/* Usage:	void *  alt_mmu_TTBR0_get (void);	*/
	printf("-- Translation Table Base Register 0 (TTBR0) --\n");
	printf("alt_mmu_TTBR0_get() .... result=0x%08X\n", (int)alt_mmu_TTBR0_get());
	/* Usage:	void *  alt_mmu_TTBR1_get (void);	*/
	printf("-- Translation Table Base Register 1 (TTBR1) --\n");
	printf("alt_mmu_TTBR1_get() .... result=0x%08X\n", (int)alt_mmu_TTBR1_get());
	
	/* API to get the following register values not provided in HWLib. */
	uint32_t ttbr0;			/* Translation Table Base Register 0 (TTBR0) */
	uint32_t ttbr1;			/* Translation Table Base Register 1 (TTBR1) */
	uint32_t ttbcr;			/* Translation Table Base Control Register (TTBCR) */
	uint32_t dacr;			/* Domain Access Control Register (DACR) */
	uint32_t contextidr;	/* Context ID Register (CONTEXTIDR) */
	__asm("MRC p15, 0, %0, c2, c0, 0" : "=r" (ttbr0));
	__asm("MRC p15, 0, %0, c2, c0, 1" : "=r" (ttbr1));
	__asm("MRC p15, 0, %0, c2, c0, 2" : "=r" (ttbcr));
	__asm("MRC p15, 0, %0, c3, c0, 0" : "=r" (dacr));
	__asm("MRC p15, 0, %0, c13, c0, 1" : "=r" (contextidr));
	printf("-- TTBRx real values, and the other register values. --\n");
	printf("TTBR0      = 0x%08X\n", (int)ttbr0);
	printf("TTBR1      = 0x%08X\n", (int)ttbr1);
	printf("TTBCR      = 0x%08X\n", (int)ttbcr);
	printf("DACR       = 0x%08X\n", (int)dacr);
	printf("CONTEXTIDR = 0x%08X\n", (int)contextidr);

	return;
}
#endif

/********************************************************************************//*!
 * @fn  void sample_ecc_init(void)
 * @brief  Initializing the sample program. 
 * 
 * @details  Call the initialization functions for the ECC Management (onchip-RAM).  
 *           Other, do the following.
 *           -# Configure the Helio Board GPIO settings. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/20  ---  #13689  新規作成. 
 **//*******************************************************************************/
void sample_ecc_init(void)
{
	ALT_STATUS_CODE result_code;
	
	
	/* Usage: ALT_STATUS_CODE alt_ecc_is_enabled(const ALT_ECC_RAM_ENUM_t ram_block); */
	result_code = alt_ecc_is_enabled(ALT_ECC_RAM_OCRAM);
	switch(result_code)
	{
	case ALT_E_TRUE:
		/* Usage: ALT_STATUS_CODE alt_ecc_stop(const ALT_ECC_RAM_ENUM_t ram_block); */
		result_code = alt_ecc_stop(ALT_ECC_RAM_OCRAM);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_ecc_stop() result=%d\n", (int)result_code);
		}
		break;
	case ALT_E_FALSE:
		/* NOP */
		break;
	default:
		printf("ERROR!!: alt_ecc_is_enabled() result=%d\n", (int)result_code);
		return;
	}
	
	/* Usage: ALT_STATUS_CODE alt_ecc_start(const ALT_ECC_RAM_ENUM_t ram_block); */
	/**** HWLib Document has a note about how to use this API to. (Below, some of the highlights.) */
	/* The decision on whether to enable ECC protection for the peripheral embedded
	 * RAM block should be made before commencing normal operational use of the peripheral.
	 * Ideally, ECC protection for a peripheral should be enabled immediately after calling
	 * the peripheral's initialization function and calling the alt_ecc_init() function
	 * designating the applicable peripheral embedded RAM block.
	 *
	 * For example, the proper initialization sequence for enabling ECC for the QSPI controller embedded RAM block is:
	 *     alt_qspi_init();			// Initialize the QSPI controller
	 *     alt_qspi_enable();		// Enable the QSPI controller.
	 *     alt_ecc_start(ALT_ECC_RAM_QSPI);		// Bring up ECC protection for QSPI.
	 */
#ifdef soc_a10
	result_code = alt_ecc_init();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_ecc_init() result=%d\n", (int)result_code);
	}
#endif /* soc_a10 */

	result_code = alt_ecc_start(ALT_ECC_RAM_OCRAM);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_ecc_start() result=%d\n", (int)result_code);
	}

	/* Initializes the on-chip RAM space. */
	memset((void*)OCRAMF_VIRTUAL_ADDR, 0, OCRAMF_SIZE);

#ifdef soc_cv_av
	// ECC (for On-Chip RAM) Corrected and Uncorrected Interrupt GIC Config 
	sample_ecc_setting_gic(ALT_INT_INTERRUPT_RAM_ECC_CORRECTED_IRQ);
	sample_ecc_setting_gic(ALT_INT_INTERRUPT_RAM_ECC_UNCORRECTED_IRQ);
#else /* soc_cv_av */
	// ECC Global Interrupt (DERR_GLOBAL, SERR_GLOBAL) GIC Config
	sample_ecc_setting_gic(ALT_INT_INTERRUPT_SERR_GLOBAL);
	sample_ecc_setting_gic(ALT_INT_INTERRUPT_DERR_GLOBAL);
#endif /* soc_cv_av */
	
	return;
}

/********************************************************************************//*!
 * @fn  void sample_ecc_setting_gic(ALT_INT_INTERRUPT_t int_id) 
 * @brief  Setting of GIC (ECC for On-Chip RAM Corredted/Uncorrected interrupt).
 * 
 * @details    In order to receive a ECC for On-Chip RAM Corredted/Uncorrected interrupt setting of GIC.
 * @param[in] int_id  ALT_INT_INTERRUPT_RAM_ECC_CORRECTED_IRQ or ALT_INT_INTERRUPT_RAM_ECC_UNCORRECTED_IRQ 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/20  ---  #13689  新規作成. 
 **//*******************************************************************************/
void sample_ecc_setting_gic(ALT_INT_INTERRUPT_t int_id)
{
	ALT_STATUS_CODE result_code;
	uint32_t ui32temp = 0;
	ALT_INT_TRIGGER_t trigger_type = ALT_INT_TRIGGER_NA;
	alt_int_callback_t callbackp;
	
	switch(int_id)
	{
#ifdef soc_cv_av
	case ALT_INT_INTERRUPT_RAM_ECC_CORRECTED_IRQ:
		printf("interrupt setting (GIC: ALT_INT_INTERRUPT_RAM_ECC_CORRECTED_IRQ)\n");
		callbackp = sample_ecc_corrected_callback;
		break;
	case ALT_INT_INTERRUPT_RAM_ECC_UNCORRECTED_IRQ:
		printf("interrupt setting (GIC: ALT_INT_INTERRUPT_RAM_ECC_UNCORRECTED_IRQ)\n");
		callbackp = sample_ecc_uncorrected_callback;
		break;
#else /* soc_cv_av */
	case ALT_INT_INTERRUPT_SERR_GLOBAL:
		printf("interrupt setting (GIC: ALT_INT_INTERRUPT_SERR_GLOBAL)\n");
		callbackp = sample_ecc_corrected_callback;
		break;
	case ALT_INT_INTERRUPT_DERR_GLOBAL:
		printf("interrupt setting (GIC: ALT_INT_INTERRUPT_DERR_GLOBAL)\n");
		callbackp = sample_ecc_uncorrected_callback;
		break;
#endif /* soc_cv_av */
	default:
		printf("ERROR!!: sample_ecc_setting_gic() 1st argument is invalid.\n");
		return;
	}
	
	/* Usage: ALT_STATUS_CODE alt_int_isr_register(ALT_INT_INTERRUPT_t int_id, alt_int_callback_t callback, void* context);	*/
	/*  typedef void(* alt_int_callback_t)(uint32_t icciar, void *context) */
	result_code = alt_int_isr_register(int_id, callbackp, NULL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_isr_register() result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_int_dist_target_set(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t target);	*/
	result_code = alt_int_dist_target_set(int_id, 1);		// arg2=targetCPU (Only Core#0)
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_target_set() result=%d\n", (int)result_code);
	}
	
	/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_set(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t trigger_type);	*/
	result_code = alt_int_dist_trigger_set(int_id, ALT_INT_TRIGGER_LEVEL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_trigger_set() result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_int_dist_priority_set(ALT_INT_INTERRUPT_t int_id, uint32_t priority);	*/
	result_code = alt_int_dist_priority_set(int_id, 99);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_priority_set() result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_int_dist_enable(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_enable(int_id);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable() result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_int_dist_secure_enable(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_secure_enable(int_id);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable() result=%d\n", (int)result_code);
	}

	// Print setting value 
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_secure(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_enabled(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_pending(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_active(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_priority_get(ALT_INT_INTERRUPT_t int_id, uint32_t* priority);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_target_get(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t* target);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_get(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t* trigger_type);	*/
	/****************************************************************************************/
	printf("- alt_int_dist_is_secure() --> result=%d\n", (int)alt_int_dist_is_secure(int_id));
	printf("- alt_int_dist_is_enabled() --> result=%d\n", (int)alt_int_dist_is_enabled(int_id));
	printf("- alt_int_dist_is_pending() --> result=%d\n", (int)alt_int_dist_is_pending(int_id));
	printf("- alt_int_dist_is_active() --> result=%d\n", (int)alt_int_dist_is_active(int_id));
	printf("- alt_int_dist_priority_get() --> result=%d out=0x%08X\n", (int)alt_int_dist_priority_get(int_id,&ui32temp), (int)ui32temp);
	printf("- alt_int_dist_target_get() --> result=%d out=0x%08X\n", (int)alt_int_dist_target_get(int_id,&ui32temp), (int)ui32temp);
	printf("- alt_int_dist_trigger_get() --> result=%d out=%d\n", (int)alt_int_dist_trigger_get(int_id,&trigger_type), (int)trigger_type);
	printf("\n");
	
	return;
}

/********************************************************************************//*!
 * @fn  void sample_ecc_corredted_callback(uint32_t icciar, void *context) 
 * @brief  On-Chip RAM ECC Corredted Interrupt(ALT_INT_INTERRUPT_RAM_ECC_CORRECTED_IRQ) callback function.
 * 
 * @details  working opportunity an interrupt from the ECC(for On-Chip RAM), indicate 
 *           an interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued. 
 * @param[in] context  The user provided context.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/20  ---  #13689  新規作成. 
 **//*******************************************************************************/
void sample_ecc_corrected_callback(uint32_t icciar, void *context)
#ifdef soc_cv_av
{
	ALT_STATUS_CODE result_code;
	uint32_t ecc_status;

	/* Usage: ALT_STATUS_CODE alt_int_dist_pending_clear(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_RAM_ECC_CORRECTED_IRQ);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_ecc_status_get(const ALT_ECC_RAM_ENUM_t ram_block, uint32_t* status); */
	result_code = alt_ecc_status_get(ALT_ECC_RAM_OCRAM, &ecc_status);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_ecc_status_get() result=%d\n", (int)result_code);
	}
	//printf("[INTERRUPT]On-Chip RAM ECC Corrected Interrupt is occured!! status=0x%08X\n", (int)ecc_status);
	util_intlog_record(ALT_INT_INTERRUPT_RAM_ECC_CORRECTED_IRQ, (int)icciar, (int)ecc_status);

	// Clear ECC Status 
	/* Usage: ALT_STATUS_CODE alt_ecc_status_clear(const ALT_ECC_RAM_ENUM_t ram_block, const uint32_t ecc_mask); */
	result_code = alt_ecc_status_clear(ALT_ECC_RAM_OCRAM, ecc_status);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_ecc_status_clear() result=%d\n", (int)result_code);
	}
	
	return;
}
#else /* soc_cv_av */
{
	ALT_STATUS_CODE result_code;
	uint32_t ecc_status;

	/* Usage: ALT_STATUS_CODE alt_int_dist_pending_clear(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_SERR_GLOBAL);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_ecc_status_get(const ALT_ECC_RAM_ENUM_t ram_block, uint32_t* status); */
	result_code = alt_ecc_status_get(ALT_ECC_RAM_OCRAM, &ecc_status);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_ecc_status_get() result=%d\n", (int)result_code);
	}
	//printf("[INTERRUPT]On-Chip RAM ECC Uncorrected Interrupt is occured!! status=0x%08X\n", (int)ecc_status);
	util_intlog_record(ALT_INT_INTERRUPT_SERR_GLOBAL, (int)icciar, (int)ecc_status);

	// Clear ECC Status
	/* Usage: ALT_STATUS_CODE alt_ecc_status_clear(const ALT_ECC_RAM_ENUM_t ram_block, const uint32_t ecc_mask); */
	result_code = alt_ecc_status_clear(ALT_ECC_RAM_OCRAM, ecc_status);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_ecc_status_clear() result=%d\n", (int)result_code);
	}

	return;
}
#endif /* soc_cv_av */

/********************************************************************************//*!
 * @fn  void sample_ecc_uncorredted_callback(uint32_t icciar, void *context) 
 * @brief  On-Chip RAM ECC Uncorredted Interrupt(ALT_INT_INTERRUPT_RAM_ECC_UNCORRECTED_IRQ) callback function.
 * 
 * @details  working opportunity an interrupt from the ECC(for On-Chip RAM), indicate 
 *           an interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued. 
 * @param[in] context  The user provided context.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/20  ---  #13689  新規作成. 
 **//*******************************************************************************/
void sample_ecc_uncorrected_callback(uint32_t icciar, void *context)
#ifdef soc_cv_av
{
	ALT_STATUS_CODE result_code;
	uint32_t ecc_status;

	/* Usage: ALT_STATUS_CODE alt_int_dist_pending_clear(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_RAM_ECC_UNCORRECTED_IRQ);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_ecc_status_get(const ALT_ECC_RAM_ENUM_t ram_block, uint32_t* status); */
	result_code = alt_ecc_status_get(ALT_ECC_RAM_OCRAM, &ecc_status);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_ecc_status_get() result=%d\n", (int)result_code);
	}
	//printf("[INTERRUPT]On-Chip RAM ECC Uncorrected Interrupt is occured!! status=0x%08X\n", (int)ecc_status);
	util_intlog_record(ALT_INT_INTERRUPT_RAM_ECC_UNCORRECTED_IRQ, (int)icciar, (int)ecc_status);

	// Clear ECC Status 
	/* Usage: ALT_STATUS_CODE alt_ecc_status_clear(const ALT_ECC_RAM_ENUM_t ram_block, const uint32_t ecc_mask); */
	result_code = alt_ecc_status_clear(ALT_ECC_RAM_OCRAM, ecc_status);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_ecc_status_clear() result=%d\n", (int)result_code);
	}
	
	return;
}
#else /* soc_cv_av */
{
	ALT_STATUS_CODE result_code;
	uint32_t ecc_status;

	/* Usage: ALT_STATUS_CODE alt_int_dist_pending_clear(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_DERR_GLOBAL);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_ecc_status_get(const ALT_ECC_RAM_ENUM_t ram_block, uint32_t* status); */
	result_code = alt_ecc_status_get(ALT_ECC_RAM_OCRAM, &ecc_status);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_ecc_status_get() result=%d\n", (int)result_code);
	}
	//printf("[INTERRUPT]On-Chip RAM ECC Uncorrected Interrupt is occured!! status=0x%08X\n", (int)ecc_status);
	util_intlog_record(ALT_INT_INTERRUPT_DERR_GLOBAL, (int)icciar, (int)ecc_status);

	// Clear ECC Status
	/* Usage: ALT_STATUS_CODE alt_ecc_status_clear(const ALT_ECC_RAM_ENUM_t ram_block, const uint32_t ecc_mask); */
	result_code = alt_ecc_status_clear(ALT_ECC_RAM_OCRAM, ecc_status);
	if(result_code != ALT_E_SUCCESS){
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_ecc_status_clear() result=%d\n", (int)result_code);
	}

	return;
}
#endif /* soc_cv_av */

/********************************************************************************//*!
 * @fn  void sample_ecc_test_init(void) 
 * @brief  Initializing the sample program. 
 * 
 * @details  Initialize and Configure the Cache, MMU, and ECC. 
 *           Other, do the following.
 *           -# Setting for the Address Space Remap Register. 
 *           -# Setting for the L2 Cache Address Filter. 
 *           -# Call Initialize ECC Sample Program sub-function.
 *           -# Call MMU setting sub-function.
 *           -# Configure the Helio Board GPIO settings. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/20  ---  #13689  新規作成. 
 **//*******************************************************************************/
void sample_ecc_test_init(void)
{
#if 0 /* cpu0_init に実装済み */
	ALT_STATUS_CODE result_code;
	size_t trans_table_bytes;
	// Remap Setting Value
	ALT_ADDR_SPACE_MPU_ATTR_t			mpu_attr = ALT_ADDR_SPACE_MPU_ZERO_AT_OCRAM;
	ALT_ADDR_SPACE_NONMPU_ATTR_t		nonmpu_attr = ALT_ADDR_SPACE_NONMPU_ZERO_AT_OCRAM;
	ALT_ADDR_SPACE_H2F_BRIDGE_ATTR_t	h2f_attr = ALT_ADDR_SPACE_H2F_ACCESSIBLE;
	ALT_ADDR_SPACE_LWH2F_BRIDGE_ATTR_t	lwh2f_attr = ALT_ADDR_SPACE_LWH2F_ACCESSIBLE;

	
	/* bridge interface to initialize. */
	//alt_bridge_init(ALT_BRIDGE_H2F, NULL, NULL);
	//alt_bridge_init(ALT_BRIDGE_LWH2F, NULL, NULL);
	/* Setting for the Address Space Remap Register. */
	result_code = alt_addr_space_remap(mpu_attr, nonmpu_attr, h2f_attr, lwh2f_attr);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_addr_space_remap() .... result=%d\n", (int)result_code);
	}
	/* Setting for the L2 Cache Address Filter. */
	result_code = alt_l2_addr_filter_cfg_set(0x00010000, 0x40000000);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_l2_addr_filter_cfg_set(0x00010000, 0x40000000) .... result=%d\n", (int)result_code);
	}
#endif /* cpu0_init に実装済み */

#if 0 // main() 内で cpu0_init() 実施済み
	cpu0_init();
#endif

#if 0 /* cpu0_init に実装済み */
	printf("L1/L2 Cache all Disable!!!! \n");
	result_code = alt_cache_system_disable();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_cache_system_disable() result=%d\n", (int)result_code);
	}
#endif /* cpu0_init に実装済み */
	
	/* Initialize ECC Sample Program. */
	printf("ECC Enable & Initialize!!!! \n");
	sample_ecc_init();

#if 0 /* cpu0_init に実装済み */
	/* Usage:	ALT_STATUS_CODE  alt_mmu_init (void);	*/
	result_code = alt_mmu_init();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_mmu_init() .... result=%d\n", (int)result_code);
	}
	
	/* Usage:	size_t  alt_mmu_va_space_storage_required (const ALT_MMU_MEM_REGION_t *mem_regions, const size_t num_mem_regions);	*/
	/************************************************************/
	/* target discriptor : ALT_MMU_MEM_REGION_t MemoryRegion[] 	*/
	/* 						MemoryRegion[0] .. ON-CHIP RAM		*/
	/* 						MemoryRegion[1] .. SDRAM1			*/
	/* 						MemoryRegion[2] .. SDRAM2			*/
	/* 						MemoryRegion[3] .. SDRAM3 (include .text)	*/
	/* 						MemoryRegion[4] .. HPS2FPGA			*/
	/* 						MemoryRegion[5] .. LW_HPS2FPGA		*/
	/* 						MemoryRegion[6] .. PERIPHRALS		*/
	/* 						MemoryRegion[7] .. BOOTROM			*/
	/* 						MemoryRegion[8] .. MPU_SCU			*/
	/* 						MemoryRegion[9] .. MPU_L2			*/
	/* 						MemoryRegion[10] . ON-CHIP RAM		*/
	/************************************************************/
	trans_table_bytes = alt_mmu_va_space_storage_required(MemoryRegion, ValidMemoryRegion);
	printf("alt_mmu_va_space_storage_required() ... result=0x%08X(%d) bytes\n", 
		(int)trans_table_bytes, (int)trans_table_bytes);
	
	printf("MMU Virtual Address Setting & Enable.\n");
	sample_mmu_init();
	sample_mmu_print_registers();
	
	printf("MMU Enable.\n");
	result_code = alt_mmu_enable();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_mmu_enable() result=%d\n", (int)result_code);
	}
#endif /* cpu0_init に実装済み */
	
	return;
}

/********************************************************************************//*!
 * @fn  void sample_ecc_test_uninit(void) 
 * @brief  Uninitializing the sample program. 
 * 
 * @details  Call the uninitialization function for ECC Management API. 
 *           Do nothing.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/20  ---  #13689  新規作成. 
 **//*******************************************************************************/
void sample_ecc_test_uninit(void)
{
	return;
}

/********************************************************************************//*!
 * @fn  void sample_ecc_test_print_usage(void) 
 * @brief  Display usage of the sample program. 
 * 
 * @details  Following information displays on the console.
 *            - Overview of HPS for push button functions (or operations)
 *            - Overview of HPS for DIP switch functions (or operations)
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/20  ---  #13689  新規作成. 
 **//*******************************************************************************/
void sample_ecc_test_print_usage(void)
{
	printf("\n");
	printf("+--- Usage ( Function of PUSH SW and SLIDE SW )  ---------------+\n");
	printf("+ PUSH SW #0 .... Exit Test loop!!!\n");
	printf("+ PUSH SW #1 .... Cache Clean / Test for ECC Error Detection.\n");
	printf("+ PUSH SW #2 .... Cache Purge / Test for ECC Error Detection.\n");
	printf("+ PUSH SW #3 .... Memory Check (Test for ECC Error Injection and Detection)\n");
	printf("+ SLIDE SW#0 .... ECC Double bit Error Injection ON/OFF\n");
	printf("+ SLIDE SW#1 .... ECC Single bit Error Injection ON/OFF\n");
	printf("+ SLIDE SW#2 .... Debug Mode ON/OFF (ON .. force write-through, linefill disable mode)\n");
	printf("+ SLIDE SW#3 .... L1/L2 Cache ON/OFF\n");
	printf("+---------------------------------------------------------------+\n\n");
	
#if 0
ループ中にHPS用PUSHSWの操作を検出した場合、それぞれ以下の処理を実行します。
	- HPS用PUSHSW0 ... ループを抜けてプログラム終了します。(uninit処理を実行)
	- HPS用PUSHSW1 ... Cache Cleanを行った上でテスト領域へのReadアクセスを行います。(ECC割り込みの発生確認) 
	- HPS用PUSHSW2 ... Cache Purge(Clean&Invalidate)を行った上でテスト領域へのReadアクセスを行います。(ECC割り込みの発生確認) 
	- HPS用PUSHSW3 ... DIPSW1,2の設定に応じてECCエラーのインジェクションおよびメモリチェックを行います。
HPS用DIPSWは以下の動作選択に利用します。
	- HPS用DIPSW1 .... ECC Double Bit Error(uncorredtable)インジェクション設定 {OFF:無効／ON:有効}
	- HPS用DIPSW2 .... ECC Single Bit Error(correctable)インジェクション設定 {OFF:無効／ON:有効}
	- HPS用DIPSW3 .... L2C-310 Debug Mode設定 {OFF:無効／ON:有効}
	　　（ON:有効にするとCacheが強制 Write-Through かつ linefill無効 で動作するモードに入ります。）
	- HPS用DIPSW4 .... L1/L2 Cache設定 {OFF:無効／ON:有効}
#endif
	
	return;
}

/********************************************************************************//*!
 * @fn  void sample_ecc_test_main(void) 
 * @brief  Main routine of the sample program. 
 * 
 * @details  Infinite loop structure works. Make the detection of various types 
 *           of switches in the loop, it performs in various testing processes.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/20  ---  #13689  新規作成. 
 **//*******************************************************************************/
void sample_ecc_test_main(void)
{
	ALT_STATUS_CODE result_code;
#if 99-99
	bool system_cache_en = false;
#else
	bool system_cache_en = true;	// main() 内で cpu0_init() により alt_cache_system_enable() 実施済み
#endif
	uint32_t switch_raw = 0;
	uint32_t switch_raw_bk = 0;
	uint32_t switch_detect = 0;
	uint32_t switch_detect_on = 0;
	uint32_t switch_number = 0;
	bool disp_usage = true;

	util_intlog_print();

	/* ################################################ */
	/* ## Implement the test setting process here!!! ## */
	/* ################################################ */
	
	printf("==== Start While(1) loop process!!! ============\n");
	switch_raw_bk = sample_detect_switch();
	switch_raw_bk ^= SAMPLE_SWITCH_BIT_SLIDE2;
	switch_raw_bk ^= SAMPLE_SWITCH_BIT_SLIDE3;
	while(1)
	{
		if(disp_usage)
		{
			sample_ecc_test_print_usage();
			disp_usage = false;
		}

		// ---- Check the Slide-Switch and Push-Switch. ---- 
		switch_raw = sample_detect_switch();
		switch_detect = switch_raw ^ switch_raw_bk;
		switch_detect_on  |= switch_detect & switch_raw;
		
		// Slide-Switch 
		if(switch_detect & SAMPLE_SWITCH_BIT_SLIDEALL)
		{
			switch_number = switch_raw & SAMPLE_SWITCH_BIT_NUM;
			switch_number *= 1;		// To avoid warnings.

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE0){
				//printf("SAMPLE_SWITCH_BIT_SLIDE0\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE1){
				//printf("SAMPLE_SWITCH_BIT_SLIDE1\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE2){
				//printf("SAMPLE_SWITCH_BIT_SLIDE2\n");
				/* Setting L2 Cache Debug Register */
				if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE2){
					/* bit#1: DWB: Disable write-back, force WT = 1 (Force write-through behavior.) */
					/* bit#0: DCL: Disable cache linefill = 1 (Disable cache linefills.) */
					printf("Enter L2C Debug Mode (Force WT behavior, Disable cache linefills).\n");
					alt_write_word(ALT_MPUL2_DEBUG_CTRL_ADDR, (alt_read_word(ALT_MPUL2_DEBUG_CTRL_ADDR)|0x3));
				} else {
					/* bit#1: DWB: Disable write-back, force WT = 0 (Enable write-back behavior. This is the default.) */
					/* bit#0: DCL: Disable cache linefill = 0 (Enable cache linefills. This is the default.) */
					printf("Exit L2C Debug Mode (Enable WB behavior, Enable cache linefills).\n");
					alt_write_word(ALT_MPUL2_DEBUG_CTRL_ADDR, (alt_read_word(ALT_MPUL2_DEBUG_CTRL_ADDR)&~0x3));
				}
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE3){
				//printf("SAMPLE_SWITCH_BIT_SLIDE3\n");
				/* Setting L1/L2 Cache */
				if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE3){
					if(system_cache_en != true){
						printf("L1/L2 Cache all Enable.\n");
						result_code = alt_cache_system_enable();
						if(result_code != ALT_E_SUCCESS){
							printf("ERROR!!: alt_cache_system_enable() result=%d\n", (int)result_code);
						}
					}
					system_cache_en = true;
				} else {
					if(system_cache_en != false){
						printf("L1/L2 Cache all Disable.\n");
						result_code = alt_cache_system_disable();
						if(result_code != ALT_E_SUCCESS){
							printf("ERROR!!: alt_cache_system_disable() result=%d\n", (int)result_code);
						}
					}
					system_cache_en = false;
				}
			}
			disp_usage = false;
		}
		
		
		// Push-Switch 0
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH0)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH0;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH0\n");
			break;	// Exit Push-Switch 0 and 1 becomes ON.
		}
		// Push-Switch 1
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH1)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH1;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH1\n");
			if(system_cache_en == true){
				if((result_code = alt_cache_system_clean((void*)OCRAM0_PHYSICAL_ADDR, 0x80)) != ALT_E_SUCCESS)	printf("ERROR!!: ");
				printf("alt_cache_system_clean(OCRAM0_PHYSICAL_ADDR, 0x80) result=%d\n", (int)result_code);
			}
			/* ECC Check: Memory Read Access */
			sample_memread_for_ecccheck((void*)OCRAM0_PHYSICAL_ADDR, 0x80);

			if(system_cache_en == true){
				if((result_code = alt_cache_system_clean((void*)OCRAMF_PHYSICAL_ADDR, 0x80)) != ALT_E_SUCCESS)	printf("ERROR!!: ");
				printf("alt_cache_system_clean(OCRAMF_PHYSICAL_ADDR, 0x80) result=%d\n", (int)result_code);
			}
			/* ECC Check: Memory Read Access */
			sample_memread_for_ecccheck((void*)OCRAMF_PHYSICAL_ADDR, 0x80);
			disp_usage = true;
		}
		// Push-Switch 2
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH2)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH2;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH2\n");
			if(system_cache_en == true){
				if((result_code = alt_cache_system_purge((void*)OCRAM0_PHYSICAL_ADDR, 0x80)) != ALT_E_SUCCESS)	printf("ERROR!!: ");
				printf("alt_cache_system_purge(OCRAM0_PHYSICAL_ADDR, 0x80) result=%d\n", (int)result_code);
			}
			/* ECC Check: Memory Read Access */
			sample_memread_for_ecccheck((void*)OCRAM0_PHYSICAL_ADDR, 0x80);

			if(system_cache_en == true){
				if((result_code = alt_cache_system_purge((void*)OCRAMF_PHYSICAL_ADDR, 0x80)) != ALT_E_SUCCESS)	printf("ERROR!!: ");
				printf("alt_cache_system_purge(OCRAMF_PHYSICAL_ADDR, 0x80) result=%d\n", (int)result_code);
			}
			/* ECC Check: Memory Read Access */
			sample_memread_for_ecccheck((void*)OCRAMF_PHYSICAL_ADDR, 0x80);
			disp_usage = true;
		}
		// Push-Switch 3
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH3)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH3;

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH3\n");
			printf("Start on-chip Memory Check\n");
			sample_memchk_by_memset((uint32_t*)(OCRAM0_PHYSICAL_ADDR + 0x00), 16);
			sample_memchk_by_memset((uint32_t*)(OCRAMF_PHYSICAL_ADDR + 0x10), 16);
			sample_memchk_by_memset(&testbuf_on_onchipram[0x20/sizeof(uint32_t)], 16);
			printf("\n");
			sample_memchk_by_byte((uint8_t*)(OCRAM0_PHYSICAL_ADDR + 0x30), 4);
			sample_memchk_by_byte((uint8_t*)(OCRAMF_PHYSICAL_ADDR + 0x34), 4);
			sample_memchk_by_byte((uint8_t*)&testbuf_on_onchipram[0x38/sizeof(uint32_t)], 4);
			printf("\n");
			sample_memchk_by_halfword((uint16_t*)(OCRAM0_PHYSICAL_ADDR + 0x40), 4);
			sample_memchk_by_halfword((uint16_t*)(OCRAMF_PHYSICAL_ADDR + 0x44), 4);
			sample_memchk_by_halfword((uint16_t*)&testbuf_on_onchipram[0x48/sizeof(uint32_t)], 4);
			printf("\n");
			sample_memchk_by_word((uint32_t*)(OCRAM0_PHYSICAL_ADDR + 0x50), 4);
			sample_memchk_by_word((uint32_t*)(OCRAMF_PHYSICAL_ADDR + 0x54), 4);
			sample_memchk_by_word(&testbuf_on_onchipram[0x58/sizeof(uint32_t)], 4);
			printf("\n");
			sample_memchk_by_wordword((uint64_t*)(OCRAM0_PHYSICAL_ADDR + 0x60), 8);
			sample_memchk_by_wordword((uint64_t*)(OCRAMF_PHYSICAL_ADDR + 0x68), 8);
			sample_memchk_by_wordword((uint64_t*)&testbuf_on_onchipram[0x70/sizeof(uint32_t)], 8);
			printf("\nFinish!!! on-chip Memory Check\n");
			disp_usage = true;
		}
		
		util_intlog_print();

		switch_raw_bk = switch_raw;
	}
	printf("==== End While(1) loop process. ================\n");

	util_intlog_print();

	return;
}

// All APIs to be tested in this sample.
/***********************************************************************/
/* Usage: ALT_STATUS_CODE alt_ecc_start(const ALT_ECC_RAM_ENUM_t ram_block); */
/* Usage: ALT_STATUS_CODE alt_ecc_stop(const ALT_ECC_RAM_ENUM_t ram_block); */
/* Usage: ALT_STATUS_CODE alt_ecc_is_enabled(const ALT_ECC_RAM_ENUM_t ram_block); */
/* Usage: ALT_STATUS_CODE alt_ecc_status_get(const ALT_ECC_RAM_ENUM_t ram_block, uint32_t *status); */
/* Usage: ALT_STATUS_CODE alt_ecc_status_clear(const ALT_ECC_RAM_ENUM_t ram_block, const uint32_t ecc_mask); */
/* Usage: ALT_STATUS_CODE alt_ecc_serr_inject(const ALT_ECC_RAM_ENUM_t ram_block); */
/* Usage: ALT_STATUS_CODE alt_ecc_derr_inject(const ALT_ECC_RAM_ENUM_t ram_block); */
/***********************************************************************/

////////////////////////////// Common Functions //////////////////////////////////////// 

///////////////////// Test Functions /////////////////////////////////
void sample_memchk_by_byte(volatile uint8_t* start_address, int bytes)
{
	int i;
	volatile uint32_t switch_raw = 0;

	printf("\nsample_memchk_by_byte() START! ----> adr=0x%08X size=0x%08X\n", (int)start_address, bytes);
	for(i=0; i<bytes; i+=sizeof(uint8_t)){
		switch_raw = sample_detect_switch();

#ifdef soc_cv_av
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM);
#else /* soc_cv_av */
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
#endif /* soc_cv_av */

		*start_address = (uint8_t)0x55;
		if(*start_address != (uint8_t)0x55)	printf("MEMCHK NG! adr=0x%08X set=0x55 ref=0x%02X\n", (int)start_address, (int)*start_address);

#ifdef soc_cv_av
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM);
#else /* soc_cv_av */
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
#endif /* soc_cv_av */

		*start_address = (uint8_t)0xAA;
		if(*start_address != (uint8_t)0xAA)	printf("MEMCHK NG! adr=0x%08X set=0xAA ref=0x%02X\n", (int)start_address, (int)*start_address);

		start_address++;
	}
	printf("\n----> sample_memchk_by_byte() FINISH!\n\n");
	return;
}

void sample_memchk_by_halfword(volatile uint16_t* start_address, int bytes)
{
	int i;
	volatile uint32_t switch_raw = 0;

	printf("\nsample_memchk_by_halfword() START! ----> adr=0x%08X size=0x%08X\n", (int)start_address, bytes);
	for(i=0; i<bytes; i+=sizeof(uint16_t)){
		switch_raw = sample_detect_switch();

#ifdef soc_cv_av
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM);
#else /* soc_cv_av */
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
#endif /* soc_cv_av */

		*start_address = (uint16_t)0x5555;
		if(*start_address != (uint16_t)0x5555)	printf("MEMCHK NG! adr=0x%08X set=0x5555 ref=0x%04X\n", (int)start_address, (int)*start_address);

#ifdef soc_cv_av
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM);
#else /* soc_cv_av */
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
#endif /* soc_cv_av */

		*start_address = (uint16_t)0xAAAA;
		if(*start_address != (uint16_t)0xAAAA)	printf("MEMCHK NG! adr=0x%08X set=0xAAAA ref=0x%04X\n", (int)start_address, (int)*start_address);

		start_address++;
	}
	printf("\n----> sample_memchk_by_halfword() FINISH!\n\n");
	return;
}

void sample_memchk_by_word(volatile uint32_t* start_address, int bytes)
{
	int i;
	volatile uint32_t switch_raw = 0;

	printf("\nsample_memchk_by_word() START! ----> adr=0x%08X size=0x%08X\n", (int)start_address, bytes);
	for(i=0; i<bytes; i+=sizeof(uint32_t)){
		switch_raw = sample_detect_switch();

#ifdef soc_cv_av
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM);
#else /* soc_cv_av */
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
#endif /* soc_cv_av */

		*start_address = (uint32_t)0x55555555;
		if(*start_address != (uint32_t)0x55555555)	printf("MEMCHK NG! adr=0x%08X set=0x55555555 ref=0x%08X\n", (int)start_address, (int)*start_address);

#ifdef soc_cv_av
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM);
#else /* soc_cv_av */
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
#endif /* soc_cv_av */

		*start_address = (uint32_t)0xAAAAAAAA;
		if(*start_address != (uint32_t)0xAAAAAAAA)	printf("MEMCHK NG! adr=0x%08X set=0xAAAAAAAA ref=0x%08X\n", (int)start_address, (int)*start_address);

		start_address++;
	}
	printf("\n----> sample_memchk_by_word() FINISH!\n\n");
	return;
}

void sample_memchk_by_wordword(volatile uint64_t* start_address, int bytes)
{
	int i;
	volatile uint32_t switch_raw = 0;

	printf("\nsample_memchk_by_wordword() START! ----> adr=0x%08X size=0x%08X\n", (int)start_address, bytes);
	for(i=0; i<bytes; i+=sizeof(uint64_t)){
		switch_raw = sample_detect_switch();

#ifdef soc_cv_av
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM);
#else /* soc_cv_av */
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
#endif /* soc_cv_av */

		*start_address = (uint64_t)0x5555555555555555UL;
		if(*start_address != (uint64_t)0x5555555555555555UL)	printf("MEMCHK NG! adr=0x%08X set=0x5555555555555555 ref=0x%016llX\n", (int)start_address, (long long int)*start_address);

#ifdef soc_cv_av
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM);
#else /* soc_cv_av */
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
#endif /* soc_cv_av */

		*start_address = (uint64_t)0xAAAAAAAAAAAAAAAAUL;
		if(*start_address != (uint64_t)0xAAAAAAAAAAAAAAAAUL)	printf("MEMCHK NG! adr=0x%08X set=0xAAAAAAAAAAAAAAAA ref=0x%016llX\n", (int)start_address, (long long int)*start_address);

		start_address++;
	}
	printf("\n----> sample_memchk_by_wordword() FINISH!\n\n");
	return;
}

void sample_memchk_by_memset(volatile uint32_t* start_address, int bytes)
{
	int i;
	volatile uint8_t* memchk_p;
	volatile uint32_t switch_raw = 0;

	printf("\nsample_memchk_by_memset() START! ----> adr=0x%08X size=0x%08X\n", (int)start_address, bytes);
	{
		switch_raw = sample_detect_switch();

#ifdef soc_cv_av
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM);
#else /* soc_cv_av */
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
#endif /* soc_cv_av */

		memset((void*)start_address, 0x55, bytes);
		memchk_p = (uint8_t*)start_address;
		for(i=0; i<bytes; i++, memchk_p++){
			if(*memchk_p != 0x55)	printf("MEMCHK NG! adr=0x%08X set=0x55 ref=0x%02X\n", (int)memchk_p, (int)*memchk_p);
		}

#ifdef soc_cv_av
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM);
#else /* soc_cv_av */
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0)	(void)alt_ecc_derr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
		if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1)	(void)alt_ecc_serr_inject(ALT_ECC_RAM_OCRAM, (uint32_t)start_address, 0);
#endif /* soc_cv_av */

		memset((void*)start_address, 0xAA, bytes);
		memchk_p = (uint8_t*)start_address;
		for(i=0; i<bytes; i++, memchk_p++){
			if(*memchk_p != 0xAA)	printf("MEMCHK NG! adr=0x%08X set=0xAA ref=0x%02X\n", (int)memchk_p, (int)*memchk_p);
		}
	}
	printf("\n----> sample_memchk_by_memset() FINISH!\n\n");
	return;
}

/* ECC Check: Memory Read Access */
void sample_memread_for_ecccheck(void* start_address, int bytes)
{
	volatile uint8_t* u8srcp;
	volatile uint16_t* u16srcp;
	volatile uint32_t* u32srcp;
	volatile uint64_t* u64srcp;
	volatile uint8_t u8dst __WARN_UNUSED__;
	volatile uint16_t u16dst __WARN_UNUSED__;
	volatile uint32_t u32dst __WARN_UNUSED__;
	volatile uint64_t u64dst __WARN_UNUSED__;

	printf("\nsample_memread_for_ecccheck() by byte! ----> adr=0x%08X size=0x%08X\n", (int)start_address, bytes);
//	for(u8srcp = (uint8_t*)start_address; u8srcp < (uint8_t*)(start_address + bytes); u8srcp++){
	for(u8srcp = (uint8_t*)start_address; u8srcp < (uint8_t*)((int)start_address + bytes); u8srcp++){
		printf("Read by byte: 0x%08X\n", (int)u8srcp);
		fflush(stdout);
		u8dst = *u8srcp;
	}
	printf("\nsample_memread_for_ecccheck() by halfword! ----> adr=0x%08X size=0x%08X\n", (int)start_address, bytes);
//	for(u16srcp = (uint16_t*)start_address; u16srcp < (uint16_t*)(start_address + bytes); u16srcp++){
	for(u16srcp = (uint16_t*)start_address; u16srcp < (uint16_t*)((int)start_address + bytes); u16srcp++){
		printf("Read by halfword: 0x%08X\n", (int)u16srcp);
		fflush(stdout);
		u16dst = *u16srcp;
	}
	printf("\nsample_memread_for_ecccheck() by word! ----> adr=0x%08X size=0x%08X\n", (int)start_address, bytes);
//	for(u32srcp = (uint32_t*)start_address; u32srcp < (uint32_t*)(start_address + bytes); u32srcp++){
	for(u32srcp = (uint32_t*)start_address; u32srcp < (uint32_t*)((int)start_address + bytes); u32srcp++){
		printf("Read by word: 0x%08X\n", (int)u32srcp);
		fflush(stdout);
		u32dst = *u32srcp;
	}
	printf("\nsample_memread_for_ecccheck() by wordword! ----> adr=0x%08X size=0x%08X\n", (int)start_address, bytes);
//	for(u64srcp = (uint64_t*)start_address; u64srcp < (uint64_t*)(start_address + bytes); u64srcp++){
	for(u64srcp = (uint64_t*)start_address; u64srcp < (uint64_t*)((int)start_address + bytes); u64srcp++){
		printf("Read by wordword: 0x%08X\n", (int)u64srcp);
		fflush(stdout);
		u64dst = *u64srcp;
	}
	printf("\n----> sample_memread_for_ecccheck() FINISH!\n\n");
	return;
}


/************************************/
/*	Test Command (ECC OnchipRAM)	*/
/************************************/
int sample_ecc_test_cmd(char* options)
{
    ALT_STATUS_CODE status = ALT_E_SUCCESS;
    
    printf("\r\nUser Application Start!\r\n");

    //
    // Start the interrupt system
    //
    if (status == ALT_E_SUCCESS)
    {
    	util_intlog_init();
        status = socfpga_int_start();
    }

    // Printing Current PLL Setting. And Set Global Timer for Time Measurement.
    util_time_init();

    //// Sample Function Call!!! |---->
    printf(">>>> Execute sample_ecc_test_init(); !!! <<<<\r\n");
    sample_ecc_test_init();

    printf(">>>> Execute sample_ecc_test_main(); !!! <<<<\r\n");
    sample_ecc_test_main();

    printf(">>>> Execute sample_ecc_test_uninit(); !!! <<<<\r\n");
    sample_ecc_test_uninit();
    //// Sample Function Call!!! <----|

    // Printing All Results of Time Measurement. And Uninit Global Timer.
    util_time_uninit();

    //
    // Stop the interrupt system
    //
    socfpga_int_stop();
	util_intlog_print();

    printf("Finished running the sample !!!\r\n");
    if (status == ALT_E_SUCCESS)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}
/***********************************************************************************
 * end of file 
 ***********************************************************************************/
