﻿/********************************************************************************//*!
 * @file  sample_gptmr.c 
 * @brief  To use HWLib for the General-Purpose Timer example program. 
 * 
 * @details  Run the API category below.
 *           - The General Purpose Timer Manager API  
 *               - Enable, Disable, and Status  
 *               - Counters Interface  
 *               - Interrupts  
 *               - Mode Control 
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 * @date 2014/08/18  ---  -----  v14.0化 (誤植訂正、操作説明の表示を追加). 
 * @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
 * @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
 * @date 2018/07/02  ---  #25940  All in One ベアメタル・アプリケーション v17.1 への対応を実施.
 * @date 2019/01/18  ---  -----   All in One ベアメタル・アプリケーション v18.1 への対応を実施.
 * @date 2025/06/30  ---  -----   All in One ベアメタル・アプリケーション v24.1 への対応を実施. "alt_printf.h" のインク―ルド追加.
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes 
 ***********************************************************************************/
#include <stdio.h>
#include <string.h>
#include "hwlib.h"
#include "socal/socal.h"
#include "socal/hps.h"
#include "alt_generalpurpose_io.h"
#include "alt_globaltmr.h"
#include "alt_clock_manager.h"
#include "alt_interrupt.h"
#include "alt_timers.h"
#include "alt_watchdog.h"

#include "sample_app_setting.h"
#include "util_time_measurement.h"
#include "util_interrupt_log.h"
#include "alt_printf.h"

/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/
static void sample_gptmr_print_specific_counter(ALT_GPT_TIMER_t tmr_id);
static void sample_gptmr_print_all_counter(void);
static void sample_gptmr_print_all_counter2(void);
static void sample_gptmr_print_specific_settings_status(ALT_GPT_TIMER_t tmr_id);
static void sample_gptmr_print_specific_settings(ALT_GPT_TIMER_t tmr_id);
static void sample_gptmr_print_all_settings(void);
static void sample_gptmr_print_all_settings2(void);
static void sample_gptmr_print_all_settings3(void);
static void sample_gptmr_start_all(void);
static void sample_gptmr_stop_all(void);
static void sample_gptmr_reset_all(void);
static void sample_gptmr_ienable_all(void);
static void sample_gptmr_idisable_all(void);
static void sample_gptmr_print_setting_gic(ALT_INT_INTERRUPT_t int_id);
static void sample_gptmr_print_all_int_settings(void);
void sample_gptmr_test_init(void);
void sample_gptmr_test_uninit(void);
void sample_gptmr_setting_gic(void);
void sample_gptmr_callback_OSC1_TMR0(uint32_t icciar, void *context);
void sample_gptmr_callback_OSC1_TMR1(uint32_t icciar, void *context);
void sample_gptmr_callback_SP_TMR0(uint32_t icciar, void *context);
void sample_gptmr_callback_SP_TMR1(uint32_t icciar, void *context);
static void sample_gptmr_changemode_and_reset(ALT_GPT_TIMER_t tmr_id, ALT_GPT_RESTART_MODE_t mode);
void sample_gptmr_test_print_usage(void);
void sample_gptmr_test_main(void);
int  sample_gptmr_test_cmd(char* options);
// prototype (Common Functions) //

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/

/***********************************************************************************
 *  variables 
 ***********************************************************************************/

/***********************************************************************************
 *  functions 
 ***********************************************************************************/

/********************************************************************************//*!
 * @fn  static void sample_gptmr_print_specific_counter(ALT_GPT_TIMER_t tmr_id) 
 * @brief  Display the value of specific general-purpose timer count registers. 
 * 
 * @details  Displays the register value that you can get with the following APIs.
 *           - alt_gpt_counter_get(). 
 *           - alt_gpt_reset_value_get(). 
 *           - alt_gpt_maxcounter_get(). 
 *           - alt_gpt_curtime_millisecs_get(). 
 *           - alt_gpt_time_millisecs_get(). 
 *           - alt_gpt_maxtime_millisecs_get(). 
 * @param[in] tmr_id  general-purpose timer identification types. 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_print_specific_counter(ALT_GPT_TIMER_t tmr_id)
{
	printf("CounterValue[cur/ini/max]= 0x%08X / 0x%08X / 0x%08X\n",
			(int)alt_gpt_counter_get(tmr_id),
			(int)alt_gpt_reset_value_get(tmr_id),
			(int)alt_gpt_maxcounter_get(tmr_id));
	printf("MilliSeconds[cur/ini/max]= %d / %d /%d\n",
			(int)alt_gpt_curtime_millisecs_get(tmr_id),
			(int)alt_gpt_time_millisecs_get(tmr_id),
			(int)alt_gpt_maxtime_millisecs_get(tmr_id));

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_print_all_counter(void) 
 * @brief  Display the value of all timer counters. 
 * 
 * @details  Call sample_gptmr_print_specific_counter for all timers (including 
 *           the WDT, Global Timer).
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_print_all_counter(void)
{
	printf("<< ALT_GPT_CPU_GLOBAL_TMR >>\n");
	sample_gptmr_print_specific_counter(ALT_GPT_CPU_GLOBAL_TMR);
	printf("\n");

	printf("<< ALT_GPT_CPU_PRIVATE_TMR >>\n");
	sample_gptmr_print_specific_counter(ALT_GPT_CPU_PRIVATE_TMR);
	printf("\n");

	printf("<< ALT_GPT_CPU_WDTGPT_TMR >>\n");
	sample_gptmr_print_specific_counter(ALT_GPT_CPU_WDTGPT_TMR);
	printf("\n");

	printf("<< ALT_GPT_OSC1_TMR0 >>\n");
	sample_gptmr_print_specific_counter(ALT_GPT_OSC1_TMR0);
	printf("\n");

	printf("<< ALT_GPT_OSC1_TMR1 >>\n");
	sample_gptmr_print_specific_counter(ALT_GPT_OSC1_TMR1);
	printf("\n");

	printf("<< ALT_GPT_SP_TMR0 >>\n");
	sample_gptmr_print_specific_counter(ALT_GPT_SP_TMR0);
	printf("\n");

	printf("<< ALT_GPT_SP_TMR1 >>\n");
	sample_gptmr_print_specific_counter(ALT_GPT_SP_TMR1);
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_print_all_counter2(void) 
 * @brief  Display the value of all general-purpose timer counters. 
 * 
 * @details  Call sample_gptmr_print_specific_counter for the general-purpose 
 *           timers of all (except for the WDT, Global Timer).
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_print_all_counter2(void)
{
	printf("<< ALT_GPT_OSC1_TMR0 >>\n");
	sample_gptmr_print_specific_counter(ALT_GPT_OSC1_TMR0);
	printf("\n");

	printf("<< ALT_GPT_OSC1_TMR1 >>\n");
	sample_gptmr_print_specific_counter(ALT_GPT_OSC1_TMR1);
	printf("\n");

	printf("<< ALT_GPT_SP_TMR0 >>\n");
	sample_gptmr_print_specific_counter(ALT_GPT_SP_TMR0);
	printf("\n");

	printf("<< ALT_GPT_SP_TMR1 >>\n");
	sample_gptmr_print_specific_counter(ALT_GPT_SP_TMR1);
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_print_specific_settings_status(ALT_GPT_TIMER_t tmr_id) 
 * @brief  Display the value of specific general-purpose timer setting status. 
 * 
 * @details  Displays timer setting status that you can get with the following APIs.
 *           - alt_gpt_mode_get(). 
 *           - alt_gpt_tmr_is_running(). 
 *           - alt_gpt_int_is_enabled(). 
 *           - alt_gpt_int_is_pending(). 
 * @param[in] tmr_id  general-purpose timer identification types. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_print_specific_settings_status(ALT_GPT_TIMER_t tmr_id)
{
	/* Usage:  int32_t alt_gpt_mode_get(ALT_GPT_TIMER_t tmr_id);	*/
	printf("alt_gpt_mode_get() ............... result=%d\n", (int)alt_gpt_mode_get(tmr_id));
	/* Usage:  ALT_STATUS_CODE alt_gpt_tmr_is_running(ALT_GPT_TIMER_t tmr_id);	*/
	printf("alt_gpt_tmr_is_running() ......... result=%d\n", (int)alt_gpt_tmr_is_running(tmr_id));
	/* Usage:  bool alt_gpt_int_is_enabled(ALT_GPT_TIMER_t tmr_id);	*/
	printf("alt_gpt_int_is_enabled() ......... result=%d\n", (int)alt_gpt_int_is_enabled(tmr_id));
	/* Usage:  ALT_STATUS_CODE alt_gpt_int_is_pending(ALT_GPT_TIMER_t tmr_id);	*/
	printf("alt_gpt_int_is_pending() ......... result=%d\n", (int)alt_gpt_int_is_pending(tmr_id));

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_print_specific_settings(ALT_GPT_TIMER_t tmr_id) 
 * @brief  Display the value of specific general-purpose timer settings. 
 * 
 * @details  Displays timer settings that you can get with the various APIs.
 * @param[in] tmr_id  general-purpose timer identification types. 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_print_specific_settings(ALT_GPT_TIMER_t tmr_id)
{
	uint32_t	ui32val;

	sample_gptmr_print_specific_settings_status(tmr_id);

	/* Usage:  uint32_t alt_gpt_prescaler_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_prescaler_get(tmr_id);
	printf("alt_gpt_prescaler_get() .......... result=%d\n", (int)ui32val);
	/* Usage:  uint32_t alt_gpt_freq_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_freq_get(tmr_id);
	printf("alt_gpt_freq_get() ............... result=%d[Hz]\n", (int)ui32val);

	/* Usage:  uint32_t alt_gpt_counter_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_counter_get(tmr_id);
	printf("alt_gpt_counter_get() ............ result=0x%08X(%d)\n", (int)ui32val, (int)ui32val);
	/* Usage:  uint32_t alt_gpt_reset_value_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_reset_value_get(tmr_id);
	printf("alt_gpt_reset_value_get() ........ result=0x%08X(%d)\n", (int)ui32val, (int)ui32val);
	/* Usage:  uint32_t alt_gpt_maxcounter_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_maxcounter_get(tmr_id);
	printf("alt_gpt_maxcounter_get() ......... result=0x%08X(%d)\n", (int)ui32val, (int)ui32val);

	/* Usage:  uint32_t alt_gpt_time_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_time_get(tmr_id);
	printf("alt_gpt_time_get() ............... result=%d[period@sec]\n", (int)ui32val);
	/* Usage:  uint32_t alt_gpt_time_millisecs_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_time_millisecs_get(tmr_id);
	printf("alt_gpt_time_millisecs_get() ..... result=%d[period@msec]\n", (int)ui32val);
	/* Usage:  uint32_t alt_gpt_time_microsecs_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_time_microsecs_get(tmr_id);
	printf("alt_gpt_time_microsecs_get() ..... result=%d[period@usec]\n", (int)ui32val);
	/* Usage:  uint32_t alt_gpt_curtime_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_curtime_get(tmr_id);
	printf("alt_gpt_curtime_get() ............ result=%d[sec]\n", (int)ui32val);
	/* Usage:  uint32_t alt_gpt_curtime_millisecs_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_curtime_millisecs_get(tmr_id);
	printf("alt_gpt_curtime_millisecs_get() .. result=%d[msec]\n", (int)ui32val);
	/* Usage:  uint32_t alt_gpt_curtime_microsecs_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_curtime_microsecs_get(tmr_id);
	printf("alt_gpt_curtime_microsecs_get() .. result=%d[usec]\n", (int)ui32val);
	/* Usage:  uint32_t alt_gpt_curtime_nanosecs_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_curtime_nanosecs_get(tmr_id);
	printf("alt_gpt_curtime_nanosecs_get() ... result=%d[nsec]\n", (int)ui32val);
	/* Usage:  uint32_t alt_gpt_maxtime_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_maxtime_get(tmr_id);
	printf("alt_gpt_maxtime_get() ............ result=%d[period@sec]\n", (int)ui32val);
	/* Usage:  uint32_t alt_gpt_maxtime_millisecs_get(ALT_GPT_TIMER_t tmr_id);	*/
	ui32val = alt_gpt_maxtime_millisecs_get(tmr_id);
	printf("alt_gpt_maxtime_millisecs_get() .. result=%d[period@msec]\n", (int)ui32val);

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_print_all_settings(void) 
 * @brief  Display the value of all timer settings. 
 * 
 * @details  Call sample_gptmr_print_specific_settings for all timers (including 
 *           the WDT, Global Timer).
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_print_all_settings(void)
{
	printf("<< ALT_GPT_CPU_GLOBAL_TMR >>\n");
	sample_gptmr_print_specific_settings(ALT_GPT_CPU_GLOBAL_TMR);
	printf("\n");

	printf("<< ALT_GPT_CPU_PRIVATE_TMR >>\n");
	sample_gptmr_print_specific_settings(ALT_GPT_CPU_PRIVATE_TMR);
	printf("\n");

	printf("<< ALT_GPT_CPU_WDTGPT_TMR >>\n");
	sample_gptmr_print_specific_settings(ALT_GPT_CPU_WDTGPT_TMR);
	printf("\n");

	printf("<< ALT_GPT_OSC1_TMR0 >>\n");
	sample_gptmr_print_specific_settings(ALT_GPT_OSC1_TMR0);
	printf("\n");

	printf("<< ALT_GPT_OSC1_TMR1 >>\n");
	sample_gptmr_print_specific_settings(ALT_GPT_OSC1_TMR1);
	printf("\n");

	printf("<< ALT_GPT_SP_TMR0 >>\n");
	sample_gptmr_print_specific_settings(ALT_GPT_SP_TMR0);
	printf("\n");

	printf("<< ALT_GPT_SP_TMR1 >>\n");
	sample_gptmr_print_specific_settings(ALT_GPT_SP_TMR1);
	printf("\n");


	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_print_all_settings2(void) 
 * @brief  Display the value of all general-purpose timer settings. 
 * 
 * @details  Call sample_gptmr_print_specific_settings for the general-purpose 
 *           timers of all (except for the WDT, Global Timer).
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_print_all_settings2(void)
{
	printf("<< ALT_GPT_OSC1_TMR0 >>\n");
	sample_gptmr_print_specific_settings(ALT_GPT_OSC1_TMR0);
	printf("\n");

	printf("<< ALT_GPT_OSC1_TMR1 >>\n");
	sample_gptmr_print_specific_settings(ALT_GPT_OSC1_TMR1);
	printf("\n");

	printf("<< ALT_GPT_SP_TMR0 >>\n");
	sample_gptmr_print_specific_settings(ALT_GPT_SP_TMR0);
	printf("\n");

	printf("<< ALT_GPT_SP_TMR1 >>\n");
	sample_gptmr_print_specific_settings(ALT_GPT_SP_TMR1);
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_print_all_settings3(void) 
 * @brief  Display the value of all general-purpose timer setting status. 
 * 
 * @details  Call sample_gptmr_print_specific_settings_status for the general- 
 *           purpose timers of all (except for the WDT, Global Timer).
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_print_all_settings3(void)
{
	printf("<< ALT_GPT_OSC1_TMR0 >>\n");
	sample_gptmr_print_specific_settings_status(ALT_GPT_OSC1_TMR0);
	printf("\n");

	printf("<< ALT_GPT_OSC1_TMR1 >>\n");
	sample_gptmr_print_specific_settings_status(ALT_GPT_OSC1_TMR1);
	printf("\n");

	printf("<< ALT_GPT_SP_TMR0 >>\n");
	sample_gptmr_print_specific_settings_status(ALT_GPT_SP_TMR0);
	printf("\n");

	printf("<< ALT_GPT_SP_TMR1 >>\n");
	sample_gptmr_print_specific_settings_status(ALT_GPT_SP_TMR1);
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_start_all(void) 
 * @brief  Start the timers for the general-purpose timers.
 * 
 * @details  Trial related to the general-purpose timer start APIs.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_start_all(void)
{
	ALT_STATUS_CODE	result_code;

	/* Usage:  ALT_STATUS_CODE alt_gpt_tmr_start(ALT_GPT_TIMER_t tmr_id);	*/
	printf("-- HWLib test : alt_gpt_tmr_start() --\n");
	printf(" - ALT_GPT_OSC1_TMR0 timer start!!\n");
	result_code = alt_gpt_tmr_start(ALT_GPT_OSC1_TMR0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_tmr_start(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_OSC1_TMR1 timer start!!\n");
	result_code = alt_gpt_tmr_start(ALT_GPT_OSC1_TMR1);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_tmr_start(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR0 timer start!!\n");
	result_code = alt_gpt_tmr_start(ALT_GPT_SP_TMR0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_tmr_start(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR1 timer start!!\n");
	result_code = alt_gpt_tmr_start(ALT_GPT_SP_TMR1);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_tmr_start(); result=%d\n", (int)result_code);
	}
	printf("\n");


	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_stop_all(void) 
 * @brief  Stop the timers for the general-purpose timers.
 * 
 * @details  Trial related to the general-purpose timer stop APIs.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_stop_all(void)
{
	ALT_STATUS_CODE	result_code;

	/* Usage:  ALT_STATUS_CODE alt_gpt_tmr_stop(ALT_GPT_TIMER_t tmr_id);	*/
	printf("-- HWLib test : alt_gpt_tmr_stop() --\n");
	printf(" - ALT_GPT_OSC1_TMR0 timer stop!!\n");
	result_code = alt_gpt_tmr_stop(ALT_GPT_OSC1_TMR0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_tmr_stop(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_OSC1_TMR1 timer stop!!\n");
	result_code = alt_gpt_tmr_stop(ALT_GPT_OSC1_TMR1);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_tmr_stop(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR0 timer stop!!\n");
	result_code = alt_gpt_tmr_stop(ALT_GPT_SP_TMR0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_tmr_stop(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR1 timer stop!!\n");
	result_code = alt_gpt_tmr_stop(ALT_GPT_SP_TMR1);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_tmr_stop(); result=%d\n", (int)result_code);
	}
	printf("\n");


	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_reset_all(void) 
 * @brief  Reset the timers for the general-purpose timers.
 * 
 * @details  Trial related to the general-purpose timer reset APIs.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_reset_all(void)
{
	ALT_STATUS_CODE	result_code;

	/* Usage:  ALT_STATUS_CODE alt_gpt_tmr_reset(ALT_GPT_TIMER_t tmr_id);	*/
	printf("-- HWLib test : alt_gpt_tmr_reset() --\n");
	printf(" - ALT_GPT_OSC1_TMR0 timer reset!!\n");
	result_code = alt_gpt_tmr_reset(ALT_GPT_OSC1_TMR0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_tmr_reset(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_OSC1_TMR1 timer reset!!\n");
	result_code = alt_gpt_tmr_reset(ALT_GPT_OSC1_TMR1);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_tmr_reset(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR0 timer reset!!\n");
	result_code = alt_gpt_tmr_reset(ALT_GPT_SP_TMR0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_tmr_reset(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR1 timer reset!!\n");
	result_code = alt_gpt_tmr_reset(ALT_GPT_SP_TMR1);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_tmr_reset(); result=%d\n", (int)result_code);
	}
	printf("\n");


	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_ienable_all(void) 
 * @brief  Enabling interrupts of all general-purpose timers. 
 * 
 * @details  Trial related to the interrupt enable APIs.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_ienable_all(void)
{
	ALT_STATUS_CODE	result_code;

	/* Usage:  ALT_STATUS_CODE alt_gpt_int_if_pending_clear(ALT_GPT_TIMER_t tmr_id);	*/
	printf("-- HWLib test : alt_gpt_int_if_pending_clear() --\n");
	printf("alt_gpt_int_if_pending_clear(ALT_GPT_OSC1_TMR0); result=%d\n", (int)alt_gpt_int_if_pending_clear(ALT_GPT_OSC1_TMR0));
	printf("alt_gpt_int_if_pending_clear(ALT_GPT_OSC1_TMR1); result=%d\n", (int)alt_gpt_int_if_pending_clear(ALT_GPT_OSC1_TMR1));
	printf("alt_gpt_int_if_pending_clear(ALT_GPT_SP_TMR0); result=%d\n", (int)alt_gpt_int_if_pending_clear(ALT_GPT_SP_TMR0));
	printf("alt_gpt_int_if_pending_clear(ALT_GPT_SP_TMR1); result=%d\n", (int)alt_gpt_int_if_pending_clear(ALT_GPT_SP_TMR1));
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_enable(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_enable(ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable(TIMER_OSC1_0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_enable(ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable(TIMER_OSC1_1_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_enable(ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable(TIMER_L4SP_0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_enable(ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable(TIMER_L4SP_1_IRQ) result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_int_dist_secure_enable(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_secure_enable(ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_secure_enable(TIMER_OSC1_0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_secure_enable(ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_secure_enable(TIMER_OSC1_1_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_secure_enable(ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_secure_enable(TIMER_L4SP_0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_secure_enable(ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_secure_enable(TIMER_L4SP_1_IRQ) result=%d\n", (int)result_code);
	}

	/* Usage:  ALT_STATUS_CODE alt_gpt_int_enable(ALT_GPT_TIMER_t tmr_id);	*/
	printf("-- HWLib test : alt_gpt_int_enable() --\n");
	printf(" - ALT_GPT_OSC1_TMR0 timer interrupt enable!!\n");
	result_code = alt_gpt_int_enable(ALT_GPT_OSC1_TMR0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_int_enable(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_OSC1_TMR1 timer interrupt enable!!\n");
	result_code = alt_gpt_int_enable(ALT_GPT_OSC1_TMR1);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_int_enable(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR0 timer interrupt enable!!\n");
	result_code = alt_gpt_int_enable(ALT_GPT_SP_TMR0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_int_enable(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR1 timer interrupt enable!!\n");
	result_code = alt_gpt_int_enable(ALT_GPT_SP_TMR1);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_int_enable(); result=%d\n", (int)result_code);
	}
	printf("\n");


	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_idisable_all(void) 
 * @brief  Disabling interrupts of all general-purpose timers. 
 * 
 * @details  Trial related to the interrupt disable APIs. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_idisable_all(void)
{
	ALT_STATUS_CODE	result_code;

	/* Usage:  ALT_STATUS_CODE alt_gpt_int_disable(ALT_GPT_TIMER_t tmr_id);	*/
	printf("-- HWLib test : alt_gpt_int_disable() --\n");
	printf(" - ALT_GPT_OSC1_TMR0 timer interrupt disable!!\n");
	result_code = alt_gpt_int_disable(ALT_GPT_OSC1_TMR0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_int_disable(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_OSC1_TMR1 timer interrupt disable!!\n");
	result_code = alt_gpt_int_disable(ALT_GPT_OSC1_TMR1);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_int_disable(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR0 timer interrupt disable!!\n");
	result_code = alt_gpt_int_disable(ALT_GPT_SP_TMR0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_int_disable(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR1 timer interrupt disable!!\n");
	result_code = alt_gpt_int_disable(ALT_GPT_SP_TMR1);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_int_disable(); result=%d\n", (int)result_code);
	}
	printf("\n");


	return;
}

/********************************************************************************//*!
 * @fn  void sample_gptmr_test_init(void) 
 * @brief  Initializing the sample program. 
 * 
 * @details  Call the initialization function for the general-purpose timer API. 
 *           Other, do the following.
 *           -# Configure the Helio Board GPIO settings. 
 *           -# Setting of general-purpose timers (Use HWLib). 
 *           -# Setting of GIC (General-Purpose timer interrupt). 
 *           -# Print general-purpose timer settings. 
 *           -# Start the timers for the general-purpose timers. 
 *           -# Enabling interrupts of all general-purpose timers.  
 *           -# Print GIC settings of all general-purpose timer interrupts. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gptmr_test_init(void)
{
	ALT_STATUS_CODE result_code;

	/* Usage:  ALT_STATUS_CODE alt_gpt_mode_set(ALT_GPT_TIMER_t tmr_id, ALT_GPT_RESTART_MODE_t mode);	*/
	/* Usage:  ALT_STATUS_CODE alt_gpt_counter_set(ALT_GPT_TIMER_t tmr_id, uint32_t val);	*/
	/* Usage:  ALT_STATUS_CODE alt_gpt_prescaler_set(ALT_GPT_TIMER_t tmr_id, uint32_t val);	*/

#if 0	// main() 内で cpu0_init() 実施済み
	cpu0_init();
#endif

	/* Usage:  ALT_STATUS_CODE alt_gpt_all_tmr_init(void);	*/
	printf("-- HWLib test : alt_gpt_all_tmr_init() --\n");
	result_code = alt_gpt_all_tmr_init();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_all_tmr_init() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	printf("======== Print Initial GPTimer Settings ========\n");
	sample_gptmr_print_all_settings();

	// GPT Timer Stop & Interrupt Disable
	sample_gptmr_stop_all();
	sample_gptmr_idisable_all();

	/* Usage:  ALT_STATUS_CODE alt_gpt_prescaler_set(ALT_GPT_TIMER_t tmr_id, uint32_t val);	*/
	/* ##	Sets the clock prescaler value of the specified timer. Valid for CPU_PRIVATE_TMR and
	 * 		CPU_GLOBAL_TMR. Returns an error if called with a tmr_id of OSC1_TMR0, OSC1_TMR1,
	 * 		SP_TMR0, or SP_TMR1 since they have no prescaler.	*/
	/* 		--> Do Test using arg1 of ALT_GPT_CPU_WDTGPT_TMR. */
	printf("-- HWLib test : alt_gpt_prescaler_set(ALT_GPT_CPU_WDTGPT_TMR, 0) --\n");
	result_code = alt_gpt_prescaler_set(ALT_GPT_CPU_WDTGPT_TMR , 0);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_prescaler_set(); result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage:  ALT_STATUS_CODE alt_gpt_counter_set(ALT_GPT_TIMER_t tmr_id, uint32_t val);	*/
	printf("-- HWLib test : alt_gpt_counter_set() --\n");
	printf(" - ALT_GPT_OSC1_TMR0 counter set to 500,000,000 @25MHz (20sec)\n");
	result_code = alt_gpt_counter_set(ALT_GPT_OSC1_TMR0, 500000000);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_counter_set(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_OSC1_TMR1 counter set to 1,000,000,000 @25MHz (40sec)\n");
	result_code = alt_gpt_counter_set(ALT_GPT_OSC1_TMR1, 1000000000);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_counter_set(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR0 counter set to 500,000,000 @100MHz (5sec)\n");
	result_code = alt_gpt_counter_set(ALT_GPT_SP_TMR0, 500000000);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_counter_set(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR1 counter set to 1,000,000,000 @100MHz (10sec)\n");
	result_code = alt_gpt_counter_set(ALT_GPT_SP_TMR1, 1000000000);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_counter_set(); result=%d\n", (int)result_code);
	}
	printf("\n");

	/* Usage:  ALT_STATUS_CODE alt_gpt_mode_set(ALT_GPT_TIMER_t tmr_id, ALT_GPT_RESTART_MODE_t mode);	*/
	printf("-- HWLib test : alt_gpt_mode_set() --\n");
	printf(" - ALT_GPT_OSC1_TMR0 response mode set to ALT_GPT_RESTART_MODE_PERIODIC\n");
	result_code = alt_gpt_mode_set(ALT_GPT_OSC1_TMR0, ALT_GPT_RESTART_MODE_PERIODIC);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_mode_set(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_OSC1_TMR1 response mode set to ALT_GPT_RESTART_MODE_PERIODIC\n");
	result_code = alt_gpt_mode_set(ALT_GPT_OSC1_TMR1, ALT_GPT_RESTART_MODE_PERIODIC);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_mode_set(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR0 response mode set to ALT_GPT_RESTART_MODE_PERIODIC\n");
	result_code = alt_gpt_mode_set(ALT_GPT_SP_TMR0, ALT_GPT_RESTART_MODE_PERIODIC);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_mode_set(); result=%d\n", (int)result_code);
	}
	printf(" - ALT_GPT_SP_TMR1 response mode set to ALT_GPT_RESTART_MODE_PERIODIC\n");
	result_code = alt_gpt_mode_set(ALT_GPT_SP_TMR1, ALT_GPT_RESTART_MODE_PERIODIC);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_mode_set(); result=%d\n", (int)result_code);
	}
	printf("\n");


	// Interrupt GIC Config
	sample_gptmr_setting_gic();

	printf("======== Print Test GPTimer Settings ========\n");
	sample_gptmr_print_all_settings();

	printf("--- All GPT Start!!! ---\n");
	sample_gptmr_start_all();
	sample_gptmr_print_all_settings3();

	printf("--- All GPT Interrupts Enable!!! ---\n");
	sample_gptmr_ienable_all();
	sample_gptmr_print_all_settings3();

	printf("======== Print Test Interrupt Settings ========\n");
	sample_gptmr_print_all_int_settings();


	return;
}

/********************************************************************************//*!
 * @fn  void sample_gptmr_test_uninit(void) 
 * @brief  Uninitializing the sample program. 
 * 
 * @details  Call the uninitialization function for the general-purpose timer API.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gptmr_test_uninit(void)
{
	ALT_STATUS_CODE result_code;

	/* Usage:  ALT_STATUS_CODE alt_gpt_all_tmr_uninit(void);	*/
	printf("-- HWLib test : alt_gpt_all_tmr_uninit() --\n");
	result_code = alt_gpt_all_tmr_uninit();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_all_tmr_uninit() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");


	return;
}


/********************************************************************************//*!
 * @fn  static void sample_gptmr_print_setting_gic(ALT_INT_INTERRUPT_t int_id) 
 * @brief  Print GIC settings of specific general-purpose timer interrupts.
 * 
 * @details  Displays the specific interrupt settings of detail.
 * @param[in] int_id  interrupt identification types.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_print_setting_gic(ALT_INT_INTERRUPT_t int_id)
{
	uint32_t ui32temp = 0;
	ALT_INT_TRIGGER_t trigger_type = ALT_INT_TRIGGER_NA;

	//// Print initial value
	///****************************************************************************************/
	///* Usage: ALT_STATUS_CODE alt_int_dist_is_secure(ALT_INT_INTERRUPT_t int_id);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_is_enabled(ALT_INT_INTERRUPT_t int_id);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_is_pending(ALT_INT_INTERRUPT_t int_id);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_is_active(ALT_INT_INTERRUPT_t int_id);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_priority_get(ALT_INT_INTERRUPT_t int_id, uint32_t* priority);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_target_get(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t* target);	*/
	///* Usage: ALT_STATUS_CODE alt_int_dist_trigger_get(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t* trigger_type);	*/
	///****************************************************************************************/
	printf("alt_int_dist_is_secure() ...... result=%d\n", (int)alt_int_dist_is_secure(int_id));
	printf("alt_int_dist_is_enabled() ..... result=%d\n", (int)alt_int_dist_is_enabled(int_id));
	printf("alt_int_dist_is_pending() ..... result=%d\n", (int)alt_int_dist_is_pending(int_id));
	printf("alt_int_dist_is_active() ...... result=%d\n", (int)alt_int_dist_is_active(int_id));
	printf("alt_int_dist_priority_get() ... result=%d out=0x%08X\n", (int)alt_int_dist_priority_get(int_id, &ui32temp), (int)ui32temp);
	printf("alt_int_dist_target_get() ..... result=%d out=0x%08X\n", (int)alt_int_dist_target_get(int_id, &ui32temp), (int)ui32temp);
	printf("alt_int_dist_trigger_get() .... result=%d out=%d\n", (int)alt_int_dist_trigger_get(int_id, &trigger_type), (int)trigger_type);

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_print_all_int_settings(void) 
 * @brief  Print GIC settings of all general-purpose timer interrupts. 
 * 
 * @details  Displays the three types of interrupt settings.
 *           - ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ 
 *           - ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ 
 *           - ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ 
 *           - ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_print_all_int_settings(void)
{
	printf("<< ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ >>\n");
	sample_gptmr_print_setting_gic(ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ);
	printf("\n");

	printf("<< ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ >>\n");
	sample_gptmr_print_setting_gic(ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ);
	printf("\n");

	printf("<< ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ >>\n");
	sample_gptmr_print_setting_gic(ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ);
	printf("\n");

	printf("<< ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ >>\n");
	sample_gptmr_print_setting_gic(ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ);
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  void sample_gptmr_setting_gic(void) 
 * @brief  Setting of GIC (General-Purpose timer interrupt).
 * 
 * @details    In order to receive a general-purpose timer interrupt setting of GIC.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gptmr_setting_gic(void)
{
	ALT_STATUS_CODE result_code;


	/* Usage: ALT_STATUS_CODE alt_int_isr_register(ALT_INT_INTERRUPT_t int_id, alt_int_callback_t callback, void* context);	*/
	/*  typedef void(* alt_int_callback_t)(uint32_t icciar, void *context) */
	result_code = alt_int_isr_register(ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ, sample_gptmr_callback_OSC1_TMR0, NULL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_isr_register(TIMER_OSC1_0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_isr_register(ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ, sample_gptmr_callback_OSC1_TMR1, NULL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_isr_register(TIMER_OSC1_1_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_isr_register(ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ, sample_gptmr_callback_SP_TMR0, NULL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_isr_register(TIMER_L4SP_0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_isr_register(ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ, sample_gptmr_callback_SP_TMR1, NULL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_isr_register(TIMER_L4SP_1_IRQ) result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_int_dist_target_set(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t target);	*/
	result_code = alt_int_dist_target_set(ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ, 1);		// arg2=targetCPU (Only Core#0)
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_target_set(TIMER_OSC1_0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_target_set(ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ, 1);		// arg2=targetCPU (Only Core#0)
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_target_set(TIMER_OSC1_1_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_target_set(ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ, 1);		// arg2=targetCPU (Only Core#0)
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_target_set(TIMER_L4SP_0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_target_set(ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ, 1);		// arg2=targetCPU (Only Core#0)
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_target_set(TIMER_L4SP_1_IRQ) result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_set(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t trigger_type);	*/
	result_code = alt_int_dist_trigger_set(ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ, ALT_INT_TRIGGER_LEVEL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_trigger_set(TIMER_OSC1_0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_trigger_set(ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ, ALT_INT_TRIGGER_LEVEL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_trigger_set(TIMER_OSC1_1_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_trigger_set(ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ, ALT_INT_TRIGGER_LEVEL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_trigger_set(TIMER_L4SP_0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_trigger_set(ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ, ALT_INT_TRIGGER_LEVEL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_trigger_set(TIMER_L4SP_1_IRQ) result=%d\n", (int)result_code);
	}

	/* Usage: ALT_STATUS_CODE alt_int_dist_priority_set(ALT_INT_INTERRUPT_t int_id, uint32_t priority);	*/
	result_code = alt_int_dist_priority_set(ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ, 99);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_priority_set(TIMER_OSC1_0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_priority_set(ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ, 99);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_priority_set(TIMER_OSC1_1_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_priority_set(ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ, 99);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_priority_set(TIMER_L4SP_0_IRQ) result=%d\n", (int)result_code);
	}
	result_code = alt_int_dist_priority_set(ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ, 99);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_priority_set(TIMER_L4SP_1_IRQ) result=%d\n", (int)result_code);
	}

	return;
}

/********************************************************************************//*!
 * @fn  void sample_gptmr_callback_OSC1_TMR0(uint32_t icciar, void *context) 
 * @brief  L4_OSC1 Timer0 (ALT_GPT_OSC1_TMR0) interrupt callback function.
 * 
 * @details  working opportunity an interrupt from the ALT_GPT_OSC1_TMR0, indicate 
 *           an interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued. 
 * @param[in] context  The user provided context.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gptmr_callback_OSC1_TMR0(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;

	// WDT Pending Clear
	/* Usage:  ALT_STATUS_CODE alt_gpt_int_clear_pending(ALT_GPT_TIMER_t tmr_id);	*/
	result_code = alt_gpt_int_clear_pending(ALT_GPT_OSC1_TMR0);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT1]ERROR!!: alt_gpt_int_clear_pending(ALT_GPT_OSC1_TMR0) result=%d\n", (int)result_code);
	}

	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT1]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	//fprintf(stderr, "[INTERRUPT1] ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ is occurred!!\n");
	util_intlog_record(ALT_INT_INTERRUPT_TIMER_OSC1_0_IRQ, (int)icciar, *(int*)context);
	//sample_gptmr_print_specific_counter(ALT_GPT_OSC1_TMR0);

	return;
}

/********************************************************************************//*!
 * @fn  void sample_gptmr_callback_OSC1_TMR1(uint32_t icciar, void *context) 
 * @brief  L4_OSC1 Timer1 (ALT_GPT_OSC1_TMR1) interrupt callback function.
 * 
 * @details  working opportunity an interrupt from the ALT_GPT_OSC1_TMR1, indicate 
 *           an interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued. 
 * @param[in] context  The user provided context.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gptmr_callback_OSC1_TMR1(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;

	// WDT Pending Clear
	/* Usage:  ALT_STATUS_CODE alt_gpt_int_clear_pending(ALT_GPT_TIMER_t tmr_id);	*/
	result_code = alt_gpt_int_clear_pending(ALT_GPT_OSC1_TMR1);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT2]ERROR!!: alt_gpt_int_clear_pending(ALT_GPT_OSC1_TMR1) result=%d\n", (int)result_code);
	}

	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT2]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	//fprintf(stderr, "[INTERRUPT2] ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ is occurred!!\n");
	util_intlog_record(ALT_INT_INTERRUPT_TIMER_OSC1_1_IRQ, (int)icciar, *(int*)context);
	//sample_gptmr_print_specific_counter(ALT_GPT_OSC1_TMR1);

	return;
}

/********************************************************************************//*!
 * @fn  void sample_gptmr_callback_SP_TMR0(uint32_t icciar, void *context) 
 * @brief  L4_SP Timer0 (ALT_GPT_SP_TMR0) interrupt callback function.
 * 
 * @details  working opportunity an interrupt from the ALT_GPT_SP_TMR0, indicate 
 *           an interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued. 
 * @param[in] context  The user provided context.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gptmr_callback_SP_TMR0(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;

	// WDT Pending Clear
	/* Usage:  ALT_STATUS_CODE alt_gpt_int_clear_pending(ALT_GPT_TIMER_t tmr_id);	*/
	result_code = alt_gpt_int_clear_pending(ALT_GPT_SP_TMR0);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT3]ERROR!!: alt_gpt_int_clear_pending(ALT_GPT_SP_TMR0) result=%d\n", (int)result_code);
	}

	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT3]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	//fprintf(stderr, "[INTERRUPT3] ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ is occurred!!\n");
	util_intlog_record(ALT_INT_INTERRUPT_TIMER_L4SP_0_IRQ, (int)icciar, *(int*)context);
	//sample_gptmr_print_specific_counter(ALT_GPT_SP_TMR0);

	return;
}

/********************************************************************************//*!
 * @fn  void sample_gptmr_callback_SP_TMR1(uint32_t icciar, void *context) 
 * @brief  L4_SP Timer1 (ALT_GPT_SP_TMR1) interrupt callback function.
 * 
 * @details  working opportunity an interrupt from the ALT_GPT_SP_TMR1, indicate 
 *           an interrupt, and the interrupt pending bit clear.
 * @param[in] icciar  The callback to use when the given interrupt is issued. 
 * @param[in] context  The user provided context.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gptmr_callback_SP_TMR1(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;

	// WDT Pending Clear
	/* Usage:  ALT_STATUS_CODE alt_gpt_int_clear_pending(ALT_GPT_TIMER_t tmr_id);	*/
	result_code = alt_gpt_int_clear_pending(ALT_GPT_SP_TMR1);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT4]ERROR!!: alt_gpt_int_clear_pending(ALT_GPT_SP_TMR1) result=%d\n", (int)result_code);
	}

	// GIC Pending Clear
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT4]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	//fprintf(stderr, "[INTERRUPT4] ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ is occurred!!\n");
	util_intlog_record(ALT_INT_INTERRUPT_TIMER_L4SP_1_IRQ, (int)icciar, *(int*)context);
	//sample_gptmr_print_specific_counter(ALT_GPT_SP_TMR1);

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_gptmr_changemode_and_reset(ALT_GPT_TIMER_t tmr_id, ALT_GPT_RESTART_MODE_t mode) 
 * @brief  Change mode and reset the specific general-purpose timer. 
 * 
 * @details  Change the specified general-purpose timer response mode. 
 *           Then reset the timer. 
 * @param[in] tmr_id  general-purpose timer identification types. 
 * @param[in] mode  ALT_GPT_RESTART_MODE_PERIODIC / ALT_GPT_RESTART_MODE_ONESHOT. 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_gptmr_changemode_and_reset(ALT_GPT_TIMER_t tmr_id, ALT_GPT_RESTART_MODE_t mode)
{
	ALT_STATUS_CODE	result_code;

	switch(mode)
	{
	case ALT_GPT_RESTART_MODE_PERIODIC:
		printf("Timer(tmr_id=%d) response mode set to ALT_GPT_RESTART_MODE_PERIODIC\n", (int)tmr_id);
		break;
	default:	/* case ALT_GPT_RESTART_MODE_ONESHOT: */
		printf("Timer(tmr_id=%d) response mode set to ALT_GPT_RESTART_MODE_ONESHOT\n", (int)tmr_id);
		break;
	}
	result_code = alt_gpt_mode_set(tmr_id, mode);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_mode_set(); result=%d\n", (int)result_code);
	}

	/* Usage:  ALT_STATUS_CODE alt_gpt_tmr_reset(ALT_GPT_TIMER_t tmr_id);	*/
	printf("-- HWLib test : alt_gpt_tmr_reset() --\n");
	result_code = alt_gpt_tmr_reset(tmr_id);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_gpt_tmr_reset(); result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  void sample_gptmr_test_print_usage(void) 
 * @brief  Display usage of the sample program. 
 * 
 * @details  Following information displays on the console.
 *            - Overview of HPS for push button functions (or operations)
 *            - Overview of HPS for DIP switch functions (or operations)
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/18  ---  #xxxxx  新規作成. 
 **//*******************************************************************************/
void sample_gptmr_test_print_usage(void)
{
	printf("\n");
	printf("+--- Usage ( Function of PUSH SW and SLIDE SW )  ---------------+\n");
	printf("+ PUSH SW #0 .... Exit Test loop!!!\n");
	printf("+ PUSH SW #1 .... Display the timer counters and each timeout time(msec).\n");
	printf("+ PUSH SW #2 .... Display the informations of each timers.(mode,enable,interrupts..)\n");
	printf("+ PUSH SW #3 .... Reset all general-purpose timer counters.\n");
	printf("+ SLIDE SW#0:3 .. Select timer mode. {Periodic(ON) / One-Shot(OFF)}\n");
	printf("   Each switch is as follows:\n");
	printf("   #0:OSC1 timer0, #1:OSC1 timer1, #3:SP timer0, #4:SP timer1\n");
	printf("+---------------------------------------------------------------+\n\n");
	
#if 0
	　無限ループ内で以下の処理を行う。
	　 - HPS用DIPSW1-4の変更契機で、各GPTの動作モードを変更します。
	　　 また、モード変更を行った場合はタイマを再起動します。
	　　 各DIPSWの対応は下記の通りです。
	　　・DIPSW1 .. OSC1 timer 0 
	　　・DIPSW2 .. OSC1 timer 1 
	　　・DIPSW3 .. SP timer 0 
	　　・DIPSW4 .. SP timer 1 
	　　（各DIPSWは、ONならPERIODIC(User-defined count mode)、
	　　　OFFならONESHOT(Free-running mode)になります。）

	　 - HPS用PUSHSW1の押下契機で 4個のGPTのカウンタ値およびタイムアウトまでの
	　　 残時間（ミリ秒）を表示します。

	　 - HPS用PUSHSW2の押下契機で 4個のGPTの動作情報（モード、タイマイネーブル、
	　　 割り込みイネーブル、割り込みペンディング）を表示します。

	　 - HPS用PUSHSW3の押下契機で 4個のGPT全てを再起動（リセット）します。

	　 - HPS用PUSHSW0の押下契機で無限ループを終了します。
	　　 (テストプログラム終了)
#endif
	
	return;
}

/********************************************************************************//*!
 * @fn  void sample_gptmr_test_main(void) 
 * @brief  Main routine of the sample program. 
 * 
 * @details  Infinite loop structure works. Make the detection of various types 
 *           of switches in the loop, it performs in various testing processes.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/28  ---  #6334  新規作成. 
 * @date 2014/01/29  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_gptmr_test_main(void)
{
	uint32_t switch_raw = 0;
	uint32_t switch_raw_bk = 0;
	uint32_t switch_detect = 0;
	uint32_t switch_detect_on = 0;
	uint32_t switch_number = 0;
	bool disp_usage = true;
	
	// GPT Timer Restart
	sample_gptmr_reset_all();
	sample_gptmr_print_all_counter();
	util_intlog_print();

	printf("==== Start While(1) loop process!!! ============\n");
	switch_raw_bk = sample_detect_switch();
	while(1)
	{
		if(disp_usage)
		{
			sample_gptmr_test_print_usage();
			disp_usage = false;
		}

		// ---- Check the Slide-Switch and Push-Switch. ---- 
		switch_raw = sample_detect_switch();
		switch_detect = switch_raw ^ switch_raw_bk;
		switch_detect_on  |= switch_detect & switch_raw;
		
		// Slide-Switch 
		if(switch_detect & SAMPLE_SWITCH_BIT_SLIDEALL)
		{
			switch_number = switch_raw & SAMPLE_SWITCH_BIT_NUM;
			switch_number *= 1;		// To avoid warnings.

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE0){
				//printf("SAMPLE_SWITCH_BIT_SLIDE0\n");
				(void)alt_gpt_tmr_stop(ALT_GPT_OSC1_TMR0);
				if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE0){
					sample_gptmr_changemode_and_reset(ALT_GPT_OSC1_TMR0, ALT_GPT_RESTART_MODE_PERIODIC);
				} else {
					sample_gptmr_changemode_and_reset(ALT_GPT_OSC1_TMR0, ALT_GPT_RESTART_MODE_ONESHOT);
				}
				(void)alt_gpt_tmr_reset(ALT_GPT_OSC1_TMR0);
				sample_gptmr_print_specific_counter(ALT_GPT_OSC1_TMR0);
				sample_gptmr_print_specific_settings_status(ALT_GPT_OSC1_TMR0);
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE1){
				//printf("SAMPLE_SWITCH_BIT_SLIDE1\n");
				(void)alt_gpt_tmr_stop(ALT_GPT_OSC1_TMR1);
				if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE1){
					sample_gptmr_changemode_and_reset(ALT_GPT_OSC1_TMR1, ALT_GPT_RESTART_MODE_PERIODIC);
				} else {
					sample_gptmr_changemode_and_reset(ALT_GPT_OSC1_TMR1, ALT_GPT_RESTART_MODE_ONESHOT);
				}
				(void)alt_gpt_tmr_reset(ALT_GPT_OSC1_TMR1);
				sample_gptmr_print_specific_counter(ALT_GPT_OSC1_TMR1);
				sample_gptmr_print_specific_settings_status(ALT_GPT_OSC1_TMR1);
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE2){
				//printf("SAMPLE_SWITCH_BIT_SLIDE2\n");
				(void)alt_gpt_tmr_stop(ALT_GPT_SP_TMR0);
				if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE2){
					sample_gptmr_changemode_and_reset(ALT_GPT_SP_TMR0, ALT_GPT_RESTART_MODE_PERIODIC);
				} else {
					sample_gptmr_changemode_and_reset(ALT_GPT_SP_TMR0, ALT_GPT_RESTART_MODE_ONESHOT);
				}
				(void)alt_gpt_tmr_reset(ALT_GPT_SP_TMR0);
				sample_gptmr_print_specific_counter(ALT_GPT_SP_TMR0);
				sample_gptmr_print_specific_settings_status(ALT_GPT_SP_TMR0);
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE3){
				//printf("SAMPLE_SWITCH_BIT_SLIDE3\n");
				(void)alt_gpt_tmr_stop(ALT_GPT_SP_TMR1);
				if(switch_raw & SAMPLE_SWITCH_BIT_SLIDE3){
					sample_gptmr_changemode_and_reset(ALT_GPT_SP_TMR1, ALT_GPT_RESTART_MODE_PERIODIC);
				} else {
					sample_gptmr_changemode_and_reset(ALT_GPT_SP_TMR1, ALT_GPT_RESTART_MODE_ONESHOT);
				}
				(void)alt_gpt_tmr_reset(ALT_GPT_SP_TMR1);
				sample_gptmr_print_specific_counter(ALT_GPT_SP_TMR1);
				sample_gptmr_print_specific_settings_status(ALT_GPT_SP_TMR1);
			}
			disp_usage = false;
		}
		
		
		// Push-Switch 0
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH0)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH0;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH0\n");
			sample_gptmr_idisable_all();

			printf("======== Print Last GPTimer Settings ========\n");
			sample_gptmr_print_all_settings2();

			// PGT Timer Stop
			sample_gptmr_stop_all();

			break;	// Exit Push-Switch 0 and 1 becomes ON.
		}
		// Push-Switch 1
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH1)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH1;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH1\n");
			// Print OSC1_TMR0 and OSC1_TMR1 counter!!
			// Print SP_TMR0 and SP_TMR1 counter!!
			printf("<< ALT_GPT_OSC1_TMR0 >>\n");
			sample_gptmr_print_specific_counter(ALT_GPT_OSC1_TMR0);
			printf("<< ALT_GPT_OSC1_TMR1 >>\n");
			sample_gptmr_print_specific_counter(ALT_GPT_OSC1_TMR1);
			printf("<< ALT_GPT_SP_TMR0 >>\n");
			sample_gptmr_print_specific_counter(ALT_GPT_SP_TMR0);
			printf("<< ALT_GPT_SP_TMR1 >>\n");
			sample_gptmr_print_specific_counter(ALT_GPT_SP_TMR1);
			printf("\n");
			disp_usage = true;
		}
		// Push-Switch 2
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH2)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH2;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH2\n");
			sample_gptmr_print_all_settings3();
			disp_usage = true;
		}
		// Push-Switch 3
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH3)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH3;

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH3\n");
			sample_gptmr_reset_all();
			sample_gptmr_print_all_counter2();
			sample_gptmr_print_all_settings3();
			disp_usage = true;
		}
		
		util_intlog_print();

		switch_raw_bk = switch_raw;
	}
	printf("==== End While(1) loop process. ================\n");

	util_intlog_print();

	return;
}


/********************************************/
/*	Test Command (General-Purpose Timer)	*/
/********************************************/
int sample_gptmr_test_cmd(char* options)
{
	volatile uint32_t	waitstep;

    ALT_STATUS_CODE status = ALT_E_SUCCESS;
    
    printf("\r\nUser Application Start!\r\n");

    //
    // Start the interrupt system
    //
    if (status == ALT_E_SUCCESS)
    {
    	util_intlog_init();
        status = socfpga_int_start();
    }

    // Wait for 1024steps
    for(waitstep=1024; waitstep>0; waitstep--)
    {}


    // Printing Current PLL Setting. And Set Global Timer for Time Measurement.
    util_time_init();

    //// Sample Function Call!!! |---->
    printf(">>>> Execute sample_gptmr_test_init(); !!! <<<<\r\n");
    sample_gptmr_test_init();

    printf(">>>> Execute sample_gptmr_test_main(); !!! <<<<\r\n");
    sample_gptmr_test_main();

    printf(">>>> Execute sample_gptmr_test_uninit(); !!! <<<<\r\n");
    sample_gptmr_test_uninit();
    //// Sample Function Call!!! <----|

    // Printing All Results of Time Measurement. And Uninit Global Timer.
    util_time_uninit();

    //
    // Stop the interrupt system
    //
    socfpga_int_stop();
	util_intlog_print();

    printf("Finished running the sample !!!\r\n");
    if (status == ALT_E_SUCCESS)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}

// All APIs
/***********************************************************************/
/* Usage:  ALT_STATUS_CODE alt_gpt_all_tmr_uninit(void);	*/
/* Usage:  ALT_STATUS_CODE alt_gpt_all_tmr_init(void);	*/
/* Usage:  ALT_STATUS_CODE alt_gpt_tmr_stop(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  ALT_STATUS_CODE alt_gpt_tmr_start(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  ALT_STATUS_CODE alt_gpt_tmr_is_running(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  ALT_STATUS_CODE alt_gpt_tmr_reset(ALT_GPT_TIMER_t tmr_id);	*/

/* Usage:  ALT_STATUS_CODE alt_gpt_counter_set(ALT_GPT_TIMER_t tmr_id, uint32_t val);	*/
/* Usage:  uint32_t alt_gpt_counter_get(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  uint32_t alt_gpt_reset_value_get(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  uint32_t alt_gpt_maxcounter_get(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  ALT_STATUS_CODE alt_gpt_prescaler_set(ALT_GPT_TIMER_t tmr_id, uint32_t val);	*/
/* Usage:  uint32_t alt_gpt_prescaler_get(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  uint32_t alt_gpt_freq_get(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  uint32_t alt_gpt_time_get(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  uint32_t alt_gpt_time_millisecs_get(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  uint32_t alt_gpt_time_microsecs_get(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  uint32_t alt_gpt_curtime_get(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  uint32_t alt_gpt_curtime_millisecs_get(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  uint32_t alt_gpt_curtime_microsecs_get(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  uint32_t alt_gpt_curtime_nanosecs_get(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  uint32_t alt_gpt_maxtime_get(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  uint32_t alt_gpt_maxtime_millisecs_get(ALT_GPT_TIMER_t tmr_id);	*/

/* Usage:  ALT_STATUS_CODE alt_gpt_int_disable(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  ALT_STATUS_CODE alt_gpt_int_enable(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  bool alt_gpt_int_is_enabled(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  ALT_STATUS_CODE alt_gpt_int_clear_pending(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  ALT_STATUS_CODE alt_gpt_int_is_pending(ALT_GPT_TIMER_t tmr_id);	*/
/* Usage:  ALT_STATUS_CODE alt_gpt_int_if_pending_clear(ALT_GPT_TIMER_t tmr_id);	*/

/* Usage:  ALT_STATUS_CODE alt_gpt_mode_set(ALT_GPT_TIMER_t tmr_id, ALT_GPT_RESTART_MODE_t mode);	*/
/* Usage:  int32_t alt_gpt_mode_get(ALT_GPT_TIMER_t tmr_id);	*/
/***********************************************************************/

////////////////////////////// Common Functions //////////////////////////////////////// 

/***********************************************************************************
 * end of file 
 ***********************************************************************************/
