﻿/********************************************************************************//*!
 * @file  sample_time_measurement.c 
 * @brief  Sample program implements the time measurement. 
 * 
 * @details  Time measurement program 'util_time_measurement.c' and 
 *           'util_time_measurement.h' sample was used.
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/23  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 * @date 2014/08/18  ---  -----  v14.0化 (誤植訂正、操作説明の表示を追加). 
 * @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
 * @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
 * @date 2018/07/02  ---  #25940  All in One ベアメタル・アプリケーション v17.1 への対応を実施.
 * @date 2019/01/18  ---  -----   All in One ベアメタル・アプリケーション v18.1 への対応を実施.
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes 
 ***********************************************************************************/
#include <stdio.h>
#include <string.h>
#include "hwlib.h"
#include "socal/socal.h"
#include "socal/hps.h"
#include "alt_generalpurpose_io.h"

#include "sample_app_setting.h"
#include "util_time_measurement.h"
#include "util_interrupt_log.h"
#include "alt_printf.h"

/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/
void sample_time_measurement_init(void);
void sample_time_measurement_test_main(void);
void sample_time_measurement_test_print_usage(void);
int  sample_time_measurement_test_cmd(char* options);

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/

/***********************************************************************************
 *  variables 
 ***********************************************************************************/

/***********************************************************************************
 *  functions 
 ***********************************************************************************/

/********************************************************************************//*!
 * @fn  void sample_time_measurement_init(void) 
 * @brief  Initialize for this sample programs. 
 * 
 * @details  Call the initialization time measurement program 'util_time_init()', 
 *           and Helio Board GPIO settings function 'sample_gpio_config_helioboard()'.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/23  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_time_measurement_init(void)
{
	/* Usage: bool alt_clk_is_in_safe_mode(ALT_CLK_SAFE_DOMAIN_t clk_domain;	*/
	printf("-- test : util_time_init() --\n");
    util_time_init();

	util_intlog_init();
#if 0	// main() 内で cpu0_init() 実施済み
	cpu0_init();
#endif

	return;
}

/********************************************************************************//*!
 * @fn  void sample_time_measurement_test_main(void) 
 * @brief  Main routine of the sample program. 
 * 
 * @details  Infinite loop structure works. Make the detection of various types 
 *           of switches in the loop, it performs in various testing processes.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/23  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_time_measurement_test_main(void)
{
	uint32_t switch_raw = 0;
	uint32_t switch_raw_bk = 0;
	uint32_t switch_detect = 0;
	uint32_t switch_detect_on = 0;
	uint32_t switch_number = 0;
	bool disp_usage = true;
	
	printf("==== Start While(1) loop process!!! ============\n");
	switch_raw_bk = sample_detect_switch();
	while(1)
	{
		if(disp_usage)
		{
			sample_time_measurement_test_print_usage();
			disp_usage = false;
		}

		// ---- Check the Slide-Switch and Push-Switch. ---- 
		switch_raw = sample_detect_switch();
		switch_detect = switch_raw ^ switch_raw_bk;
		switch_detect_on  |= switch_detect & switch_raw;
		
		// Slide-Switch 
		if(switch_detect & SAMPLE_SWITCH_BIT_SLIDEALL)
		{
			switch_number = switch_raw & SAMPLE_SWITCH_BIT_NUM;
			switch_number *= 1;		// To avoid warnings.

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE0){
				//printf("SAMPLE_SWITCH_BIT_SLIDE0\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE1){
				//printf("SAMPLE_SWITCH_BIT_SLIDE1\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE2){
				//printf("SAMPLE_SWITCH_BIT_SLIDE2\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE3){
				//printf("SAMPLE_SWITCH_BIT_SLIDE3\n");
			}
			disp_usage = false;
		}
		
		
		// Push-Switch 0
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH0)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH0;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH0\n");

			/* Usage: void util_time_print_result_all(void);	*/
			printf("-- test : util_time_print_result_all() --\n");
			util_time_print_result_all(enumUTIL_TIME_TARGET_by_BOTH);

			break;	// Exit Push-Switch 0 and 1 becomes ON.
		}
		// Push-Switch 1
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH1)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH1;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH1\n");

			/* Usage: void util_time_print_result_all_and_clear(void);	*/
			printf("-- test : util_time_print_result_all_and_clear() --\n");
			util_time_print_result_all_and_clear(enumUTIL_TIME_TARGET_by_BOTH);

			disp_usage = true;
		}
		// Push-Switch 2
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH2)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH2;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH2\n");

			/* Usage: void util_time_record_start_point(uint32_t index);	*/
			printf("-- test : util_time_record_start_point(%d) --\n", (int)switch_number);
			util_time_record_start_point(switch_number);

			disp_usage = true;
		}
		// Push-Switch 3
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH3)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH3;

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH3\n");

			/* Usage: void util_time_record_end_point(uint32_t index);	*/
			printf("-- test : util_time_record_end_point(%d) --\n", (int)switch_number);
			util_time_record_end_point(switch_number);
			/* Usage: void util_time_print_result_by_counter(uint32_t index);	*/
			printf("-- test : util_time_print_result_by_counter(%d) --\n", (int)switch_number);
			util_time_print_result_by_counter(switch_number);
			/* Usage: void util_time_print_result_by_seconds(uint32_t index);	*/
			printf("-- test : util_time_print_result_by_seconds(%d) --\n", (int)switch_number);
			util_time_print_result_by_seconds(switch_number);

			disp_usage = true;
		}
		
		util_intlog_print();

		switch_raw_bk = switch_raw;
	}
	printf("==== End While(1) loop process. ================\n");

	util_intlog_print();

	return;
}

/********************************************************************************//*!
 * @fn  void sample_time_measurement_test_print_usage(void) 
 * @brief  Display usage of the sample program. 
 * 
 * @details  Following information displays on the console.
 *            - Overview of HPS for push button functions (or operations)
 *            - Overview of HPS for DIP switch functions (or operations)
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/18  ---  #xxxxx  新規作成. 
 **//*******************************************************************************/
void sample_time_measurement_test_print_usage(void)
{
	printf("\n");
	printf("+--- Usage ( Function of PUSH SW and SLIDE SW )  ---------------+\n");
	printf("+ PUSH SW #0 .... Exit Test loop!!! (and print all results.)\n");
	printf("+ PUSH SW #1 .... Display all results and clear.\n");
	printf("+ PUSH SW #2 .... Run the recording function for start point.\n");
	printf("+ PUSH SW #3 .... Run the recording function for end point.\n");
	printf("                         (And display the recent result.)\n");
	printf("+ SLIDE SW#0:3 .. Select the index that identifies the measurement results.\n");
	printf("+---------------------------------------------------------------+\n\n");
	
#if 0
	→無限ループを行い、HPS用PUSHSW,DIPSWの操作を契機に下記の
	　処理を呼び出します。(PUSHSW0で終了)
	　
	　 - DIPSW1:4 .. 計測対象識別用のindex値として利用します。

	　 - PUSHSW0 .. 計測結果の全件表示を行い無限ループから抜けます。
	　　　　　　　　（util_time_print_result_all）(★テスト終了)

	　 - PUSHSW1 .. 計測結果の全件表示＆クリアを呼び出します。
	　　　　　　　　（util_time_print_result_all_and_clear）

	　 - PUSHSW2 .. DIPSWをindex値として計測開始を記録。
	　　　　　　　　（util_time_record_start_point）
	
	　 - PUSHSW3 .. DIPSWをindex値として計測終了を記録し、
	　　　　　　　　１件分の計測結果を表示します。
	　　　　　　　　（util_time_record_end_point,
	　　　　　　　　　util_time_print_result_by_counter,
	　　　　　　　　　util_time_print_result_by_seconds）
	
	★ポイント★
	無限ループ処理が走っている状態で、PUSHSW2を押した後、
	任意の秒数待機した上でPUSHSW3を押してみて下さい。
	計測結果として、待機した時間に応じた時間が表示されます。
#endif
	
	return;
}


/************************************/
/*	Test Command (MMU Management)	*/
/************************************/
int sample_time_measurement_test_cmd(char* options)
{
    ALT_STATUS_CODE status = ALT_E_SUCCESS;
    
    printf("\r\nUser Application Start!\r\n");

    //
    // Start the interrupt system
    //
    if (status == ALT_E_SUCCESS)
    {
        status = socfpga_int_start();
    }

    //// Sample Function Call!!! |---->
    printf(">>>> Execute sample_time_measurement_init(); !!! <<<<\r\n");
    sample_time_measurement_init();

    printf(">>>> Execute sample_time_measurement_test_main(); !!! <<<<\r\n");
    sample_time_measurement_test_main();
    //// Sample Function Call!!! <----|

    //
    // Stop the interrupt system
    //
    socfpga_int_stop();

    printf("Finished running the sample !!!\r\n");
    if (status == ALT_E_SUCCESS)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}

// All Functions
/***********************************************************************/
/* Usage: void util_time_init(void);	*/
/* Usage: void util_time_record_start_point(uint32_t index);	*/
/* Usage: void util_time_record_end_point(uint32_t index);	*/
/* Usage: void util_time_print_result_by_counter(uint32_t index);	*/
/* Usage: void util_time_print_result_by_seconds(uint32_t index);	*/
/* Usage: void util_time_print_result_all(void);	*/
/* Usage: void util_time_print_result_all_and_clear(void);	*/
/***********************************************************************/

/***********************************************************************************
 * end of file 
 ***********************************************************************************/
