﻿/********************************************************************************//*!
 * @file  sample_app_setting.c
 * @brief  Initial setting of the HPS and the Board.
 * 
 * @details  Initial setting of the HPS and the Board.
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2015/03/02  ---  #18042  新規作成.
 * @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
 * @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
 * @date 2018/06/29  ---  -----  DE10-Nano ボードへの対応を追加.
 * @date 2021/04/23  ---  -----  SoC EDS v20.1 / Arm DS 2020.1 への対応を実施. 
 * 
 * @attention 
 * Copyright (C) 2013-2021 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes
 ***********************************************************************************/
#include <stdio.h>
//HWLib Headers

#include "alt_address_space.h"
#include "alt_interrupt.h"
#include "alt_system_manager.h"
#include "alt_mmu.h"
#include "alt_cache.h"
#include "alt_bridge_manager.h"
#include "hwlib.h"
#include "socal/socal.h"
#include "socal/hps.h"
#include "socal/alt_sysmgr.h"
#include "socal/alt_rstmgr.h"
#ifdef soc_cv_av
#include "socal/alt_acpidmap.h"
#endif /* soc_cv_av */
#include "sample_app_setting.h"
#include "util_time_measurement.h"
#include "altera_avalon_pio_regs.h"
#include "hps_system.h"
#include "alt_printf.h"

/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/
void sample_config_switch(void);

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/
/* GPIO definition for Helio Board. */
/** memo: port asign ***************/
/* Signal Name <--> HPS Port Name  */
/* USER_DIPSW_HPS0 <--> HPS_GPI13  */
/* USER_DIPSW_HPS1 <--> HPS_GPI12  */
/* USER_DIPSW_HPS2 <--> HPS_GPI11  */
/* USER_DIPSW_HPS3 <--> HPS_GPI10  */
/* USER_PB_HPS0 <--> HPS_GPI9      */
/* USER_PB_HPS1 <--> HPS_GPI8      */
/* USER_PB_HPS2 <--> HPS_GPI7      */
/* USER_PB_HPS3 <--> HPS_GPI6      */
/***********************************/
#define HELIO_GPIOPORTC_DIPSW0		(ALT_GPIO_BIT26)
#define HELIO_GPIOPORTC_DIPSW1		(ALT_GPIO_BIT25)
#define HELIO_GPIOPORTC_DIPSW2		(ALT_GPIO_BIT24)
#define HELIO_GPIOPORTC_DIPSW3		(ALT_GPIO_BIT23)
#define HELIO_GPIOPORTC_DIPSWALL	((HELIO_GPIOPORTC_DIPSW0)|(HELIO_GPIOPORTC_DIPSW1)|(HELIO_GPIOPORTC_DIPSW2)|(HELIO_GPIOPORTC_DIPSW3))
#define HELIO_GPIOPORTC_DIPSWLSB	(23)
#define HELIO_GPIOPORTC_PUSHSW2		(ALT_GPIO_BIT22)
#define HELIO_GPIOPORTC_PUSHSW0		(ALT_GPIO_BIT21)
#define HELIO_GPIOPORTC_PUSHSW3		(ALT_GPIO_BIT20)
#define HELIO_GPIOPORTC_PUSHSW1		(ALT_GPIO_BIT19)
#define HELIO_GPIOPORTC_PUSHSWALL	((HELIO_GPIOPORTC_PUSHSW0)|(HELIO_GPIOPORTC_PUSHSW1)|(HELIO_GPIOPORTC_PUSHSW2)|(HELIO_GPIOPORTC_PUSHSW3))
#define HELIO_GPIOPORTC_SWALL		((HELIO_GPIOPORTC_DIPSWALL)|(HELIO_GPIOPORTC_PUSHSWALL))

/* GPIO definition for Atlas Board. */
/** memo: port asign **********************/
/* Part Name <--> HPS Port Name           */
/* SLIDE SW (SW0) <--> FPGA PIO (L10)     */
/* SLIDE SW (SW1) <--> FPGA PIO (L9)      */
/* SLIDE SW (SW2) <--> FPGA PIO (H6)      */
/* SLIDE SW (SW3) <--> FPGA PIO (H5)      */
/* TACT SW (KEY0) <--> FPGA PIO (AH17)    */
/* TACT SW (KEY1) <--> FPGA PIO (AH16)    */
/* TACT SW (KEY2) HPS_KEY <--> HPS_GPIO54 */
/******************************************/
#define ATLAS_AVALONPIO_DIPSW0		(0x00000001)
#define ATLAS_AVALONPIO_DIPSW1		(0x00000002)
#define ATLAS_AVALONPIO_DIPSW2		(0x00000004)
#define ATLAS_AVALONPIO_DIPSW3		(0x00000008)
#define ATLAS_AVALONPIO_PUSH0		(0x00000001)
#define ATLAS_AVALONPIO_PUSH1		(0x00000002)
#define ATLAS_GPIOPORTB_PUSH2		(ALT_GPIO_BIT25)

//HPS_KEY <--> HPS_GPIO54 : GPIO1-25
//HPS_LED <--> HPS_GPIO53 : GPIO1-24

//User Interface (FPGA)
//KEY[1..0]
//SW[3..0]
//LED[7..0]

/* GPIO definition for Sodia Board. */
/** memo: port asign ***************/
/* Part Name <--> HPS Port Name    */
/* HPS DIP1 (S9) <--> HPS_GPI4     */
/* HPS DIP2 (S9) <--> HPS_GPI5     */
/* HPS DIP3 (S9) <--> HPS_GPI6     */
/* HPS DIP4 (S9) <--> HPS_GPI7     */
/* HPS PB0 (S10) <--> HPS_GPI8     */
/* HPS PB1 (S11) <--> HPS_GPI9     */
/* HPS PB2 (S12) <--> HPS_GPI10    */
/* HPS PB3 (S13) <--> HPS_GPI11    */
/***********************************/
#define SODIA_GPIOPORTC_DIPSW0		(ALT_GPIO_BIT17)
#define SODIA_GPIOPORTC_DIPSW1		(ALT_GPIO_BIT18)
#define SODIA_GPIOPORTC_DIPSW2		(ALT_GPIO_BIT19)
#define SODIA_GPIOPORTC_DIPSW3		(ALT_GPIO_BIT20)
#define SODIA_GPIOPORTC_DIPSWALL	((SODIA_GPIOPORTC_DIPSW0)|(SODIA_GPIOPORTC_DIPSW1)|(SODIA_GPIOPORTC_DIPSW2)|(SODIA_GPIOPORTC_DIPSW3))
#define SODIA_GPIOPORTC_DIPSWLSB	(17)
#define SODIA_GPIOPORTC_PUSHSW0		(ALT_GPIO_BIT21)
#define SODIA_GPIOPORTC_PUSHSW1		(ALT_GPIO_BIT22)
#define SODIA_GPIOPORTC_PUSHSW2		(ALT_GPIO_BIT23)
#define SODIA_GPIOPORTC_PUSHSW3		(ALT_GPIO_BIT24)
#define SODIA_GPIOPORTC_PUSHSWALL	((SODIA_GPIOPORTC_PUSHSW0)|(SODIA_GPIOPORTC_PUSHSW1)|(SODIA_GPIOPORTC_PUSHSW2)|(SODIA_GPIOPORTC_PUSHSW3))
#define SODIA_GPIOPORTC_SWALL		((SODIA_GPIOPORTC_DIPSWALL)|(SODIA_GPIOPORTC_PUSHSWALL))

/* GPIO definition for Cyclone V SoC Development Kit. */
/** memo: port asign ***************/
/* Signal Name <--> HPS Port Name  */
/* USER_DIPSW_HPS0 <--> HPS_GPI4   */
/* USER_DIPSW_HPS1 <--> HPS_GPI5   */
/* USER_DIPSW_HPS2 <--> HPS_GPI6   */
/* USER_DIPSW_HPS3 <--> HPS_GPI7   */
/* USER_PB_HPS0 <--> HPS_GPI8      */
/* USER_PB_HPS1 <--> HPS_GPI9      */
/* USER_PB_HPS2 <--> HPS_GPI10     */
/* USER_PB_HPS3 <--> HPS_GPI11     */
/***********************************/
#define C5SOCDK_GPIOPORTC_DIPSW0		(ALT_GPIO_BIT17)
#define C5SOCDK_GPIOPORTC_DIPSW1		(ALT_GPIO_BIT18)
#define C5SOCDK_GPIOPORTC_DIPSW2		(ALT_GPIO_BIT19)
#define C5SOCDK_GPIOPORTC_DIPSW3		(ALT_GPIO_BIT20)
#define C5SOCDK_GPIOPORTC_DIPSWALL	((C5SOCDK_GPIOPORTC_DIPSW0)|(C5SOCDK_GPIOPORTC_DIPSW1)|(C5SOCDK_GPIOPORTC_DIPSW2)|(C5SOCDK_GPIOPORTC_DIPSW3))
#define C5SOCDK_GPIOPORTC_DIPSWLSB	(17)
#define C5SOCDK_GPIOPORTC_PUSHSW0		(ALT_GPIO_BIT21)
#define C5SOCDK_GPIOPORTC_PUSHSW1		(ALT_GPIO_BIT22)
#define C5SOCDK_GPIOPORTC_PUSHSW2		(ALT_GPIO_BIT23)
#define C5SOCDK_GPIOPORTC_PUSHSW3		(ALT_GPIO_BIT24)
#define C5SOCDK_GPIOPORTC_PUSHSWALL	((C5SOCDK_GPIOPORTC_PUSHSW0)|(C5SOCDK_GPIOPORTC_PUSHSW1)|(C5SOCDK_GPIOPORTC_PUSHSW2)|(C5SOCDK_GPIOPORTC_PUSHSW3))
#define C5SOCDK_GPIOPORTC_SWALL		((C5SOCDK_GPIOPORTC_DIPSWALL)|(C5SOCDK_GPIOPORTC_PUSHSWALL))

/* GPIO definition for Arria 10 SoC Development Kit. */
/** memo: port asign ***************/
/* Signal Name <--> HPS Port Name  */
/* USER_DIPSW_HPS0 <--> ????????   */
/* USER_DIPSW_HPS1 <--> ????????   */
/* USER_DIPSW_HPS2 <--> GPIO1_IO18(N23) */
/* USER_DIPSW_HPS3 <--> GPIO1_IO9 (J23) */
/*  :   HPS側のIO Pinアサインが不明なのでFPGAを利用..  */
/***********************************/
#define A10SOCDK_AVALONPIO_DIPSW0		(0x00000001)
#define A10SOCDK_AVALONPIO_DIPSW1		(0x00000002)
#define A10SOCDK_AVALONPIO_DIPSW2		(0x00000004)
#define A10SOCDK_AVALONPIO_DIPSW3		(0x00000008)
#define A10SOCDK_AVALONPIO_DIPSWALL		(0x0000000F)
#define A10SOCDK_AVALONPIO_PUSH0		(0x00000001)
#define A10SOCDK_AVALONPIO_PUSH1		(0x00000002)
#define A10SOCDK_AVALONPIO_PUSH2		(0x00000004)
#define A10SOCDK_AVALONPIO_PUSH3		(0x00000008)
#define A10SOCDK_AVALONPIO_PUSHALL		(0x0000000F)

// definition for enable_neon_and_vfp() 
#define ALT_CPU_CPACR_ASEDIS_SET_MSK	(0x80000000)
#define ALT_CPU_CPACR_D32DIS_SET_MSK	(1 << 30)
#define ALT_CPU_CPACR_CP11_SET_MSK		(3 << 22)		/* 11b: Full Access */
#define ALT_CPU_CPACR_CP10_SET_MSK		(3 << 20)		/* 11b: Full Access */

// Auxiliary Control Register. See Cortex-A9, section 4.3.10.
#define ALT_CPU_ACTLR_SMP_SET_MSK                (1 << 6)
// Non-secure Access Control Register. See Cortex-A9, section 4.3.13.
#define ALT_CPU_NSACR_NSSMP_SET_MSK              (1 << 18)

// SCU functions
#ifdef soc_cv_av
#define SOCFPGA5XS1_BSP_SCU_BASE		(unsigned int)(0xFFFEC000)
#else /* soc_cv_av */
#define SOCFPGA5XS1_BSP_SCU_BASE		(unsigned int)(0xFFFFC000)
#endif /* soc_cv_av */
#define SOCFPGA5XS1_BSP_SCU_CTLR		(*(volatile unsigned int *)(SOCFPGA5XS1_BSP_SCU_BASE + 0x0))
#define SOCFPGA5XS1_BSP_SCU_INV_WAY		(*(volatile unsigned int *)(SOCFPGA5XS1_BSP_SCU_BASE + 0xC))

/* MMU table buffer size */
#define	MMU_TLB_HEAP_SIZE	( 32 * _KB )

/* TLB memory should be aligned 16KB */
#define	MMU_TLB_ALIGNMENT		(16*_KB)

/******** Definition Memory Region ********/
#define MAX_MEMORY_REGION_NUM	16

/***********************************************************************************
 *  variables
 ***********************************************************************************/
//typedef struct ALT_MMU_MEM_REGION_s
//{
//  void *                 va;        /*!< The beginning virtual address for the memory
//                                     *   region. The address must be aligned to one of 4KiB,
//                                     *   64KiB, 1MiB, or 16MiB boundaries.
//                                     */
//  void *                 pa;        /*!< The beginning physical address mapping for the
//                                     *   virtual address of the memory region. The address
//                                     *   must be aligned to one of 4KiB, 64KiB, 1MiB, or 16MiB
//                                     *   boundaries.
//                                     */
//  uint32_t               size;      /*!< The size of the memory region in bytes. The size
//                                     *   must be a multiple of 4KiB, 64KiB, 1MiB, or 16MiB
//                                     *   sizes.
//                                     */
//  ALT_MMU_AP_t           access;    /*!< The access permissions for the memory region. */
//  ALT_MMU_ATTR_t         attributes;/*!< The memory region attributes. These attributes
//                                     *   determine the memory type (ordering), cache
//                                     *   policy, and as a possible side effect, the
//                                     *   shareablity of the memory region.
//                                     */
//  ALT_MMU_TTB_S_t        shareable; /*!< The shareability of the memory region. */
//  ALT_MMU_TTB_XN_t       execute;   /*!< Whether instructions can be executed from this
//                                     *   memory region.
//                                     */
//  ALT_MMU_TTB_NS_t       security;  /*!< Controls whether address translations made from
//                                     *   the secure state translate physical address in
//                                     *   the secure or non-secure address map.
//                                     */
//} ALT_MMU_MEM_REGION_t;
static int	ValidMemoryRegion = 14;		/*!< The number of valid descriptors. */
static ALT_MMU_MEM_REGION_t MemoryRegion[MAX_MEMORY_REGION_NUM] = {
	/* [0] SDRAM0	*//* For Mutex using LDREX/STREX. */
	/* -- Cortex-A9 MPCore テクニカルリファレンス マニュアル [2.4.2. ACPの制限] -- */
	/* LDREX/STREX機構をACP経由で使用し、AxUSER[0] = 1およびAxCACHE[1] = 1に
	 * マークされているコヒーレントメモリ領域に排他アクセスを行うことはできません。
	 * ただし、AxUSER[0] = 0またはAxCACHE[1] = 0にマークされている非コヒーレントメモリ領域に
	 * ついては、ACP経由のLDREX/STREX機構が完全にサポートされます。 */
	/* ==> ALT_MMU_TTB_S_NON_SHAREABLE & ALT_MMU_ATTR_NC */
	{ (void*)SDRAM0_VIRTUAL_ADDR, (void*)SDRAM0_PHYSICAL_ADDR, SDRAM0_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_NC,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [1] SDRAMx	*/
	{ (void*)SDRAMx_VIRTUAL_ADDR, (void*)SDRAMx_PHYSICAL_ADDR, SDRAMx_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_WBA,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [2] SDRAM1	*/
	{ (void*)SDRAM1_VIRTUAL_ADDR, (void*)SDRAM1_PHYSICAL_ADDR, SDRAM1_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_NC,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [3] SDRAM2	*/
	{ (void*)SDRAM2_VIRTUAL_ADDR, (void*)SDRAM2_PHYSICAL_ADDR, SDRAM2_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_WT,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [4] SDRAM3	*/
	{ (void*)SDRAM3_VIRTUAL_ADDR, (void*)SDRAM3_PHYSICAL_ADDR, SDRAM3_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_WB,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [5] SDRAM4	*/
	{ (void*)SDRAM4_VIRTUAL_ADDR, (void*)SDRAM4_PHYSICAL_ADDR, SDRAM4_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_WBA,				/* attributes, */
	  //ALT_MMU_ATTR_WBA_NC,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [6] ACPWINDOW	*/
	{ (void*)ACPWINDOW_VIRTUAL_ADDR, (void*)ACPWINDOW_PHYSICAL_ADDR, ACPWINDOW_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_NC,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [7] HPS2FPGA	*/
	{ (void*)HPS2FPGA_VIRTUAL_ADDR, (void*)HPS2FPGA_PHYSICAL_ADDR, HPS2FPGA_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_DEVICE,			/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [8] LW_HPS2FPGA	*/
	{ (void*)LW_HPS2FPGA_VIRTUAL_ADDR, (void*)LW_HPS2FPGA_PHYSICAL_ADDR, LW_HPS2FPGA_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_DEVICE,			/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_ENABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
#ifdef soc_cv_av
	/* [9] PERIPHRALS	*/
	{ (void*)PERIPHRALS_VIRTUAL_ADDR, (void*)PERIPHRALS_PHYSICAL_ADDR, PERIPHRALS_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_STRONG,			/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_ENABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [10] BOOTROM	*/
	{ (void*)BOOTROM_VIRTUAL_ADDR, (void*)BOOTROM_PHYSICAL_ADDR, BOOTROM_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_READ_ONLY,			/* access, */
	  ALT_MMU_ATTR_WBA,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [11] MPU_SCU	*/
	{ (void*)MPUSCU_VIRTUAL_ADDR, (void*)MPUSCU_PHYSICAL_ADDR, MPUSCU_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_STRONG,			/* attributes, */
	  ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, */
	  ALT_MMU_TTB_XN_ENABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [12] MPU_L2	*/
	{ (void*)MPUL2_VIRTUAL_ADDR, (void*)MPUL2_PHYSICAL_ADDR, MPUL2_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_STRONG,			/* attributes, */
	  ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, */
	  ALT_MMU_TTB_XN_ENABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [13] RAM	*/
	{ (void*)RAM_VIRTUAL_ADDR, (void*)RAM_PHYSICAL_ADDR, RAM_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_WBA,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
#else /* soc_cv_av */
	/* [9] A10 PERIPHRALS	*/
	{ (void*)PERIPHRALS_VIRTUAL_ADDR, (void*)PERIPHRALS_PHYSICAL_ADDR, PERIPHRALS_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_STRONG,			/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_ENABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [10] A10 RAM	*/
	{ (void*)RAM_VIRTUAL_ADDR, (void*)RAM_PHYSICAL_ADDR, RAM_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_WBA,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [11] A10 BOOTROM	*/
	{ (void*)BOOTROM_VIRTUAL_ADDR, (void*)BOOTROM_PHYSICAL_ADDR, BOOTROM_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_READ_ONLY,			/* access, */
	  ALT_MMU_ATTR_WBA,				/* attributes, */
	  ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [12] A10 MPU_SCU	*/
	{ (void*)MPUSCU_VIRTUAL_ADDR, (void*)MPUSCU_PHYSICAL_ADDR, MPUSCU_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_STRONG,			/* attributes, */
	  ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, */
	  ALT_MMU_TTB_XN_ENABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [13] A10 MPU_L2	*/
	{ (void*)MPUL2_VIRTUAL_ADDR, (void*)MPUL2_PHYSICAL_ADDR, MPUL2_SIZE,	/* va, pa, size, */
	  ALT_MMU_AP_FULL_ACCESS,		/* access, */
	  ALT_MMU_ATTR_STRONG,			/* attributes, */
	  ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, */
	  ALT_MMU_TTB_XN_ENABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
#endif /* soc_cv_av */
	/* [14] Undefined */
	{ 0, 0, 0,	/* va, pa, size, */
	  ALT_MMU_AP_NO_ACCESS,			/* access, */
	  ALT_MMU_ATTR_FAULT,			/* attributes, */
	  ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	},
	/* [15] Undefined */
	{ 0, 0, 0,	/* va, pa, size, */
	  ALT_MMU_AP_NO_ACCESS,			/* access, */
	  ALT_MMU_ATTR_FAULT,			/* attributes, */
	  ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, */
	  ALT_MMU_TTB_XN_DISABLE,		/* execute, */
	  ALT_MMU_TTB_NS_SECURE			/* security */
	}
};

uint32_t mmu_tlb_heap[MMU_TLB_HEAP_SIZE] __attribute__ ((aligned MMU_TLB_ALIGNMENT)) ;
uint32_t *mmu_tlb_l1 = NULL;

/***********************************************************************************
 *  functions 
 ***********************************************************************************/
static void enable_neon_and_vfp(void)
{
	uint32_t regdata = 0;
	
	// Refer to: Cortex(TM)-A9 NEON(TM) Media Processing Engine TRM.
	// "Example 2-1 Enabling Advanced SIMD and VFP"
	// --------------------------------------------------------------------
	// MRC p15,0,r0,c1,c0,2 ; Read CPACR into r0
	// ORR r0,r0,#(3<<20) ; OR in User and Privileged access for CP10
	// ORR r0,r0,#(3<<22) ; OR in User and Privileged access for CP11
	// BIC r0, r0, #(3<<30) ; Clear ASEDIS/D32DIS if set
	// MCR p15,0,r0,c1,c0,2 ; Store new access permissions into CPACR
	// ISB ; Ensure side-effect of CPACR is visible
	// MOV r0,#(1<<30) ; Create value with FPEXC (bit 30) set in r0
	// VMSR FPEXC,r0 ; Enable VFP and SIMD extensions
	// --------------------------------------------------------------------

	// MRC p15, 0,<Rd>, c1, c0, 2; Read Coprocessor Access Control Register
#ifdef ARM_COMPILER	// 99
#if   defined(__ARMCOMPILER_VERSION)
    __asm("MRC p15, 0, %[regdata], c1, c0, 2" : [regdata] "=r" (regdata));
#elif defined(__ARMCC_VERSION)
    __asm("MRC p15, 0, regdata, c1, c0, 2");
#endif
#endif
#ifdef GNU_COMPILER	// 99
	__asm__ volatile("MRC p15, 0, %0, c1, c0, 2"
			: "=r" (regdata)
			: );
#endif
	
	regdata &= (uint32_t)~(ALT_CPU_CPACR_ASEDIS_SET_MSK | ALT_CPU_CPACR_D32DIS_SET_MSK);
	regdata |= (ALT_CPU_CPACR_CP11_SET_MSK | ALT_CPU_CPACR_CP10_SET_MSK);
	
	// MCR p15, 0,<Rd>, c1, c0, 2; Write Coprocessor Access Control Register
#ifdef ARM_COMPILER	// 99
#if   defined(__ARMCOMPILER_VERSION)
    __asm("MCR p15, 0, %[regdata], c1, c0, 2" : : [regdata] "r" (regdata));
#elif defined(__ARMCC_VERSION)
    __asm("MCR p15, 0, regdata, c1, c0, 2");
#endif
#endif
#ifdef GNU_COMPILER	// 99
	__asm__ volatile("MCR p15, 0, %0, c1, c0, 2"
			:
			: "r" (regdata));
#endif

	// ISB ; Ensure side-effect of CPACR is visible
#ifdef ARM_COMPILER	// 99
    __asm("ISB");
#endif
#ifdef GNU_COMPILER	// 99
	__asm__ volatile("ISB"
			:
			: );
#endif

	// MOV r0,#(1<<30) ; Create value with FPEXC (bit 30) set
	regdata = (1 << 30);

	// VMSR FPEXC,r0 ; Enable VFP and SIMD extensions
#ifdef ARM_COMPILER	// 99
#if   defined(__ARMCOMPILER_VERSION)
    __asm("VMSR FPEXC, %[regdata]" : [regdata] "=r" (regdata));
#elif defined(__ARMCC_VERSION)
    __asm("VMSR FPEXC, regdata");
#endif
#endif
#ifdef GNU_COMPILER	// 99
	__asm__ volatile("VMSR FPEXC, %0"
			:
			: "r" (regdata));
#endif

	return;
}

static void enable_actlr_smp(void)
{
	uint32_t regdata;

	// MRC p15, 0, <Rt>, c1, c0, 1; Read ACTLR
#ifdef ARM_COMPILER	// 99
#if   defined(__ARMCOMPILER_VERSION)
    __asm("MRC p15, 0, %[regdata], c1, c0, 1" : [regdata] "=r" (regdata));
#elif defined(__ARMCC_VERSION)
    __asm("MRC p15, 0, regdata, c1, c0, 1");
#endif
#endif
#ifdef GNU_COMPILER	// 99
	__asm__ volatile("MRC p15, 0, %0, c1, c0, 1"
			: "=r" (regdata)
			: );
#endif

	regdata |= ALT_CPU_ACTLR_SMP_SET_MSK;

	// MCR p15, 0, <Rt>, c1, c0, 1; Write ACTLR
#ifdef ARM_COMPILER	// 99
#if   defined(__ARMCOMPILER_VERSION)
    __asm("MCR p15, 0, %[regdata], c1, c0, 1" : : [regdata] "r" (regdata));
#elif defined(__ARMCC_VERSION)
    __asm("MCR p15, 0, regdata, c1, c0, 1");
#endif
#endif
#ifdef GNU_COMPILER	// 99
	__asm__ volatile("MCR p15, 0, %0, c1, c0, 1"
			:
			: "r" (regdata));
#endif

	return;
}

static void enable_nsacr_nssmp(void)
{
	uint32_t regdata;

	// MRC p15, 0, <Rt>, c1, c1, 2; Read NSACR
#ifdef ARM_COMPILER	// 99
#if   defined(__ARMCOMPILER_VERSION)
    __asm("MRC p15, 0, %[regdata], c1, c1, 2" : [regdata] "=r" (regdata));
#elif defined(__ARMCC_VERSION)
    __asm("MRC p15, 0, regdata, c1, c1, 2");
#endif
#endif
#ifdef GNU_COMPILER
	__asm__ volatile("MRC p15, 0, %0, c1, c1, 2"
			: "=r" (regdata)
			: );
#endif

	regdata |= ALT_CPU_NSACR_NSSMP_SET_MSK;

	// MCR p15, 0, <Rt>, c1, c1, 2; Write NSACR
#ifdef ARM_COMPILER	// 99
#if   defined(__ARMCOMPILER_VERSION)
    __asm("MCR p15, 0, %[regdata], c1, c1, 2" : : [regdata] "r" (regdata));
#elif defined(__ARMCC_VERSION)
    __asm("MCR p15, 0, regdata, c1, c1, 2");
#endif
#endif
#ifdef GNU_COMPILER
	__asm__ volatile("MCR p15, 0, %0, c1, c1, 2"
			:
			: "r" (regdata));
#endif

	return;
}

void* sample_mmu_tlb_alloc(const size_t size, void * context)
{
	void* p = NULL;

	/* User defined function used for allocating storage for first and second level translation tables. */

	if(size >= sizeof(mmu_tlb_heap)){
		printf("[C0] ERROR!!: mmu_tlb_alloc() 1st argument ""size"" is too large!\r\n");
	} else if(((uint32_t)context % MMU_TLB_ALIGNMENT) != 0){
		printf("[C0] ERROR!!: mmu_tlb_alloc() 2st argument ""context"" is not aligned!\r\n");
	} else {
		p = context;
	}

	return p ;
}

void sample_mmu_init_and_enable(void)
{
	ALT_STATUS_CODE result_code;
	
	/* Usage:	ALT_STATUS_CODE  alt_mmu_init (void);	*/
	result_code = alt_mmu_init();
	if(result_code != ALT_E_SUCCESS){
		printf("[C0] ERROR!!: alt_mmu_init() .... result=%d\r\n", (int)result_code);
	}
	
	/* disable MMU */
	result_code = alt_mmu_disable();
	if(result_code != ALT_E_SUCCESS){
		printf("[C0] ERROR!!: alt_mmu_disable() .... result=%d\r\n", (int)result_code);
	}

	/* create MMU tables */
	/* Usage:	ALT_STATUS_CODE  alt_mmu_va_space_create (uint32_t **ttb1, const ALT_MMU_MEM_REGION_t *mem_regions, const size_t num_mem_regions, alt_mmu_ttb_alloc_t ttb_alloc, void *ttb_alloc_context);	*/
	result_code = alt_mmu_va_space_create(&mmu_tlb_l1, MemoryRegion, ValidMemoryRegion, sample_mmu_tlb_alloc, (void*)mmu_tlb_heap );
	if(result_code != ALT_E_SUCCESS){
		printf("[C0] ERROR!!: alt_mmu_va_space_create() .... result=%d\r\n", (int)result_code);
	}
	
	if(result_code == ALT_E_SUCCESS){
		/* enable MMU */
		/* Usage:	ALT_STATUS_CODE  alt_mmu_va_space_enable (const uint32_t *ttb1);	*/
		result_code = alt_mmu_va_space_enable(mmu_tlb_l1);
		if(result_code != ALT_E_SUCCESS){
			printf("[C0] ERROR!!: alt_mmu_va_space_enable() .... result=%d\r\n", (int)result_code);
		}
	}
	
	if(result_code != ALT_E_SUCCESS){
		printf("[C0] MMU Init error\r\n" );
	}
	
	return;
}

#ifdef soc_a10
static ALT_STATUS_CODE sample_a10_addr_space_setting(void)
{
	uint32_t addr_filt_end;

	/* Set the L2 filtering start address to 0. */
    addr_filt_end = (alt_read_word(ALT_L2_CACHE_ADDR_FILTERING_END_ADDR) &
                              ALT_L2_CACHE_ADDR_FILTERING_END_ADDR_MASK);
    return alt_l2_addr_filter_cfg_set(0x0, addr_filt_end);
}
#endif /* soc_a10 */

int cpu0_init(void)
{
	ALT_STATUS_CODE status = ALT_E_SUCCESS;
	
	// Enabling Advanced SIMD and VFP 
	enable_neon_and_vfp();
	
	// ACTLR.SMP=1 / NSACR.NS_SMP=1
	enable_actlr_smp();
	enable_nsacr_nssmp();
	
	// Initialize SCU (SCU is a shared resource in the MPCore. So, performed only by Core0.)
	SOCFPGA5XS1_BSP_SCU_CTLR	= 0x0;							/* Disable SCU */
	SOCFPGA5XS1_BSP_SCU_INV_WAY = 0xFFFF;						/* Invalidate SCU Tag RAM */
	SOCFPGA5XS1_BSP_SCU_CTLR	= 0x1;							/* Enable SCU */
	
#ifdef soc_cv_av
	// Setting signal of AXI transaction (AxUSER[0]=1). This setting necesary for coherent transfer via ACP.
    alt_write_word(ALT_ACPIDMAP_DYNRD_ADDR, ALT_ACPIDMAP_DYNRD_PAGE_SET(0) | ALT_ACPIDMAP_DYNRD_USER_SET(1));
    alt_write_word(ALT_ACPIDMAP_DYNWR_ADDR, ALT_ACPIDMAP_DYNWR_PAGE_SET(0) | ALT_ACPIDMAP_DYNWR_USER_SET(1));
#endif /* soc_cv_av */

	// initialize GIC Distributor registers
	alt_int_global_init();
	
	// Setting Global Timer for time measurement. 
	util_time_init();
	
#ifdef soc_cv_av
	//Remap Address Map for CPU1 boot. Need to set BootROM at 0x0.
	if (status == ALT_E_SUCCESS)
	{
		status = alt_addr_space_remap(ALT_ADDR_SPACE_MPU_ZERO_AT_BOOTROM,
										ALT_ADDR_SPACE_NONMPU_ZERO_AT_SDRAM,
										ALT_ADDR_SPACE_H2F_ACCESSIBLE,
										ALT_ADDR_SPACE_LWH2F_ACCESSIBLE);
	}

	// ** If you want to reset the CPU1 implemented here. (need to setting CPU1 start address)
    // - Set CPU1 start address
    // - Release a reset for CPU1

	status = alt_mpu_addr_space_remap_0_to_sdram();
	if(status != ALT_E_SUCCESS){
		printf("[C0] ERROR!!: alt_mpu_addr_space_remap_0_to_sdram() .... result=%d\r\n", (int)status);
	}
#else /* soc_cv_av */

	// ** If you want to reset the CPU1 implemented here. (need to setting CPU1 start address)
    // - Set CPU1 start address
    // - Release a reset for CPU1

	status = sample_a10_addr_space_setting();
	if(status != ALT_E_SUCCESS){
		printf("[C0] ERROR!!: sample_a10_addr_space_setting() .... result=%d\r\n", (int)status);
	}
#endif /* soc_cv_av */

    if(status == ALT_E_SUCCESS)
    {
    	alt_bridge_init(ALT_BRIDGE_LWH2F
#ifdef soc_cv_av
    		, NULL, NULL
#endif /* soc_cv_av */
    		);
    }

    if(status == ALT_E_SUCCESS)
    {
    	alt_bridge_init(ALT_BRIDGE_H2F
#ifdef soc_cv_av
    		, NULL, NULL
#endif /* soc_cv_av */
    		);
    }

    if(status == ALT_E_SUCCESS)
    {
    	alt_bridge_init(ALT_BRIDGE_F2S
#ifdef soc_cv_av
    		, NULL, NULL
#endif /* soc_cv_av */
    		);
    }

    if(status == ALT_E_SUCCESS)
    {
    	alt_bridge_init(ALT_BRIDGE_F2H
#ifdef soc_cv_av
    		, NULL, NULL
#endif /* soc_cv_av */
    		);
    }

#if 99	/* These are implemented on sample_mmu.c */
	// Setting MMU & Enable 
	sample_mmu_init_and_enable();
#endif	/* These are implemented on sample_mmu.c */
	
	// Cache Enable 
	alt_cache_system_enable();
	
	// initialize & enable CPU interrupt
	alt_int_cpu_init();
	alt_int_cpu_enable_all();
	// enable GIC global interrupt (Core0 only)
	alt_int_global_enable_all();
	__asm("CPSIE i");
	
	// Deassert GPIO reset : for switch input / and reset other peripherals
#ifdef soc_cv_av
	alt_clrbits_word(ALT_RSTMGR_PERMODRST_ADDR,
		(ALT_RSTMGR_PERMODRST_GPIO0_SET_MSK | ALT_RSTMGR_PERMODRST_GPIO1_SET_MSK | ALT_RSTMGR_PERMODRST_GPIO2_SET_MSK | \
		 ALT_RSTMGR_PERMODRST_L4WD0_SET_MSK | ALT_RSTMGR_PERMODRST_L4WD1_SET_MSK | \
		 ALT_RSTMGR_PERMODRST_OSC1TMR0_SET_MSK | ALT_RSTMGR_PERMODRST_OSC1TMR1_SET_MSK | \
		 ALT_RSTMGR_PERMODRST_SPTMR0_SET_MSK | ALT_RSTMGR_PERMODRST_SPTMR1_SET_MSK | \
		 ALT_RSTMGR_PERMODRST_DMA_SET_MSK ));
#endif /* soc_cv_av */
#ifdef soc_a10
	alt_clrbits_word(ALT_RSTMGR_PER1MODRST_ADDR,
		(ALT_RSTMGR_PER1MODRST_GPIO0_SET_MSK | ALT_RSTMGR_PER1MODRST_GPIO1_SET_MSK | ALT_RSTMGR_PER1MODRST_GPIO2_SET_MSK | \
		 ALT_RSTMGR_PER1MODRST_WD0_SET_MSK | ALT_RSTMGR_PER1MODRST_WD1_SET_MSK | \
		 ALT_RSTMGR_PER1MODRST_L4SYSTMR0_SET_MSK | ALT_RSTMGR_PER1MODRST_L4SYSTMR1_SET_MSK | \
		 ALT_RSTMGR_PER1MODRST_SPTMR0_SET_MSK | ALT_RSTMGR_PER1MODRST_SPTMR1_SET_MSK \
		 ));
	alt_clrbits_word(ALT_RSTMGR_PER0MODRST_ADDR,
		(ALT_RSTMGR_PER0MODRST_DMA_SET_MSK));
#endif /* soc_a10 */
	
	// Execute board setting.
	sample_config_switch();
	
	return 0;
}

#if 99
//
// Initializes and enables the interrupt controller.
//
ALT_STATUS_CODE socfpga_int_start(void)
{
    ALT_STATUS_CODE status = ALT_E_SUCCESS;

    //
    // Initialize the global and CPU interrupt items
    //

    if (status == ALT_E_SUCCESS)
    {
        status = alt_int_global_init();
        if (status != ALT_E_SUCCESS)
        {
            printf("ERROR: alt_int_global_init() failed, %d.\r\n", (int)status);
        }
    }

    if (status == ALT_E_SUCCESS)
    {
        status = alt_int_cpu_init();
        if (status != ALT_E_SUCCESS)
        {
            printf("ERROR: alt_int_cpu_init() failed, %d.\r\n", (int)status);
        }
    }

    //
    // Enable the CPU and global interrupt
    //

    if (status == ALT_E_SUCCESS)
    {
        status = alt_int_cpu_enable();
        if (status != ALT_E_SUCCESS)
        {
            printf("ERROR: alt_int_cpu_enable() failed, %d.\r\n", (int)status);
        }
    }

    if (status == ALT_E_SUCCESS)
    {
        status = alt_int_global_enable();
        if (status != ALT_E_SUCCESS)
        {
            printf("ERROR: alt_int_global_enable() failed, %d.\r\n", (int)status);
        }
    }

    return status;
}

void socfpga_int_stop(void)
{
    //
    // Disable the global, CPU, and distributor interrupt
    //

    if (alt_int_global_disable() != ALT_E_SUCCESS)
    {
        printf("WARN: alt_int_global_disable() return non-SUCCESS.\r\n");
    }

    if (alt_int_cpu_disable() != ALT_E_SUCCESS)
    {
        printf("WARN: alt_int_cpu_disable() return non-SUCCESS.\r\n");
    }

    //
    // Uninitialize the CPU and global data structures.
    //

    if (alt_int_cpu_uninit() != ALT_E_SUCCESS)
    {
        printf("WARN: alt_int_cpu_uninit() return non-SUCCESS.\r\n");
    }

    if (alt_int_global_uninit() != ALT_E_SUCCESS)
    {
        printf("WARN: alt_int_global_uninit() return non-SUCCESS.\r\n");
    }
}
#endif // 99

#if TARGET_BOARD_NO==1	/* 1:helio */
/********************************************************************************//*!
 * @fn  static void sample_gpio_config_helioboard(void) 
 * @brief  Configure the Helio Board GPIO settings. 
 * 
 * @details  Performs the following processing. 
 *           - Gets the contents of the current port settings to gpio_config_array. 
 *           - Global timer counter value of the measurement at the start
 *           - Setting DIPSW and PUSHSW via bit index configuration func. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/13  ---  #13144  新規作成. 
 **//*******************************************************************************/
static void sample_gpio_config_helioboard(void)
{
	ALT_STATUS_CODE result_code;
	ALT_GPIO_CONFIG_RECORD_t gpio_config_array[10];
	ALT_GPIO_CONFIG_RECORD_t* p_gpiocfg;

	// Setting OUT-PORT（GPIO61） via normal configuration func.
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_port_config(	ALT_GPIO_PORT_t gpio_pid,
	 * 												uint32_t mask,
	 * 												ALT_GPIO_PIN_DIR_t dir,
	 * 												ALT_GPIO_PIN_TYPE_t type,
	 * 												ALT_GPIO_PIN_POL_t pol,
	 * 												ALT_GPIO_PIN_DEBOUNCE_t debounc,
	 * 												uint32_t data); 						*/
	/****************************************************************************************/
	result_code = alt_gpio_port_config(ALT_GPIO_PORTC, ALT_GPIO_BIT3,
			ALT_GPIO_PIN_OUTPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_BIT3);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config() result=%d\n", (int)result_code);
	}

	// Gets the contents of the current port settings to gpio_config_array.
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_group_config_get(ALT_GPIO_CONFIG_RECORD_t* config_array, uint32_t len);	*/
	/****************************************************************************************/
	gpio_config_array[0].signal_number = ALT_GPIO_1BIT_61;
	gpio_config_array[1].signal_number = ALT_HLGPI_6;
	gpio_config_array[2].signal_number = ALT_HLGPI_7;
	gpio_config_array[3].signal_number = ALT_HLGPI_8;
	gpio_config_array[4].signal_number = ALT_HLGPI_9;
	gpio_config_array[5].signal_number = ALT_HLGPI_10;
	gpio_config_array[6].signal_number = ALT_HLGPI_11;
	gpio_config_array[7].signal_number = ALT_HLGPI_12;
	gpio_config_array[8].signal_number = ALT_HLGPI_13;
	gpio_config_array[9].signal_number = ALT_END_OF_GPIO_SIGNALS;
	result_code = alt_gpio_group_config_get(gpio_config_array, 10);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_group_config_get() result=%d\n", (int)result_code);
	}

	// Setting DIPSW and PUSHSW via bit index configuration func.
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_group_config(ALT_GPIO_CONFIG_RECORD_t* config_array, uint32_t len); */
	/****************************************************************************************/
	for(int i=1; i<10; i++)
	{
		p_gpiocfg = &gpio_config_array[i];
		p_gpiocfg->direction = ALT_GPIO_PIN_INPUT;
		p_gpiocfg->type = ALT_GPIO_PIN_LEVEL_TRIG_INT;
		p_gpiocfg->polarity = ALT_GPIO_PIN_ACTIVE_HIGH;
		p_gpiocfg->debounce = ALT_GPIO_PIN_NODEBOUNCE;
		p_gpiocfg->data = ALT_GPIO_PIN_DATAZERO;
	}
	result_code = alt_gpio_group_config(gpio_config_array, 10);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_group_config() result=%d\n", (int)result_code);
	}

	return;
}

static uint32_t sample_detect_switch_helioboard(void)
{
	uint32_t gpio_portc;
	uint32_t switch_result = 0;
	
	gpio_portc = alt_gpio_port_data_read(ALT_GPIO_PORTC, HELIO_GPIOPORTC_SWALL);
	gpio_portc = ~gpio_portc & HELIO_GPIOPORTC_SWALL;	// bit inversion and mask ({ON=0,OFF=1}->{ON=1,OFF=0})
	switch_result |= (gpio_portc & HELIO_GPIOPORTC_DIPSW0) ? SAMPLE_SWITCH_BIT_SLIDE0 : 0;
	switch_result |= (gpio_portc & HELIO_GPIOPORTC_DIPSW1) ? SAMPLE_SWITCH_BIT_SLIDE1 : 0;
	switch_result |= (gpio_portc & HELIO_GPIOPORTC_DIPSW2) ? SAMPLE_SWITCH_BIT_SLIDE2 : 0;
	switch_result |= (gpio_portc & HELIO_GPIOPORTC_DIPSW3) ? SAMPLE_SWITCH_BIT_SLIDE3 : 0;
	switch_result |= (gpio_portc & HELIO_GPIOPORTC_PUSHSW0) ? SAMPLE_SWITCH_BIT_PUSH0 : 0;
	switch_result |= (gpio_portc & HELIO_GPIOPORTC_PUSHSW1) ? SAMPLE_SWITCH_BIT_PUSH1 : 0;
	switch_result |= (gpio_portc & HELIO_GPIOPORTC_PUSHSW2) ? SAMPLE_SWITCH_BIT_PUSH2 : 0;
	switch_result |= (gpio_portc & HELIO_GPIOPORTC_PUSHSW3) ? SAMPLE_SWITCH_BIT_PUSH3 : 0;
	
	return switch_result;
}

#elif TARGET_BOARD_NO==2	/* 2:atlas */
/********************************************************************************//*!
 * @fn  static void sample_gpio_config_atlasboard(void) 
 * @brief  Configure the Atlas Board GPIO settings. 
 * 
 * @details  Performs the following processing. 
 *           - Gets the contents of the current port settings to gpio_config_array. 
 *           - Global timer counter value of the measurement at the start
 *           - Setting DIPSW and PUSHSW via bit index configuration func. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2016/06/20  ---  #xxxxx  新規作成. 
 **//*******************************************************************************/
static void sample_gpio_config_atlasboard(void)
{
	ALT_STATUS_CODE result_code;

	//HPS_KEY <--> HPS_GPIO54	: GPIO1-25 
	//HPS_LED <--> HPS_GPIO53 : GPIO1-24 
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_port_config(	ALT_GPIO_PORT_t gpio_pid,
	 * 												uint32_t mask,
	 * 												ALT_GPIO_PIN_DIR_t dir,
	 * 												ALT_GPIO_PIN_TYPE_t type,
	 * 												ALT_GPIO_PIN_POL_t pol,
	 * 												ALT_GPIO_PIN_DEBOUNCE_t debounc,
	 * 												uint32_t data); 						*/
	/****************************************************************************************/
	result_code = alt_gpio_port_config(ALT_GPIO_PORTB, ALT_GPIO_BIT25,
			ALT_GPIO_PIN_INPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_PIN_DATAZERO);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config(HPS_KEY) result=%d\n", (int)result_code);
	}
	
	result_code = alt_gpio_port_config(ALT_GPIO_PORTB, ALT_GPIO_BIT24,
			ALT_GPIO_PIN_OUTPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_BIT24);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config(HPS_LED) result=%d\n", (int)result_code);
	}

	return;
}

static uint32_t sample_detect_switch_atlasboard(void)
{
	uint32_t gpio_portb;
	uint32_t button_pio;
	uint32_t dipsw_pio;
	uint32_t switch_result;
	
	gpio_portb = alt_gpio_port_data_read(ALT_GPIO_PORTB, ATLAS_GPIOPORTB_PUSH2);
	gpio_portb = ~gpio_portb & ATLAS_GPIOPORTB_PUSH2;	// bit inversion and mask ({ON=0,OFF=1}->{ON=1,OFF=0})

	button_pio = IORD_ALTERA_AVALON_PIO_DATA(IOADDR_ALTERA_AVALON_PIO_DATA(BUTTON_PIO_BASE));
	dipsw_pio =  IORD_ALTERA_AVALON_PIO_DATA(IOADDR_ALTERA_AVALON_PIO_DATA(DIPSW_PIO_BASE));

	switch_result = dipsw_pio & 0x0000000F;
	if((~button_pio & ATLAS_AVALONPIO_PUSH0)&&(~button_pio & ATLAS_AVALONPIO_PUSH1)){
		switch_result |= SAMPLE_SWITCH_BIT_PUSH0;
	} else {
		switch_result |= (~button_pio & ATLAS_AVALONPIO_PUSH0) ? SAMPLE_SWITCH_BIT_PUSH1 : 0;
		switch_result |= (~button_pio & ATLAS_AVALONPIO_PUSH1) ? SAMPLE_SWITCH_BIT_PUSH2 : 0;
	}
	switch_result |= (gpio_portb & ATLAS_GPIOPORTB_PUSH2) ? SAMPLE_SWITCH_BIT_PUSH3 : 0;

	return switch_result;
}

#elif TARGET_BOARD_NO==3	/* 3:sodia */
/********************************************************************************//*!
 * @fn  static void sample_gpio_config_sodiaboard(void) 
 * @brief  Configure the Sodia Board GPIO settings. 
 * 
 * @details  Performs the following processing. 
 *           - Gets the contents of the current port settings to gpio_config_array. 
 *           - Global timer counter value of the measurement at the start
 *           - Setting DIPSW and PUSHSW via bit index configuration func. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2016/06/20  ---  #xxxxx  新規作成. 
 **//*******************************************************************************/
static void sample_gpio_config_sodiaboard(void)
{
	ALT_STATUS_CODE result_code;
	ALT_GPIO_CONFIG_RECORD_t gpio_config_array[9];
	ALT_GPIO_CONFIG_RECORD_t* p_gpiocfg;

	// Setting OUT-PORT（GPIO41,42,43,44） via normal configuration func.
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_port_config(	ALT_GPIO_PORT_t gpio_pid,
	 * 												uint32_t mask,
	 * 												ALT_GPIO_PIN_DIR_t dir,
	 * 												ALT_GPIO_PIN_TYPE_t type,
	 * 												ALT_GPIO_PIN_POL_t pol,
	 * 												ALT_GPIO_PIN_DEBOUNCE_t debounc,
	 * 												uint32_t data); 						*/
	/****************************************************************************************/
	result_code = alt_gpio_port_config(ALT_GPIO_PORTB, ALT_GPIO_BIT15,
			ALT_GPIO_PIN_OUTPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_BIT15);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config(PORTB-BIT15) result=%d\n", (int)result_code);
	}
	result_code = alt_gpio_port_config(ALT_GPIO_PORTB, ALT_GPIO_BIT14,
			ALT_GPIO_PIN_OUTPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_BIT14);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config(PORTB-BIT14) result=%d\n", (int)result_code);
	}
	result_code = alt_gpio_port_config(ALT_GPIO_PORTB, ALT_GPIO_BIT13,
			ALT_GPIO_PIN_OUTPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_BIT13);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config(PORTB-BIT13) result=%d\n", (int)result_code);
	}
	result_code = alt_gpio_port_config(ALT_GPIO_PORTB, ALT_GPIO_BIT12,
			ALT_GPIO_PIN_OUTPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_BIT12);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config(PORTB-BIT12) result=%d\n", (int)result_code);
	}

	// Gets the contents of the current port settings to gpio_config_array.
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_group_config_get(ALT_GPIO_CONFIG_RECORD_t* config_array, uint32_t len);	*/
	/****************************************************************************************/
	gpio_config_array[0].signal_number = ALT_HLGPI_4;
	gpio_config_array[1].signal_number = ALT_HLGPI_5;
	gpio_config_array[2].signal_number = ALT_HLGPI_6;
	gpio_config_array[3].signal_number = ALT_HLGPI_7;
	gpio_config_array[4].signal_number = ALT_HLGPI_8;
	gpio_config_array[5].signal_number = ALT_HLGPI_9;
	gpio_config_array[6].signal_number = ALT_HLGPI_10;
	gpio_config_array[7].signal_number = ALT_HLGPI_11;
	gpio_config_array[8].signal_number = ALT_END_OF_GPIO_SIGNALS;
	result_code = alt_gpio_group_config_get(gpio_config_array, 9);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_group_config_get() result=%d\n", (int)result_code);
	}

	// Setting DIPSW and PUSHSW via bit index configuration func.
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_group_config(ALT_GPIO_CONFIG_RECORD_t* config_array, uint32_t len); */
	/****************************************************************************************/
	for(int i=0; i<9; i++)
	{
		p_gpiocfg = &gpio_config_array[i];
		p_gpiocfg->direction = ALT_GPIO_PIN_INPUT;
		p_gpiocfg->type = ALT_GPIO_PIN_LEVEL_TRIG_INT;
		p_gpiocfg->polarity = ALT_GPIO_PIN_ACTIVE_HIGH;
		p_gpiocfg->debounce = ALT_GPIO_PIN_NODEBOUNCE;
		p_gpiocfg->data = ALT_GPIO_PIN_DATAZERO;
	}
	result_code = alt_gpio_group_config(gpio_config_array, 9);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_group_config() result=%d\n", (int)result_code);
	}

	return;
}

static uint32_t sample_detect_switch_sodiaboard(void)
{
	uint32_t gpio_portc;
	uint32_t switch_result;
	
	gpio_portc = alt_gpio_port_data_read(ALT_GPIO_PORTC, SODIA_GPIOPORTC_SWALL);
	gpio_portc = ~gpio_portc & SODIA_GPIOPORTC_SWALL;	// bit inversion and mask ({ON=0,OFF=1}->{ON=1,OFF=0})
	switch_result = (uint32_t)((gpio_portc & SODIA_GPIOPORTC_DIPSWALL) >> SODIA_GPIOPORTC_DIPSWLSB);
	switch_result |= (gpio_portc & SODIA_GPIOPORTC_PUSHSW0) ? SAMPLE_SWITCH_BIT_PUSH0 : 0;
	switch_result |= (gpio_portc & SODIA_GPIOPORTC_PUSHSW1) ? SAMPLE_SWITCH_BIT_PUSH1 : 0;
	switch_result |= (gpio_portc & SODIA_GPIOPORTC_PUSHSW2) ? SAMPLE_SWITCH_BIT_PUSH2 : 0;
	switch_result |= (gpio_portc & SODIA_GPIOPORTC_PUSHSW3) ? SAMPLE_SWITCH_BIT_PUSH3 : 0;
	
	return switch_result;
}

#elif TARGET_BOARD_NO==4	/* 4:c5socdk */
/********************************************************************************//*!
 * @fn  static void sample_gpio_config_c5socdkboard(void) 
 * @brief  Configure the Cyclone V SoC Development Kit GPIO settings. 
 * 
 * @details  Performs the following processing. 
 *           - Gets the contents of the current port settings to gpio_config_array. 
 *           - Global timer counter value of the measurement at the start
 *           - Setting DIPSW and PUSHSW via bit index configuration func. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2016/06/20  ---  #xxxxx  新規作成. 
 **//*******************************************************************************/
static void sample_gpio_config_c5socdkboard(void)
{
	ALT_STATUS_CODE result_code;
	ALT_GPIO_CONFIG_RECORD_t gpio_config_array[9];
	ALT_GPIO_CONFIG_RECORD_t* p_gpiocfg;

	// Setting OUT-PORT（GPIO41,42,43,44） via normal configuration func.
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_port_config(	ALT_GPIO_PORT_t gpio_pid,
	 * 												uint32_t mask,
	 * 												ALT_GPIO_PIN_DIR_t dir,
	 * 												ALT_GPIO_PIN_TYPE_t type,
	 * 												ALT_GPIO_PIN_POL_t pol,
	 * 												ALT_GPIO_PIN_DEBOUNCE_t debounc,
	 * 												uint32_t data); 						*/
	/****************************************************************************************/
	result_code = alt_gpio_port_config(ALT_GPIO_PORTB, ALT_GPIO_BIT15,
			ALT_GPIO_PIN_OUTPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_BIT15);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config(PORTB-BIT15) result=%d\n", (int)result_code);
	}
	result_code = alt_gpio_port_config(ALT_GPIO_PORTB, ALT_GPIO_BIT14,
			ALT_GPIO_PIN_OUTPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_BIT14);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config(PORTB-BIT14) result=%d\n", (int)result_code);
	}
	result_code = alt_gpio_port_config(ALT_GPIO_PORTB, ALT_GPIO_BIT13,
			ALT_GPIO_PIN_OUTPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_BIT13);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config(PORTB-BIT13) result=%d\n", (int)result_code);
	}
	result_code = alt_gpio_port_config(ALT_GPIO_PORTB, ALT_GPIO_BIT12,
			ALT_GPIO_PIN_OUTPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_BIT12);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config(PORTB-BIT12) result=%d\n", (int)result_code);
	}

	// Gets the contents of the current port settings to gpio_config_array.
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_group_config_get(ALT_GPIO_CONFIG_RECORD_t* config_array, uint32_t len);	*/
	/****************************************************************************************/
	gpio_config_array[0].signal_number = ALT_HLGPI_4;
	gpio_config_array[1].signal_number = ALT_HLGPI_5;
	gpio_config_array[2].signal_number = ALT_HLGPI_6;
	gpio_config_array[3].signal_number = ALT_HLGPI_7;
	gpio_config_array[4].signal_number = ALT_HLGPI_8;
	gpio_config_array[5].signal_number = ALT_HLGPI_9;
	gpio_config_array[6].signal_number = ALT_HLGPI_10;
	gpio_config_array[7].signal_number = ALT_HLGPI_11;
	gpio_config_array[8].signal_number = ALT_END_OF_GPIO_SIGNALS;
	result_code = alt_gpio_group_config_get(gpio_config_array, 9);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_group_config_get() result=%d\n", (int)result_code);
	}

	// Setting DIPSW and PUSHSW via bit index configuration func.
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_group_config(ALT_GPIO_CONFIG_RECORD_t* config_array, uint32_t len); */
	/****************************************************************************************/
	for(int i=0; i<9; i++)
	{
		p_gpiocfg = &gpio_config_array[i];
		p_gpiocfg->direction = ALT_GPIO_PIN_INPUT;
		p_gpiocfg->type = ALT_GPIO_PIN_LEVEL_TRIG_INT;
		p_gpiocfg->polarity = ALT_GPIO_PIN_ACTIVE_HIGH;
		p_gpiocfg->debounce = ALT_GPIO_PIN_NODEBOUNCE;
		p_gpiocfg->data = ALT_GPIO_PIN_DATAZERO;
	}
	result_code = alt_gpio_group_config(gpio_config_array, 9);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_group_config() result=%d\n", (int)result_code);
	}

	return;
}

static uint32_t sample_detect_switch_c5socdkboard(void)
{
	uint32_t gpio_portc;
	uint32_t switch_result = 0;
	
	gpio_portc = alt_gpio_port_data_read(ALT_GPIO_PORTC, C5SOCDK_GPIOPORTC_SWALL);
	gpio_portc = ~gpio_portc & C5SOCDK_GPIOPORTC_SWALL;	// bit inversion and mask ({ON=0,OFF=1}->{ON=1,OFF=0})

	switch_result |= (gpio_portc & C5SOCDK_GPIOPORTC_DIPSW0) ? SAMPLE_SWITCH_BIT_SLIDE0 : 0;
	switch_result |= (gpio_portc & C5SOCDK_GPIOPORTC_DIPSW1) ? SAMPLE_SWITCH_BIT_SLIDE1 : 0;
	switch_result |= (gpio_portc & C5SOCDK_GPIOPORTC_DIPSW2) ? SAMPLE_SWITCH_BIT_SLIDE2 : 0;
	switch_result |= (gpio_portc & C5SOCDK_GPIOPORTC_DIPSW3) ? SAMPLE_SWITCH_BIT_SLIDE3 : 0;
	switch_result |= (gpio_portc & C5SOCDK_GPIOPORTC_PUSHSW0) ? SAMPLE_SWITCH_BIT_PUSH0 : 0;
	switch_result |= (gpio_portc & C5SOCDK_GPIOPORTC_PUSHSW1) ? SAMPLE_SWITCH_BIT_PUSH1 : 0;
	switch_result |= (gpio_portc & C5SOCDK_GPIOPORTC_PUSHSW2) ? SAMPLE_SWITCH_BIT_PUSH2 : 0;
	switch_result |= (gpio_portc & C5SOCDK_GPIOPORTC_PUSHSW3) ? SAMPLE_SWITCH_BIT_PUSH3 : 0;
	
	return switch_result;
}
#elif TARGET_BOARD_NO==5	/* 5:a10socdk */
/********************************************************************************//*!
 * @fn  static void sample_gpio_config_a10socdkboard(void) 
 * @brief  Configure the Arria 10 SoC Development Kit GPIO settings. 
 * 
 * @details  Performs the following processing. 
 *           - Gets the contents of the current port settings to gpio_config_array. 
 *           - Global timer counter value of the measurement at the start
 *           - Setting DIPSW and PUSHSW via bit index configuration func. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2017/04/14  ---  #xxxxx  新規作成. 
 **//*******************************************************************************/
static void sample_gpio_config_a10socdkboard(void)
{
	return;
}

static uint32_t sample_detect_switch_a10socdkboard(void)
{
	uint32_t button_pio;
	uint32_t dipsw_pio;
	uint32_t switch_result;
	
	button_pio = IORD_ALTERA_AVALON_PIO_DATA(IOADDR_ALTERA_AVALON_PIO_DATA(BUTTON_PIO_BASE));
	dipsw_pio =  IORD_ALTERA_AVALON_PIO_DATA(IOADDR_ALTERA_AVALON_PIO_DATA(DIPSW_PIO_BASE));

	switch_result = ~dipsw_pio & 0x0000000F;
	switch_result |= (~button_pio & A10SOCDK_AVALONPIO_PUSHALL) << 4;

	return switch_result;
}
#elif TARGET_BOARD_NO==6	/* 6:de10nano */
/********************************************************************************//*!
 * @fn  static void sample_gpio_config_de10nanoboard(void) 
 * @brief  Configure the DE10-Nano Board GPIO settings. 
 * 
 * @details  Performs the following processing. 
 *           - Gets the contents of the current port settings to gpio_config_array. 
 *           - Global timer counter value of the measurement at the start
 *           - Setting DIPSW and PUSHSW via bit index configuration func. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2018/06/29  ---  #xxxxx  新規作成. 
 **//*******************************************************************************/
/* Same as the Atlas Board. */
static void sample_gpio_config_de10nanoboard(void)
{
	ALT_STATUS_CODE result_code;

	//HPS_KEY <--> HPS_GPIO54 : GPIO1-25 
	//HPS_LED <--> HPS_GPIO53 : GPIO1-24 
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_gpio_port_config(	ALT_GPIO_PORT_t gpio_pid,
	 * 												uint32_t mask,
	 * 												ALT_GPIO_PIN_DIR_t dir,
	 * 												ALT_GPIO_PIN_TYPE_t type,
	 * 												ALT_GPIO_PIN_POL_t pol,
	 * 												ALT_GPIO_PIN_DEBOUNCE_t debounc,
	 * 												uint32_t data); 						*/
	/****************************************************************************************/
	result_code = alt_gpio_port_config(ALT_GPIO_PORTB, ALT_GPIO_BIT25,
			ALT_GPIO_PIN_INPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_PIN_DATAZERO);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config(HPS_KEY) result=%d\n", (int)result_code);
	}
	
	result_code = alt_gpio_port_config(ALT_GPIO_PORTB, ALT_GPIO_BIT24,
			ALT_GPIO_PIN_OUTPUT, ALT_GPIO_PIN_LEVEL_TRIG_INT, ALT_GPIO_PIN_ACTIVE_HIGH, ALT_GPIO_PIN_NODEBOUNCE,
			ALT_GPIO_BIT24);
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_gpio_port_config(HPS_LED) result=%d\n", (int)result_code);
	}

	return;
}

/* Same as the Atlas Board. */
static uint32_t sample_detect_switch_de10nanoboard(void)
{
	uint32_t gpio_portb;
	uint32_t button_pio;
	uint32_t dipsw_pio;
	uint32_t switch_result;
	
	gpio_portb = alt_gpio_port_data_read(ALT_GPIO_PORTB, ATLAS_GPIOPORTB_PUSH2);
	gpio_portb = ~gpio_portb & ATLAS_GPIOPORTB_PUSH2;	// bit inversion and mask ({ON=0,OFF=1}->{ON=1,OFF=0})

	button_pio = IORD_ALTERA_AVALON_PIO_DATA(IOADDR_ALTERA_AVALON_PIO_DATA(BUTTON_PIO_BASE));
	dipsw_pio =  IORD_ALTERA_AVALON_PIO_DATA(IOADDR_ALTERA_AVALON_PIO_DATA(DIPSW_PIO_BASE));

	switch_result = dipsw_pio & 0x0000000F;
	if((~button_pio & ATLAS_AVALONPIO_PUSH0)&&(~button_pio & ATLAS_AVALONPIO_PUSH1)){
		switch_result |= SAMPLE_SWITCH_BIT_PUSH0;
	} else {
		switch_result |= (~button_pio & ATLAS_AVALONPIO_PUSH0) ? SAMPLE_SWITCH_BIT_PUSH1 : 0;
		switch_result |= (~button_pio & ATLAS_AVALONPIO_PUSH1) ? SAMPLE_SWITCH_BIT_PUSH2 : 0;
	}
	switch_result |= (gpio_portb & ATLAS_GPIOPORTB_PUSH2) ? SAMPLE_SWITCH_BIT_PUSH3 : 0;

	return switch_result;
}
#endif	/* TARGET_BOARD_NO */

void sample_config_switch(void)
{
#if TARGET_BOARD_NO==1	/* 1:helio */
	sample_gpio_config_helioboard();
#elif TARGET_BOARD_NO==2	/* 2:atlas */
	sample_gpio_config_atlasboard();
#elif TARGET_BOARD_NO==3	/* 3:sodia */
	sample_gpio_config_sodiaboard();
#elif TARGET_BOARD_NO==4	/* 4:c5socdk */
	sample_gpio_config_c5socdkboard();
#elif TARGET_BOARD_NO==5	/* 5:a10socdk */
	sample_gpio_config_a10socdkboard();
#elif TARGET_BOARD_NO==6	/* 6:de10nano */
	sample_gpio_config_de10nanoboard();
#endif	/* TARGET_BOARD_NO */
	return;
}

uint32_t sample_detect_switch(void)
{
	uint32_t switch_result = 0;
#if TARGET_BOARD_NO==1	/* 1:helio */
	switch_result = sample_detect_switch_helioboard();
#elif TARGET_BOARD_NO==2	/* 2:atlas */
	switch_result = sample_detect_switch_atlasboard();
#elif TARGET_BOARD_NO==3	/* 3:sodia */
	switch_result = sample_detect_switch_sodiaboard();
#elif TARGET_BOARD_NO==4	/* 4:c5socdk */
	switch_result = sample_detect_switch_c5socdkboard();
#elif TARGET_BOARD_NO==5	/* 5:a10socdk */
	switch_result = sample_detect_switch_a10socdkboard();
#elif TARGET_BOARD_NO==6	/* 6:de10nano */
	switch_result = sample_detect_switch_de10nanoboard();
#endif	/* TARGET_BOARD_NO */
	return switch_result;
}


/***********************************************************************************
 * end of file 
 ***********************************************************************************/



