/******************************************************************************
 *
 * Copyright 2014 Altera Corporation. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of the author may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 ******************************************************************************/

/*
 * This file contains minimal retarget support to make printf work over UART for the ARMCC compiler
 */

#include <stdio.h>
#include <string.h>
#include <hwlib.h>
#include "alt_printf.h"
#include <rt_sys.h>

#ifdef __cplusplus
extern "C" {
#endif

#define DEFAULT_HANDLE 0x100
#define STDIN_FILENO  0
#define STDOUT_FILENO 1

/*
 * These names are special strings which will be recognized by 
 * _sys_open and will cause it to return the standard I/O handles, instead
 * of opening a real file.
 */
const char __stdin_name[] ="STDIN";
const char __stdout_name[]="STDOUT";
const char __stderr_name[]="STDERR";

#define UART_MAX_LEN 128
char log_buf[UART_MAX_LEN];

/*
 * Open a file. May return -1 if the file failed to open. We do not require
 * this function to do anything. Simply return a dummy handle.
 */
FILEHANDLE _sys_open(const char * name, int openmode)
{
	if(name == NULL)
	{
		return -1;
		
	} else if(strcmp(name, __stdin_name) == 0)
	{
		return STDIN_FILENO;
		
	} else if(strcmp(name, __stdout_name) == 0)
	{
		return STDOUT_FILENO;
	}
	
    return DEFAULT_HANDLE;
}

/*
 * Close a file. Should return 0 on success or a negative value on error.
 * Not required in this implementation. Always return success.
 */
int _sys_close(FILEHANDLE fh)
{
    return 0; //return success
}

/*
 * Write to a file. Returns 0 on success, negative on error, and the number
 * of characters _not_ written on partial success. This implementation sends
 * a buffer of size 'len' to the UART.
 */
int _sys_write(FILEHANDLE fh, const unsigned char * ptr,
               unsigned len, int mode)
{
    unsigned max_len = len;
    unsigned return_value = 0;

    if (max_len > UART_MAX_LEN)
    {
        max_len = UART_MAX_LEN;
        return_value = len - max_len;
    }

    snprintf(log_buf, max_len+1, (const char *) ptr);

    alt_printf(log_buf);
    
    return return_value;   
}

/*
 * Read from a file. Can return:
 * Returns 0 for now, not implemented. 
 */
int _sys_read(FILEHANDLE fh, unsigned char * buf,
              unsigned len, int mode)
{
	char *ptr = (char*)buf;
	char ch;
	int index = 0;
	int charlimit = len - 2;
    ALT_PRINTF_MEM_INFO_t *pinfo = (ALT_PRINTF_MEM_INFO_t *)DEFAULT_TERM;
	
	/* Fails if length less than 2 */
	if(len < 2)
	{
		return -1;
	}
	len = (len > 128) ? 128 : len;
	charlimit = len - 2;
	
	/* Print each character to UART */
	while(index < charlimit)
	{
		ch = pinfo->getc_function((FILE *)pinfo);
		
		if((ch >= ' ')&&('~' >= ch)){
			ptr[index++] = ch;
			/* Echo back */
			pinfo->putc_function(ch, (FILE *)pinfo);
		} else if(ch == '\b'){
			if(index > 0){
				ptr[index--] = '\0';
		    	/* Echo back */
				pinfo->putc_function('\b', (FILE *)pinfo);
				pinfo->putc_function(' ', (FILE *)pinfo);
				pinfo->putc_function('\b', (FILE *)pinfo);
			}
		} else if((ch == '\r')||(ch == '\n')){
			ptr[index++] = '\r';
			ptr[index++] = '\n';
			ptr[index] = '\0';
			pinfo->putc_function('\n', (FILE *)pinfo);
			break;			/* input finish */
		} else if(ch == '\x03'){
			index = 0;
			ptr[index++] = ch;
			ptr[index++] = '\n';
			ptr[index] = '\0';
			pinfo->putc_function('\n', (FILE *)pinfo);
			break;			/* input cancel */
		} else {
			ptr[index++] = ' ';
			/* Echo back */
			pinfo->putc_function(ch, (FILE *)pinfo);
		}
	}
	
	return len - index;
}

/*
 * Writes a character to the output channel. This function is used
 * for last-resort error message output.
 */
void _ttywrch(int ch)
{
    // Convert correctly for endianness change
    char ench=ch;
    
    alt_printf(&ench);
}

/*
 * Return non-zero if the argument file is connected to a terminal.
 */
int _sys_istty(FILEHANDLE fh)
{
    return 0; //connecting to a UART buffered output
}

/*
 * Move the file position to a given offset from the file start.
 * Returns >=0 on success, <0 on failure. Seeking is not supported for the 
 * UART.
 */
int _sys_seek(FILEHANDLE fh, long pos)
{
    return -1; // error
}

/*
 * Flush any OS buffers associated with fh, ensuring that the file
 * is up to date on disk. Result is >=0 if OK, negative for an
 * error.
 */
int _sys_ensure(FILEHANDLE fh)
{
    return 0; // success
}

/*
 * Return the current length of a file, or <0 if an error occurred.
 * _sys_flen is allowed to reposition the file pointer (so Unix can
 * implement it with a single lseek, for example), since it is only
 * called when processing SEEK_END relative fseeks, and therefore a
 * call to _sys_flen is always followed by a call to _sys_seek.
 */
long _sys_flen(FILEHANDLE fh)
{
    return 0;
}

/*
 * Return the name for temporary file number sig in the buffer
 * name. Returns 0 on failure. maxlen is the maximum name length
 * allowed.
 */
int _sys_tmpnam(char * name, int sig, unsigned maxlen)
{
    return 0; // fail, not supported
}

/*
 * Terminate the program, passing a return code back to the user.
 * This function may not return.
 */
void _sys_exit(int returncode)
{
    while(1) {};
}

#ifdef __cplusplus
}
#endif
