﻿/********************************************************************************//*!
 * @file  sample_app_setting.c
 * @brief  L2 MMU initial setting of the HPS and the Board.
 * 
 * @details  L2 MMU initial setting of the HPS and the Board.
 * 
 * @note  nothing. 
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes
 ***********************************************************************************/
#include "l2mmu_setting.h"

#include <stdio.h>
//HWLib Headers
#include "alt_address_space.h"
#include "alt_interrupt.h"
#include "alt_system_manager.h"
#include "alt_mmu.h"
#include "alt_cache.h"
#include "hwlib.h"
#include "socal/socal.h"
#include "socal/hps.h"
#include "socal/alt_sysmgr.h"
#include "socal/alt_rstmgr.h"
#include "socal/alt_acpidmap.h"


/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/
// definition for enable_neon_and_vfp() 
#define ALT_CPU_CPACR_ASEDIS_SET_MSK	((unsigned)1 << 31)
#define ALT_CPU_CPACR_D32DIS_SET_MSK	(1 << 30)
#define ALT_CPU_CPACR_CP11_SET_MSK		(3 << 22)		/* 11b: Full Access */
#define ALT_CPU_CPACR_CP10_SET_MSK		(3 << 20)		/* 11b: Full Access */

// Auxiliary Control Register. See Cortex-A9, section 4.3.10.
#define ALT_CPU_ACTLR_SMP_SET_MSK                (1 << 6)
// Non-secure Access Control Register. See Cortex-A9, section 4.3.13.
#define ALT_CPU_NSACR_NSSMP_SET_MSK              (1 << 18)

// SCU functions
#define SOCFPGA5XS1_BSP_SCU_BASE		(unsigned int)(0xFFFEC000)
#define SOCFPGA5XS1_BSP_SCU_CTLR		(*(volatile unsigned int *)(SOCFPGA5XS1_BSP_SCU_BASE + 0x0))
#define SOCFPGA5XS1_BSP_SCU_INV_WAY		(*(volatile unsigned int *)(SOCFPGA5XS1_BSP_SCU_BASE + 0xC))

/* MMU table buffer size */
#define	MMU_TLB_HEAP_SIZE	( 32 * _KB )

/* TLB memory should be aligned 16KB */
#define	MMU_TLB_ALIGNMENT		(16*_KB)

/******** Definition Memory Region ********/
#define MAX_MEMORY_REGION_NUM	16

/***********************************************************************************
 *  variables
 ***********************************************************************************/
//typedef struct ALT_MMU_MEM_REGION_s
//{
//  void *                 va;        /*!< The beginning virtual address for the memory
//                                     *   region. The address must be aligned to one of 4KiB,
//                                     *   64KiB, 1MiB, or 16MiB boundaries.
//                                     */
//  void *                 pa;        /*!< The beginning physical address mapping for the
//                                     *   virtual address of the memory region. The address
//                                     *   must be aligned to one of 4KiB, 64KiB, 1MiB, or 16MiB
//                                     *   boundaries.
//                                     */
//  uint32_t               size;      /*!< The size of the memory region in bytes. The size
//                                     *   must be a multiple of 4KiB, 64KiB, 1MiB, or 16MiB
//                                     *   sizes.
//                                     */
//  ALT_MMU_AP_t           access;    /*!< The access permissions for the memory region. */
//  ALT_MMU_ATTR_t         attributes;/*!< The memory region attributes. These attributes
//                                     *   determine the memory type (ordering), cache
//                                     *   policy, and as a possible side effect, the
//                                     *   shareablity of the memory region.
//                                     */
//  ALT_MMU_TTB_S_t        shareable; /*!< The shareability of the memory region. */
//  ALT_MMU_TTB_XN_t       execute;   /*!< Whether instructions can be executed from this
//                                     *   memory region.
//                                     */
//  ALT_MMU_TTB_NS_t       security;  /*!< Controls whether address translations made from
//                                     *   the secure state translate physical address in
//                                     *   the secure or non-secure address map.
//                                     */
//} ALT_MMU_MEM_REGION_t;
static int	ValidMemoryRegion = 13;		/*!< The number of valid descriptors. */

static ALT_MMU_MEM_REGION_t MemoryRegion[MAX_MEMORY_REGION_NUM] = {
		/* [0] SDRAM1	*/
		{ (void*)SDRAM1_VIRTUAL_ADDR, (void*)SDRAM1_PHYSICAL_ADDR, SDRAM1_SIZE,	/* va, pa, size, */
				ALT_MMU_AP_FULL_ACCESS,		/* access, */
				ALT_MMU_ATTR_WBA,				/* attributes, */
				ALT_MMU_TTB_S_SHAREABLE,		/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
				ALT_MMU_TTB_XN_DISABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [1] SDRAM2	*/
		{ (void*)SDRAM2_VIRTUAL_ADDR, (void*)SDRAM2_PHYSICAL_ADDR, SDRAM2_SIZE,	/* va, pa, size, */
				ALT_MMU_AP_FULL_ACCESS,		/* access, */
				ALT_MMU_ATTR_WBA,				/* attributes, */
				ALT_MMU_TTB_S_SHAREABLE,		/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
				ALT_MMU_TTB_XN_DISABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [2] SDRAM3	*/
		{ (void*)SDRAM3_VIRTUAL_ADDR, (void*)SDRAM3_PHYSICAL_ADDR, SDRAM3_SIZE,	/* va, pa, size, */
				ALT_MMU_AP_FULL_ACCESS,		/* access, */
				ALT_MMU_ATTR_WBA,				/* attributes, */
				ALT_MMU_TTB_S_SHAREABLE,		/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
				ALT_MMU_TTB_XN_DISABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [3] SDRAM4	*/
		{ (void*)SDRAM4_VIRTUAL_ADDR, (void*)SDRAM4_PHYSICAL_ADDR, SDRAM4_SIZE,	/* va, pa, size, */
				ALT_MMU_AP_FULL_ACCESS,		/* access, */
				ALT_MMU_ATTR_WBA,				/* attributes, */
				ALT_MMU_TTB_S_SHAREABLE,		/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
				ALT_MMU_TTB_XN_DISABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [4] SDRAM5	*/
		{ (void*)SDRAM5_VIRTUAL_ADDR, (void*)SDRAM5_PHYSICAL_ADDR, SDRAM5_SIZE,	/* va, pa, size, */
				ALT_MMU_AP_FULL_ACCESS,		/* access, */
				ALT_MMU_ATTR_WBA,				/* attributes, */
				ALT_MMU_TTB_S_SHAREABLE,		/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
				ALT_MMU_TTB_XN_DISABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [5] ACPWINDOW	*/
		{ (void*)ACPWINDOW_VIRTUAL_ADDR, (void*)ACPWINDOW_PHYSICAL_ADDR, ACPWINDOW_SIZE,	/* va, pa, size, */
				ALT_MMU_AP_FULL_ACCESS,		/* access, */
				ALT_MMU_ATTR_NC,				/* attributes, */
				ALT_MMU_TTB_S_SHAREABLE,		/* shareable, *//* This value must be match with TTBR0.S if TLB on this area */
				ALT_MMU_TTB_XN_DISABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [6] HPS2FPGA	*/
		{ (void*)HPS2FPGA_VIRTUAL_ADDR, (void*)HPS2FPGA_PHYSICAL_ADDR, HPS2FPGA_SIZE,	/* va, pa, size, */
				ALT_MMU_AP_FULL_ACCESS,		/* access, */
				ALT_MMU_ATTR_DEVICE,			/* attributes, */
				ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
				ALT_MMU_TTB_XN_DISABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [7] LW_HPS2FPGA	*/
		{ (void*)LW_HPS2FPGA_VIRTUAL_ADDR, (void*)LW_HPS2FPGA_PHYSICAL_ADDR, LW_HPS2FPGA_SIZE,	/* va, pa, size, */
				ALT_MMU_AP_FULL_ACCESS,		/* access, */
				ALT_MMU_ATTR_DEVICE,			/* attributes, */
				ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
				ALT_MMU_TTB_XN_ENABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [8] PERIPHRALS	*/
		{ (void*)PERIPHRALS_VIRTUAL_ADDR, (void*)PERIPHRALS_PHYSICAL_ADDR, PERIPHRALS_SIZE,	/* va, pa, size, */
				ALT_MMU_AP_FULL_ACCESS,		/* access, */
				ALT_MMU_ATTR_STRONG,			/* attributes, */
				ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
				ALT_MMU_TTB_XN_ENABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [9] BOOTROM	*/
		{ (void*)BOOTROM_VIRTUAL_ADDR, (void*)BOOTROM_PHYSICAL_ADDR, BOOTROM_SIZE,	/* va, pa, size, */
				ALT_MMU_AP_READ_ONLY,			/* access, */
				ALT_MMU_ATTR_WBA,				/* attributes, */
				ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
				ALT_MMU_TTB_XN_DISABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [10] MPU_SCU	*/
		{ (void*)MPUSCU_VIRTUAL_ADDR, (void*)MPUSCU_PHYSICAL_ADDR, MPUSCU_SIZE,	/* va, pa, size, */
				ALT_MMU_AP_FULL_ACCESS,		/* access, */
				ALT_MMU_ATTR_STRONG,			/* attributes, */
				ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, */
				ALT_MMU_TTB_XN_ENABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [11] MPU_L2	*/
		{ (void*)MPUL2_VIRTUAL_ADDR, (void*)MPUL2_PHYSICAL_ADDR, MPUL2_SIZE,	/* va, pa, size, */
				ALT_MMU_AP_FULL_ACCESS,		/* access, */
				ALT_MMU_ATTR_STRONG,			/* attributes, */
				ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, */
				ALT_MMU_TTB_XN_ENABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [12] RAM	*/
		{ (void*)RAM_VIRTUAL_ADDR, (void*)RAM_PHYSICAL_ADDR, RAM_SIZE,	/* va, pa, size, */
				ALT_MMU_AP_FULL_ACCESS,		/* access, */
				ALT_MMU_ATTR_WBA,				/* attributes, */
				ALT_MMU_TTB_S_SHAREABLE,		/* shareable, */
				ALT_MMU_TTB_XN_DISABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [13] Undefined */
		{ 0, 0, 0,	/* va, pa, size, */
				ALT_MMU_AP_NO_ACCESS,			/* access, */
				ALT_MMU_ATTR_FAULT,			/* attributes, */
				ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, */
				ALT_MMU_TTB_XN_DISABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [14] Undefined */
		{ 0, 0, 0,	/* va, pa, size, */
				ALT_MMU_AP_NO_ACCESS,			/* access, */
				ALT_MMU_ATTR_FAULT,			/* attributes, */
				ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, */
				ALT_MMU_TTB_XN_DISABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		},
		/* [15] Undefined */
		{ 0, 0, 0,	/* va, pa, size, */
				ALT_MMU_AP_NO_ACCESS,			/* access, */
				ALT_MMU_ATTR_FAULT,			/* attributes, */
				ALT_MMU_TTB_S_NON_SHAREABLE,	/* shareable, */
				ALT_MMU_TTB_XN_DISABLE,		/* execute, */
				ALT_MMU_TTB_NS_SECURE			/* security */
		}
};

uint32_t mmu_tlb_heap[MMU_TLB_HEAP_SIZE] __attribute__ ((aligned MMU_TLB_ALIGNMENT)) ;
uint32_t *mmu_tlb_l1 = NULL;

/***********************************************************************************
 *  functions 
 ***********************************************************************************/
static void enable_neon_and_vfp(void)
{
	uint32_t regdata;

	// Refer to: Cortex(TM)-A9 NEON(TM) Media Processing Engine TRM.
	// "Example 2-1 Enabling Advanced SIMD and VFP"
	// --------------------------------------------------------------------
	// MRC p15,0,r0,c1,c0,2 ; Read CPACR into r0
	// ORR r0,r0,#(3<<20) ; OR in User and Privileged access for CP10
	// ORR r0,r0,#(3<<22) ; OR in User and Privileged access for CP11
	// BIC r0, r0, #(3<<30) ; Clear ASEDIS/D32DIS if set
	// MCR p15,0,r0,c1,c0,2 ; Store new access permissions into CPACR
	// ISB ; Ensure side-effect of CPACR is visible
	// MOV r0,#(1<<30) ; Create value with FPEXC (bit 30) set in r0
	// VMSR FPEXC,r0 ; Enable VFP and SIMD extensions
	// --------------------------------------------------------------------

	// MRC p15, 0,<Rd>, c1, c0, 2; Read Coprocessor Access Control Register
#ifdef __ARMCOMPILER_VERSION
	__asm("MRC p15, 0, %[regdata]0, c1, c0, 2" : [regdata] "=r" (regdata) : );
#endif
#ifdef __ARMCC_VERSION
	__asm("MRC p15, 0, regdata, c1, c0, 2");
#endif
#ifdef __GNUCC_VERSION
	__asm__ volatile("MRC p15, 0, %0, c1, c0, 2" : "=r" (regdata) : );
#endif

	regdata &= ~(ALT_CPU_CPACR_ASEDIS_SET_MSK | ALT_CPU_CPACR_D32DIS_SET_MSK);
	regdata |= (ALT_CPU_CPACR_CP11_SET_MSK | ALT_CPU_CPACR_CP10_SET_MSK);

	// MCR p15, 0,<Rd>, c1, c0, 2; Write Coprocessor Access Control Register
#ifdef __ARMCOMPILER_VERSION
	__asm("MCR p15, 0, %[regdata], c1, c0, 2" : : [regdata] "r" (regdata));
#endif
#ifdef __ARMCC_VERSION
	__asm("MCR p15, 0, regdata, c1, c0, 2");
#endif
#ifdef __GNUCC_VERSION
	__asm__ volatile("MCR p15, 0, %0, c1, c0, 2" : : "r" (regdata));
#endif

	// ISB ; Ensure side-effect of CPACR is visible
#ifdef __ARMCOMPILER_VERSION
	__asm("ISB" : : );
#endif
#ifdef __ARMCC_VERSION
	__asm("ISB");
#endif
#ifdef __GNUCC_VERSION
	__asm__ volatile("ISB" : : );
#endif

	// MOV r0,#(1<<30) ; Create value with FPEXC (bit 30) set
	regdata = (1 << 30);

	// VMSR FPEXC,r0 ; Enable VFP and SIMD extensions
#ifdef __ARMCOMPILER_VERSION
	__asm("VMSR FPEXC, %[regdata]" : : [regdata] "r" (regdata));
#endif
#ifdef __ARMCC_VERSION
	__asm("VMSR FPEXC, regdata");
#endif
#ifdef __GNUCC_VERSION
	__asm__ volatile("VMSR FPEXC, %0" : : "r" (regdata));
#endif

	return;
}

static void enable_actlr_smp(void)
{
	uint32_t regdata;

	// MRC p15, 0, <Rt>, c1, c0, 1; Read ACTLR
#ifdef __ARMCOMPILER_VERSION
	__asm("MRC p15, 0, %[regdata]0, c1, c0, 1" : [regdata] "=r" (regdata) : );
#endif
#ifdef __ARMCC_VERSION
	__asm("MRC p15, 0, regdata, c1, c0, 1");
#endif
#ifdef __GNUCC_VERSION
	__asm__ volatile("MRC p15, 0, %0, c1, c0, 1" : "=r" (regdata) : );
#endif

	regdata |= ALT_CPU_ACTLR_SMP_SET_MSK;

	// MCR p15, 0, <Rt>, c1, c0, 1; Write ACTLR
#ifdef __ARMCOMPILER_VERSION
	__asm("MCR p15, 0, %[regdata], c1, c0, 1" : : [regdata] "r" (regdata));
#endif
#ifdef __ARMCC_VERSION
	__asm("MCR p15, 0, regdata, c1, c0, 1");
#endif
#ifdef __GNUCC_VERSION
	__asm__ volatile("MCR p15, 0, %0, c1, c0, 1" : : "r" (regdata));
#endif

	return;
}

static void enable_nsacr_nssmp(void)
{
	uint32_t regdata;

	// MRC p15, 0, <Rt>, c1, c1, 2; Read NSACR
#ifdef __ARMCOMPILER_VERSION
	__asm("MRC p15, 0, %[regdata]0, c1, c1, 2" : [regdata] "=r" (regdata) : );
#endif
#ifdef __ARMCC_VERSION
	__asm("MRC p15, 0, regdata, c1, c1, 2");
#endif
#ifdef __GNUCC_VERSION
	__asm__ volatile("MRC p15, 0, %0, c1, c1, 2" : "=r" (regdata) : );
#endif

	regdata |= ALT_CPU_NSACR_NSSMP_SET_MSK;

	// MCR p15, 0, <Rt>, c1, c1, 2; Write NSACR
#ifdef __ARMCOMPILER_VERSION
	__asm("MCR p15, 0, %[regdata], c1, c1, 2" : : [regdata] "r" (regdata));
#endif
#ifdef __ARMCC_VERSION
	__asm("MCR p15, 0, regdata, c1, c1, 2");
#endif
#ifdef __GNUCC_VERSION
	__asm__ volatile("MCR p15, 0, %0, c1, c1, 2" : : "r" (regdata));
#endif

	return;
}

void* sample_mmu_tlb_alloc(const size_t size, void * context)
{
	void* p = NULL;

	/* User defined function used for allocating storage for first and second level translation tables. */

	if(size >= sizeof(mmu_tlb_heap)){
		printf("[C0] ERROR!!: mmu_tlb_alloc() 1st argument ""size"" is too large!\r\n");
	} else if(((uint32_t)context % MMU_TLB_ALIGNMENT) != 0){
		printf("[C0] ERROR!!: mmu_tlb_alloc() 2st argument ""context"" is not aligned!\r\n");
	} else {
		p = context;
	}

	return p ;
}

void sample_mmu_init_and_enable(void)
{
	ALT_STATUS_CODE result_code;

	/* Usage:	ALT_STATUS_CODE  alt_mmu_init (void);	*/
	result_code = alt_mmu_init();
	if(result_code != ALT_E_SUCCESS){
		printf("[C0] ERROR!!: alt_mmu_init() .... result=%d\r\n", (int)result_code);
	}

	/* disable MMU */
	result_code = alt_mmu_disable();
	if(result_code != ALT_E_SUCCESS){
		printf("[C0] ERROR!!: alt_mmu_disable() .... result=%d\r\n", (int)result_code);
	}

	/* create MMU tables */
	/* Usage:	ALT_STATUS_CODE  alt_mmu_va_space_create (uint32_t **ttb1, const ALT_MMU_MEM_REGION_t *mem_regions, const size_t num_mem_regions, alt_mmu_ttb_alloc_t ttb_alloc, void *ttb_alloc_context);	*/
	result_code = alt_mmu_va_space_create(&mmu_tlb_l1, MemoryRegion, ValidMemoryRegion, sample_mmu_tlb_alloc, (void*)mmu_tlb_heap );
	if(result_code != ALT_E_SUCCESS){
		printf("[C0] ERROR!!: alt_mmu_va_space_create() .... result=%d\r\n", (int)result_code);
	}

	if(result_code == ALT_E_SUCCESS){
		/* enable MMU */
		/* Usage:	ALT_STATUS_CODE  alt_mmu_va_space_enable (const uint32_t *ttb1);	*/
		result_code = alt_mmu_va_space_enable(mmu_tlb_l1);
		if(result_code != ALT_E_SUCCESS){
			printf("[C0] ERROR!!: alt_mmu_va_space_enable() .... result=%d\r\n", (int)result_code);
		}
	}

	if(result_code != ALT_E_SUCCESS){
		printf("[C0] MMU Init error\r\n" );
	}

	return;
}

int cpu0_l2mmu_init(void)
{
	ALT_STATUS_CODE status = ALT_E_SUCCESS;

	// Enabling Advanced SIMD and VFP 
	enable_neon_and_vfp();

	// ACTLR.SMP=1 / NSACR.NS_SMP=1
	enable_actlr_smp();
	enable_nsacr_nssmp();

	// Initialize SCU (SCU is a shared resource in the MPCore. So, performed only by Core0.)
	SOCFPGA5XS1_BSP_SCU_CTLR	= 0x0;							/* Disable SCU */
	SOCFPGA5XS1_BSP_SCU_INV_WAY = 0xFFFF;						/* Invalidate SCU Tag RAM */
	SOCFPGA5XS1_BSP_SCU_CTLR	= 0x1;							/* Enable SCU */

	// Setting signal of AXI transaction (AxUSER[0]=1). This setting necesary for coherent transfer via ACP.
	alt_write_word(ALT_ACPIDMAP_DYNRD_ADDR, ALT_ACPIDMAP_DYNRD_PAGE_SET(0) | ALT_ACPIDMAP_DYNRD_USER_SET(1));
	alt_write_word(ALT_ACPIDMAP_DYNWR_ADDR, ALT_ACPIDMAP_DYNWR_PAGE_SET(0) | ALT_ACPIDMAP_DYNWR_USER_SET(1));

	// initialize GIC Distributor registers
	alt_int_global_init();

	status = alt_mpu_addr_space_remap_0_to_sdram();
	if(status != ALT_E_SUCCESS){
		printf("[C0] ERROR!!: alt_mpu_addr_space_remap_0_to_sdram() .... result=%d\r\n", (int)status);
	}

	// Setting MMU & Enable 
	sample_mmu_init_and_enable();

	// Cache Enable 
	alt_cache_system_enable();

	// initialize & enable CPU interrupt
	alt_int_cpu_init();
	alt_int_cpu_enable_all();
	// enable GIC global interrupt (Core0 only)
	alt_int_global_enable_all();
	__asm("CPSIE i");

	return 0;
}

/***********************************************************************************
 * end of file 
 ***********************************************************************************/
