﻿/********************************************************************************//*!
 * @file  sample_clock_manager.c 
 * @brief  To use HWLib for the Clock Manager example program. 
 * 
 * @details  Run the API category below.
 *           - The Clock Manager API  
 *               - Clock Manager Status  
 *               - Safe Mode Options  
 *               - PLL Bypass Control  
 *               - Clock Gating Control  
 *               - Clock Source Selection  
 *               - Clock Frequency Control  
 *               - Clock Manager Interrupt Management  
 *               - Clock Group Configuration 
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/11  ---  #6221  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 * @date 2014/08/18  ---  -----  v14.0化 (誤植訂正、操作説明の表示を追加). 
 * @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
 * @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
 * @date 2018/07/02  ---  #25940  All in One ベアメタル・アプリケーション v17.1 への対応を実施.
 * @date 2019/01/18  ---  -----   All in One ベアメタル・アプリケーション v18.1 への対応を実施.
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes 
 ***********************************************************************************/

#include <stdbool.h>
#include <stdio.h>
#include <string.h>
#include "socal/socal.h"
#include "socal/hps.h"
#include "alt_clock_manager.h"
#include "alt_generalpurpose_io.h"
#include "alt_globaltmr.h"
#include "alt_interrupt.h"
#include "hwlib.h"
#include "socal/alt_clkmgr.h"

#include "sample_app_setting.h"
#include "util_time_measurement.h"
#include "util_interrupt_log.h"
#include "alt_printf.h"

/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/
void sample_clkmgr_init(void);
static void sample_globaltimer_config(void);

void sample_clkmgr_setting_gic(void);
void sample_clkmgr_callback(uint32_t icciar, void *context);
void sample_clkmgr_test_customize_config(void);
static void sample_clkmgr_print_plllock(void);
#ifdef soc_cv_av
static void sample_clkmgr_print_freq_and_devider(ALT_CLK_t pll, int divion);
static void sample_clkmgr_print_pllconfig(ALT_CLK_t pll, ALT_CLK_PLL_CFG_t* pllconfig);
#endif /* soc_cv_av */
void sample_clkmgr_test_print_pllconfig(void);
void sample_clkmgr_test_main(void);
void sample_clkmgr_test_print_usage(void);
int  sample_clkmgr_test_cmd(char* options);

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/
#ifdef soc_cv_av
#define ALT_PLLLOCK_STATUSBIT_ACHV	((ALT_MAIN_PLL_LOCK_ACHV)|(ALT_PERIPH_PLL_LOCK_ACHV)|(ALT_SDR_PLL_LOCK_ACHV))
#define ALT_PLLLOCK_STATUSBIT_LOST	((ALT_MAIN_PLL_LOCK_LOST)|(ALT_PERIPH_PLL_LOCK_LOST)|(ALT_SDR_PLL_LOCK_LOST))
#else /* soc_cv_av */
#define ALT_CLKMGR_INTREN_ADDR  ALT_CAST(void *, (ALT_CAST(char *, ALT_CLKMGR_CLKMGR_ADDR) + ALT_CLKMGR_CLKMGR_INTREN_OFST))
#define ALT_CLKMGR_INTR_ADDR	ALT_CAST(void *, (ALT_CAST(char *, ALT_CLKMGR_CLKMGR_ADDR) + ALT_CLKMGR_CLKMGR_INTR_OFST))
#define ALT_CLKMGR_STAT_ADDR	ALT_CAST(void *, (ALT_CAST(char *, ALT_CLKMGR_CLKMGR_ADDR) + ALT_CLKMGR_CLKMGR_STAT_OFST))
#define ALT_PLLLOCK_STATUSBIT_ACHV	((ALT_CLKMGR_CLKMGR_INTRS_MAINPLLACHIEVED_SET_MSK)|(ALT_CLKMGR_CLKMGR_INTRS_PERPLLACHIEVED_SET_MSK))
#define ALT_PLLLOCK_STATUSBIT_LOST	((ALT_CLKMGR_CLKMGR_INTRS_MAINPLLLOST_SET_MSK)|(ALT_CLKMGR_CLKMGR_INTRS_PERPLLLOST_SET_MSK))
#endif /* soc_cv_av */
#define ALT_PLLLOCK_STATUSBIT_ALL	((ALT_PLLLOCK_STATUSBIT_ACHV)|(ALT_PLLLOCK_STATUSBIT_LOST))

/***********************************************************************************
 *  variables 
 ***********************************************************************************/
volatile uint32_t last_clk_lock_status = 0;
#ifdef soc_cv_av
ALT_CLK_PLL_CFG_t mainpllconfig;
ALT_CLK_PLL_CFG_t peripllconfig;
ALT_CLK_PLL_CFG_t sdrapllconfig;
#endif /* soc_cv_av */

/***********************************************************************************
 *  functions 
 ***********************************************************************************/

/********************************************************************************//*!
 * @fn  void sample_clkmgr_init(void) 
 * @brief  Initializing the sample program. 
 * 
 * @details  Call the initialization functions. 
 *           Do the following.
 *           -# Configure the Helio Board GPIO settings. 
 *           -# Configure the Global Timer settings. 
 *           -# Setting of GIC (Clock Manager(Pll Lock Status) interrupt). 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/11  ---  #6221  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_clkmgr_init(void)
{
#ifdef soc_cv_av
	ALT_STATUS_CODE result_code;
#endif /* soc_cv_av */

	// Clock Manager Interrupt Disable
#ifdef soc_cv_av
	/* Usage: ALT_STATUS_CODE alt_clk_irq_disable(ALT_CLK_PLL_LOCK_STATUS_t lock_stat_mask);	*/
	printf("-- HWLib test : alt_clk_irq_disable() --\n");
#ifdef GNU_COMPILER	// 99
	result_code = (ALT_STATUS_CODE)alt_clk_irq_disable(ALT_PLLLOCK_STATUSBIT_ALL);
#endif
#ifdef ARM_COMPILER	// 99
	result_code = alt_clk_irq_disable((ALT_CLK_PLL_LOCK_STATUS_t)ALT_PLLLOCK_STATUSBIT_ALL);
#endif
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_irq_disable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");
#else /* soc_cv_av */
	alt_clrbits_word(ALT_CLKMGR_INTREN_ADDR, ALT_PLLLOCK_STATUSBIT_ALL);
#endif /* soc_cv_av */

#if 0	// main() 内で cpu0_init() 実施済み
	cpu0_init();
#endif
	sample_globaltimer_config();

	// Clock Manager Interrupt GIC Config
	sample_clkmgr_setting_gic();

	// Clock Manager Interrupt Enable
#ifdef soc_cv_av
	/* Usage: ALT_STATUS_CODE alt_clk_irq_enable(ALT_CLK_PLL_LOCK_STATUS_t lock_stat_mask);	*/
	printf("-- HWLib test : alt_clk_irq_enable() --\n");
#ifdef GNU_COMPILER	// 99
	result_code = alt_clk_irq_enable(ALT_PLLLOCK_STATUSBIT_ALL);
#endif
#ifdef ARM_COMPILER	// 99
	result_code = alt_clk_irq_enable((ALT_CLK_PLL_LOCK_STATUS_t)ALT_PLLLOCK_STATUSBIT_ALL);
#endif
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_irq_enable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");
#else /* soc_cv_av */
	alt_setbits_word(ALT_CLKMGR_INTREN_ADDR, ALT_PLLLOCK_STATUSBIT_ALL);
#endif /* soc_cv_av */

	return;
}

/********************************************************************************//*!
 * @fn  static void sample_globaltimer_config(void) 
 * @brief  Global timer configuration.
 * 
 * @details  The global timer configuration by using the HWLib APIs. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/11  ---  #6221  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void sample_globaltimer_config(void)
{
	ALT_STATUS_CODE	result_code;

	// Parameter setting for the global timer.
	/***********************************************************************/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_prescaler_set(uint32_t val);	*/
	/*			val ... Valid range is 0-255. Actual clock divisor ratio is this number plus one. */
	/***********************************************************************/
	result_code = alt_globaltmr_prescaler_set(0);	// Set the minimum value(0).
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_prescaler_set() result=%d\n", (int)result_code);
	}

	// Function ON/OFF Setting: Timer=ON, CompMode=OFF, AutoInc=OFF, Interrupt=OFF
	/***********************************************************************/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_comp_mode_stop(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_autoinc_mode_stop(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_int_disable(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_start(void);	*/
	/***********************************************************************/
	result_code = alt_globaltmr_comp_mode_stop();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_comp_mode_stop() result=%d\n", (int)result_code);
	}
	result_code = alt_globaltmr_autoinc_mode_stop();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_autoinc_mode_stop() result=%d\n", (int)result_code);
	}
	result_code = alt_globaltmr_int_disable();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_int_disable() result=%d\n", (int)result_code);
	}
	result_code = alt_globaltmr_start();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_start() result=%d\n", (int)result_code);
	}

	return;
}

/********************************************************************************//*!
 * @fn  void sample_clkmgr_setting_gic(void) 
 * @brief  Setting of GIC (Clock Manager interrupt).
 * 
 * @details    In order to receive a Clock Manager(PLL Lock Status) interrupt setting of GIC.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/11  ---  #6221  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_clkmgr_setting_gic(void)
{
	ALT_STATUS_CODE result_code;
	uint32_t ui32temp = 0;
	ALT_INT_TRIGGER_t trigger_type = ALT_INT_TRIGGER_NA;

	// Print initial value (GIC: ALT_INT_INTERRUPT_CLKMGR_IRQ)
	/****************************************************************************************/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_secure(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_enabled(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_pending(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_is_active(ALT_INT_INTERRUPT_t int_id);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_priority_get(ALT_INT_INTERRUPT_t int_id, uint32_t* priority);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_target_get(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t* target);	*/
	/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_get(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t* trigger_type);	*/
	/****************************************************************************************/
	printf("-- HWLib test : alt_int_dist_is_secure() --\n");
	printf("arg1=ALT_INT_INTERRUPT_CLKMGR_IRQ --> result=%d\n", (int)alt_int_dist_is_secure(ALT_INT_INTERRUPT_CLKMGR_IRQ));
	printf("\n");
	printf("-- HWLib test : alt_int_dist_is_enabled() --\n");
	printf("arg1=ALT_INT_INTERRUPT_CLKMGR_IRQ --> result=%d\n", (int)alt_int_dist_is_enabled(ALT_INT_INTERRUPT_CLKMGR_IRQ));
	printf("\n");
	printf("-- HWLib test : alt_int_dist_is_pending() --\n");
	printf("arg1=ALT_INT_INTERRUPT_CLKMGR_IRQ --> result=%d\n", (int)alt_int_dist_is_pending(ALT_INT_INTERRUPT_CLKMGR_IRQ));
	printf("\n");
	printf("-- HWLib test : alt_int_dist_is_active() --\n");
	printf("arg1=ALT_INT_INTERRUPT_CLKMGR_IRQ --> result=%d\n", (int)alt_int_dist_is_active(ALT_INT_INTERRUPT_CLKMGR_IRQ));
	printf("\n");
	printf("-- HWLib test : alt_int_dist_priority_get() --\n");
	printf("arg1=ALT_INT_INTERRUPT_CLKMGR_IRQ --> result=%d out=0x%08X\n", (int)alt_int_dist_priority_get(ALT_INT_INTERRUPT_CLKMGR_IRQ,&ui32temp), (int)ui32temp);
	printf("\n");
	printf("-- HWLib test : alt_int_dist_target_get() --\n");
	printf("arg1=ALT_INT_INTERRUPT_CLKMGR_IRQ --> result=%d out=0x%08X\n", (int)alt_int_dist_target_get(ALT_INT_INTERRUPT_CLKMGR_IRQ,&ui32temp), (int)ui32temp);
	printf("\n");
	printf("-- HWLib test : alt_int_dist_trigger_get() --\n");
	printf("arg1=ALT_INT_INTERRUPT_CLKMGR_IRQ --> result=%d out=%d\n", (int)alt_int_dist_trigger_get(ALT_INT_INTERRUPT_CLKMGR_IRQ,&trigger_type), (int)trigger_type);
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_isr_register(ALT_INT_INTERRUPT_t int_id, alt_int_callback_t callback, void* context);	*/
	/*  typedef void(* alt_int_callback_t)(uint32_t icciar, void *context) */
	printf("-- HWLib test : alt_int_isr_register() --\n");
	result_code = alt_int_isr_register(ALT_INT_INTERRUPT_CLKMGR_IRQ, sample_clkmgr_callback, NULL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_isr_register() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_target_set(ALT_INT_INTERRUPT_t int_id, alt_int_cpu_target_t target);	*/
	printf("-- HWLib test : alt_int_dist_target_set() --\n");
	result_code = alt_int_dist_target_set(ALT_INT_INTERRUPT_CLKMGR_IRQ, 1);		// arg2=targetCPU (Only Core#0)
	if(result_code != ALT_E_SUCCESS)
	{
		printf("ERROR!!: alt_int_dist_target_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_trigger_set(ALT_INT_INTERRUPT_t int_id, ALT_INT_TRIGGER_t trigger_type);	*/
	printf("-- HWLib test : alt_int_dist_trigger_set() --\n");
	result_code = alt_int_dist_trigger_set(ALT_INT_INTERRUPT_CLKMGR_IRQ, ALT_INT_TRIGGER_EDGE);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_trigger_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_priority_set(ALT_INT_INTERRUPT_t int_id, uint32_t priority);	*/
	printf("-- HWLib test : alt_int_dist_priority_set() --\n");
	result_code = alt_int_dist_priority_set(ALT_INT_INTERRUPT_CLKMGR_IRQ, 99);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_priority_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_enable(ALT_INT_INTERRUPT_t int_id);	*/
	printf("-- HWLib test : alt_int_dist_enable() --\n");
	result_code = alt_int_dist_enable(ALT_INT_INTERRUPT_CLKMGR_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	/* Usage: ALT_STATUS_CODE alt_int_dist_secure_enable(ALT_INT_INTERRUPT_t int_id);	*/
	printf("-- HWLib test : alt_int_dist_secure_enable() --\n");
	result_code = alt_int_dist_secure_enable(ALT_INT_INTERRUPT_CLKMGR_IRQ);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_int_dist_enable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("\n");

	return;
}

/********************************************************************************//*!
 * @fn  void sample_clkmgr_callback(uint32_t icciar, void *context) 
 * @brief  Clock Manager (ALT_INT_INTERRUPT_CLKMGR_IRQ) interrupt callback function.
 * 
 * @details  working opportunity an interrupt from the ALT_INT_INTERRUPT_CLKMGR_IRQ, 
 *           indicate an interrupt, and the interrupt pending bit clear.
 *           Specifically, by the following factors informs interrupts.
 *           - Main PLL Achieved Lock
 *           - Main PLL Lost Lock
 *           - Peripheral PLL Achieved Lock
 *           - Peripheral PLL Lost Lock
 *           - SDRAM PLL Achieved Lock
 *           - SDRAM PLL Lost Lock
 * @param[in] icciar  The callback to use when the given interrupt is issued. 
 * @param[in] context  The user provided context.  
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/11  ---  #6221  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_clkmgr_callback(uint32_t icciar, void *context)
{
	ALT_STATUS_CODE result_code;

	/* Usage: ALT_STATUS_CODE alt_int_dist_pending_clear(ALT_INT_INTERRUPT_t int_id);	*/
	result_code = alt_int_dist_pending_clear(ALT_INT_INTERRUPT_CLKMGR_IRQ);
	if(result_code != ALT_E_SUCCESS)
	{
		fprintf(stderr, "[INTERRUPT]ERROR!!: alt_int_dist_pending_clear() result=%d\n", (int)result_code);
	}

	/* Usage: uint32_t alt_clk_lock_status_get(void);	*/
#ifdef soc_cv_av
	last_clk_lock_status = alt_clk_lock_status_get();
#else /* soc_cv_av */
	last_clk_lock_status = alt_read_word(ALT_CLKMGR_INTR_ADDR);
#endif /* soc_cv_av */

	///* Usage: ALT_STATUS_CODE alt_clk_lock_status_clear(ALT_CLK_PLL_LOCK_STATUS_t lock_stat_mask);	*/
	//printf("[INTERRUPT]Clock Manager Interrupt is occurred!! Status=0x%08X\n", (int)last_clk_lock_status);
	util_intlog_record(ALT_INT_INTERRUPT_CLKMGR_IRQ, (int)icciar, (int)last_clk_lock_status);

	return;
}

/********************************************************************************//*!
 * @fn  void sample_clkmgr_test_customize_config(void) 
 * @brief  Try customizing the clock manager settings. 
 * 
 * @details  Performs all major clock Manager API for customizing the settings. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/11  ---  #6221  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_clkmgr_test_customize_config(void)
{
#ifdef soc_cv_av
	ALT_STATUS_CODE	result_code;

	/***********************************************************************/
	/* Usage: ALT_STATUS_CODE alt_clk_pll_bypass_disable(ALT_CLK_t pll);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_pll_bypass_enable(ALT_CLK_t pll, bool use_input_mux);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_clock_disable(ALT_CLK_t clk);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_clock_enable(ALT_CLK_t clk);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_source_set(ALT_CLK_t clk, ALT_CLK_t ref_clk);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_ext_clk_freq_set(ALT_CLK_t clk, alt_freq_t freq);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_pll_cfg_set(ALT_CLK_t pll, const ALT_CLK_PLL_CFG_t* pll_cfg);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_pll_vco_cfg_set(ALT_CLK_t pll, uint32_t mult, uint32_t div);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_pll_guard_band_set(ALT_CLK_t pll, uint32_t guard_band);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_divider_set(ALT_CLK_t clk, uint32_t div);	*/
	/***********************************************************************/
	/* Usage: ALT_STATUS_CODE alt_clk_pll_bypass_enable(ALT_CLK_t pll, bool use_input_mux);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_pll_bypass_disable(ALT_CLK_t pll);	*/
	printf("-- HWLib test : alt_clk_pll_bypass_enable() --\n");
	result_code = alt_clk_pll_bypass_enable(ALT_CLK_PERIPHERAL_PLL, true);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_pll_bypass_enable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("-- HWLib test : alt_clk_pll_bypass_disable() --\n");
	result_code = alt_clk_pll_bypass_disable(ALT_CLK_PERIPHERAL_PLL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_pll_bypass_disable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	/* Usage: ALT_STATUS_CODE alt_clk_clock_disable(ALT_CLK_t clk);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_clock_enable(ALT_CLK_t clk);	*/
	printf("-- HWLib test : alt_clk_clock_disable() --\n");
	result_code = alt_clk_clock_disable(ALT_CLK_GPIO_DB);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_clock_disable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	printf("-- HWLib test : alt_clk_clock_enable() --\n");
	result_code = alt_clk_clock_enable(ALT_CLK_GPIO_DB);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_clock_enable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	/* Usage: ALT_STATUS_CODE alt_clk_source_set(ALT_CLK_t clk, ALT_CLK_t ref_clk);	*/
	printf("-- HWLib test : alt_clk_source_set() --\n");
	result_code = alt_clk_source_set(ALT_CLK_PERIPHERAL_PLL, ALT_CLK_IN_PIN_OSC2);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_source_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	/* Usage: ALT_STATUS_CODE alt_clk_ext_clk_freq_set(ALT_CLK_t clk, alt_freq_t freq);	*/
	printf("-- HWLib test : alt_clk_ext_clk_freq_set() --\n");
	result_code = alt_clk_ext_clk_freq_set(ALT_CLK_IN_PIN_OSC1, (alt_freq_t)25000000);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_ext_clk_freq_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	/* Usage: ALT_STATUS_CODE alt_clk_pll_bypass_enable(ALT_CLK_t pll, bool use_input_mux);	*/
	printf("-- HWLib test : alt_clk_pll_bypass_enable() --\n");
	result_code = alt_clk_pll_bypass_enable(ALT_CLK_PERIPHERAL_PLL, true);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_pll_bypass_enable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	/* Usage: ALT_STATUS_CODE alt_clk_pll_cfg_set(ALT_CLK_t pll, const ALT_CLK_PLL_CFG_t* pll_cfg);	*/
	peripllconfig.ref_clk = ALT_CLK_IN_PIN_OSC1;
	printf("-- HWLib test : alt_clk_pll_cfg_set() --\n");
	result_code = alt_clk_pll_cfg_set(ALT_CLK_PERIPHERAL_PLL, &peripllconfig);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_pll_cfg_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	/* Usage: ALT_STATUS_CODE alt_clk_pll_bypass_disable(ALT_CLK_t pll);	*/
	printf("-- HWLib test : alt_clk_pll_bypass_disable() --\n");
	result_code = alt_clk_pll_bypass_disable(ALT_CLK_PERIPHERAL_PLL);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_pll_bypass_disable() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
#if 99-99
	/* Usage: ALT_STATUS_CODE alt_clk_pll_vco_cfg_set(ALT_CLK_t pll, uint32_t mult, uint32_t div);	*/
	printf("-- HWLib test : alt_clk_pll_vco_cfg_set() --\n");
	result_code =  alt_clk_pll_vco_cfg_set(ALT_CLK_PERIPHERAL_PLL, (uint32_t)160, (uint32_t)4);
	result_code |= alt_clk_pll_vco_cfg_set(ALT_CLK_PERIPHERAL_PLL, (uint32_t)80, (uint32_t)2);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_pll_vco_cfg_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
#endif
	/* Usage: ALT_STATUS_CODE alt_clk_pll_guard_band_set(ALT_CLK_t pll, uint32_t guard_band);	*/
	printf("-- HWLib test : alt_clk_pll_guard_band_set() --\n");
	result_code = alt_clk_pll_guard_band_set(ALT_CLK_PERIPHERAL_PLL, (uint32_t)20);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_pll_guard_band_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}
	/* Usage: ALT_STATUS_CODE alt_clk_divider_set(ALT_CLK_t clk, uint32_t div);	*/
	printf("-- HWLib test : alt_clk_divider_set() --\n");
	result_code = alt_clk_divider_set(ALT_CLK_GPIO_DB, (uint32_t)5);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_divider_set() result=%d\n", (int)result_code);
	} else {
		printf("SUCCESS!!\n");
	}

#endif /* soc_cv_av */
	return;
}

/********************************************************************************//*!
 * @fn  static void sample_clkmgr_print_freq_and_devider(ALT_CLK_t pll, int divion) 
 * @brief  Display the setting of specified PLL. 
 * 
 * @details  PLL specified target shown setting values for frequency and frequency divider. 
 * @param[in] pll  Specify the target PLL (type of ALT_CLK_t) 
 * @param[in] divion  Divider settings display ON(!0)/OFF(0) 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/11  ---  #6221  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
#ifdef soc_cv_av
static void sample_clkmgr_print_freq_and_devider(ALT_CLK_t pll, int divion)
{
	ALT_STATUS_CODE	result_code;
	uint32_t ui32div = 0;
	alt_freq_t freq = 0;

	/* Usage: ALT_STATUS_CODE alt_clk_freq_get(ALT_CLK_t clk, alt_freq_t* freq);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_divider_get(ALT_CLK_t clk, uint32_t* div);	*/
	result_code = alt_clk_freq_get(pll, &freq);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_freq_get() result=%d\n", (int)result_code);
	}
	if(divion)
	{
		result_code = alt_clk_divider_get(pll, &ui32div);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_clk_divider_get() result=%d\n", (int)result_code);
		}
		printf("Frequency=%10d (Hz) [Divider=%d]\n", (int)freq, (int)ui32div);
	} else {
		printf("Frequency=%10d (Hz)\n", (int)freq);
	}

	return;
}
#endif /* soc_cv_av */

/********************************************************************************//*!
 * @fn  static void sample_clkmgr_print_plllock(void) 
 * @brief  Display all PLL Lock Status. 
 * 
 * @details  Display the results obtained with the following API.  
 *           - alt_clk_pll_is_locked(ALT_CLK_MAIN_PLL) 
 *           - alt_clk_pll_is_locked(ALT_CLK_PERIPHERAL_PLL) 
 *           - alt_clk_pll_is_locked(ALT_CLK_SDRAM_PLL) 
 *           - alt_clk_lock_status_get() 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/11  ---  #6221  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
#ifdef soc_cv_av
static void sample_clkmgr_print_plllock(void)
{
	/* Usage: ALT_STATUS_CODE alt_clk_pll_is_locked(ALT_CLK_t pll);	*/
	printf("-- HWLib test : alt_clk_pll_is_locked() --\n");
	printf(" * Main PLL Lock       : %d\n", (int)alt_clk_pll_is_locked(ALT_CLK_MAIN_PLL));
	printf(" * Peripheral PLL Lock : %d\n", (int)alt_clk_pll_is_locked(ALT_CLK_PERIPHERAL_PLL));
	printf(" * SDRAM PLL Lock      : %d\n", (int)alt_clk_pll_is_locked(ALT_CLK_SDRAM_PLL));
	/* Usage: uint32_t alt_clk_lock_status_get(void);	*/
	printf("-- HWLib test : alt_clk_lock_status_get() --\n");
	last_clk_lock_status = alt_clk_lock_status_get();
	printf("Clock Manager Interrupt Status=0x%08X\n", (int)last_clk_lock_status);
	printf("\n");

	return;
}
#else /* soc_cv_av */
static void sample_clkmgr_print_plllock(void)
{
	uint32_t stat = alt_read_word(ALT_CLKMGR_STAT_ADDR);

	/* Usage: ALT_STATUS_CODE alt_clk_pll_is_locked(ALT_CLK_t pll);	*/
	printf("Clock Manager Status=0x%08X\n", (int)stat);
	printf(" * Main PLL Lock       : %d\n", (int)ALT_CLKMGR_CLKMGR_STAT_MAINPLLLOCKED_GET(stat));
	printf(" * Peripheral PLL Lock : %d\n", (int)ALT_CLKMGR_CLKMGR_STAT_PERPLLLOCKED_GET(stat));
	/* Usage: uint32_t alt_clk_lock_status_get(void);	*/
	last_clk_lock_status = alt_read_word(ALT_CLKMGR_INTR_ADDR);
	printf("Clock Manager Interrupt Status=0x%08X\n", (int)last_clk_lock_status);
	printf("\n");

	return;
}
#endif /* soc_cv_av */

/********************************************************************************//*!
 * @fn  static void sample_clkmgr_print_pllconfig(ALT_CLK_t pll, ALT_CLK_PLL_CFG_t* pllconfig) 
 * @brief  Displays the get specified PLL for more information. 
 * 
 * @details  PLL specified by the 1st argument to get details. 
 *           Storing the information in the location specified by the 2nd argument. 
 * @param[in] pll  Specify the target PLL (type of ALT_CLK_t) 
 * @param[out] pllconfig  Location of the configuration settings of the PLL 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/11  ---  #6221  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
#ifdef soc_cv_av
static void sample_clkmgr_print_pllconfig(ALT_CLK_t pll, ALT_CLK_PLL_CFG_t* pllconfig)
{
	ALT_STATUS_CODE	result_code;
	alt_freq_t freq, vcofreq;
	ALT_CLK_t tsourceclk;
	uint32_t ui32mult, ui32div, ui32guardband;

	/* Usage: ALT_CLK_t alt_clk_source_get(ALT_CLK_t clk);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_pll_vco_cfg_get(ALT_CLK_t pll, uint32_t* mult, uint32_t* div);	*/
	printf("-- HWLib test : alt_clk_source_get() --\n");
	tsourceclk = alt_clk_source_get(pll);
	printf(" * PLL Reference Clock Source: %d\n", (int)tsourceclk);
	printf("-- HWLib test : alt_clk_pll_vco_cfg_get() --\n");
	result_code = alt_clk_pll_vco_cfg_get(pll, &ui32mult, &ui32div);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_pll_vco_cfg_get() result=%d\n", (int)result_code);
	} else {
		printf(" * VCO - Multiplier (M) value: %d\n", (int)ui32mult);
		printf(" * VCO - Divider (N) value   : %d\n", (int)ui32div);
	}
	printf("-- HWLib test : alt_clk_pll_cfg_get() --\n");
	result_code = alt_clk_pll_cfg_get(pll, pllconfig);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_pll_cfg_get() result=%d\n", (int)result_code);
	} else {
		printf(" * PLL Reference Clock Source: %d\n", (int)pllconfig->ref_clk);
		printf(" * VCO - Multiplier (M) value: %d (=>%d)\n", (int)pllconfig->mult, (int)pllconfig->mult+1);
		printf(" * VCO - Divider (N) value   : %d (=>%d)\n", (int)pllconfig->div, (int)pllconfig->div+1);
		for(int i=0; i<6; i++){
			printf(" * C%d - {PhaseShift, Divider}: {%d, %d (=>%d x K)}\n", i,
					(int)pllconfig->pshift[i], (int)pllconfig->cntrs[i], (int)pllconfig->cntrs[i]+1);
		}
		printf(" # in the main PLL, K=2 for C0, and K=4 for C1,C2. Others are all K=1.\n");
		/* Usage: uint32_t alt_clk_pll_guard_band_get(ALT_CLK_t pll);	*/
		/* Usage: ALT_STATUS_CODE alt_clk_pll_vco_freq_get(ALT_CLK_t pll, alt_freq_t* freq);	*/
		printf("-- HWLib test : alt_clk_pll_guard_band_get() --\n");
		ui32guardband = alt_clk_pll_guard_band_get(pll);
		printf("-- HWLib test : alt_clk_pll_vco_freq_get() --\n");
		result_code = alt_clk_pll_vco_freq_get(pll, &vcofreq);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_clk_pll_vco_freq_get() result=%d\n", (int)result_code);
		}
		printf(" * Gurad Band Value            : %3d (%%)\n", (int)ui32guardband);
		printf(" * VCO Frequency (Fvco)        : %10d (Hz)\n", (int)vcofreq);
		/* Usage: ALT_STATUS_CODE alt_clk_pll_is_bypassed(ALT_CLK_t pll);	*/
		/* Usage: ALT_STATUS_CODE alt_clk_is_enabled(ALT_CLK_t clk);	*/
		printf("-- HWLib test : alt_clk_pll_is_bypassed(),alt_clk_is_enabled() --\n");
		printf(" * Is bypassed(%d) or not(%d)  : %d\n", (int)ALT_E_TRUE, (int)ALT_E_FALSE, (int)alt_clk_pll_is_bypassed(pll));
		printf(" * Is enabled(%d) or not(%d)   : %d\n", (int)ALT_E_TRUE, (int)ALT_E_FALSE, (int)alt_clk_is_enabled(pll));
		/* Usage: ALT_STATUS_CODE alt_clk_freq_get(ALT_CLK_t clk, alt_freq_t* freq);	*/
		printf("-- HWLib test : alt_clk_freq_get() --\n");
		result_code = alt_clk_freq_get(pllconfig->ref_clk, &freq);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_clk_freq_get() result=%d\n", (int)result_code);
		} else {
			printf(" * Clock Frequency - Fin=%10d (Hz)\n", (int)freq);
		}
	}
}
#endif /* soc_cv_av */

/********************************************************************************//*!
 * @fn  void sample_clkmgr_test_print_pllconfig(void) 
 * @brief  Get the information related to clock Manager and displays. 
 * 
 * @details  Displays the following information specifically.
 *           - Input Clock Frequency Value 
 *           - Output Clock Frequency and Divider Value 
 *           - setting information of Main PLL 
 *           - setting information of Peripheral PLL 
 *           - setting information of SDRAM PLL 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/11  ---  #6221  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
#ifdef soc_cv_av
void sample_clkmgr_test_print_pllconfig(void)
{
	alt_freq_t freq[6];

	printf("==== Input Clock Frequency Value =====\n");
	printf("ALT_CLK_IN_PIN_OSC1(%d):\t", (int)ALT_CLK_IN_PIN_OSC1);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_IN_PIN_OSC1, 0);
	printf("ALT_CLK_IN_PIN_OSC2(%d):\t", (int)ALT_CLK_IN_PIN_OSC2);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_IN_PIN_OSC2, 0);
	printf("ALT_CLK_F2H_PERIPH_REF(%d):\t", (int)ALT_CLK_F2H_PERIPH_REF);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_F2H_PERIPH_REF, 0);
	printf("ALT_CLK_F2H_SDRAM_REF(%d):\t", (int)ALT_CLK_F2H_SDRAM_REF);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_F2H_SDRAM_REF, 0);
	printf("\n");
	printf("==== Output Clock Frequency and Divider Value =====\n");
	printf("ALT_CLK_MPU(%d):        \t", (int)ALT_CLK_MPU);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_MPU, 1);
	printf("ALT_CLK_MPU_L2_RAM(%d): \t", (int)ALT_CLK_MPU_L2_RAM);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_MPU_L2_RAM, 1);
	printf("ALT_CLK_MPU_PERIPH(%d): \t", (int)ALT_CLK_MPU_PERIPH);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_MPU_PERIPH, 1);
	printf("ALT_CLK_L4_MAIN(%d):    \t", (int)ALT_CLK_L4_MAIN);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_L4_MAIN, 1);
	printf("ALT_CLK_L4_MP(%d):      \t", (int)ALT_CLK_L4_MP);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_L4_MP, 1);
	printf("ALT_CLK_L4_SP(%d):      \t", (int)ALT_CLK_L4_SP);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_L4_SP, 1);
	printf("\n");

	// Main PLL
	printf("==== setting information of Main PLL: (pll=%d) ====\n", (int)ALT_CLK_MAIN_PLL);
	sample_clkmgr_print_pllconfig(ALT_CLK_MAIN_PLL, &mainpllconfig);
	(void)alt_clk_freq_get(ALT_CLK_MAIN_PLL_C0, &freq[0]);
	(void)alt_clk_freq_get(ALT_CLK_MAIN_PLL_C1, &freq[1]);
	(void)alt_clk_freq_get(ALT_CLK_MAIN_PLL_C2, &freq[2]);
	(void)alt_clk_freq_get(ALT_CLK_MAIN_PLL_C3, &freq[3]);
	(void)alt_clk_freq_get(ALT_CLK_MAIN_PLL_C4, &freq[4]);
	(void)alt_clk_freq_get(ALT_CLK_MAIN_PLL_C5, &freq[5]);
	for(int i=0; i<6; i++){
		printf("             +-- C%d:Fout=%10d (Hz)\n", i, (int)freq[i]);
	}
	printf("\n");

	// Peripheral PLL
	printf("==== setting information of Peripheral PLL: (pll=%d) ====\n", (int)ALT_CLK_PERIPHERAL_PLL);
	sample_clkmgr_print_pllconfig(ALT_CLK_PERIPHERAL_PLL, &peripllconfig);
	(void)alt_clk_freq_get(ALT_CLK_PERIPHERAL_PLL_C0, &freq[0]);
	(void)alt_clk_freq_get(ALT_CLK_PERIPHERAL_PLL_C1, &freq[1]);
	(void)alt_clk_freq_get(ALT_CLK_PERIPHERAL_PLL_C2, &freq[2]);
	(void)alt_clk_freq_get(ALT_CLK_PERIPHERAL_PLL_C3, &freq[3]);
	(void)alt_clk_freq_get(ALT_CLK_PERIPHERAL_PLL_C4, &freq[4]);
	(void)alt_clk_freq_get(ALT_CLK_PERIPHERAL_PLL_C5, &freq[5]);
	for(int i=0; i<6; i++){
		printf("             +-- C%d:Fout=%10d (Hz)\n", i, (int)freq[i]);
	}
	printf("\n");

	// SDRAM PLL
	printf("==== setting information of SDRAM PLL: (pll=%d) ====\n", (int)ALT_CLK_SDRAM_PLL);
	sample_clkmgr_print_pllconfig(ALT_CLK_SDRAM_PLL, &sdrapllconfig);
	(void)alt_clk_freq_get(ALT_CLK_SDRAM_PLL_C0, &freq[0]);
	(void)alt_clk_freq_get(ALT_CLK_SDRAM_PLL_C1, &freq[1]);
	(void)alt_clk_freq_get(ALT_CLK_SDRAM_PLL_C2, &freq[2]);
	(void)alt_clk_freq_get(ALT_CLK_SDRAM_PLL_C3, &freq[3]);
	(void)alt_clk_freq_get(ALT_CLK_SDRAM_PLL_C4, &freq[4]);
	(void)alt_clk_freq_get(ALT_CLK_SDRAM_PLL_C5, &freq[5]);
	for(int i=0; i<6; i++){
		printf("             +-- C%d:Fout=%10d (Hz)\n", i, (int)freq[i]);
	}
	printf("\n");


	return;
}
#else /* soc_cv_av */
void sample_clkmgr_test_print_pllconfig(void)
{
	alt_freq_t freq;

	printf("==== Input Clock Frequency Value =====\n");
	alt_clk_freq_get(ALT_CLK_OSC1, &freq);
	printf("ALT_CLK_OSC1(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_OSC1, (int)freq);
	alt_clk_freq_get(ALT_CLK_F2S_FREE_CLK, &freq);
	printf("ALT_CLK_F2S_FREE_CLK(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_F2S_FREE_CLK, (int)freq);
	alt_clk_freq_get(ALT_CLK_INTOSC_HS_DIV2, &freq);
	printf("ALT_CLK_INTOSC_HS_DIV2(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_INTOSC_HS_DIV2, (int)freq);
	alt_clk_freq_get(ALT_CLK_MAIN_PLL, &freq);
	printf("ALT_CLK_MAIN_PLL(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_MAIN_PLL, (int)freq);
	alt_clk_freq_get(ALT_CLK_PERIPHERAL_PLL, &freq);
	printf("ALT_CLK_PERIPHERAL_PLL(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_PERIPHERAL_PLL, (int)freq);
	printf("\n");
	printf("==== Output Clock Frequency Value =====\n");
	printf("---- Hardware Sequenced Clocks -----\n");
	printf("<< Clock Group : MPU >>\n");
	alt_clk_freq_get(ALT_CLK_MPU, &freq);
	printf("ALT_CLK_MPU(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_MPU, (int)freq);
	alt_clk_freq_get(ALT_CLK_MPU_PERIPH, &freq);
	printf("ALT_CLK_MPU_PERIPH(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_MPU_PERIPH, (int)freq);
	alt_clk_freq_get(ALT_CLK_MPU_L2_RAM, &freq);
	printf("ALT_CLK_MPU_L2_RAM(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_MPU_L2_RAM, (int)freq);
	printf("<< Clock Group : NOC >>\n");
	alt_clk_freq_get(ALT_CLK_L3_MAIN, &freq);
	printf("ALT_CLK_L3_MAIN(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_L3_MAIN, (int)freq);
	alt_clk_freq_get(ALT_CLK_L4_SYS_FREE, &freq);
	printf("ALT_CLK_L4_SYS_FREE(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_L4_SYS_FREE, (int)freq);
	alt_clk_freq_get(ALT_CLK_L4_MAIN, &freq);
	printf("ALT_CLK_L4_MAIN(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_L4_MAIN, (int)freq);
	alt_clk_freq_get(ALT_CLK_L4_MP, &freq);
	printf("ALT_CLK_L4_MP(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_L4_MP, (int)freq);
	alt_clk_freq_get(ALT_CLK_L4_SP, &freq);
	printf("ALT_CLK_L4_SP(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_L4_SP, (int)freq);
	alt_clk_freq_get(ALT_CLK_CS_TIMER, &freq);
	printf("ALT_CLK_CS_TIMER(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_CS_TIMER, (int)freq);
	alt_clk_freq_get(ALT_CLK_CS_AT, &freq);
	printf("ALT_CLK_CS_AT(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_CS_AT, (int)freq);
	alt_clk_freq_get(ALT_CLK_CS_PDBG, &freq);
	printf("ALT_CLK_CS_PDBG(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_CS_PDBG, (int)freq);
	alt_clk_freq_get(ALT_CLK_CS_TRACE, &freq);
	printf("ALT_CLK_CS_TRACE(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_CS_TRACE, (int)freq);
	printf("<< Clock Group : FPGA/HMC Reference >>\n");
	alt_clk_freq_get(ALT_CLK_S2F_USER0, &freq);
	printf("ALT_CLK_S2F_USER0(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_S2F_USER0, (int)freq);
	alt_clk_freq_get(ALT_CLK_S2F_USER1, &freq);
	printf("ALT_CLK_S2F_USER1(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_S2F_USER1, (int)freq);
	alt_clk_freq_get(ALT_CLK_HMC_PLL_REF, &freq);
	printf("ALT_CLK_HMC_PLL_REF(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_HMC_PLL_REF, (int)freq);
	printf("---- Software Sequenced Clocks -----\n");
	alt_clk_freq_get(ALT_CLK_EMAC0, &freq);
	printf("ALT_CLK_EMAC0(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_EMAC0, (int)freq);
	alt_clk_freq_get(ALT_CLK_EMAC1, &freq);
	printf("ALT_CLK_EMAC1(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_EMAC1, (int)freq);
	alt_clk_freq_get(ALT_CLK_EMAC2, &freq);
	printf("ALT_CLK_EMAC2(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_EMAC2, (int)freq);
	alt_clk_freq_get(ALT_CLK_EMAC_PTP, &freq);
	printf("ALT_CLK_EMAC_PTP(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_EMAC_PTP, (int)freq);
	alt_clk_freq_get(ALT_CLK_GPIO, &freq);
	printf("ALT_CLK_GPIO(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_GPIO, (int)freq);
	alt_clk_freq_get(ALT_CLK_SDMMC, &freq);
	printf("ALT_CLK_SDMMC(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_SDMMC, (int)freq);
	alt_clk_freq_get(ALT_CLK_USB, &freq);
	printf("ALT_CLK_USB(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_USB, (int)freq);
	alt_clk_freq_get(ALT_CLK_SPIM, &freq);
	printf("ALT_CLK_SPIM(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_SPIM, (int)freq);
	alt_clk_freq_get(ALT_CLK_NAND, &freq);
	printf("ALT_CLK_NAND(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_NAND, (int)freq);
	alt_clk_freq_get(ALT_CLK_QSPI, &freq);
	printf("ALT_CLK_QSPI(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_QSPI, (int)freq);
	printf("\n");

	return;
}
#endif /* soc_cv_av */

/********************************************************************************//*!
 * @fn  void sample_clkmgr_test_print_pllconfig2(void)  
 * @brief  Get the information related to clock Manager and displays. 
 * 
 * @details  Displays the following information specifically.
 *           - Input Clock Frequency Value 
 *           - Output Clock Frequency and Divider Value 
 *           - setting information of Main PLL 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/11  ---  #6221  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
#ifdef soc_cv_av
void sample_clkmgr_test_print_pllconfig2(void)
{
	alt_freq_t freq[6];

	printf("==== Input Clock Frequency Value =====\n");
	printf("ALT_CLK_IN_PIN_OSC1(%d):\t", (int)ALT_CLK_IN_PIN_OSC1);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_IN_PIN_OSC1, 0);
	printf("ALT_CLK_IN_PIN_OSC2(%d):\t", (int)ALT_CLK_IN_PIN_OSC2);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_IN_PIN_OSC2, 0);
	printf("ALT_CLK_F2H_PERIPH_REF(%d):\t", (int)ALT_CLK_F2H_PERIPH_REF);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_F2H_PERIPH_REF, 0);
	printf("ALT_CLK_F2H_SDRAM_REF(%d):\t", (int)ALT_CLK_F2H_SDRAM_REF);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_F2H_SDRAM_REF, 0);
	printf("\n");
	printf("==== Output Clock Frequency and Divider Value =====\n");
	printf("ALT_CLK_MPU(%d):        \t", (int)ALT_CLK_MPU);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_MPU, 1);
	printf("ALT_CLK_MPU_L2_RAM(%d): \t", (int)ALT_CLK_MPU_L2_RAM);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_MPU_L2_RAM, 1);
	printf("ALT_CLK_MPU_PERIPH(%d): \t", (int)ALT_CLK_MPU_PERIPH);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_MPU_PERIPH, 1);
	printf("ALT_CLK_L4_MAIN(%d):    \t", (int)ALT_CLK_L4_MAIN);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_L4_MAIN, 1);
	printf("ALT_CLK_L4_MP(%d):      \t", (int)ALT_CLK_L4_MP);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_L4_MP, 1);
	printf("ALT_CLK_L4_SP(%d):      \t", (int)ALT_CLK_L4_SP);
	sample_clkmgr_print_freq_and_devider(ALT_CLK_L4_SP, 1);
	printf("\n");

	// Main PLL
	printf("==== setting information of Main PLL: (pll=%d) ====\n", (int)ALT_CLK_MAIN_PLL);
	sample_clkmgr_print_pllconfig(ALT_CLK_MAIN_PLL, &mainpllconfig);
	(void)alt_clk_freq_get(ALT_CLK_MAIN_PLL_C0, &freq[0]);
	(void)alt_clk_freq_get(ALT_CLK_MAIN_PLL_C1, &freq[1]);
	(void)alt_clk_freq_get(ALT_CLK_MAIN_PLL_C2, &freq[2]);
	(void)alt_clk_freq_get(ALT_CLK_MAIN_PLL_C3, &freq[3]);
	(void)alt_clk_freq_get(ALT_CLK_MAIN_PLL_C4, &freq[4]);
	(void)alt_clk_freq_get(ALT_CLK_MAIN_PLL_C5, &freq[5]);
	for(int i=0; i<6; i++){
		printf("             +-- C%d:Fout=%10d (Hz)\n", i, (int)freq[i]);
	}
	printf("\n");

	return;
}
#else /* soc_cv_av */
void sample_clkmgr_test_print_pllconfig2(void)
{
	alt_freq_t freq;

	printf("==== Input Clock Frequency Value =====\n");
	alt_clk_freq_get(ALT_CLK_OSC1, &freq);
	printf("ALT_CLK_OSC1(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_OSC1, (int)freq);
	alt_clk_freq_get(ALT_CLK_MAIN_PLL, &freq);
	printf("ALT_CLK_MAIN_PLL(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_MAIN_PLL, (int)freq);
	alt_clk_freq_get(ALT_CLK_PERIPHERAL_PLL, &freq);
	printf("ALT_CLK_PERIPHERAL_PLL(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_PERIPHERAL_PLL, (int)freq);
	printf("\n");
	printf("==== Output Clock Frequency Value =====\n");
	printf("---- Hardware Sequenced Clocks -----\n");
	printf("<< Clock Group : MPU >>\n");
	alt_clk_freq_get(ALT_CLK_MPU, &freq);
	printf("ALT_CLK_MPU(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_MPU, (int)freq);
	alt_clk_freq_get(ALT_CLK_MPU_PERIPH, &freq);
	printf("ALT_CLK_MPU_PERIPH(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_MPU_PERIPH, (int)freq);
	alt_clk_freq_get(ALT_CLK_MPU_L2_RAM, &freq);
	printf("ALT_CLK_MPU_L2_RAM(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_MPU_L2_RAM, (int)freq);
	printf("<< Clock Group : NOC >>\n");
	alt_clk_freq_get(ALT_CLK_L3_MAIN, &freq);
	printf("ALT_CLK_L3_MAIN(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_L3_MAIN, (int)freq);
	alt_clk_freq_get(ALT_CLK_L4_SYS_FREE, &freq);
	printf("ALT_CLK_L4_SYS_FREE(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_L4_SYS_FREE, (int)freq);
	alt_clk_freq_get(ALT_CLK_L4_MAIN, &freq);
	printf("ALT_CLK_L4_MAIN(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_L4_MAIN, (int)freq);
	alt_clk_freq_get(ALT_CLK_L4_MP, &freq);
	printf("ALT_CLK_L4_MP(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_L4_MP, (int)freq);
	alt_clk_freq_get(ALT_CLK_L4_SP, &freq);
	printf("ALT_CLK_L4_SP(%d):\tFrequency=%10d (Hz)\n", (int)ALT_CLK_L4_SP, (int)freq);
	printf("\n");

	return;
}
#endif /* soc_cv_av */

/********************************************************************************//*!
 * @fn  ALT_STATUS_CODE sample_clkmgr_change_mainpll(uint32_t ui32targetM)
 * @brief  Change the Main PLL frequency according to the M value. 
 * 
 * @details  Process by following these steps. 
 *           -# Judge M Value and Main PLL Current Lock Status. (Next disagreement only) 
 *           -# Get Current MainPLL Configuration Data 
 *           -# Record Current MainPLL Multiplier Value and Set Next Value. 
 *           -# Set Next MainPLL Configuration Data. 
 *           -# Print New Frequency!! 
 * @param[in] ui32targetM  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/11  ---  #6221  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
#ifdef soc_cv_av
ALT_STATUS_CODE sample_clkmgr_change_mainpll(uint32_t ui32targetM)
{
	static uint32_t ui32currentM = 63;
	static uint32_t ui32currentMbk = 63;
	ALT_STATUS_CODE	result_code = ALT_E_TRUE;

	// Judge M Value and Main PLL Current Lock Status.
	if(ui32currentM != ui32targetM)
	{
		if((alt_clk_lock_status_get() & ALT_CLKMGR_INTER_MAINPLLLOCKED_SET_MSK) != 0){
#ifdef GNU_COMPILER	// 99
			ALT_CLK_GROUP_RAW_CFG_t tMainGroupCfg;
#endif
#ifdef ARM_COMPILER	// 99
			ALT_CLK_GROUP_RAW_CFG_t* tMainGroupCfg = 0;
#endif
			alt_freq_t mpubasefreq;

			// Get Current MainPLL Configuration Data
			/* Usage: ALT_STATUS_CODE alt_clk_group_cfg_raw_get(ALT_CLK_GRP_t clk_group, ALT_CLK_GROUP_RAW_CFG_t* clk_group_raw_cfg);	*/
#ifdef GNU_COMPILER	// 99
			result_code = alt_clk_group_cfg_raw_get(ALT_MAIN_PLL_CLK_GRP, &tMainGroupCfg);
#endif
#ifdef ARM_COMPILER	// 99
			result_code = alt_clk_group_cfg_raw_get(ALT_MAIN_PLL_CLK_GRP, (ALT_CLK_GROUP_RAW_CFG_t*)&tMainGroupCfg);
#endif
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_clk_group_cfg_raw_get() result=%d\n", (int)result_code);
			} else {

				// Record Current MainPLL Multiplier Value and Set Next Value.
#ifdef GNU_COMPILER	// 99
				ui32currentMbk = tMainGroupCfg.clkgrp.mainpllgrp.fld.vco.numer;
#endif
#ifdef ARM_COMPILER	// 99
				ui32currentMbk = tMainGroupCfg->clkgrp.mainpllgrp.fld.vco.numer;
#endif
				if(ui32targetM > ui32currentM){
					ui32currentM = ui32currentMbk + 1;
				} else if(ui32targetM < ui32currentM){
					ui32currentM = ui32currentMbk - 1;
				} else {
					ui32currentM = ui32currentMbk;
				}

#ifdef GNU_COMPILER	// 99
				tMainGroupCfg.clkgrp.mainpllgrp.fld.vco.numer = ui32currentM;
#endif
#ifdef ARM_COMPILER	// 99
				tMainGroupCfg->clkgrp.mainpllgrp.fld.vco.numer = ui32currentM;
#endif
				// Set Next MainPLL Configuration Data
				/* Usage: ALT_STATUS_CODE alt_clk_group_cfg_raw_set(const ALT_CLK_GROUP_RAW_CFG_t* clk_group_raw_cfg);	*/
#ifdef GNU_COMPILER	// 99
				result_code = alt_clk_group_cfg_raw_set(&tMainGroupCfg);
#endif
#ifdef ARM_COMPILER	// 99
				result_code = alt_clk_group_cfg_raw_set((ALT_CLK_GROUP_RAW_CFG_t*)&tMainGroupCfg);
#endif
				if(result_code != ALT_E_SUCCESS){
					printf("ERROR!!: alt_clk_group_cfg_raw_set() result=%d\n", (int)result_code);
				} else {
					printf("##Change mpu_base_clk Multiplier: %d --> %d\n", (int)ui32currentMbk, (int)ui32currentM);
					// Print New Frequency
					result_code = alt_clk_freq_get(ALT_CLK_MPU, &mpubasefreq);
					if(result_code != ALT_E_SUCCESS){
						printf("ERROR!!: alt_clk_freq_get() result=%d\n", (int)result_code);
					} else {
						printf(" * mpu_base_clk Frequency = %10d (Hz)\n", (int)mpubasefreq);
					}
				}
			}
		}

		result_code = ALT_E_FALSE;
	}

	return result_code;
}
#else /* soc_cv_av */
ALT_STATUS_CODE sample_clkmgr_change_mainpll(uint32_t ui32targetM)
{
	static uint32_t ui32currentM = 0xBF;
	static uint32_t ui32currentMbk = 0xBF;
	ALT_STATUS_CODE	result_code = ALT_E_TRUE;

	// Judge M Value and Main PLL Current Lock Status.
	if(ui32currentM != ui32targetM)
	{
		if((alt_read_word(ALT_CLKMGR_CLKMGR_STAT_ADDR) & ALT_CLKMGR_CLKMGR_STAT_MAINPLLLOCKED_SET_MSK) != 0){
			alt_freq_t mpubasefreq;

			// Get Current MainPLL Configuration Data
			// Record Current MainPLL Multiplier Value and Set Next Value.
			ui32currentMbk = ALT_CLKMGR_MAINPLL_VCO1_NUMER_GET(alt_read_word(ALT_CLKMGR_MAINPLL_VCO1_ADDR));

			if(ui32targetM > ui32currentM){
				ui32currentM = ui32currentMbk + 1;
			} else if(ui32targetM < ui32currentM){
				ui32currentM = ui32currentMbk - 1;
			} else {
				ui32currentM = ui32currentMbk;
			}

			// Set Next MainPLL Configuration Data
			alt_replbits_word(ALT_CLKMGR_MAINPLL_VCO1_ADDR, ALT_CLKMGR_MAINPLL_VCO1_NUMER_SET_MSK, ui32currentM);

			printf("##Change mpu_base_clk Multiplier: %d --> %d\n", (int)ui32currentMbk, (int)ui32currentM);
			fflush(stdout);
			// Print New Frequency
			result_code = alt_clk_freq_get(ALT_CLK_MPU, &mpubasefreq);
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_clk_freq_get() result=%d\n", (int)result_code);
			} else {
				printf(" * mpu_base_clk Frequency = %10d (Hz)\n", (int)mpubasefreq);
			}
		}

		result_code = ALT_E_FALSE;
	}

	return result_code;
}
#endif /* soc_cv_av */

/********************************************************************************//*!
 * @fn  void sample_clkmgr_test_main(void) 
 * @brief  Main routine of the sample program. 
 * 
 * @details  Infinite loop structure works. Make the detection of various types 
 *           of switches in the loop, it performs in various testing processes.
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/11  ---  #6221  新規作成. 
 * @date 2014/01/30  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void sample_clkmgr_test_main(void)
{
	uint32_t ui32high, ui32low;
	uint32_t ui32lowbk = 0;
	
	uint32_t switch_raw = 0;
	uint32_t switch_raw_bk = 0;
	uint32_t switch_detect = 0;
	uint32_t switch_detect_on = 0;
	uint32_t switch_number = 0;
	bool disp_usage = true;
	
	uint32_t ui32targetM = 63;
	ALT_STATUS_CODE	result_code;
	bool issafenml, issafedbg __WARN_UNUSED__;

	util_intlog_print();

	printf("==== Start While(1) loop process!!! ============\n");
	switch_raw_bk = sample_detect_switch();
	while(1)
	{
		sample_clkmgr_change_mainpll(ui32targetM);
		if(disp_usage)
		{
			sample_clkmgr_test_print_usage();
			disp_usage = false;
		}

		// ---- Check the Slide-Switch and Push-Switch. ---- 
		switch_raw = sample_detect_switch();
		switch_detect = switch_raw ^ switch_raw_bk;
		switch_detect_on  |= switch_detect & switch_raw;
		
		// ---- Check the Global Timer Registers. ----
		result_code = alt_globaltmr_get(&ui32high, &ui32low);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_globaltmr_get() result=%d\n", (int)result_code);
		}
		
		// Slide-Switch 
		if(switch_detect & SAMPLE_SWITCH_BIT_SLIDEALL)
		{
			switch_number = switch_raw & SAMPLE_SWITCH_BIT_NUM;
			switch_number *= 1;		// To avoid warnings.

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE0){
				//printf("SAMPLE_SWITCH_BIT_SLIDE0\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE1){
				//printf("SAMPLE_SWITCH_BIT_SLIDE1\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE2){
				//printf("SAMPLE_SWITCH_BIT_SLIDE2\n");
			}
			if(switch_detect & SAMPLE_SWITCH_BIT_SLIDE3){
				//printf("SAMPLE_SWITCH_BIT_SLIDE3\n");
			}
			
			// Detect Main Pll Multiplier Value from DIPSW1-4 Value.
#ifdef soc_cv_av
			ui32targetM = ((switch_number + 1) * 4) - 1;
#else /* soc_cv_av *//* At arria10 (1.2 GHz), the maximum value of M is 191. */
			ui32targetM = ((switch_number + 1) * 12) - 1;
			if(ui32targetM < 26)	ui32targetM = 26;
#endif /* soc_cv_av */
			
			disp_usage = false;
		}
		
		
		// Push-Switch 0
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH0)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH0;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH0\n");
			break;	// Exit Push-Switch 0 and 1 becomes ON.
		}
		// Push-Switch 1
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH1)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH1;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH1\n");
#ifdef soc_cv_av
			/* Usage: bool alt_clk_is_in_safe_mode(ALT_CLK_SAFE_DOMAIN_t clk_domain;	*/
			printf("-- HWLib test : alt_clk_is_in_safe_mode() --\n");
			issafenml = alt_clk_is_in_safe_mode(ALT_CLK_DOMAIN_NORMAL);
			issafedbg = alt_clk_is_in_safe_mode(ALT_CLK_DOMAIN_DEBUG);
			printf("Clock is in Safe Mode (ALT_CLK_DOMAIN_NORMAL): %d\n", issafenml);
			printf("Clock is in Safe Mode (ALT_CLK_DOMAIN_DEBUG) : %d\n", issafedbg);
			if(issafenml || issafedbg){
				/* Usage: ALT_STATUS_CODE alt_clk_safe_mode_clear(void);	*/
				printf("-- HWLib test : alt_clk_safe_mode_clear() --\n");
				result_code = alt_clk_safe_mode_clear();
				if(result_code != ALT_E_SUCCESS){
					printf("ERROR!!: alt_clk_safe_mode_clear() result=%d\n", (int)result_code);
				} else {
					printf("SUCCESS!!\n");
				}
			}
#else /* soc_cv_av */
			printf("The name of Safe Mode has changed to Boot Mode in Arria 10.\n");
			printf("clkmgr_clkmgrp.ctrl = 0x%08X\n", (int)alt_read_word(ALT_CLKMGR_CLKMGR_CTL_ADDR));
			issafenml = ALT_CLKMGR_CLKMGR_CTL_BOOTMOD_GET(alt_read_word(ALT_CLKMGR_CLKMGR_CTL_ADDR));
			printf("Clock is in Boot Mode: %d\n", issafenml);
			if(issafenml){
				alt_clrbits_word(ALT_CLKMGR_CLKMGR_CTL_ADDR, ALT_CLKMGR_CLKMGR_CTL_BOOTMOD_SET_MSK);
			}
#endif /* soc_cv_av */
			disp_usage = true;
		}
		// Push-Switch 2
		if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH2)
		&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH2;
			
			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH2\n");
#ifdef soc_cv_av
			/* Usage: ALT_STATUS_CODE alt_clk_lock_status_clear(ALT_CLK_PLL_LOCK_STATUS_t lock_stat_mask);	*/
			printf("-- HWLib test : alt_clk_lock_status_clear() --\n");
#ifdef GNU_COMPILER	// 99
			result_code = alt_clk_lock_status_clear(ALT_PLLLOCK_STATUSBIT_ALL);
#endif
#ifdef ARM_COMPILER	// 99
			result_code = alt_clk_lock_status_clear((ALT_CLK_PLL_LOCK_STATUS_t)ALT_PLLLOCK_STATUSBIT_ALL);
#endif
			if(result_code != ALT_E_SUCCESS){
				printf("ERROR!!: alt_clk_lock_status_clear() result=%d\n", (int)result_code);
			} else {
				printf("SUCCESS!!\n");
#else /* soc_cv_av */
			{
#endif /* soc_cv_av */
				sample_clkmgr_print_plllock();
			}
			sample_clkmgr_test_print_pllconfig2();
			disp_usage = true;
		}
		// Push-Switch 3
		//if( (switch_detect_on & SAMPLE_SWITCH_BIT_PUSH3)
		//&&(!(switch_raw & SAMPLE_SWITCH_BIT_PUSHALL)))
		if(switch_raw & SAMPLE_SWITCH_BIT_PUSH3)
		{
			switch_detect_on &= ~SAMPLE_SWITCH_BIT_PUSH3;

			/* ######################################## */
			/* ## Implement the test process here!!! ## */
			/* ######################################## */
			//printf("SAMPLE_SWITCH_BIT_PUSH3\n");
			printf("GlobalTimer Counter = 0x%08X:%08X\n", (int)ui32high, (int)ui32low);
		} else {
			// Global Timer Counter Value
			if((ui32low ^ ui32lowbk) & 0xF0000000)
			{
				printf("GlobalTimer Counter = 0x%08X:%08X\n", (int)ui32high, (int)ui32low);
			}
		}
		
		util_intlog_print();

		ui32lowbk = ui32low;
		switch_raw_bk = switch_raw;
	}
	printf("==== End While(1) loop process. ================\n");

	util_intlog_print();

	return;
}

/********************************************************************************//*!
 * @fn  void sample_clkmgr_test_print_usage(void) 
 * @brief  Display usage of the sample program. 
 * 
 * @details  Following information displays on the console.
 *            - Overview of HPS for push button functions (or operations)
 *            - Overview of HPS for DIP switch functions (or operations)
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/08/18  ---  #xxxxx  新規作成. 
 **//*******************************************************************************/
void sample_clkmgr_test_print_usage(void)
{
	printf("\n");
	printf("+--- Usage ( Function of PUSH SW and SLIDE SW )  ---------------+\n");
	printf("+ PUSH SW #0 .... Exit Test loop!!! \n");
	printf("+ PUSH SW #1 .... Display the Safe Mode status and clear.\n");
	printf("+ PUSH SW #2 .... Display the PLL lock/unlock status and clear.\n");
	printf("+ PUSH SW #3 .... Display the Global Timer counter.\n");
	printf("                  (Display continuously during push-down)\n");
	printf("+ SLIDE SW#0:3 .. Change Main clock frequency dynamically.\n");
	printf("     Slowest SLIDE SW[0:3]=0000, and 1111 is the fastest.\n");
	printf("+---------------------------------------------------------------+\n\n");
	
#if 0
	※HPS用DIPSW1-4の切り替え契機でMain PLLの M(1-4096) を
 	　変更するサンプルです。（HPSのMainクロック周波数切り替えを試す。）

	 HPS用DIPSW1-4の値に応じて、Mainクロックを下記の通りに変更します。
	 [DIPSW1-4]
	 [0000] .. M=4  mpu_clk=50MHz
	 [0001] .. M=8  mpu_clk=100MHz
	 [0010] .. M=12 mpu_clk=150MHz
	 [0011] .. M=16 mpu_clk=200MHz
	 [0100] .. M=20 mpu_clk=250MHz
	 [0101] .. M=24 mpu_clk=300MHz
	 [0110] .. M=28 mpu_clk=350MHz
	 [0111] .. M=32 mpu_clk=400MHz
	 [1000] .. M=36 mpu_clk=450MHz
	 [1001] .. M=40 mpu_clk=500MHz
	 [1010] .. M=44 mpu_clk=550MHz
	 [1011] .. M=48 mpu_clk=600MHz
	 [1100] .. M=52 mpu_clk=650MHz
	 [1101] .. M=56 mpu_clk=700MHz
	 [1110] .. M=60 mpu_clk=750MHz
	 [1111] .. M=64 mpu_clk=800MHz

	　※注意事項： MainPLLを変更する際には、1回の変化を20%以内とする必要あり。
	　-- quote from "Cyclone V Device Handbook Volume3" --- 
	　Once a PLL is locked, changes to any PLL VCO frequency that are 20 percent or less
	　do not cause the PLL to lose lock. Iteratively changing the VCO frequency in
	　increments of 20 percent or less allow a slow ramp of the VCO base frequency without
	　loss of lock.For example, to change a VCO frequency by 40% without losing lock,
	　change the frequency by 20%, then change it again by 16.7%.
	
	　無限ループ内で以下の処理を行う。
	　 - GlobalTimerカウンタ値の下から28bit目以上（下位32bitの
	　　 bit#31:28）が変化する毎にカウンタ値をコンソール出力。
	　　 HPS用PUSHSW3が押されている間は無条件にコンソール出力。
	　　 (Timerカウンタの進み具合の確認用)
	
	　 - HPS用PUSHSW2の押下契機で カテゴリ：Clock Manager Status の
	　　 APIを試行します。PLL Lock/Unlock Statusのクリアおよび表示。
	　　 (Clock Manager Status APIの試行)
 
	　 - HPS用PUSHSW1の押下契機で カテゴリ：Safe Mode Options の
	　　 APIを試行します。Safe Modeステータスの表示およびクリア 
	　　 (Safe Mode Options APIの試行)
	
	　 - HPS用DIPSW1-4の変更契機で、PLLのMultiplier設定値の変更および
	　　 変更後のクロック周波数の表示を試みます。
	   Main PLLの M(1-4096)を対象に、 DIPSW1をMSB、DIPSW4をLSBとして
	   　 (4bit値 × 4)をMultiplierの設定値に適用します。
	  　  ※  尚、PLLの変更中はsecureモードで動作させるようにPLL変更前後で
	  　　 下記のAPIを実行します。

	　 - HPS用PUSHSW0の押下契機で無限ループを終了します。
	　　 (テストプログラム終了)
#endif
	
	return;
}


/************************************/
/*	Test Command (Clock Manager)	*/
/************************************/
int sample_clkmgr_test_cmd(char* options)
{
    ALT_STATUS_CODE status = ALT_E_SUCCESS;
    
    printf("\r\nUser Application Start!\r\n");

    //
    // Start the interrupt system
    //
    if (status == ALT_E_SUCCESS)
    {
    	util_intlog_init();
        status = socfpga_int_start();
    }

    //// Sample Function Call!!! |---->
    printf(">>>> Execute sample_clkmgr_init(); !!! <<<<\r\n");
    sample_clkmgr_init();
	util_intlog_print();

    printf(">>>> Execute sample_clkmgr_test_print_pllconfig(); !!! <<<<\r\n");
    sample_clkmgr_test_print_pllconfig();

    printf(">>>> Execute sample_clkmgr_test_customize_config(); !!! <<<<\r\n");
	sample_clkmgr_test_customize_config();

    printf(">>>> Re-Execute sample_clkmgr_test_print_pllconfig(); !!! <<<<\r\n");
	sample_clkmgr_test_print_pllconfig();

    printf(">>>> sample_clkmgr_test(); !!! <<<<\r\n");
    sample_clkmgr_test_main();
    //// Sample Function Call!!! <----|

    //
    // Stop the interrupt system
    //
    socfpga_int_stop();
	util_intlog_print();

    printf("Finished running the sample !!!\r\n");
    if (status == ALT_E_SUCCESS)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}

// All APIs
/***********************************************************************/
/* Usage: ALT_STATUS_CODE alt_clk_lock_status_clear(ALT_CLK_PLL_LOCK_STATUS_t lock_stat_mask);	*/
/* Usage: uint32_t alt_clk_lock_status_get(void);	*/
/* Usage: ALT_STATUS_CODE alt_clk_pll_is_locked(ALT_CLK_t pll);	*/
/* Usage: ALT_STATUS_CODE alt_clk_safe_mode_clear(void);	*/
/* Usage: bool alt_clk_is_in_safe_mode(ALT_CLK_SAFE_DOMAIN_t clk_domain;	*/
/* Usage: ALT_STATUS_CODE alt_clk_pll_bypass_disable(ALT_CLK_t pll);	*/
/* Usage: ALT_STATUS_CODE alt_clk_pll_bypass_enable(ALT_CLK_t pll, bool use_input_mux);	*/
/* Usage: ALT_STATUS_CODE alt_clk_pll_is_bypassed(ALT_CLK_t pll);	*/
/* Usage: ALT_STATUS_CODE alt_clk_clock_disable(ALT_CLK_t clk);	*/
/* Usage: ALT_STATUS_CODE alt_clk_clock_enable(ALT_CLK_t clk);	*/
/* Usage: ALT_STATUS_CODE alt_clk_is_enabled(ALT_CLK_t clk);	*/
/* Usage: ALT_CLK_t alt_clk_source_get(ALT_CLK_t clk);	*/
/* Usage: ALT_STATUS_CODE alt_clk_source_set(ALT_CLK_t clk, ALT_CLK_t ref_clk);	*/
/* Usage: ALT_STATUS_CODE alt_clk_ext_clk_freq_set(ALT_CLK_t clk, alt_freq_t freq);	*/
/* Usage: alt_freq_t alt_clk_ext_clk_freq_get(ALT_CLK_t clk);	*/
/* Usage: ALT_STATUS_CODE alt_clk_pll_cfg_get(ALT_CLK_t pll, ALT_CLK_PLL_CFG_t* pll_cfg);	*/
/* Usage: ALT_STATUS_CODE alt_clk_pll_cfg_set(ALT_CLK_t pll, const ALT_CLK_PLL_CFG_t* pll_cfg);	*/
/* Usage: ALT_STATUS_CODE alt_clk_pll_vco_cfg_get(ALT_CLK_t pll, uint32_t* mult, uint32_t* div);	*/
/* Usage: ALT_STATUS_CODE alt_clk_pll_vco_cfg_set(ALT_CLK_t pll, uint32_t mult, uint32_t div);	*/
/* Usage: ALT_STATUS_CODE alt_clk_pll_vco_freq_get(ALT_CLK_t pll, alt_freq_t* freq);	*/
/* Usage: uint32_t alt_clk_pll_guard_band_get(ALT_CLK_t pll);	*/
/* Usage: ALT_STATUS_CODE alt_clk_pll_guard_band_set(ALT_CLK_t pll, uint32_t guard_band);	*/
/* Usage: ALT_STATUS_CODE alt_clk_divider_get(ALT_CLK_t clk, uint32_t* div);	*/
/* Usage: ALT_STATUS_CODE alt_clk_divider_set(ALT_CLK_t clk, uint32_t div);	*/
/* Usage: ALT_STATUS_CODE alt_clk_freq_get(ALT_CLK_t clk, alt_freq_t* freq);	*/
/* Usage: ALT_STATUS_CODE alt_clk_irq_disable(ALT_CLK_PLL_LOCK_STATUS_t lock_stat_mask);	*/
/* Usage: ALT_STATUS_CODE alt_clk_irq_enable(ALT_CLK_PLL_LOCK_STATUS_t lock_stat_mask);	*/
/* Usage: ALT_STATUS_CODE alt_clk_group_cfg_raw_get(ALT_CLK_GRP_t clk_group, ALT_CLK_GROUP_RAW_CFG_t* clk_group_raw_cfg);	*/
/* Usage: ALT_STATUS_CODE alt_clk_group_cfg_raw_set(const ALT_CLK_GROUP_RAW_CFG_t* clk_group_raw_cfg);	*/
/***********************************************************************/

/***********************************************************************************
 * end of file 
 ***********************************************************************************/
