/* file GSFI_Operation_sample.c
 *
 * Copyright (C) 2021 MACNICA,Inc. All Rights Reserved.
 *  This software is licensed "AS IS".
 *  Please perform use of this software by a user's own responsibility and expense.
 *  It cannot guarantee in the maker side about the damage which occurred
 *  by the ab-ility not to use or use this software, and all damage that occurred secondarily.
 * Created on: 2021/07/15
 */
#include "io.h"
#include <unistd.h>
#include <stdio.h>
#include <string.h>

#include "GSFI_Operation_sample.h"
#include "sys/alt_cache.h"
#include "sys/errno.h"
#include "system.h"
/***********************************************************************************
 * flash_info_init
 *
 * GSFI ̃p[^ݒ̏ Flash ROM ̏\̂Ɋi[܂B
 * gp Flash ROM ̎dlɍ킹ĕύXĎgpB
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *@- gsfi_csr_base: GSFI  CSR BASE Address
 *@- gsfi_mem_base: GSFI  MEM BASE Address
 *@- gsfi_mem_span: GSFI  ݒ肵 Flash ROM ̗e
 *@- gsfi_csr_span: GSFI  CSR ̃[e
 *@- io_mode: Standard DUAL QUAD ̏
 * Returns:
 * - 0: Open 
 * - -19(ENODEV): GSFI Őݒ肵eʂƁAڑĂ Flash ROM ̗eʂĂȂƂ
 * - -5(EIO): Addressing ̐ݒɎsƂ
 ***********************************************************************************/
int flash_info_init(gsfi_flash_info *flash_info, alt_u32 gsfi_csr_base, alt_u32 gsfi_mem_base, alt_u32 gsfi_mem_span, alt_u32 gsfi_csr_span, alt_u32 io_mode, alt_u32 data_size){
	int ret_code = 0;
	int device_id = 0x00FF0000;
	int num_device_table = 0;

	static const gsfi_flash_const flash_info_table[]={
			{1,    32768, 4096,  4,    3}, // num_device_table = 0  for 1
			{2,    0,     0,     0,    0}, // num_device_table = 1  for 2
			{4,    65536, 4096,  8,    3}, // num_device_table = 2  for 4
			{8,    65536, 4096,  16,   3}, // num_device_table = 3  for 8
			{16,   65536, 4096,  32,   3}, // num_device_table = 4  for 16
			{32,   65536, 4096,  64,   3}, // num_device_table = 5  for 32
			{64,   65536, 4096,  128,  3}, // num_device_table = 6  for 64
			{128,  65536, 4096,  256,  3}, // num_device_table = 7  for 128
			{256,  65536, 4096,  512,  4}, // num_device_table = 8  for 256
			{512,  65536, 4096,  1024, 4}, // num_device_table = 9  for 512
			{1024, 65536, 4096,  2048, 4}, // num_device_table = 10 for 1024
			{0,    0,     0,     0,    0}, // num_device_table = 11 for None
	};

	flash_info->gsfi_csr_base  = gsfi_csr_base;
	flash_info->gsfi_mem_base  = gsfi_mem_base;
	flash_info->gsfi_csr_span  = gsfi_csr_span;
	flash_info->gsfi_mem_span  = gsfi_mem_span;
	flash_info->io_mode        = io_mode;
	flash_info->data_size      = data_size;
	device_id =  (device_id & read_device_id(flash_info)) >> 16;

	if(flash_info->gsfi_mem_span == 131072 && device_id == 0x10){          // 1       131072  id = 0x10
		num_device_table = 0;
	}else if(flash_info->gsfi_mem_span == 262144){                         // 2       262144
		ret_code = -ENODEV;
		num_device_table = 1;
	}else if(flash_info->gsfi_mem_span == 524288 && device_id == 0x13){    // 4       524288  id = 0x13
		num_device_table = 2;
	}else if(flash_info->gsfi_mem_span == 1048576 && device_id == 0x14){   // 8      1048576  id = 0x14
		num_device_table = 3;
	}else if(flash_info->gsfi_mem_span == 2097152 && device_id == 0x15){   // 16     2097152  id = 0x15
		num_device_table = 4;
	}else if(flash_info->gsfi_mem_span == 4194304 && device_id == 0x16){   // 32     4194304  id = 0x16
		num_device_table = 5;
	}else if(flash_info->gsfi_mem_span == 8388608 && device_id == 0x17){   // 64     8388608  id = 0x17
		num_device_table = 6;
	}else if(flash_info->gsfi_mem_span == 16777216 && device_id == 0x18){  // 128   16777216  id = 0x18
		num_device_table = 7;
	}else if(flash_info->gsfi_mem_span == 33554432 && device_id == 0x19){  // 256   33554432  id = 0x19
		num_device_table = 8;
	}else if(flash_info->gsfi_mem_span == 67108864 && device_id == 0x20){  // 512   67108864  id = 0x20
		num_device_table = 9;
	}else if(flash_info->gsfi_mem_span == 134217728 && device_id == 0x21){ // 1024 134217728  id = 0x21
		num_device_table = 10;
	}else{                                                                 // not match
		ret_code = -ENODEV;
		num_device_table = 11;
	}

	flash_info->flash_rom        = flash_info_table[num_device_table].flash_rom;
	flash_info->sector_size      = flash_info_table[num_device_table].sector_size;
	flash_info->subsector_size   = flash_info_table[num_device_table].subsector_size;
	flash_info->number_of_sector = flash_info_table[num_device_table].number_of_sector;
	flash_info->addressing_mode  = flash_info_table[num_device_table].addressing_mode;
	// Addressing Mode Set
	if(flash_info->addressing_mode == 0x4){
		IOWR(flash_info->gsfi_csr_base,GSFI_CONTROL_REG,0x00000101);
#if MACRONIX_ISSI
		enable_4byte_for_mx25L(flash_info);
#endif
	}else if(flash_info->addressing_mode == 0x3){
		IOWR(flash_info->gsfi_csr_base,GSFI_CONTROL_REG,0x00000001);
#if MACRONIX_ISSI
		disable_4byte_for_mx25L(flash_info);
#endif
	}else{
		ret_code = EIO;
	}
	return ret_code;
}
/***********************************************************************************
 * change_clock_rate
 *
 * GSFI DCLK̎gύX܂B
 * 0x1 ~ 0x10 ȊO̒lw肵ꍇAError ɂȂ܂B
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *@- clock_rate: 0x1 ~ 0x10
 * Returns:
 * - clock_rate: Success
 * - -5(EINVAL): clock rate ̒lAdl𒴂Ăꍇ
 ***********************************************************************************/
int change_clock_rate(gsfi_flash_info *flash_info, alt_u32 clock_rate){
	if(clock_rate > 0 && clock_rate <= 0x10){
		IOWR(flash_info->gsfi_csr_base, GSFI_SPI_CLOCK_REG, clock_rate);
		return IORD(flash_info->gsfi_csr_base, GSFI_SPI_CLOCK_REG);
	}else{
		return -EINVAL;
	}
}
/***********************************************************************************
 * change_io_mode
 *
 * GSFI  IO Mode ύX܂B
 * 0 ~ 2 ȊO̒lw肵ꍇAErrorɂȂ܂B
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *@- io_mode_id: 0 ~ 2
 * Returns:
 * - 0: Success
 * - -22(EINVAL): io_mode_id ̒lAdl𒴂Ăꍇ
 *
 ***********************************************************************************/
int change_io_mode(gsfi_flash_info *flash_info, alt_u32 io_mode_id){
	int cmd_set = 0;
#if MACRONIX_ISSI
	int read_status0;
	int read_status1;
#endif
	switch(io_mode_id){

	case STANDARD:
#if MACRONIX_ISSI
		read_status0 = read_status_register(flash_info);
		read_status0 = read_status0 & ~0x40;
		write_status_reg(flash_info, read_status0); // Status Register [6]@Write Standard for MX25L IS25L
#endif
		cmd_set = (
				(GSFI_READ_INST_DUMMY_CYCLE_MASK & READ_BYTES_DUMMY << GSFI_READ_INST_DUMMY_CYCLE_BIT)
				+(READ_BYTES_CMD)
		);
		IOWR(flash_info->gsfi_csr_base,GSFI_READ_INST_REG,cmd_set); //0x00000003
		cmd_set = (
				(GSFI_WRITE_INST_POLLING_OPECODE_MASK & READ_STATUS_CMD << GSFI_WRITE_INST_POLLING_OPECODE_BIT)
				+(WRITE_BYTES_CMD)
		);
		IOWR(flash_info->gsfi_csr_base,GSFI_WRITE_INST_REG,cmd_set); //0x00000502
		cmd_set = io_mode_id;
		IOWR(flash_info->gsfi_csr_base,GSFI_OPERATE_PROTCOL_REG,cmd_set); //0x00000000

		break;

	case DUAL:
		cmd_set = (
				(GSFI_READ_INST_DUMMY_CYCLE_MASK & EXTENDED_DUAL_FAST_READ_DUMMY << GSFI_READ_INST_DUMMY_CYCLE_BIT)
				+(EXTENDED_DUAL_FAST_READ_CMD)
		);
		IOWR(flash_info->gsfi_csr_base,GSFI_READ_INST_REG,cmd_set); //0x0000008BB
		cmd_set = (
				(GSFI_WRITE_INST_POLLING_OPECODE_MASK & READ_STATUS_CMD << GSFI_WRITE_INST_POLLING_OPECODE_BIT)
				+(EXTENDED_DUAL_FAST_WRITE_CMD)
		);
		IOWR(flash_info->gsfi_csr_base,GSFI_WRITE_INST_REG,cmd_set); //0x0000005D2
		cmd_set = (
				(GSFI_OPERATE_PROTCOL_READ_DATA_MASK & io_mode_id << GSFI_OPERATE_PROTCOL_READ_DATA_BIT)
				+(GSFI_OPERATE_PROTCOL_READ_ADD_MASK & io_mode_id << GSFI_OPERATE_PROTCOL_READ_ADD_BIT)
				+(GSFI_OPERATE_PROTCOL_WRITE_DATA_MASK & io_mode_id << GSFI_OPERATE_PROTCOL_WRITE_DATA_BIT)
				+(GSFI_OPERATE_PROTCOL_WRITE_ADD_MASK  & io_mode_id << GSFI_OPERATE_PROTCOL_WRITE_ADD_BIT)
				//+(io_mode_id)
		);
		IOWR(flash_info->gsfi_csr_base,GSFI_OPERATE_PROTCOL_REG,cmd_set); //0x00011110
		break;

	case QUAD:
#if MACRONIX_ISSI
		read_status1 = read_status_register(flash_info);
		read_status1 = read_status1 | 0x40;
		write_status_reg(flash_info, read_status1); // Status Register [6]@Write QUAD for MX25L IS25L
#endif
		cmd_set = (
				(GSFI_READ_INST_DUMMY_CYCLE_MASK & EXTENDED_QUAD_FAST_READ_DUMMY << GSFI_READ_INST_DUMMY_CYCLE_BIT)
				+(EXTENDED_QUAD_FAST_READ_CMD)
		);
		IOWR(flash_info->gsfi_csr_base,GSFI_READ_INST_REG,cmd_set); //0x000000AEB
		cmd_set = (
				(GSFI_WRITE_INST_POLLING_OPECODE_MASK & READ_STATUS_CMD << GSFI_WRITE_INST_POLLING_OPECODE_BIT)
				+(EXTENDED_QUAD_FAST_WRITE_CMD)
		);
		IOWR(flash_info->gsfi_csr_base,GSFI_WRITE_INST_REG,cmd_set); //0x000000512
		cmd_set = (
				(GSFI_OPERATE_PROTCOL_READ_DATA_MASK  & io_mode_id << GSFI_OPERATE_PROTCOL_READ_DATA_BIT)
				+(GSFI_OPERATE_PROTCOL_READ_ADD_MASK   & io_mode_id << GSFI_OPERATE_PROTCOL_READ_ADD_BIT)
				+(GSFI_OPERATE_PROTCOL_WRITE_DATA_MASK  & io_mode_id << GSFI_OPERATE_PROTCOL_WRITE_DATA_BIT)
#if !ADDRESS_IO_MODE
				+(GSFI_OPERATE_PROTCOL_WRITE_ADD_MASK   & io_mode_id << GSFI_OPERATE_PROTCOL_WRITE_ADD_BIT)
#endif
				//+(io_mode_id)
		);
		IOWR(flash_info->gsfi_csr_base,GSFI_OPERATE_PROTCOL_REG,cmd_set); //00022220
		break;
	default:
		return -EINVAL;
	}
	return 0;
}
/***********************************************************************************
 * read_various_register
 *
 * lX Read Register R}hɑΉĂ܂B
 * Read lԂ܂B
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *@- cmd: Operation Codes
 *@- num_data_bytes: Read Data Bytes
 *
 * Returns:
 * - read_config_register
 ***********************************************************************************/
int read_various_register(gsfi_flash_info *flash_info, alt_u32 cmd, alt_u32 num_data_bytes){
	int cmd_set = 0;
	num_data_bytes = num_data_bytes << GSFI_CMD_SET_NUM_DATA_BYTES_BIT;
	cmd_set = (
			(GSFI_CMD_SET_NUM_DATA_BYTES_MASK & num_data_bytes)
			+(GSFI_CMD_SET_DATA_TYPE_MASK)
			+(cmd)
	);
	IOWR(flash_info->gsfi_csr_base,GSFI_CMD_SET_REG,cmd_set);
	IOWR(flash_info->gsfi_csr_base,GSFI_CMD_CONTROL_REG,GSFI_CMD_CONTROL_START_MASK);
	return IORD(flash_info->gsfi_csr_base,GSFI_CMD_READ_DATA0_REG);

}
/***********************************************************************************
 * read_device_id
 *
 * read_device_id 擾āAlԂ܂B
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *
 * Returns:
 * - read_status_register
 ***********************************************************************************/
int read_device_id(gsfi_flash_info *flash_info){
	return read_various_register(flash_info, READ_DEVICE_ID_CMD, READ_DEVICE_ID_BYTES);
}
/***********************************************************************************
 * read_status_register
 *
 * read_status_register 擾āAlԂ܂B
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *
 * Returns:
 * - read_status_register
 ***********************************************************************************/
int read_status_register(gsfi_flash_info *flash_info){
	return read_various_register(flash_info, READ_STATUS_CMD, READ_STATUS_BYTES);
}
/***********************************************************************************
 * read_flag_status_register
 *
 * read_flag_status_register 擾āAlԂ܂B
 * EPCQA ́Aread_flag_status_register@̂ŁAӁB
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *
 * Returns:
 * - read_flag_status_register
 ***********************************************************************************/
int read_flag_status_register(gsfi_flash_info *flash_info){
	return read_various_register(flash_info, READ_FLAG_STATUS_CMD, READ_FLAG_STATUS_BYTES);
}
/***********************************************************************************
 * read_config_register
 *
 * read_config_register 擾āAlԂ܂B
 *  *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *
 * Returns:
 * - read_config_register
 ***********************************************************************************/
int read_config_register(gsfi_flash_info *flash_info){
	return read_various_register(flash_info, READ_NON_VOLA_CONFIG_CMD, READ_NON_VOLA_CONFIG_BYTES);
}
/***********************************************************************************
 * write_various_register
 *
 * lX Write Register R}hɑΉĂ܂B
 * Write Enable ʊ֐ŎsĂǂ݂ĂB
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *@- cmd: Operation Codes
 *@- num_data_bytes: Read Data Bytes
 *@- write_data: Write Data
 *
 * Returns:
 * - 0: Not Error
 * - counter: WIP Check Timeout
 ***********************************************************************************/
int write_various_register(gsfi_flash_info *flash_info, alt_u32 cmd, alt_u32 num_data_bytes, alt_u32 write_data){
	int cmd_set = 0;
	num_data_bytes = num_data_bytes << GSFI_CMD_SET_NUM_DATA_BYTES_BIT;
	cmd_set = (
			(GSFI_CMD_SET_NUM_DATA_BYTES_MASK & num_data_bytes)
			+(cmd)
	);
	IOWR(flash_info->gsfi_csr_base,GSFI_CMD_SET_REG,cmd_set);
	IOWR(flash_info->gsfi_csr_base,GSFI_CMD_WRITE_DATA0_REG,write_data);
	IOWR(flash_info->gsfi_csr_base,GSFI_CMD_CONTROL_REG,GSFI_CMD_CONTROL_START_MASK);
	return poll_for_write_in_progress(flash_info);
}
/***********************************************************************************
 * write_enable
 *
 * write_enable Lɂ܂B
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *
 * Returns:
 * - none
 ***********************************************************************************/
void write_enable(gsfi_flash_info *flash_info){
	write_various_register(flash_info, WRITE_ENABLE_CMD, WRITE_ENABLE_BYTES, 0x0);
}
/***********************************************************************************
 * write_status
 *
 * Status Register  Data  Wrtie ܂B
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 * - write_data: Write Data
 *
 * Returns:
 * - 0(0): Not Error
 * - counter: WIP Check Timeout
 ***********************************************************************************/
int write_status_reg(gsfi_flash_info *flash_info, alt_u32 write_data){
	write_enable(flash_info);
	return write_various_register(flash_info, WRITE_STATUS_CMD, WRITE_STATUS_BYTES, write_data);
}

/***********************************************************************************
 * enable_4byte_for_mx25L
 *
 * Macronix  ISSI  Flash ROM  4 byte Addressing ̐ݒLɂ܂B
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *
 * Returns:
 * - none
 ***********************************************************************************/
void enable_4byte_for_mx25L(gsfi_flash_info *flash_info){
	write_various_register(flash_info, ENABLE_4BYTE_FOR_MX25L_CMD, ENABLE_4BYTE_FOR_MX25L_BYTES, 0x0);
}
/***********************************************************************************
 * disable_4byte_for_mx25L
 *
 * Macronix  ISSI  Flash ROM  3 byte Addressing ̐ݒLɂ܂B
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *
 * Returns:
 * - none
 ***********************************************************************************/
void disable_4byte_for_mx25L(gsfi_flash_info *flash_info){
	write_various_register(flash_info, DISABLE_4BYTE_FOR_MX25L_CMD, DISABLE_4BYTE_FOR_MX25L_BYTES, 0x0);
}
/***********************************************************************************
 * write_register_for_sector_protect
 *
 * ̊֐́ASector Protect s܂B
 *@sector_lock_id  Flash ROM ̎dlɍĂȂꍇ́AError ɂȂ܂B
 *@ݒAݒeقȂꍇ́AStatus Register \āAError ɂȂ܂B
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *@- sector_lock_id: Block protection bits in Flash ROM ==> Bit4 | Bit3 | Bit2 | Bit1 | Bit0
 *                                                           TB  | BP3  | BP2  | BP1  | BP0
 * Returns:
 * - 0: Not Error
 * - 22(EINVAL): gpĂ Flash ROM ̎dlƁAsector_lock_id  TB BP ̐ݒ肪ĂȂꍇ
 * - 5(EIO): Verify Error
 * - counter: Timeout Error
 ***********************************************************************************/
int write_sector_protect(gsfi_flash_info *flash_info, alt_u32 sector_lock_id){
	int locked_mask = 0x7C;
	int locked_bit = 2;
	int cmd_set = 0;
	int tb_bp_num = 0;
	int ret_code = 0;

	if(flash_info->flash_rom <= 32){
		tb_bp_num = 0xF;
	}else{
		tb_bp_num = 0x1F;
	}
	if(sector_lock_id >=0 && sector_lock_id <= tb_bp_num){
		write_enable(flash_info);
		cmd_set = sector_lock_id << locked_bit;
		ret_code = write_status_reg(flash_info, cmd_set);
	}else{
		return -EINVAL;
	}
	if(((read_status_register(flash_info) & locked_mask) >> locked_bit) != sector_lock_id){
		return EIO;
	}
	return ret_code;
}
/***********************************************************************************
 * poll_for_write_in_progress
 *
 * ̊֐́AWIP Check s܂B
 * WIP Check ATIMEOUT_1US_VALUE ̎Ԉȏ㑱ꍇ́AError ɂȂ܂B
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *
 * Returns:
 * - 0: Not Error
 * - counter: Timeout Error
 ***********************************************************************************/
int poll_for_write_in_progress(gsfi_flash_info *flash_info){
	alt_u32 timeout = TIMEOUT_1US_VALUE;
	alt_u32 counter = 0;
	while((read_status_register(flash_info) & 0x00000001) == 0x00000001){
		usleep(1); /* delay 1us */
#if TIMEOUT_1US_VALUE > 0
		if(timeout <= counter )
		{
			return counter;
		}
		counter++;
#endif
	}
	return 0;
}
/***********************************************************************************
 * validate_write_sector_arguments
 *
 * ̊֐́Aw肵 Offset  Length LdlɈᔽĂȂmF܂B
 *@ Offset  0 傫
 *@ Length  0 ȏォ
 *@ Offset  Flash ROM ̗eʂ𒴂ĂȂ
 *@ w Offset  Length Offset  Sector 𒴂Ȃ
 *
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *@- offset: Read Address you want to read
 *@- length:@ Data Length
 *
 * Returns:
 * - 0: Not Error
 * - -22(EINVAL): w肵ł Flash ROM ̎dl𖞂ĂȂꍇ
 ***********************************************************************************/
int validate_write_sector_arguments(gsfi_flash_info *flash_info, alt_u32 offset, alt_u32 length){
	alt_u32 number_of_sector = offset / flash_info->sector_size ;
	alt_u32 sector_end_address = flash_info->sector_size * (number_of_sector +1 ) - 0x1;
	alt_u32 write_end_address = length + offset - 0x1;
	if(
			(offset < 0)
			|| (offset >= flash_info->gsfi_mem_span)
			|| (length <= 0)
			|| (write_end_address > sector_end_address)
	){
		return -EINVAL;
	}
	return 0;
}
/***********************************************************************************
 * erase_sector
 *
 * ̊֐́Aw肵 Sector Offset  Erase ܂B
 * WIP Check sAErase ܂ Wait ܂B
 *
 * w肵Sector Offset LdlɈᔽĂȂmF܂B
 *@ Sector Offset 0傫
 *@ Sector Offset  Flash ROM ̗eʂ𒴂ĂȂ
 *@ Sector Offset  Sector BASE Address 
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *@- sector_offset: Erase Sector BASE Address
 *
 * Returns:
 * - 0: Not Error
 * - -22(EINVAL): w肵ł́AFlash ROM ̎dl𖞂ĂȂꍇ
 * - counter: WIP Check Timeout
 ***********************************************************************************/
int erase_sector(gsfi_flash_info *flash_info, int sector_offset){
	int cmd_set = 0;
	if((sector_offset < 0) || (sector_offset >= flash_info->gsfi_mem_span) || (sector_offset % flash_info->sector_size))
	{
		return -EINVAL;
	}
	cmd_set = (
			(flash_info->addressing_mode << GSFI_CMD_SET_NUM_ADD_BYTES_BIT)+
			(ERASE_SECTOR_CMD)
	);
	write_enable(flash_info);
	IOWR(flash_info->gsfi_csr_base,GSFI_CMD_SET_REG,cmd_set); //0x000004D8 or 0x000003D8
	IOWR(flash_info->gsfi_csr_base,GSFI_CMD_ADD_REG,sector_offset);
	IOWR(flash_info->gsfi_csr_base,GSFI_CMD_CONTROL_REG,GSFI_CMD_CONTROL_START_MASK);
	return poll_for_write_in_progress(flash_info);
}
/***********************************************************************************
 * read_memory
 *
 * ̊֐́Aw肵AhXɂf[^w肵 Read obt@[Ɋi[ ܂B
 *
 * w肵 Offset  length LdlɈᔽĂȂmF܂B
 *@ offset  0 傫
 *@ length  0 ȏォ
 *@ offset Flash ROM ̗eʂ𒴂ĂȂ
 *
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *@- *dest_addr: Read Buffer Pointer
 *@- offset: Read Address you want to read
 *@- length:@ Data Length
 *
 * Returns:
 * - 0: Not Error
 * - -22(EINVAL): w肵ł́AFlash ROM ̎dl𖞂ĂȂꍇ
 ***********************************************************************************/
int read_memory(gsfi_flash_info *flash_info, void *dest_addr, alt_u32 offset, alt_u32 length){
	int ret_code = 0;
	if((offset < 0) || (offset >= flash_info->gsfi_mem_span) || (length <= 0)){
		ret_code = -EINVAL;
	}else{
		if(ALT_CPU_DCACHE_SIZE != 0){
			alt_dcache_flush_no_writeback((alt_u8*)flash_info->gsfi_mem_base + offset, length);
		}
		memcpy(dest_addr, (alt_u8*)flash_info->gsfi_mem_base + offset, length);
	}
	return ret_code;
}
/***********************************************************************************
 * write_memory
 *
 * ̊֐́Aw肵 Write Buffer ̃f[^w肵 Offset  Write ܂B
 * validate_write_sector_arguments() sāAFlash ROM ̎dlɂ@Write ɂȂ邩mF܂B
 * Write Data  Word ̃ACgĂȂꍇA Write ܂B
 *
 * Arguments:
 *@- *flash_info: Pointer to Flash Device Structure.
 *@- *data_addr: Write Buffer Pointer
 *@- offset: Read Address you want to read
 *@- length:@ Data Length
 *
 * Returns:
 * - 0: Not Error
 * - -22(EINVAL): Error validate_write_sector_arguments()
 * - 5(EIO): Padding Error
 ***********************************************************************************/
int write_memory(gsfi_flash_info *flash_info, const void *data_addr, alt_u32 write_offset, alt_u32 length){
	int ret_code = 0;
	alt_u32 buffer_offset = 0;
	alt_u32 remaining_length = length;

	ret_code = validate_write_sector_arguments(flash_info, write_offset, length);
	if(ret_code == 0){
		/*
		 * Do writes one 32-bit word at a time.
		 * We need to make sure that we pad the first few bytes so they're word aligned if they are
		 * not already.
		 */
		while (remaining_length > 0)
		{
			alt_u32 word_to_write = 0xFFFFFFFF; // initialize word to write to blank word
			alt_u32 padding = 0; // bytes to pad the next word that is written
			alt_u32 bytes_to_copy = sizeof(alt_u32); // number of bytes from source to copy

			// we need to make sure the write is word aligned
			// this should only be true at most 1 time
			if (0 != (write_offset & (sizeof(alt_u32) - 1)))
			{
				//  data is not word aligned
				//  calculate padding bytes need to add before start of a data offset
				padding = write_offset & (sizeof(alt_u32) - 1);
				// update variables to account for padding being added
				bytes_to_copy -= padding;
				if(bytes_to_copy > remaining_length)
				{
					bytes_to_copy = remaining_length;
				}
				write_offset = write_offset - padding;
				if(0 != (write_offset & (sizeof(alt_u32) - 1)))
				{
					return EIO;
				}
			}
			else
			{
				if(bytes_to_copy > remaining_length)
				{
					bytes_to_copy = remaining_length;
				}
			}
			// prepare the word to be written
			memcpy((((void*)&word_to_write)) + padding, ((void*)data_addr) + buffer_offset, bytes_to_copy);
			buffer_offset += bytes_to_copy;
			remaining_length -= bytes_to_copy;
			// write to flash 32 bits at a time
			IOWR_32DIRECT(flash_info->gsfi_mem_base, write_offset, word_to_write);
			// update current offset
			write_offset = write_offset + sizeof(alt_u32);
		}
	}
	return ret_code;
}
