﻿/********************************************************************************//*!
 * @file  util_time_measurement.c 
 * @brief  Time measurement programs.
 * 
 * @details  Provides a mechanism to measure the time interval specified, using 
 *           the global timer with clock Manager.
 * 
 * @note  nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 * @date 2014/02/26  ---  #13246  計測結果表示用関数の仕様を一部変更.
 * @date 2015/03/02  ---  #18042  計測結果表示用のメッセージを特定のサンプル用に変更.
 * @date 2015/06/25  ---  #20263  v15.0対応/文字コード変更(UTF-8) 他. 
 * @date 2016/07/26  ---  -----  v16.0での確認およびHelio以外の評価ボードへの対応を実施.
 * 
 * @attention 
 * Copyright (C) 2013-2019 MACNICA,Inc. All Rights Reserved.\n
 *   This software is licensed "AS IS". 
 *   Please perform use of this software by a user's own responsibility and expense.
 *   It cannot guarantee in the maker side about the damage which occurred by the ab-
 *   ility not to use or use this software, and all damage that occurred secondarily.
 **//*******************************************************************************/

/***********************************************************************************
 *  includes 
 ***********************************************************************************/
#include <stdio.h>
#include "hwlib.h"
#include "alt_globaltmr.h"
#include "alt_clock_manager.h"
#include "util_time_measurement.h"
#include "alt_printf.h"

/***********************************************************************************
 *  externs 
 ***********************************************************************************/

/***********************************************************************************
 *  proto types 
 ***********************************************************************************/
static void util_time_globaltimer_config(void);
static void util_time_print_freq_and_devider(ALT_CLK_t pll, int divion);
static void util_time_print_clock_information(void);
static void util_time_record_init(UtilTimeBuf_st* pMeBuf);
static void util_time_calc_diffval(UtilTimeBuf_st* pMeBuf, int64_t* i64count);
static void util_time_print_result_by_c(uint32_t index);
static void util_time_print_result_by_s(uint32_t index);

/***********************************************************************************
 *  definitions (define, enum, typedef, etc..) 
 ***********************************************************************************/

/***********************************************************************************
 *  variables 
 ***********************************************************************************/
/*! Measurement information records (Results recording buffer). */
UtilTimeBuf_st	stUtilTimeBuf[UTIL_TIME_MAX_TRACE];

/*! Measurement Point Messages. */
const char* MeasPointMessage[UTIL_TIME_MAX_TRACE] =
{
		"",	/* [0] */
		"",	/* [1] */
		"",	/* [2] */
		"",	/* [3] */
		"",	/* [4] */
		"",	/* [5] */
		"",	/* [6] */
		"",	/* [7] */
		"",	/* [8] */
		"",	/* [9] */
		"",	/* [10] */
		"",	/* [11] */
		"",	/* [12] */
		"",	/* [13] */
		"",	/* [14] */
		"",	/* [15] */
		"",	/* [16] */
		"",	/* [17] */
		"",	/* [18] */
		"",	/* [19] */
		"",	/* [20] */
		"",	/* [21] */
		"",	/* [22] */
		"",	/* [23] */
		"",	/* [24] */
		"",	/* [25] */
		"",	/* [26] */
		"",	/* [27] */
		"",	/* [28] */
		"",	/* [29] */
		"",	/* [30] */
		""	/* [31] */
};

/***********************************************************************************
 *  functions 
 ***********************************************************************************/

/********************************************************************************//*!
 * @fn  static void util_time_globaltimer_config(void) 
 * @brief  Global timer configuration.
 * 
 * @details  The global timer configuration by using the HWLib APIs. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void util_time_globaltimer_config(void)
{
	ALT_STATUS_CODE	result_code;
	
	// Initialize
	/***********************************************************************/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_init(void);	*/
	/***********************************************************************/
	result_code = alt_globaltmr_init();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_init() result=%d\n", (int)result_code);
	}

	// Parameter setting for the global timer.
	/***********************************************************************/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_prescaler_set(uint32_t val);	*/
	/*			val ... Valid range is 0-255. Actual clock divisor ratio is this number plus one. */
	/***********************************************************************/
	result_code = alt_globaltmr_prescaler_set(UTIL_TIME_GLOBALTIMER_PRESCALE);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_prescaler_set() result=%d\n", (int)result_code);
	}

	// Function ON/OFF Setting: Timer=ON, CompMode=OFF, AutoInc=OFF, Interrupt=OFF
	/***********************************************************************/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_comp_mode_stop(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_autoinc_mode_stop(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_int_disable(void);	*/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_start(void);	*/
	/***********************************************************************/
	result_code = alt_globaltmr_comp_mode_stop();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_comp_mode_stop() result=%d\n", (int)result_code);
	}
	result_code = alt_globaltmr_autoinc_mode_stop();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_autoinc_mode_stop() result=%d\n", (int)result_code);
	}
	result_code = alt_globaltmr_int_disable();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_int_disable() result=%d\n", (int)result_code);
	}
	result_code = alt_globaltmr_start();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_start() result=%d\n", (int)result_code);
	}
}

/********************************************************************************//*!
 * @fn  static void util_time_print_freq_and_devider(ALT_CLK_t pll, int divion) 
 * @brief  Display the clock Manager settings (Specified CLK). 
 * 
 * @details  Displays the specified CLK output clock frequency, and Divider settings. 
 * @param[in] pll  CLK for information display 
 * @param[in] divion  Divider settings display ON(!0)/OFF(0) 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void util_time_print_freq_and_devider(ALT_CLK_t pll, int divion)
{
	ALT_STATUS_CODE	result_code;
	alt_freq_t freq = 0;

	/* Usage: ALT_STATUS_CODE alt_clk_freq_get(ALT_CLK_t clk, alt_freq_t* freq);	*/
	/* Usage: ALT_STATUS_CODE alt_clk_divider_get(ALT_CLK_t clk, uint32_t* div);	*/
	result_code = alt_clk_freq_get(pll, &freq);
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_clk_freq_get() result=%d\n", (int)result_code);
	}
#ifdef soc_cv_av
	if(divion)
	{
		uint32_t ui32div = 0;
		result_code = alt_clk_divider_get(pll, &ui32div);
		if(result_code != ALT_E_SUCCESS){
			printf("ERROR!!: alt_clk_divider_get() result=%d\n", (int)result_code);
		}
		printf("Frequency=%10d (Hz) [Divider=%d]\n", (int)freq, (int)ui32div);
	} else
#endif /* soc_cv_av */
	{
		printf("Frequency=%10d (Hz)\n", (int)freq);
	}

	return;
}

/********************************************************************************//*!
 * @fn  static void util_time_print_clock_information(void) 
 * @brief  Display the clock Manager settings (ALL). 
 * 
 * @details  Displays three PLLs status, and several clocks output frequency and 
 *           Divider settings. 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void util_time_print_clock_information(void)
#ifdef soc_cv_av
{
	printf("==== PLL Lock Status Information =====\n");
	printf(" * Main PLL Lock       : %d\n", (int)alt_clk_pll_is_locked(ALT_CLK_MAIN_PLL));
	printf(" * Peripheral PLL Lock : %d\n", (int)alt_clk_pll_is_locked(ALT_CLK_PERIPHERAL_PLL));
	printf(" * SDRAM PLL Lock      : %d\n", (int)alt_clk_pll_is_locked(ALT_CLK_SDRAM_PLL));
	printf("Clock Manager Interrupt Status=0x%08X\n", (int)alt_clk_lock_status_get());
	printf("\n");
	
	printf("==== Input Clock Frequency Value =====\n");
	printf("ALT_CLK_IN_PIN_OSC1(%d):\t", (int)ALT_CLK_IN_PIN_OSC1);
	util_time_print_freq_and_devider(ALT_CLK_IN_PIN_OSC1, 0);
	printf("ALT_CLK_IN_PIN_OSC2(%d):\t", (int)ALT_CLK_IN_PIN_OSC2);
	util_time_print_freq_and_devider(ALT_CLK_IN_PIN_OSC2, 0);
	printf("ALT_CLK_F2H_PERIPH_REF(%d):\t", (int)ALT_CLK_F2H_PERIPH_REF);
	util_time_print_freq_and_devider(ALT_CLK_F2H_PERIPH_REF, 0);
	printf("ALT_CLK_F2H_SDRAM_REF(%d):\t", (int)ALT_CLK_F2H_SDRAM_REF);
	util_time_print_freq_and_devider(ALT_CLK_F2H_SDRAM_REF, 0);
	printf("\n");
	printf("==== Output Clock Frequency and Divider Value =====\n");
	printf("ALT_CLK_MPU(%d):        \t", (int)ALT_CLK_MPU);
	util_time_print_freq_and_devider(ALT_CLK_MPU, 1);
	printf("ALT_CLK_MPU_L2_RAM(%d): \t", (int)ALT_CLK_MPU_L2_RAM);
	util_time_print_freq_and_devider(ALT_CLK_MPU_L2_RAM, 1);
	printf("ALT_CLK_MPU_PERIPH(%d): \t", (int)ALT_CLK_MPU_PERIPH);
	util_time_print_freq_and_devider(ALT_CLK_MPU_PERIPH, 1);
	printf("ALT_CLK_L4_MAIN(%d):    \t", (int)ALT_CLK_L4_MAIN);
	util_time_print_freq_and_devider(ALT_CLK_L4_MAIN, 1);
	printf("ALT_CLK_L4_MP(%d):      \t", (int)ALT_CLK_L4_MP);
	util_time_print_freq_and_devider(ALT_CLK_L4_MP, 1);
	printf("ALT_CLK_L4_SP(%d):      \t", (int)ALT_CLK_L4_SP);
	util_time_print_freq_and_devider(ALT_CLK_L4_SP, 1);
	printf("\n");
	printf("ALT_CLK_QSPI(%d):       \t", (int)ALT_CLK_QSPI);
	util_time_print_freq_and_devider(ALT_CLK_QSPI, 0);
	printf("ALT_CLK_NAND_X(%d):     \t", (int)ALT_CLK_NAND_X);
	util_time_print_freq_and_devider(ALT_CLK_NAND_X, 0);
	printf("ALT_CLK_NAND_X(%d):     \t", (int)ALT_CLK_NAND);
	util_time_print_freq_and_devider(ALT_CLK_NAND, 0);
	printf("ALT_CLK_SDMMC(%d):      \t", (int)ALT_CLK_SDMMC);
	util_time_print_freq_and_devider(ALT_CLK_SDMMC, 0);
	printf("\n");
	
	return;
}
#else /* soc_cv_av */
{
	printf("==== Input Clock Frequency Value =====\n");
	printf("ALT_CLK_OSC1(%d):\t\t", (int)ALT_CLK_OSC1);
	util_time_print_freq_and_devider(ALT_CLK_OSC1, 0);
	//printf("ALT_CLK_F2S_FREE_CLK(%d):\t", (int)ALT_CLK_F2S_FREE_CLK);
	//util_time_print_freq_and_devider(ALT_CLK_F2S_FREE_CLK, 0);
	//printf("ALT_CLK_INTOSC_HS_DIV2(%d):\t", (int)ALT_CLK_INTOSC_HS_DIV2);
	//util_time_print_freq_and_devider(ALT_CLK_INTOSC_HS_DIV2, 0);
	printf("\n");
	//printf("==== PLL Clock Frequency Value =====\n");
	//printf("ALT_CLK_MAIN_PLL(%d):\t", (int)ALT_CLK_MAIN_PLL);
	//util_time_print_freq_and_devider(ALT_CLK_MAIN_PLL, 0);
	//printf("ALT_CLK_PERIPHERAL_PLL(%d):\t", (int)ALT_CLK_PERIPHERAL_PLL);
	//util_time_print_freq_and_devider(ALT_CLK_PERIPHERAL_PLL, 0);
	//printf("ALT_CLK_HMC_PLL_REF(%d):\t", (int)ALT_CLK_HMC_PLL_REF);
	//util_time_print_freq_and_devider(ALT_CLK_HMC_PLL_REF, 0);
	//printf("\n");
	printf("==== Output Clock Frequency Value =====\n");
	printf("ALT_CLK_MPU(%d):\t\t\t", (int)ALT_CLK_MPU);
	util_time_print_freq_and_devider(ALT_CLK_MPU, 0);
	printf("ALT_CLK_MPU_PERIPH(%d):\t", (int)ALT_CLK_MPU_PERIPH);
	util_time_print_freq_and_devider(ALT_CLK_MPU_PERIPH, 0);
	printf("ALT_CLK_MPU_L2_RAM(%d):\t", (int)ALT_CLK_MPU_L2_RAM);
	util_time_print_freq_and_devider(ALT_CLK_MPU_L2_RAM, 0);
	printf("ALT_CLK_L3_MAIN(%d):\t\t", (int)ALT_CLK_L3_MAIN);
	util_time_print_freq_and_devider(ALT_CLK_L3_MAIN, 0);
	printf("ALT_CLK_L4_SYS_FREE(%d):\t", (int)ALT_CLK_L4_SYS_FREE);
	util_time_print_freq_and_devider(ALT_CLK_L4_SYS_FREE, 0);
	printf("ALT_CLK_L4_MAIN(%d):\t", (int)ALT_CLK_L4_MAIN);
	util_time_print_freq_and_devider(ALT_CLK_L4_MAIN, 0);
	printf("ALT_CLK_L4_MP(%d):\t\t", (int)ALT_CLK_L4_MP);
	util_time_print_freq_and_devider(ALT_CLK_L4_MP, 0);
	printf("ALT_CLK_L4_SP(%d):\t\t", (int)ALT_CLK_L4_SP);
	util_time_print_freq_and_devider(ALT_CLK_L4_SP, 0);
	printf("\n");
	printf("ALT_CLK_CS_TIMER(%d):\t", (int)ALT_CLK_CS_TIMER);
	util_time_print_freq_and_devider(ALT_CLK_CS_TIMER, 0);
	printf("ALT_CLK_CS_AT(%d):\t\t", (int)ALT_CLK_CS_AT);
	util_time_print_freq_and_devider(ALT_CLK_CS_AT, 0);
	printf("ALT_CLK_CS_PDBG(%d):\t", (int)ALT_CLK_CS_PDBG);
	util_time_print_freq_and_devider(ALT_CLK_CS_PDBG, 0);
	printf("ALT_CLK_CS_TRACE(%d):\t", (int)ALT_CLK_CS_TRACE);
	util_time_print_freq_and_devider(ALT_CLK_CS_TRACE, 0);
	printf("\n");
	//printf("ALT_CLK_S2F_USER0(%d):\t", (int)ALT_CLK_S2F_USER0);
	//util_time_print_freq_and_devider(ALT_CLK_S2F_USER0, 0);
	//printf("ALT_CLK_S2F_USER1(%d):\t", (int)ALT_CLK_S2F_USER1);
	//util_time_print_freq_and_devider(ALT_CLK_S2F_USER1, 0);
	//printf("ALT_CLK_EMAC0(%d):\t", (int)ALT_CLK_EMAC0);
	//util_time_print_freq_and_devider(ALT_CLK_EMAC0, 0);
	//printf("ALT_CLK_EMAC1(%d):\t", (int)ALT_CLK_EMAC1);
	//util_time_print_freq_and_devider(ALT_CLK_EMAC1, 0);
	//printf("ALT_CLK_EMAC2(%d):\t", (int)ALT_CLK_EMAC2);
	//util_time_print_freq_and_devider(ALT_CLK_EMAC2, 0);
	//printf("ALT_CLK_EMAC_PTP(%d):\t", (int)ALT_CLK_EMAC_PTP);
	//util_time_print_freq_and_devider(ALT_CLK_EMAC_PTP, 0);
	//printf("ALT_CLK_GPIO(%d):\t", (int)ALT_CLK_GPIO);
	//util_time_print_freq_and_devider(ALT_CLK_GPIO, 0);
	printf("ALT_CLK_SDMMC(%d):\t\t", (int)ALT_CLK_SDMMC);
	util_time_print_freq_and_devider(ALT_CLK_SDMMC, 0);
	printf("ALT_CLK_QSPI(%d):\t\t", (int)ALT_CLK_QSPI);
	util_time_print_freq_and_devider(ALT_CLK_QSPI, 0);
	printf("\n");
	
	return;
}
#endif /* soc_cv_av */

/********************************************************************************//*!
 * @fn  static void util_time_record_init(UtilTimeBuf_st* pMeBuf) 
 * @brief  Initialize a measurement information record. 
 * 
 * @details  Clear the 1 measure information records. And status to the enumUTIL_TIME_STATUS_INIT.
 * @param[in,out] pMeBuf  The target record 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
static void util_time_record_init(UtilTimeBuf_st* pMeBuf)
{
	pMeBuf->status = enumUTIL_TIME_STATUS_INIT;
	pMeBuf->start = 0;
	pMeBuf->end = 0;
}

/********************************************************************************//*!
 * @fn  static void util_time_calc_diffval(uint32_t index) 
 * @brief  Print a measurement result by counters. 
 * 
 * @details  Calculate the following value.
 *           - Global timer counter differential value measurement interval
 * @param[in] pMeBuf  The target record 
 * @retval int64_t  result_value (differential counter value)
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2015/02/18  ---  -----  新規作成(リファクタリングにてカウンタ値の差分値計算を部品化). 
 **//*******************************************************************************/
static void util_time_calc_diffval(UtilTimeBuf_st* pMeBuf, int64_t* i64count)
{
	if(pMeBuf->status != enumUTIL_TIME_STATUS_END){
		*i64count = -1;
	} else {
		
		if(pMeBuf->end > pMeBuf->start){
			*i64count = pMeBuf->end - pMeBuf->start;
		} else {
			*i64count = ~(pMeBuf->start - pMeBuf->end) + 1;
		}
	}
	
	return;
}

/********************************************************************************//*!
 * @fn  static void util_time_print_result_by_c(uint32_t index) 
 * @brief  Print a measurement result by counters. 
 * 
 * @details  Print the following values.
 *           - Global timer counter differential value measurement interval
 *           - Global timer counter value of the measurement at the start
 *           - Global timer counter value at the end of measurement
 * @param[in] index  Index of measurement information record 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 * @date 2015/02/18  ---  -----  リファクタリングにてカウンタ値の差分値計算を部品化. 
 **//*******************************************************************************/
static void util_time_print_result_by_c(uint32_t index)
{
	UtilTimeBuf_st*	pMeBuf= &stUtilTimeBuf[index];
	int64_t i64count;
	
	if(pMeBuf->status != enumUTIL_TIME_STATUS_END){
		printf(" -- Unrecorded!! --\n");
	} else {
		util_time_calc_diffval(pMeBuf, &i64count);
		printf(" - Progress of GlobalTimerCounter: %lld\n", (long long int)i64count);
		printf("   [Start=0x%016llX, End=0x%016llX]\n", (long long int)pMeBuf->start, (long long int)pMeBuf->end);
	}
	
	return;
}

/********************************************************************************//*!
 * @fn  static void util_time_print_result_by_s(uint32_t index) 
 * @brief  Print a measurement result by seconds. 
 * 
 * @details  Print the following values.
 *           - Elapsed seconds of measurement interval. 
 *           - Elapsed nano seconds of measurement interval. 
 * @param[in] index  Index of measurement information record 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 * @date 2015/02/18  ---  -----  リファクタリングにてカウンタ値の差分値計算を部品化. 
 **//*******************************************************************************/
static void util_time_print_result_by_s(uint32_t index)
{
	UtilTimeBuf_st*	pMeBuf = &stUtilTimeBuf[index];
	uint64_t	ui64nsec;
	double		doublesec;
	
	if(pMeBuf->status != enumUTIL_TIME_STATUS_END){
		printf(" -- Unrecorded!! --\n");
	} else {
		
		util_time_calc_diffval(pMeBuf, (int64_t*)&ui64nsec);
		ui64nsec *= UTIL_TIME_NSEC_PER_COUNT;
		doublesec = ui64nsec / 1000;
		doublesec /= 1000;
		doublesec /= 1000;
		
#ifdef	PRINTF_HOST
		printf(" - Elapsed Seconds (nsec): %f (%lld)\n", doublesec, (long long int)ui64nsec);
#else	/* #ifdef PRINTF_UART */
		printf(" - Elapsed Nano Seconds:  %lld nsec\n", (long long int)ui64nsec);
#endif
	}
	
	return;
}

/********************************************************************************//*!
 * @fn  void util_time_init(void) 
 * @brief  Initialize for time measurement programs. 
 * 
 * @details  Initialization processing as following.
 *           - Print clock setting info "util_time_print_clock_information()". 
 *           - Setting global timer for measurement "util_time_globaltimer_config()". 
 *           - Initialize measurement informatin records "util_time_record_init()". 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  When you use the functionality of this file go please call first. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void util_time_init(void)
{
	UtilTimeBuf_st*	pMeBuf;
	int	index;
	
	util_time_print_clock_information();

	util_time_globaltimer_config();
	
	pMeBuf = &stUtilTimeBuf[0];
	for(index=0; index<UTIL_TIME_MAX_TRACE; index++, pMeBuf++){
		util_time_record_init(pMeBuf);
	}
	
	return;
}

/********************************************************************************//*!
 * @fn  void util_time_uninit(void) 
 * @brief  Uninitialize for time measurement programs. 
 * 
 * @details  Uninitialization processing as following.
 *           - Call function of global timer uninit "alt_globaltmr_uninit()". 
 *           - Print all results, and clear all records "util_time_print_result_all_and_clear()". 
 * @param[in,out] void  none 
 * @retval void  none 
 * 
 * @attention  Upon termination of usage of this file, please call at the end. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void util_time_uninit(void)
{
	ALT_STATUS_CODE	result_code;

	// UnInitialize
	/***********************************************************************/
	/* Usage: ALT_STATUS_CODE alt_globaltmr_uninit(void);	*/
	/***********************************************************************/
	result_code = alt_globaltmr_uninit();
	if(result_code != ALT_E_SUCCESS){
		printf("ERROR!!: alt_globaltmr_uninit() result=%d\n", (int)result_code);
	}

	util_time_print_result_all_and_clear(enumUTIL_TIME_TARGET_by_BOTH);

	return;
}

/********************************************************************************//*!
 * @fn  void util_time_record_start_point(uint32_t index) 
 * @brief  Records the start point of measurement. 
 * 
 * @details  Saves the current value of the global timer counter (64-bit). 
 *           And the State of the record sets the measurement starts 
 *           (enumUTIL_TIME_STATUS_START). 
 * @param[in] index  Index of measurement information record 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void util_time_record_start_point(uint32_t index)
{
	UtilTimeBuf_st*	pMeBuf;
	
	if(index >= UTIL_TIME_MAX_TRACE){
		printf("ERROR!!: arg1 will be less than %d.  %s[%d] \n", (int)UTIL_TIME_MAX_TRACE, __FILE__, __LINE__);
		return;
	} else {
		pMeBuf = &stUtilTimeBuf[index];
	}
	
	// Record the current global timer counter value to "start".
	pMeBuf->start = alt_globaltmr_get64();
	pMeBuf->status = enumUTIL_TIME_STATUS_START;
}

/********************************************************************************//*!
 * @fn  void util_time_record_end_point(uint32_t index) 
 * @brief  Records the end point of measurement. 
 * 
 * @details  Saves the current value of the global timer counter (64-bit). 
 *           And the State of the record sets the measurement ends 
 *           (enumUTIL_TIME_STATUS_END). 
 * @param[in] index  Index of measurement information record 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        Prior to calling "util_time_record_start_point()". 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void util_time_record_end_point(uint32_t index)
{
	UtilTimeBuf_st*	pMeBuf;
	
	if(index >= UTIL_TIME_MAX_TRACE){
		printf("ERROR!!: arg1 will be less than %d.  %s[%d] \n", (int)UTIL_TIME_MAX_TRACE, __FILE__, __LINE__);
		return;
	} else {
		pMeBuf = &stUtilTimeBuf[index];
	}
	
	if(pMeBuf->status != enumUTIL_TIME_STATUS_START)
	{
		printf("ERROR!!: Call util_time_record_start_point() first.  %s[%d] \n", __FILE__, __LINE__);
		util_time_record_init(pMeBuf);
		return;
	}
	
	// Record the current global timer counter value to "end".
	pMeBuf->end = alt_globaltmr_get64();
	pMeBuf->status = enumUTIL_TIME_STATUS_END;
}

/********************************************************************************//*!
 * @fn  void util_time_print_result_by_counter(uint32_t index) 
 * @brief  Print a measurement result by counters. (Print headings.)
 * 
 * @details  First of all, to print the labels. Then call to print a measurement 
 *           result processing (util_time_print_result_by_c).
 * @param[in] index  Index of measurement information record 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void util_time_print_result_by_counter(uint32_t index)
{
	if(index >= UTIL_TIME_MAX_TRACE){
		printf("ERROR!!: arg1 will be less than %d.  %s[%d] \n", (int)UTIL_TIME_MAX_TRACE, __FILE__, __LINE__);
		return;
	}

	printf("== Time Mesurement Result by Counter ==\n");
	printf("[TIME#%2d]\n", (int)index);
	util_time_print_result_by_c(index);
	printf("\n");
	
	return;
}

/********************************************************************************//*!
 * @fn  util_time_print_result_by_seconds(uint32_t index) 
 * @brief  Print a measurement result by seconds. (Print headings.)
 * 
 * @details  First of all, to print the labels. Then call to print a measurement 
 *           result processing (util_time_print_result_by_s).
 * @param[in] index  Index of measurement information record 
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 **//*******************************************************************************/
void util_time_print_result_by_seconds(uint32_t index)
{
	if(index >= UTIL_TIME_MAX_TRACE){
		printf("ERROR!!: arg1 will be less than %d.  %s[%d] \n", (int)UTIL_TIME_MAX_TRACE, __FILE__, __LINE__);
		return;
	}

	printf("== Time Mesurement Result by Seconds ==\n");
	printf("[TIME#%2d]\n", (int)index);
	util_time_print_result_by_s(index);
	printf("\n");
	
	return;
}

/********************************************************************************//*!
 * @fn  void util_time_print_result_all(void) 
 * @brief  Print all measurement results. 
 * 
 * @details  Print all measurement results by specified content.
 * @param[in] printby  Specify the print content
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 * @date 2014/02/26  ---  #13246  引数追加（printby）.
 **//*******************************************************************************/
void util_time_print_result_all(UtilTimePrintTarget_et printby)
{
	int	index;
	
	printf("==== Time Mesurement Results (ALL) ====\n");
	for(index=0; index<UTIL_TIME_MAX_TRACE; index++){
		printf("[TIME#%2d] %s\n", (int)index, MeasPointMessage[index]);
		if(printby != enumUTIL_TIME_TARGET_by_SECOND){
			util_time_print_result_by_c(index);
		}
		if(printby != enumUTIL_TIME_TARGET_by_COUNT){
			util_time_print_result_by_s(index);
		}
	}
	printf("\n");
	
	return;
}

/********************************************************************************//*!
 * @fn  void util_time_print_result_partial(int startid, int endid)
 * @brief  Print measurement results of specified part.
 *
 * @details  Print partial measurement results by specified content.
 * @param[in] startid  First print target measurement index
 * @param[in] endid  Last print target measurement index
 * @param[in] printby  Specify the print content
 * @retval void  none
 *
 * @attention  nothing.
 * @pre        nothing.
 * @post       nothing.
 * @note       nothing.
 *
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2014/02/26  ---  #13246  新規作成.
 **//*******************************************************************************/
void util_time_print_result_partial(int startid, int endid, UtilTimePrintTarget_et printby)
{
	int	index;

	printf("==== Time Mesurement Results (%d ~ %d) ====\n", startid, endid);
	for(index=startid; index<=endid; index++){
		printf("[TIME#%2d] %s\n", (int)index, MeasPointMessage[index]);
		if(printby != enumUTIL_TIME_TARGET_by_SECOND){
			util_time_print_result_by_c(index);
		}
		if(printby != enumUTIL_TIME_TARGET_by_COUNT){
			util_time_print_result_by_s(index);
		}
	}

	return;
}

/********************************************************************************//*!
 * @fn  void util_time_print_result_all_and_clear(void) 
 * @brief  Print all results, and clear all records. 
 * 
 * @details  Performs the following processing. 
 *           - Call function of print all results "util_time_print_result_all()".
 *           - Initialization process is called for all records "util_time_record_init()".
 * @param[in] printby  Specify the print content
 * @retval void  none 
 * 
 * @attention  nothing. 
 * @pre        nothing. 
 * @post       nothing. 
 * @note       nothing. 
 * 
 * @date <b> History: \<Date\> \<Rev\> \<Keyword\> \<Details\> </b>
 * @date 2013/11/13  ---  #6242  新規作成. 
 * @date 2014/01/20  ---  -----  体裁修正（コメント整理、関数等のラベル接頭辞を変更). 
 * @date 2014/02/26  ---  #13246  引数追加（printby）.
 **//*******************************************************************************/
void util_time_print_result_all_and_clear(UtilTimePrintTarget_et printby)
{
	UtilTimeBuf_st*	pMeBuf;
	int	index;

	util_time_print_result_all(printby);

	pMeBuf = &stUtilTimeBuf[0];
	for(index=0; index<UTIL_TIME_MAX_TRACE; index++, pMeBuf++){
		util_time_record_init(pMeBuf);
	}

	return;
}

/***********************************************************************************
 * end of file 
 ***********************************************************************************/
